#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""neutron sync util"""

import ast
import copy
import hashlib
import os
import tempfile
import time
import zipfile
from datetime import datetime

import six
from oslo_config import cfg
from oslo_serialization import jsonutils
from six.moves import range
from six.moves import zip

from networking_huawei._i18n import _LI, _LE, _LW
from networking_huawei.common import constants
from networking_huawei.common.exceptions import NeutronSyncError
from networking_huawei.drivers.ac.client.service import ACRestUtils
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import fusion_sphere_alarm as fsa
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.neutron_compatible_util import ac_ml2_api as api
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.drivers.ac.db import dbif
from networking_huawei.drivers.ac.external.ext_if import ACKeyStoneIf
from networking_huawei.drivers.ac.sync import util

try:
    from neutron import context
except ImportError:
    from neutron_lib import context

LOG = ncu.ac_log.getLogger(__name__)


class ACNeutronSyncUtil(object):
    """ neutron sync common util """

    def __init__(self):
        self._start_time = None
        self._data_list = None
        self._create_data_list = None
        self._update_data_list = None
        self._neutron_sync_result = None
        self._tenant_list = None
        self._white_list = None
        self.sync_wait_flag = None
        self.quota_task_list = None

    @classmethod
    def _get_sync_id(cls, sync_param):
        try:
            req = ast.literal_eval(sync_param.get("sync_resources"))
            return req.get('id')
        except Exception as e:
            LOG.info('Get uid failed: %s', str(e))
            return 'NA'

    def _populate_white_list(self):
        """ populate white list """
        with open(ac_const.NEUTRON_SYNC_WHITELIST_FILENAME,
                  'r') as whitelist_file:
            f_size = os.path.getsize(
                ac_const.NEUTRON_SYNC_WHITELIST_FILENAME)
            # if file size is too big, empty the file
            if f_size > ac_const.LIMIT_FILE_SIZE:
                raise NeutronSyncError(reason=('Whitelist file-size is '
                                               'greater than %s' %
                                               ac_const.LIMIT_FILE_SIZE))
            white_list = whitelist_file.read()
            self._white_list = ast.literal_eval(white_list)
            # deal with bgpneighbors through the method of bgp-routes.
            if self._white_list.get("bgpneighbors"):
                if self._white_list.get("bgp-routes"):
                    tmp_bgp_routes_white = self._white_list.get("bgp-routes") + self._white_list.get("bgpneighbors")
                    self._white_list.update({"bgp-routes": list(set(tmp_bgp_routes_white))})
                else:
                    self._white_list.update({"bgp-routes": self._white_list["bgpneighbors"]})
                self._white_list.pop("bgpneighbors")
            LOG.debug('Read whitelist data: %s', self._white_list)

    def _populate_res_list(self):
        """
        Populate the resource_list based on the configured plugin resources.
        """
        resources = copy.deepcopy(self._resource_list)
        for res in resources:
            if res not in ac_const.NW_HW_NEUTRON_SYNC_SUPPORT_RES:
                self._resource_list.pop(res)

        registered_resources = copy.deepcopy(self._resource_list)
        if ncu.get_ops_version() in [ac_const.OPS_K, ac_const.OPS_L]:
            if ac_const.NW_HW_EP_GROUPS in registered_resources:
                self._resource_list.pop(ac_const.NW_HW_EP_GROUPS)
        LOG.info(_LI('Neutron sync, support resource list: %s'),
                 self._resource_list)

    def _handle_neutron_sync_except(self, admin_context, session, ex):
        error_msg = str(ex)
        fsa.ACPluginAlarm.send_neutron_sync_alarm(error_msg)
        alarm_description = {
            'operation': 'Exception while running neutron sync.',
            'reason': error_msg
        }
        self._rest.send_cloud_alarm(alarm_description, session)
        LOG.exception(_LE("Neutron sync exception: %s"), error_msg)
        # create exception reasons in db while sync error
        uuid = str(self._sync_op)
        if self._sync_op == ac_const.SYNC_OP_COMPARE_DATA:
            result = {
                'Id': uuid,
                'Status': error_msg[:100]
            }
            self.compare_db.create_db_inconsistent(admin_context, result)
        else:
            if self._sync_op in [ac_const.SYNC_OP_SYNC_DATA,
                                 ac_const.SYNC_OP_SYNC_AND_COMPARE_DATA]:
                sync_type = ac_const.INCONSISTENCY_RECOVER
            else:
                sync_type = ac_const.MULTI_INSTANCE_RECOVER
            result = {
                'Id': uuid,
                'sync_type': sync_type,
                'exec_result': 'failure',
                'error_message': error_msg[:255]
            }
            self.sync_db.create_db_record_list(admin_context, result)

    def _update_sync_start_time(self):
        neutron_sync_record = self._db_if.get_neutron_sync_record()
        sync_start_time = neutron_sync_record.sync_start_time \
            if neutron_sync_record else datetime.utcnow()
        self._start_time = time.mktime(sync_start_time.timetuple())
        # set the static neutron_sync list as entry
        self._neutron_sync_result = {
            'start_time': sync_start_time,
            ac_const.OPER_CREATE: {},
            ac_const.OPER_UPDATE: {},
            ac_const.OPER_DELETE: {}}

    def _get_resource_status(self, res_type, admin_ctx):
        """ get resource status """
        fields = ['id', 'status']
        if res_type == ac_const.NW_HW_FIP:
            fields.append('fixed_ip_address')
            fields.append('floating_ip_address')
        res_list = getattr(self.neutron_formatter, self._resource_list[
            res_type])(admin_ctx, fields=fields)
        status_dict = {res['id']: res['status'] for res in res_list}
        LOG.debug('status dict : %s', status_dict)
        return status_dict

    def _count_neutron_sync_result(self, record_list, sync_result_records,
                                   del_error_list=None):
        """ count neutron sync result """
        LOG.debug('Neutron sync statistics: %s', sync_result_records)
        sync_res_list = []
        record_list.append("Neutron sync resources list: \n")

        for method in [ac_const.OPER_CREATE, ac_const.OPER_UPDATE]:
            sync_res_list.extend(sync_result_records[method].keys())
        res_list = [res for res in ac_const.NW_HW_REPORT_STATUS_RESOURCES
                    if res in sync_res_list]
        LOG.debug('res_list: %s', res_list)

        # Update the neutron sync result based on the current status in
        # neutron DB.
        self._set_neutron_db_status(context.get_admin_context(), res_list,
                                    sync_result_records)
        sync_record_list, total_number, total_success = \
            self.handle_sync_result_records(sync_result_records, del_error_list,
                                            record_list)
        try:
            self._create_db_record_list_all(context.get_admin_context(),
                                            sync_record_list)
        except RuntimeError as ex:
            LOG.error(_LE(
                '[AC] Failed to Create sync results '
                ' in NeutronDB: %s'), str(ex))
        return total_number, total_success

    def handle_sync_result_records(self, sync_result_records,
                                   del_error_list, record_list):
        """ handle sync result records """
        total_number = total_success = 0
        if self._sync_op == ac_const.SYNC_OP_SYNC_SINGLE_INSTANCE:
            sync_type = ac_const.MULTI_INSTANCE_RECOVER
        else:
            sync_type = ac_const.INCONSISTENCY_RECOVER
        sync_record_list = []
        for method in [ac_const.OPER_DELETE, ac_const.OPER_CREATE,
                       ac_const.OPER_UPDATE]:
            for res_type in sync_result_records[method]:
                sync_record_list_tmp, total_number, total_success = \
                    self._handle_sync_result_res_type(
                        sync_result_records, del_error_list, record_list,
                        (method, res_type, total_success, total_number, sync_type))
                sync_record_list.extend(sync_record_list_tmp)

        if not sync_record_list:
            sync_record_list.append({'Id': ac_const.SYNC_NO_DIFFERENCE,
                                     'sync_type': sync_type,
                                     'exec_result': 'success'})
        return sync_record_list, total_number, total_success

    def _handle_sync_result_res_type(self, sync_result_records, del_error_list,
                                     record_list, sync_result_info):
        method, res_type, total_success, total_number, sync_type = sync_result_info

        sync_record_list = []
        for res in sync_result_records[method][res_type]:
            result = res['result']
            # if the record is deleted port, check id if the port
            # is added by deleted router, if it is, check deleted
            # router success or not, if not ,don't record
            if (method == ac_const.OPER_DELETE and
                    res_type == ac_const.NW_HW_PORTS and
                    self._check_related_router_failed(res, del_error_list)):
                continue
            if method == ac_const.OPER_DELETE and self. \
                    _check_delete_error(res['id'], del_error_list):
                result = ac_const.NW_HW_ERROR
            if result == ac_const.NW_HW_SUCCESS:
                total_success += 1
            total_number += 1
            self._util._handle_sync_record_list(
                record_list, sync_record_list, res,
                (res_type, method, result, sync_type))
        return sync_record_list, total_number, total_success

    @classmethod
    def _check_related_router_failed(cls, res, del_error_list):
        """ check whether port is related to router """
        # process port, if the port is added by deleted router
        # when delete router failed, don't record delete port success
        for obj in del_error_list:
            if res.get('related_router') == obj['uuid']:
                return True
        return False

    def _create_db_record_list_all(self, admin_ctx, sync_record_list):
        """ create db record"""
        LOG.info(_LI('[AC] Begin to Create record list in Neutron '
                     'DB for %s.'), sync_record_list)
        for record in sync_record_list:
            try:
                self.sync_db.create_db_record_list(admin_ctx, record)
            except Exception as ex:
                LOG.error(_LE('[AC] Save sync result %s '
                              'to db failed with reason: %s'), record, str(ex))

    def _set_neutron_db_status(self, admin_ctx, res_list, sync_result_records):
        """ set neutron db status """
        for res_type in res_list:
            status_dict = self._get_resource_status(res_type, admin_ctx)
            for method in [ac_const.OPER_CREATE, ac_const.OPER_UPDATE]:
                if res_type not in sync_result_records[method]:
                    continue
                for res in sync_result_records[method][res_type]:
                    self._set_sync_res_error(res, status_dict)

    @classmethod
    def _set_sync_res_error(cls, res, status_dict):
        if res['result'] == ac_const.NW_HW_SUCCESS and \
                status_dict.get(res['id']) in \
                ac_const.AC_REQUEST_ERROR_STATUS:
            LOG.debug('Set neutron sync error log for : %s',
                      res['id'])
            res['result'] = ac_const.NW_HW_ERROR

    @classmethod
    def _check_delete_error(cls, res_id, del_error_list):
        """ check delete error """
        for obj in del_error_list:
            if obj['res'] == ac_const.NW_HW_EXROUTE:
                exroute_id = "{'router_id':%s, 'nexthop':%s, " \
                             "'destination':%s}" % \
                             (obj['uuid'], obj['nexthop'],
                              obj['destination'])
                if exroute_id == res_id:
                    return True
            elif res_id == obj['uuid']:
                return True
        return False

    def _set_bind_port_level(self, network_id, host_id, dynamic_segment,
                             port_id):
        """ set bind port level """
        network_type = ac_const.NETWORK_TYPE_PREFFIX
        network_type += dynamic_segment[api.NETWORK_TYPE]
        next_segment = {
            'network-type': network_type,
            'segmentation-id': dynamic_segment[api.SEGMENTATION_ID]
        }

        bind_port_info = {
            'input': {
                'host-id': host_id,
                'network-id': network_id,
                'next-segment': next_segment,
                'port-id': port_id
            }
        }
        self._ac_rest_service.send_bind_port_request(bind_port_info)

    @classmethod
    def _bindport_filter(cls, port_info, network_info):
        """ bind port filter """
        # in hybrid scenario, only sync bindport for sriov vm port in vxlan net.
        # in network scenario, sync bindport.
        # not sync bindport for subport.
        if port_info['device_owner'] == 'trunk:subport':
            return True
        vhost_user = cfg.CONF.huawei_ac_config.vhost_user
        if vhost_user and port_info['device_owner'].startswith('compute:'):
            if port_info['binding:vnic_type'] == 'direct' and \
                    network_info['provider:network_type'] == 'vxlan':
                return False
            return True
        return False

    @classmethod
    def _process_resource_type_hw_ports(cls, res_list, total_record):
        """ process_resource_type_hw_ports """
        LOG.info("Change the sequence of port start: %s", res_list)
        parentport_res_list = []
        dhcpport_res_list = []
        for record_num in range(total_record):
            record = res_list[record_num]
            trunk_details = record.get('trunk_parent_port', None)
            if trunk_details:
                parentport_res_list.append(record)
            elif record.get('dhcp_port', None):
                dhcpport_res_list.append(record)
            else:
                parentport_res_list.insert(0, record)
        res_list = dhcpport_res_list + parentport_res_list
        LOG.info("Change the sequence of port end: %s", res_list)
        return res_list

    # Read the uuid one by one from the input list and update to the hash.
    # And once all uuid are added generate the final hash and return back to
    # the caller. SHA256(32 bytes) algorithm is used to generate the hash as it
    # was faster. We don't care about the security of hash here as here we use
    # hash to just make a short for of data to compare.
    @classmethod
    def _calculate_hash(cls, uuid_record):
        """calculate hash"""
        if not uuid_record:
            return ''
        res_sha256 = hashlib.sha256()
        for uuid in sorted(uuid_record):
            res_sha256.update(six.b(uuid))
        return res_sha256.hexdigest()

    @classmethod
    def _compare_hash_pre_process(cls, ac_hash, neutron_hash):
        """compare hash pre process"""
        for key in list(neutron_hash.keys()):
            if neutron_hash[key] == '':
                neutron_hash.pop(key)

        for key in list(ac_hash.keys()):
            if ac_hash[key] == '':
                ac_hash.pop(key)

    def _get_router_from_neutron_db(self, admin_ctx, res_type, filters=None):
        """process router port"""
        routers = getattr(self.neutron_formatter,
                          self._resource_list[res_type])(admin_ctx,
                                                         filters=filters)
        LOG.info(_LI("Process router ipv6 port of public service network : %s"
                     " and filters is :%s"), res_type, filters)
        if ncu.IS_FSP and self._sync_op != ac_const.SYNC_OP_COMPARE_DATA:
            for router in routers:
                LOG.info(_LI("Process router is : %s"), router)
                gateway_info = router.get('external_gateway_info')
                if gateway_info:
                    LOG.info(_LI("The router gateway info : %s"), gateway_info)
                    gateway_network_id = gateway_info.get('network_id')
                    tenant_id = router.get('tenant_id', None)
                    l3_plugin = ncu.get_service_plugin()['L3_ROUTER_NAT']
                    subnet_id = l3_plugin.check_gateway_and_public_network(
                        admin_ctx, gateway_network_id)
                    LOG.info(_LI("Get first subnet of public network : %s"),
                             subnet_id)
                    if subnet_id:
                        l3_plugin.add_router_interface_public_network(
                            admin_ctx, tenant_id, router['id'], subnet_id)
                    else:
                        LOG.info(_LI("No need to create public "
                                     "network ipv6 port."))
        return routers

    @classmethod
    def _process_one_subport(cls, host_id, session, subport_id):
        """process one subport"""
        rec_subport = ncu.get_port_binding(session, subport_id)
        if not rec_subport:
            return
        subport_host_id = rec_subport.get('host', None)
        if not subport_host_id or subport_host_id != host_id:
            session = dbif.ACdbInterface().get_session('write')
            with session.begin(subtransactions=True):
                LOG.info("update_subport_in_db start")
                port = ncu.get_port_binding(session, subport_id)
                if port:
                    port.host = host_id
                    session.merge(port)
                    session.flush()
                LOG.info("update_subport_in_db end")

    @classmethod
    def _get_module_name_and_container_name(cls, res_type):
        if res_type == ac_const.NW_HW_PF:
            res_type = ac_const.NW_HW_DNAT
        if res_type == "consist":
            module_name = "huawei-ac-neutron-consist"
            container_name = "huawei-ac-neutron-consist:output"
        else:
            container_name = ac_const.NEUTRON_RESOURCES_YANG.get(res_type)['container_name']
            module_name = ac_const.NEUTRON_RESOURCES_YANG.get(res_type)['module_name']
        if res_type == ac_const.NW_HW_DNAT:
            container_name = ac_const.NW_HW_DNATS
        return module_name, container_name

    def _process_file_list(self, file_list, container_name, res_type):
        resource_data = {}
        for file_name in file_list:
            download_path = self._download_resource_data_file(
                file_name, res_type)
            zip_file = zipfile.ZipFile(download_path)
            try:
                zip_data = self._get_zip_data(
                    container_name, res_type, zip_file)
            finally:
                os.remove(download_path)
            if not resource_data:
                resource_data = zip_data
                continue
            if res_type == "consist":
                if zip_data[container_name].get('port-result'):
                    resource_data[container_name]['port-result'].extend(
                        zip_data[container_name].get('port-result'))
                if zip_data[container_name].get('consist-result'):
                    resource_data[container_name]['consist-result'].extend(
                        zip_data[container_name].get('consist-result'))
            else:
                resource_data[container_name].extend(
                    zip_data[container_name])

        return resource_data

    @classmethod
    def _get_zip_data(cls, container_name, res_type, zip_file):
        """get zip data"""
        zip_data = {container_name: []}
        zip_data_consist = {container_name: {'consist-result': [],
                                             'port-result': []}}
        for z_file in zip_file.namelist():
            try:
                file_content = zip_file.read(z_file)
            except KeyError:
                raise NeutronSyncError(
                    reason='Did not find %s in zip file.' % z_file)

            data = jsonutils.loads(file_content)
            if not data or container_name not in data:
                LOG.error(_LE('Invalid data: %s'), data)
                continue
            LOG.debug("_get_resource_data_from_ac res_type %s, data %s",
                      res_type, data)
            if res_type == "consist":
                if data[container_name].get('port-result'):
                    zip_data_consist.get(container_name)['port-result']. \
                        extend(data[container_name].get('port-result'))
                if data[container_name].get('consist-result'):
                    zip_data_consist.get(container_name)['consist-result']. \
                        extend(data[container_name].get('consist-result'))
            elif res_type == ac_const.NW_HW_PORTS:
                for port_network in data[container_name]:
                    zip_data.get(container_name).extend(port_network['network-ports'])
            else:
                zip_data.get(container_name).extend(data[container_name])
        if res_type == "consist":
            return zip_data_consist
        return zip_data

    def _gather_resource_data(self, res_type, module_name):
        """gather resource data"""
        path = util.ACUtil.form_neutron_sync_url(ac_const.NW_HW_OPER_URL,
                                                 'gather-data')
        message_body = self._get_gather_message_body(res_type, module_name)
        response = self._send_gather_request_to_ac(
            (message_body, constants.REST_POST, path, res_type),
            ac_const.CONSIST_QUERY_GATHER_DATA,
            header_data=ac_const.NW_HW_NEUTRON_SYNC_REST_HEADER)

        res_json = self._get_response_data(response)
        try:
            return res_json['huawei-ac-netnsync:output']['uuid']
        except KeyError:
            LOG.error(
                _LE("Neutron sync by neutron server %s failed, "
                    "controller sent invalid data for %s gather data."),
                self.neutron_name, res_type)
            raise NeutronSyncError(
                reason='Invalid data in controller response')

    def _get_gather_message_body(self, res_type, module_name):
        if res_type == "consist":
            consistency_cond = ''
            for rec_type in self._resource_list.keys():
                if rec_type == ac_const.NW_HW_CMCC_QOS_POLICY:
                    consistency_cond = consistency_cond + \
                                       ac_const.NW_HW_QOS_POLICY + ","
                elif rec_type == ac_const.NW_HW_PF:
                    consistency_cond = consistency_cond + \
                                       ac_const.NW_HW_DNAT + ","
                else:
                    consistency_cond = consistency_cond + rec_type + ","
            consistency_cond = consistency_cond[:-1]
            message_body = ACRestUtils.fix_json(jsonutils.dumps(
                {'input': {
                    'module': [
                        {'name': module_name,
                         'filter': [{'key-name': 'cloudName',
                                     'filter-value':
                                         cfg.CONF.huawei_ac_config.cloud_name},
                                    {'key-name': 'ops-version',
                                     'filter-value': ncu.get_ops_version()},
                                    {'key-name': 'cond',
                                     'filter-value': consistency_cond}]}],
                    'gather-type': 'all'
                }}))
        else:
            message_body = ACRestUtils.fix_json(jsonutils.dumps(
                {'input': {
                    'module': [
                        {'name': module_name,
                         'filter': [{'key-name': 'cloudName',
                                     'filter-value':
                                         cfg.CONF.huawei_ac_config.cloud_name},
                                    {'key-name': 'ops-version',
                                     'filter-value': ncu.get_ops_version()}]}],
                    'gather-type': 'all'
                }}))
        return message_body

    def _gather_resource_status(self, uuid, res_type):
        """gather reosurce status"""
        path = util.ACUtil.form_neutron_sync_url(ac_const.NW_HW_OPER_URL,
                                                 'get-gather-status')
        message_body = \
            ACRestUtils.fix_json(jsonutils.dumps({'input': {'uuid': uuid}}))
        response = self._send_gather_request_to_ac(
            (message_body, constants.REST_POST, path, res_type),
            ac_const.CONSIST_QUERY_GATHER_STATUS,
            header_data=ac_const.NW_HW_NEUTRON_SYNC_REST_HEADER)
        res_json = self._get_response_data(response)
        try:
            if res_json['huawei-ac-netnsync:output']['finish-module-num'] != \
                    res_json['huawei-ac-netnsync:output']['total-module-num']:
                return ac_const.NW_HW_RETRY, []
            res_file_url = \
                res_json['huawei-ac-netnsync:output']['file-url']
            return ac_const.NW_HW_SUCCESS, res_file_url
        except KeyError:
            LOG.error(
                _LE("Neutron sync by neutron server %s failed, "
                    "controller sent invalid data for %s gather status."),
                self.neutron_name, res_type)
            raise NeutronSyncError(
                reason='Invalid data in controller response')

    @classmethod
    def _get_response_data(cls, response):
        if response.content:
            res_json = jsonutils.loads(response.content)
            if not res_json:
                raise NeutronSyncError(
                    reason='No consistency data in controller response.')
            else:
                LOG.info(_LI('Controller response: %s.'), res_json)
                return res_json
        else:
            raise NeutronSyncError(
                reason='No uuid data in controller response.')

    def _download_resource_data_file(self, file_url, res_type):
        """download reosource data fle"""
        LOG.debug('Downloading file %s for resource %s', file_url, res_type)
        message_body = ACRestUtils.fix_json(jsonutils.dumps({}))
        response = self._send_gather_request_to_ac(
            (message_body, constants.REST_GET, file_url, res_type),
            ac_const.CONSIST_QUERY_DOWNLOAD_FILE,
            header_data=ac_const.NW_HW_NEUTRON_SYNC_GET_FILE,
            need_log=False)
        if response.content:
            temp_dir = tempfile.gettempdir()
            temp_path = '%s/%s' % (temp_dir, file_url.split('/')[-1])
            LOG.debug('File will be downloaded to %s.', temp_path)
            flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
            with os.fdopen(os.open(temp_path, flags, 0o640), 'wb') as fb_out:
                for chunk in response.iter_content(chunk_size=1024):
                    if chunk:
                        fb_out.write(chunk)
            LOG.debug('Download file complete.')
            return temp_path
        else:
            raise NeutronSyncError(
                reason='No uuid data in controller response.')

    def _process_whitelist_controller(self, ac_data_list, ac_id_list,
                                      res_container):
        """process enable whitelist"""
        if cfg.CONF.huawei_ac_config.neutron_sync_enable_white_list and \
                self._white_list.get(res_container, list()):
            if self._sync_op != ac_const.SYNC_OP_SYNC_SINGLE_INSTANCE:
                # filterand remove the whilteist uuid's from ac_data_list
                ac_id_list = [each for each in ac_id_list if each not in self._white_list.get(res_container, list())]
                ac_data_list = [rs for rs in ac_data_list if
                                self._not_in(rs['uuid'], self._white_list.get(res_container, list()))]
        return ac_data_list, ac_id_list

    def _not_in(self, res_value, white_lis):
        return res_value not in white_lis

    def _process_whitelist_neutron(self, ne_data_list, neutron_id_list,
                                   res_container):
        """process enable whitelist"""
        if cfg.CONF.huawei_ac_config.neutron_sync_enable_white_list and \
                self._white_list.get(res_container, list()):
            neutron_id_list = [nid for nid in neutron_id_list if
                               self._not_in(nid, self._white_list.get(res_container, list()))]
            ne_data_list = [res for res in ne_data_list if
                            self._not_in(res['uuid'], self._white_list.get(res_container, list()))]
        return ne_data_list, neutron_id_list

    def _update_entry_update_time(self, data, res_type, rec_uuid):
        # update the attribute fields only if the resouece is available
        # in the neutron table.
        if data:
            if res_type == ac_const.NW_HW_PORTS:
                rec = self._db_if.get_port_attribute(session=None,
                                                     res_id=rec_uuid)
            else:
                rec = self._db_if.get_res_attribute(session=None,
                                                    res_id=rec_uuid,
                                                    res_type=res_type)
            if rec:
                data['updated-at'] = rec.update_time. \
                    strftime(ac_const.ISO8601_TIME_FORMAT)
                data['created-at'] = rec.create_time. \
                    strftime(ac_const.ISO8601_TIME_FORMAT)
            else:
                updated_at = ACCommonUtil.get_standard_current_time()
                data['updated-at'] = updated_at
                data['created-at'] = updated_at

    def _reorder_dhcp_ports(self):
        dhcp_ports = []
        others = []
        for data in self._data_list:
            if data.get('device-owner') == ncu.DEVICE_OWNER_DHCP:
                dhcp_ports.append(data)
            else:
                others.append(data)
        self._data_list = others + dhcp_ports

    @classmethod
    def _get_delete_data(cls, entry_info, res_type):
        """get delete data"""
        delete_data = {'uuid': entry_info['uuid'],
                       'res': res_type,
                       'tenant-name': '',
                       'status': ac_const.STAT_WAIT}
        if res_type == ac_const.NW_HW_ROUTERS:
            if 'ports' in entry_info:
                delete_data['ports'] = entry_info['ports']
            else:
                delete_data['ports'] = []
        return delete_data

    @classmethod
    def _neutron_sync_delete_info(cls, delete_data, entry_info, res_type):
        """neutron sync delete info"""
        delete_data['name'] = entry_info.get('name', ac_const.EMPTY_STR)
        delete_data['created_at'] = entry_info.get('created-at',
                                                   ac_const.EMPTY_STR)
        delete_data['updated_at'] = entry_info.get('updated-at',
                                                   ac_const.EMPTY_STR)
        delete_data['tenant-name'] = entry_info.get('tenant-name',
                                                    ac_const.EMPTY_STR)
        if res_type == ac_const.NW_HW_PORTS:
            delete_data['device-owner'] = entry_info.get(
                'device-owner', ac_const.EMPTY_STR)

    @classmethod
    def _neutron_sync_update_info(cls, update_info, entry_info):
        """neutron sync update info"""
        if 'name' in entry_info:
            update_info['name'] = entry_info['name']
        else:
            update_info['name'] = ac_const.EMPTY_STR
        if 'created_at' in entry_info:
            update_info['created_at'] = entry_info['created_at']
        else:
            update_info['created_at'] = ac_const.EMPTY_STR
        if 'updated_at' in entry_info:
            update_info['updated_at'] = entry_info['updated_at']
        else:
            update_info['updated_at'] = ac_const.EMPTY_STR

    def _update_neutron_sync_result(self):
        """ update neutron sync result """
        for method in [ac_const.OPER_CREATE, ac_const.OPER_UPDATE]:
            res_list = []
            if method == ac_const.OPER_CREATE:
                res_list = self._create_data_list
            elif method == ac_const.OPER_UPDATE:
                res_list = self._update_data_list
            for res in res_list:
                data_info = \
                    {'id': res['uuid'],
                     'name': res.get('name', ac_const.EMPTY_STR),
                     'tenant_name': res['tenant-name'],
                     'created_at': res.get('created_at', ac_const.EMPTY_STR),
                     'updated_at': res.get('updated_at', ac_const.EMPTY_STR),
                     'result': ac_const.NW_HW_SKIP_SYNC}
                if res.get('res') not in self._neutron_sync_result.get(method):
                    self._neutron_sync_result.get(method)[res['res']] = []

                self._neutron_sync_result.get(method)[res['res']].append(data_info)

    def _process_wait_for_gw_port(self, operation, rec_list, res_type):
        """process wait for gw port"""
        if res_type == ac_const.NW_HW_PORTS \
                and operation == ac_const.OPER_CREATE:
            while self.sync_wait_flag:
                time.sleep(0.5)
            # when process gateway port, set the flag
            if ncu.is_gw_port(self._data_list[rec_list[0]]):
                self.sync_wait_flag = True

    def _check_speed_limit_last_minute(self):
        """Check the number of task finished in last minute, if it reach the
        threshold, wait 10 seconds.
        """
        while True:
            time_now_sec = int(time.time())
            new_quota_task_list = \
                [time_item for time_item in self.quota_task_list
                 if time_now_sec - time_item < 60]
            self.quota_task_list = new_quota_task_list
            if len(self.quota_task_list) >= ac_const.NEUTRON_SYNC_SPEED_LIMIT:
                LOG.info("[AC]The number of tasks finished has reached %s in "
                         "last 60 seconds, need to wait.",
                         str(len(self.quota_task_list)))
                time.sleep(10)
            else:
                self.quota_task_list.append(time_now_sec)
                break

    def _process_error_res_list(self, rec_list, res_type, operation):
        for rec_index in rec_list:
            with self._lock:
                self._data_list[rec_index]['status'] = ac_const.STAT_ERR
            self._statistic_neutron_sync_request(
                res_type, operation, self._data_list[rec_index],
                ac_const.NW_HW_ERROR)

    @classmethod
    def process_entry_info(cls, entry_info, method, res_type):
        """process entry info"""
        if res_type == ac_const.NW_HW_ROUTERS and \
                method == constants.REST_UPDATE and \
                'interfaces' in entry_info:
            del entry_info['interfaces']

        # if the port is trunk parent port, when send the port
        # operation, need to set host id as none
        if ncu.IS_FSP and \
                method == constants.REST_POST and \
                res_type == ac_const.NW_HW_PORTS:
            if 'trunk-details' in entry_info \
                    and entry_info['trunk-details'] is not None:
                if 'host-id' in entry_info:
                    entry_info['host-id'] = ""
        if (res_type not in [ac_const.NW_HW_SEC_GRP_RULE,
                             ac_const.NW_HW_HEALTH_MONITOR,
                             ac_const.NW_HW_MEMBER]):
            entry_info[ac_const.CLOUD_NAME_PARAM] = \
                cfg.CONF.huawei_ac_config.cloud_name

    @classmethod
    def _check_delete_error_list(cls, before_data, after_data):
        """check delete error list"""
        del_error_list = []
        for del_obj in before_data:
            for obj_after in after_data:
                if not cls._check_resource_changed(del_obj, obj_after):
                    del_error_list.append(del_obj)
        return del_error_list

    @classmethod
    def _check_resource_changed(cls, obj_before, obj_after):
        """check resource changed"""
        if obj_before['res'] == obj_after['res'] == ac_const.NW_HW_EXROUTE:
            if (obj_before['uuid'] == obj_after['uuid'] and
                    obj_before['nexthop'] == obj_after['nexthop'] and
                    obj_before['destination'] == obj_after['destination']):
                return False
        elif obj_before['res'] == ac_const.NW_HW_BINDPORTS:
            return False
        elif obj_before['uuid'] == obj_after['uuid'] and \
                obj_before['res'] == obj_after['res']:
            return False
        return True

    @classmethod
    def get_res_status(cls, res, rest_info):
        """get res status"""

        def _check_rest_info(rest_info):
            return ('admin-state-up' in rest_info) and not rest_info['admin-state-up'] \
                and rest_info.get('device-owner') != 'neutron:LOADBALANCERV2'

        if res['res_type'] == ac_const.NW_HW_PORTS and _check_rest_info(rest_info):
            status = ac_const.NEUTRON_STATUS_DOWN
        else:
            status = ac_const.NEUTRON_STATUS_ACTIVE
        return status

    def _check_and_update_neutron_db(
            self, session, res_type, record, state):
        """check and update neutron db"""
        if state != ac_const.NEUTRON_STATUS_ERROR:
            if res_type in ac_const.NW_HW_REPORT_STATUS_RESOURCES:
                return
            state = ACRestUtils.get_neutron_status(record, res_type)
        if util.ACUtil.check_resource_have_state(res_type):
            self._db_if.update_neutron_db_state(
                session, record['uuid'], res_type, state)

    def _get_tenant_name(self, tenant_id):
        """get tenant name"""
        try:
            tenants_dict = dict(zip(
                [tenant.id for tenant in self._tenant_list],
                [tenant.name for tenant in self._tenant_list]))
            # if tenant is empty, the data is exist but the tenant
            # is deleted, so we set the tenant name as default name
            if tenant_id and tenant_id in tenants_dict:
                tenant_name = "%s(%s)" % (tenants_dict[tenant_id], tenant_id)
            else:
                tenant_name = tenant_id
        except Exception as e:
            LOG.error('[AC] Get tenant name failed, error: %s', str(e))
            tenant_name = ACKeyStoneIf.get_tenant_name_by_id_from_keystone(tenant_id)
            if tenant_name != tenant_id:
                tenant_name = "%s(%s)" % (tenant_name, tenant_id)
        return tenant_name

    def clear_sync_result(self, ctx):
        """clear sync result"""
        if self._sync_op == ac_const.SYNC_OP_COMPARE_DATA:
            self.compare_db.delete_db_compare_results(ctx)
        elif self._sync_op == ac_const.SYNC_OP_SYNC_DATA:
            self.sync_db.delete_db_sync_results(
                ctx, ac_const.INCONSISTENCY_RECOVER)
        elif self._sync_op == ac_const.SYNC_OP_SYNC_SINGLE_INSTANCE:
            self.sync_db.delete_db_sync_results(
                ctx, ac_const.MULTI_INSTANCE_RECOVER)
        elif self._sync_op == ac_const.SYNC_OP_SYNC_AND_COMPARE_DATA:
            self.compare_db.delete_db_compare_results(ctx)
            self.sync_db.delete_db_sync_results(
                ctx, ac_const.INCONSISTENCY_RECOVER)

    def _neutron_sync_update_resource(self, attr_list, ac_data, entry_info, res_type):
        """比较neutron和ac的数据是否一致，只比较update_time

        :param attr_list: neutron的update_time
        :param ac_data: ac_data
        :param entry_info: entry_info提供uuid
        :param res_type: res_type
        """
        neutron_updated_at = attr_list.get(entry_info['uuid'], None)
        ac_updated_at = ac_data['updated-at'] if ac_data else None
        if neutron_updated_at != ac_updated_at:
            self.single_ins_sync_update_resource(entry_info, res_type)

    def _cleanup_non_synced_records(self, session):
        """
        Set the status to error for the records that did not synced to
        controller during neutron_sync.
        :param session:
        :return:
        """
        self._data_list = []
        for rec in self._create_data_list:
            if self._util.check_resource_have_state(rec['res']):
                self._db_if.update_neutron_db_state(
                    session, rec['uuid'], rec['res'],
                    ac_const.NEUTRON_STATUS_ERROR)
        self._create_data_list = []

        for rec in self._update_data_list:
            if self._util.check_resource_have_state(rec['res']):
                self._db_if.update_neutron_db_state(
                    session, rec['uuid'], rec['res'],
                    ac_const.NEUTRON_STATUS_ERROR)
        self._update_data_list = []

    def single_ins_sync_update(self, entry_info, res_type):
        """single instance sync udpate"""
        try:
            if res_type == ac_const.NW_HW_FIREWALL_POLICY:
                self.single_ins_sync_update_firewall_policy(entry_info, res_type)
            else:
                self.single_ins_sync_update_resource(entry_info, res_type)
        except KeyError:
            LOG.warning(_LW('Skip update single instance sync for %s'),
                        entry_info['uuid'])
        LOG.info(_LI("Update data list: %s"), self._update_data_list)

    def single_ins_sync_update_resource(self, entry_info, res_type):
        """single instance sync update resource"""
        update_info = {'res': res_type,
                       'uuid': entry_info['uuid'],
                       'tenant-name': entry_info['tenant-name'],
                       'status': ac_const.STAT_WAIT}

        self._neutron_sync_update_info(update_info, entry_info)
        if res_type == ac_const.NW_HW_PORTS:
            update_info['device_owner'] = entry_info.get('device_owner', ac_const.EMPTY_STR)
        self._update_data_list.append(update_info)

    def single_ins_sync_update_firewall_policy(self, entry_info, res_type):
        """single instance sync update firewall policy"""
        update_info = {
            'res': res_type, 'uuid': entry_info['uuid'],
            'tenant-name': entry_info['tenant-name'],
            'status': ac_const.STAT_WAIT}
        if 'name' in entry_info:
            update_info['name'] = entry_info['name']
        else:
            update_info['name'] = ac_const.EMPTY_STR
            LOG.error("resource has no key name, resource type %s", res_type)
        if 'created_at' in entry_info:
            update_info['created_at'] = entry_info['created_at']
        else:
            update_info['created_at'] = ac_const.EMPTY_STR
            LOG.error("resource has no key created_at, resource type %s",
                      res_type)
        if 'updated_at' in entry_info:
            update_info['updated_at'] = entry_info['updated_at']
        else:
            update_info['updated_at'] = ac_const.EMPTY_STR
            LOG.error("resource has no key updated_at, resource type %s",
                      res_type)
        # add info end
        self._update_data_list.append(update_info)
