#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""worker"""

import threading
from six.moves.queue import Queue
from six.moves import range

from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei._i18n import _LE, _LI

LOG = logging.getLogger(__name__)


class ACSyncWorker(threading.Thread):
    """Thread executing tasks from a given tasks queue"""

    def __init__(self, tasks, thread_name):
        threading.Thread.__init__(self)
        self.tasks = tasks
        self.daemon = True
        self.name = thread_name
        self.start()

    def run(self):
        while True:
            func, args, kargs = self.tasks.get()
            try:
                LOG.debug("Processing function: %s in sync thread.",
                          func.__name__)
                func(self.name, *args, **kargs)
            except Exception:
                LOG.exception(_LE("Exception while processing sync thread"))
            self.tasks.task_done()


class ACSyncThreadPool(object):
    """Maintains thread pools to do the sync job"""

    def __init__(self, thread_count):
        self.task_queue = Queue(thread_count)

    def spawn_threads(self, thread_count):
        """Spawn new threads for the job"""
        LOG.debug("Creating %d threads for sync job.", thread_count)
        for itr_thread in range(thread_count):
            thread_name = 'sync_worker_%d' % itr_thread
            ACSyncWorker(self.task_queue, thread_name)

    def add_task(self, func, *args, **kargs):
        """Add new task in the queue to perform"""
        LOG.debug("Adding a new task to the queue.")
        self.task_queue.put((func, args, kargs))

    def wait_completion(self):
        """Wait for the completion of the tasks assigned in the queue."""
        LOG.debug("Wait for the thread completion started.")
        self.task_queue.join()

    def clear_task(self):
        """clear task"""
        LOG.debug("[AC] Begin to clear %s tasks in the queue",
                  self.task_queue.qsize())
        while True:
            try:
                self.task_queue.get_nowait()
            except Exception as ex:
                LOG.info(_LI("[AC] Get task failed: %s"), ex)
                break
