#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2012 Huawei Technologies Co. Ltd. All rights reserved.
"""encryptor"""

import os
import sys
import base64
import site

from networking_huawei.drivers.ac.common.constants import \
    ACCOUNT_SECURE_KEY, KEYSTONE_AUTH_SECURE_KEY, WEBSOCKET_SECURE_KEY, CFG_PATH
from networking_huawei.drivers.ac.cryptor.encrypt import aes_encrypt
from networking_huawei.drivers.ac.cryptor.cryptor import AESCryptor
from networking_huawei.drivers.ac.cryptor.pbkdf2 import enable_pbkdf2
from networking_huawei.drivers.ac.encode_convert import convert_to_bytes

PASSWORD_KEY = "pw_key"
PASSWORD_IV = "pw_iv"
KEY_IV = "key_iv"
FACTOR = "factor"

AES_ENCRYPT_LENGTH = 16
MAX_RETRY_TIMES = 3
SALT_LENGTH = 16


def get_dependency_path(data_type=ACCOUNT_SECURE_KEY):
    """get dependency path"""
    dependency_path = "networking_huawei/drivers/ac/cryptor" \
                      "/decrypt_dependency.txt"
    if data_type == KEYSTONE_AUTH_SECURE_KEY:
        dependency_path = "networking_huawei/drivers/ac/"\
                  "cryptor/decrypt_dependency_keystone.txt"
    if data_type == WEBSOCKET_SECURE_KEY:
        dependency_path = "networking_huawei/drivers/ac/"\
                  "cryptor/decrypt_dependency_websocket.txt"
    for site_path in site.getsitepackages():
        absolute_path = os.path.join(site_path, dependency_path)
        if os.path.isfile(absolute_path):
            return absolute_path
    raise IOError


def get_cfg_path():
    """get config path"""
    for site_path in site.getsitepackages():
        absolute_path = os.path.join(site_path, CFG_PATH)
        if os.path.isfile(absolute_path):
            return absolute_path
    raise IOError


def get_salt_value(factor):
    """get salt value"""
    if factor:
        missing_padding = 4 - len(factor) % 4
        if missing_padding:
            factor += '=' * missing_padding
        return base64.b64decode(factor)
    salt = get_secure_random(size=SALT_LENGTH)
    salt_path = get_cfg_path()
    try:
        flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
        with os.fdopen(os.open(salt_path, flags, 0o400), 'w') as file_point:
            file_point.write(base64.b64encode(salt).decode())
    except IOError:
        print("Failed to save salt value to %s" % salt_path)
    return salt


def get_secure_random(size=AES_ENCRYPT_LENGTH, retry=MAX_RETRY_TIMES):
    """get secure random"""
    for _ in range(retry):
        try:
            with open("/dev/random", "rb") as file_point:
                return file_point.read(size)
        except IOError:
            print("Failed to get secure random")
    raise Exception("Can not get secure random")


def encrypt_password(text, data_type=ACCOUNT_SECURE_KEY):
    """encrypt password"""
    aes_cryptor = AESCryptor(None, None, None)

    # get root key
    key = aes_cryptor._get_init_vector()
    if data_type == WEBSOCKET_SECURE_KEY:
        factor = aes_cryptor.get_websocket_key_factor()
    else:
        factor = aes_cryptor._get_cfg_factor()
    salt = get_salt_value(factor)
    root_key = enable_pbkdf2(base64.b64decode(key), salt)
    # generate work key
    work_key = get_secure_random()

    # encrypt the work key
    work_key_iv = get_secure_random()

    try:
        cipher_work_key = aes_encrypt(work_key, root_key,
                                      work_key_iv)
    except Exception:
        print("Failed to encrypt the work key")
        return None

    # save the work key in config file
    # after it is encrypted
    dependency_file = get_dependency_path(data_type)
    try:
        flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
        with os.fdopen(os.open(dependency_file, flags, 0o400), 'w') as f_out:
            if data_type == WEBSOCKET_SECURE_KEY:
                factor_record = "%s%s%s%s" % (FACTOR, "=", factor, "\n")
                f_out.write(factor_record)
            f_out.write("%s%s%s%s" % (
                PASSWORD_KEY, "=", base64.b64encode(cipher_work_key).decode(), "\n"))

            # the iv is used to encrypt password
            pw_iv = get_secure_random()
            # save the iv in config file
            f_out.write("%s%s%s%s" % (
                PASSWORD_IV, "=", base64.b64encode(pw_iv).decode(), "\n"))

            # save the iv which is used to encrypt th work key
            f_out.write("%s%s%s%s" % (
                KEY_IV, "=", base64.b64encode(work_key_iv).decode(), "\n"))
            return aes_encrypt(text, work_key, pw_iv)
    except IOError:
        print("Failed to save work key to %s" % dependency_file)
        return None


if __name__ == "__main__":
    if len(sys.argv) == 3 and sys.argv[1] and sys.argv[2] in ['1', '2', '3']:
        config_value = encrypt_password(sys.argv[1], int(sys.argv[2]))
        print(base64.b64encode(convert_to_bytes(config_value)).decode())
