#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""installation check"""

from __future__ import print_function
import os
import re
import sys
import time
import logging
import subprocess
import six
from six.moves import range
from configobj import ConfigObj
import requests.packages.urllib3
from oslo_serialization import jsonutils
import eventlet

from networking_huawei.drivers.ac.client.https_adapter import HWHTTPSAdapter

LOG = logging.getLogger(__name__)
FORMATTER = logging.Formatter('[%(levelname)s] %(message)s')
CONSOLE_HANDLER = logging.StreamHandler(sys.stdout)
CONSOLE_HANDLER.setFormatter(FORMATTER)
LOG.addHandler(CONSOLE_HANDLER)
LOG.setLevel(logging.INFO)

requests.packages.urllib3.disable_warnings()
requests = eventlet.import_patched('requests.__init__')

try:
    from networking_huawei.drivers.ac.common import security_util
except ImportError:
    LOG.error('Please ensure Agile Controller plugin installed first')
    exit(1)

CONFIG_FILE = os.path.realpath('/etc/neutron/huawei_driver_config.ini')
SG_URL_SUFFIX = '/restconf/data/huawei-ac-neutron:neutron-cfg/security-groups'
SG_MODULE = 'huawei-ac-neutron:security-groups'
SG_CONTAINER = 'security-groups'
SG_LIST = 'security-group'
STATUS = ['SUCCESS', 'FAILED']
STEPS = ['AC Northbound Account Inspection',
         'WebSocket Connection Inspection',
         'Security Group Sync Inspection']


class InspectionUtil(object):
    """inspection util"""

    def __init__(self):
        if not os.path.exists(CONFIG_FILE):
            LOG.error('Please ensure %s exists', CONFIG_FILE)
            exit(0)
        self._config = ConfigObj(CONFIG_FILE, encoding='UTF8')
        self._validate_config()
        self._username = self._config['huawei_ac_config']['ac_auth_username']
        self._password = self._get_decrypted_password()
        self._host_ip = self._config['huawei_ac_agent_config']['host_ip']
        self._hosts = self._config['huawei_ac_agent_config']['rpc_server_ip']
        self._hosts = \
            [self._hosts] if not isinstance(self._hosts, list) else self._hosts
        self._results = {host: {} for host in self._hosts}

    @property
    def results(self):
        """results"""
        return self._results

    def _validate_config(self):
        if 'huawei_ac_agent_config' not in self._config:
            LOG.error('Please ensure huawei_ac_agent_config is configured')
            exit(0)
        if 'huawei_ac_config' not in self._config:
            LOG.error('Please ensure huawei_ac_config is configured')
            exit(0)
        if 'ac_auth_username' not in self._config['huawei_ac_config']:
            LOG.error('Please ensure ac_auth_username is configured')
            exit(0)
        if 'ac_auth_password' not in self._config['huawei_ac_config']:
            LOG.error('Please ensure ac_auth_password is configured')
            exit(0)
        if 'rpc_server_ip' not in self._config['huawei_ac_agent_config']:
            LOG.error('Please ensure rpc_server_ip is configured')
            exit(0)
        if 'host_ip' not in self._config['huawei_ac_agent_config']:
            LOG.error('Please ensure host_ip is configured')
            exit(0)

    def _get_decrypted_password(self):
        password = self._config['huawei_ac_config']['ac_auth_password']
        return six.text_type(security_util.decrypt_data(password),
                             errors='ignore')

    def northbound_account_inspection(self):
        """northbound account inspection"""
        headers = {
            'Content-type': 'application/json',
            'Accept': 'application/json'
        }

        body = jsonutils.dumps({
            'userName': self._username,
            'password': self._password
        })

        token_ids = {host: None for host in self._hosts}

        for host in self._hosts:
            try:
                url = 'https://%s:18002/controller/v2/tokens' % str(host)
                with requests.session() as session:
                    adapter = HWHTTPSAdapter(
                        cert_file=None, key_file=None, password=None,
                        pool_connections=100, pool_maxsize=100)
                    session.mount("https://", adapter)
                    result = session.request(
                        'POST', url=url, headers=headers, data=body,
                        verify=False, timeout=10)
                if result.status_code == requests.codes.ok:
                    self._results[host][STEPS[0]] = STATUS[0]
                    content = jsonutils.loads(result.content)
                    token_ids[host] = content['data']['token_id']
                else:
                    self._results[host][STEPS[0]] = STATUS[1]
            except Exception:
                self._results[host][STEPS[0]] = STATUS[1]

        return token_ids

    def _jsonrpc_connection_inspection_core(self, host):
        """json rpc connection inspection call it"""
        cur_port = org_port = None
        for _ in range(10):
            process_child1 = subprocess.Popen(args=['netstat', '-anp'],
                                              stdout=subprocess.PIPE)
            process_child2 = subprocess.Popen(
                [os.path.realpath('/bin/grep'), '-w', '%s:18010' % host],
                stdin=process_child1.stdout, stdout=subprocess.PIPE)
            lines_list = subprocess.Popen(
                args=[os.path.realpath('/bin/grep'), 'ESTABLISHED'],
                stdin=process_child2.stdout, stdout=subprocess.PIPE
            ).stdout.readlines()
            lines_str = ' '.join(str(line) for line in lines_list)
            for word in lines_str.split():
                if self._host_ip not in word:
                    continue
                cur_port = word.split(':')[-1]
                if not org_port:
                    org_port = cur_port
                if cur_port != org_port:
                    self._results[host][STEPS[1]] = STATUS[1]
                    return True
            time.sleep(1)
        if not cur_port:
            self._results[host][STEPS[1]] = STATUS[1]
            return True
        return False

    def jsonrpc_connection_inspection(self):
        """json rpc connection inspection"""
        for host in self._hosts:
            if self._jsonrpc_connection_inspection_core(host):
                return
            self._results[host][STEPS[1]] = STATUS[0]

    @classmethod
    def _get_ops_default_security_groups(cls):
        cmd = 'neutron security-group-list 2>/dev/null | grep -w default'
        line_pattern = re.compile(
            r'\| [0-9a-z]{8}-[0-9a-z]{4}-[0-9a-z]{4}-[0-9a-z]{4}-[0-9a-z]{12} '
            r'\| default\s*'
            r'\| [0-9a-z]{32}.*')
        id_pattern = re.compile(
            r'[0-9a-z]{8}-[0-9a-z]{4}-[0-9a-z]{4}-[0-9a-z]{4}-[0-9a-z]{12}')
        default_sg_ids = []
        for line in os.popen(cmd).readlines():
            if not line_pattern.match(line.strip()):
                continue
            for word in line.strip().split('|'):
                if id_pattern.match(word.strip()):
                    default_sg_ids.append(word.strip())
        return default_sg_ids

    def _get_ac_default_security_groups(self):
        token_ids = self.northbound_account_inspection()
        default_sg_ids = {host: [] for host in self._hosts}

        for host in self._hosts:
            token = token_ids.get(host)
            if not token:
                continue
            default_sg_ids[host] += self._get_ac_sg_by_url(host, token)
        return default_sg_ids

    @classmethod
    def _get_ac_sg_by_url(cls, host, token):
        """ _get_ac_default_security_groups call it"""
        headers = {
            'Content-type': 'application/json',
            'Accept': 'application/json',
            'X-ACCESS-TOKEN': token
        }
        try:
            url = 'https://{}:18002{}'.format(str(host), SG_URL_SUFFIX)
            with requests.session() as session:
                adapter = HWHTTPSAdapter(
                    cert_file=None, key_file=None, password=None,
                    pool_connections=100, pool_maxsize=100)
                session.mount("https://", adapter)
                result = session.request(
                    'GET', url=url, headers=headers, verify=False, timeout=10)
                if result.status_code == requests.codes.ok:
                    content = jsonutils.loads(result.content)
                    sg_list = content[SG_MODULE][SG_LIST]
                    return [sg['uuid'] for sg in sg_list]
        except Exception as ex:
            LOG.error('_get_ac_sg_by_url occur error:%s', ex)
        return []

    def security_group_sync_inspection(self):
        """security group sync inspection"""
        ops_default_sg_ids = self._get_ops_default_security_groups()
        ac_default_sg_ids = self._get_ac_default_security_groups()
        for host in self._hosts:
            self._results[host][STEPS[2]] = STATUS[0]
            for sg_id in ops_default_sg_ids:
                if sg_id not in ac_default_sg_ids[host]:
                    self._results[host][STEPS[2]] = STATUS[1]
                    break


def format_print(results):
    """format print"""
    host_max_len = step_max_len = 4
    status_max_len = 6
    for (host, values) in results.items():
        if len(host) > host_max_len:
            host_max_len = len(host)
        for (step, status) in values.items():
            if len(step) > step_max_len:
                step_max_len = len(step)
            if len(status) > status_max_len:
                status_max_len = len(status)

    def _split_print():
        print('+-{}-+-{}-+-{}-+'.format('-' * host_max_len, '-' * step_max_len,
                                        '-' * status_max_len))

    _split_print()
    print('| HOST{} | STEP{} | STATUS{} |'.format(' ' * (host_max_len - 4),
                                                  ' ' * (step_max_len - 4),
                                                  ' ' * (status_max_len - 6)))
    _split_print()
    for (host, values) in results.items():
        for (step, status) in values.items():
            print('| {}{} | {}{} | {}{} |'.
                  format(host, ' ' * (host_max_len - len(host)),
                         step, ' ' * (step_max_len - len(step)),
                         status, ' ' * (status_max_len - len(status))))
            _split_print()


def main():
    """main function"""
    util = InspectionUtil()
    LOG.info('Inspecting...')
    util.northbound_account_inspection()
    util.jsonrpc_connection_inspection()
    util.security_group_sync_inspection()
    format_print(util.results)


if __name__ == '__main__':
    main()
