#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""installation check"""

from __future__ import print_function
import os
import re
import sys
import time
import getpass
import argparse
from subprocess import Popen, PIPE
import logging.handlers
import six
import paramiko

LOG_FILE = os.path.realpath("/var/log/neutron/modify_black_white_list.log")
CONFIG_FILE = os.path.realpath("/etc/neutron/huawei_driver_config.ini")
SCRIPT_FILE = os.path.realpath("/root/networking-huawei/tools/modify_black_white_list.py")
OPENSTACK_CLI_CRT = os.path.realpath('/etc/FSSecurity/cacert/openstack-cli_ca.crt')
PARSER_FILE = os.path.realpath("/usr/bin/install_tool/zk_cache_cfg_parser.py")
CONFIG_SEG = "huawei_ac_config"
OPS_VERSION = "OPS_version"
DEFAULT_BLACK_LIST = ['external_om', 'external_api', 'internal_base']
LIST_MATCH = 'network_list_matching'
BLACK_LIST = "network_black_list"
WHITE_LIST = "network_white_list"
MAX_RETRY_TIMES = 3
MAX_STR_LEN = 128
TIMEOUT = 10
SPECIAL_CHARS = '`~!@#$%^&*(){}[]:;<>?|\'\"/'
SPECIAL_CHAR_MAP = {
    '`': r'\`',
    '~': r'\~',
    '!': r'\!',
    '@': r'\@',
    '#': r'\#',
    '$': r'\$',
    '%': r'\%',
    '^': r'\^',
    '&': r'\&',
    '*': r'\*',
    '(': r'\(',
    ')': r'\)',
    '{': r'\{',
    '}': r'\}',
    '[': r'\[',
    ']': r'\]',
    ':': r'\:',
    ';': r'\;',
    '<': r'\<',
    '>': r'\>',
    '?': r'\?',
    '|': r'\|',
    '\'': '\\\'',
    '\"': '\\\"',
    '/': r'\/'
}

EXTEND, REMOVE, REPLACE = 1, 2, 3
CONSOLE_HANDLER = logging.StreamHandler()
FILE_HANDLER = logging.handlers.RotatingFileHandler(LOG_FILE, maxBytes=10 * 1024 * 1024, backupCount=10)
FORMATTER = logging.Formatter('%(asctime)s %(levelname)-5s [AC] %(message)s')
CONSOLE_HANDLER.setFormatter(FORMATTER)
FILE_HANDLER.setFormatter(FORMATTER)
LOG = logging.getLogger('modify_black_white_list')
LOG.addHandler(CONSOLE_HANDLER)
LOG.addHandler(FILE_HANDLER)
LOG.setLevel(logging.DEBUG)


class HostBase(object):
    """host base"""

    def __init__(self, number):
        self._number = number
        self._hostname = None
        self._fsp_password = None
        self._neutron_user = None
        self._neutron_password = None

    @property
    def number(self):
        """get number"""
        return self._number

    @number.setter
    def number(self, value):
        """set number"""
        self._number = value

    @property
    def hostname(self):
        """get host name"""
        return self._hostname

    @hostname.setter
    def hostname(self, value):
        """set host name"""
        self._hostname = value

    @property
    def fsp_password(self):
        """get fsp password"""
        return self._fsp_password

    @property
    def neutron_user(self):
        """get neutron user"""
        return self._neutron_user

    @property
    def neutron_password(self):
        """get neutron user password"""
        return self._neutron_password

    def _get_hostname(self, hostname=''):
        ensure_times = 0
        if hostname:
            self._hostname = hostname
            return

        while True:
            if ensure_times == MAX_RETRY_TIMES:
                print('INIT FAILED: max retry times exceeded, exit')
                print('=' * 80)
                exit(0)

            hostname = six.moves.input(
                'INIT INFO: please input neutron server ip: ')

            if len(hostname) > MAX_STR_LEN:
                print('INIT FAILED: hostname is too long')
                ensure_times += 1
                continue

            pattern = re.compile(r'^[\d.]+$')
            if not pattern.match(hostname):
                print('INIT FAILED: ip address format invalid')
                ensure_times += 1
                continue

            self._hostname = hostname
            return

    def _get_password(self, username):
        ensure_times = 0
        while True:
            if ensure_times == MAX_RETRY_TIMES:
                LOG.error('max retry times exceeded, exit')
                _print_split_line()
                exit(0)

            password = getpass.getpass(
                'INIT INFO: please input %s %s password: ' %
                (self._hostname, username))
            if len(password) > MAX_STR_LEN:
                LOG.error('input password is too long')
                ensure_times += 1
                continue

            return password

    def get_host_info(self, hostname='', is_fsp=True):
        """get host info"""
        self._get_hostname(hostname)
        if is_fsp:
            self._fsp_password = self._get_password('fsp')
        neutron_user = six.moves.input('INIT INFO: please input username who can change the etc: ')
        self._neutron_user = neutron_user
        self._neutron_password = self._get_password(self._neutron_user)


class ConnectBase(object):
    """connection  base"""

    def __init__(self, hostname, username, password):
        self.ssh = paramiko.SSHClient()
        self.ssh.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        self.hostname = hostname
        self.username = username
        self.password = password

    def check_connection(self, timeout=TIMEOUT):
        """check connection"""
        try:
            self.ssh.connect(
                hostname=self.hostname, username=self.username,
                password=self.password, timeout=timeout)
            shell_cmd = '{} {}'.format('ls', CONFIG_FILE)
            _, stdout, stderr = self.ssh.exec_command(shell_cmd)
            if not stdout.read() and stderr.read():
                LOG.error('%s huawei_driver_config.ini dose not exist',
                          self.hostname)
                return False
        except Exception as ex:
            LOG.error('connect host %s failed: %s', self.hostname, ex)
        else:
            return True
        finally:
            self.ssh.close()
        return True

    def invoke_execute(self, command, host, is_fsp=True):
        """invoke execute"""
        self.ssh.connect(
            hostname=self.hostname, username=self.username,
            password=self.password, timeout=TIMEOUT)
        shell = self.ssh.invoke_shell()
        shell.recv(65535)
        time.sleep(0.5)
        if is_fsp:
            root_command = "su - {} -c '{}'\n".format(host.neutron_user, command)
        else:
            root_command = "{}\n".format(command)
        shell.send(root_command)
        shell.recv(65535)
        time.sleep(0.5)
        shell.send("{}\n".format(host.neutron_password))
        shell.recv(65535)
        time.sleep(0.5)
        self.ssh.close()


class FunctionBase(object):
    """function base"""

    def __init__(self):
        self.config = six.moves.configparser.ConfigParser()
        self.config.read(CONFIG_FILE)
        self._ops_version = self._get_ops_version()
        self.pattern = re.compile(r'^[\w\s,]+$')
        self.config_file = None

    def _get_ops_version(self):
        try:
            ops_version = self.config.get(CONFIG_SEG, OPS_VERSION).strip()
            return ops_version
        except Exception as ex:
            LOG.error('get %s failed: %s', OPS_VERSION, ex)
            raise

    @property
    def is_fsp(self):
        """Check is FusionSphere or not

        :return: True or False
        """
        return self._ops_version.startswith('FusionSphere')

    def get_name_list(self, black_white):
        """get name list"""
        try:
            name_list = self.config.get(CONFIG_SEG, black_white).strip()
            name_list = name_list.split(',')
            name_list = [name.strip() for name in name_list if name.strip()]
            return name_list
        except (six.moves.configparser.NoSectionError,
                six.moves.configparser.NoOptionError) as ex:
            LOG.error('get %s failed: %s', black_white, ex)
            return [] if black_white == WHITE_LIST else DEFAULT_BLACK_LIST

    @classmethod
    def _set_name(cls, pattern, name):
        """set name: BLACK_LIST,WHITE_LIST,LIST_MATCH"""
        with open(CONFIG_FILE, 'r') as file_reader:
            lines = file_reader.readlines()
        flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
        with os.fdopen(os.open(CONFIG_FILE, flags, 0o600), 'w') as config_file:
            write_flag = False
            for line in lines:
                if re.match(r'^#*%s' % pattern, line):
                    line = '%s = %s\n' % (pattern, name)
                    write_flag = True
                config_file.write(line)
            if not write_flag:
                line = '%s = %s\n' % (pattern, name)
                config_file.write(line)

    def get_list_matching(self):
        """get list matching"""
        try:
            list_matching = self.config.get(CONFIG_SEG, LIST_MATCH).strip()
            return list_matching
        except (six.moves.configparser.NoSectionError,
                six.moves.configparser.NoOptionError) as ex:
            LOG.error('get %s failed: %s', LIST_MATCH, ex)
            return 'suffix'

    def _get_user_input(self, black_white):
        while True:
            user_input = six.moves.input('please input %s: ' % black_white)
            if self.pattern.match(user_input):
                name_list = user_input.strip().split(',')
                name_list = [name.strip() for name in name_list]
                return name_list
            LOG.error('get user input failed: %s is invalid', black_white)

    def set_black_list(self, black_string=''):
        """set black list"""
        org_black_list = self.get_name_list(BLACK_LIST)
        LOG.info('original black list: %s', org_black_list)
        black_list = black_string.split(',')
        black_list = [black.strip() for black in black_list]
        new_black_list = black_list or self._get_user_input(BLACK_LIST)
        self._set_name(BLACK_LIST, ', '.join(new_black_list))

    def set_white_list(self, white_string=''):
        """set white list"""
        org_white_list = self.get_name_list(WHITE_LIST)
        LOG.info('original white list: %s', org_white_list)
        white_list = white_string.split(',')
        white_list = [white.strip() for white in white_list]
        new_white_list = white_list or self._get_user_input(WHITE_LIST)
        self._set_name(WHITE_LIST, ', '.join(new_white_list))

    def set_list_matching(self, list_matching='suffix'):
        """set list matching"""
        org_list_matching = self.get_list_matching()
        LOG.info('original list matching: %s', org_list_matching)
        self._set_name(LIST_MATCH, list_matching)


def _run_python_in_shell(cmd):
    """shell中执行python脚本并返回执行结果

    :param cmd: list,命令
    :return: str
    """
    real_cmd = [sys.executable]
    real_cmd.extend(cmd)
    sh_process = Popen(real_cmd, stdout=PIPE)
    result = sh_process.stdout.read()
    if isinstance(result, bytes):
        result = result.decode('utf8')
    return result.strip()


def _set_env():
    dc_admin_name = _run_python_in_shell(
        [PARSER_FILE, '-s', 'cps', '-t', 'cps-web', '-c', 'dc_admin_name'])
    os.environ['OS_USERNAME'] = '{}_admin'.format(dc_admin_name)
    os.environ['OS_TENANT_NAME'] = 'dc_system_{}'.format(dc_admin_name)
    os.environ['OS_PROJECT_NAME'] = 'dc_system_{}'.format(dc_admin_name)

    os_region_name = _run_python_in_shell(
        [PARSER_FILE, '-s', 'cps', '-t', 'cps-web', '-c', 'os_region_name'])
    os.environ['OS_REGION_NAME'] = os_region_name

    os_auth_url = _run_python_in_shell(
        [PARSER_FILE, '-s', 'cps', '-t', 'cps-web', '-c', 'os_auth_url'])
    os.environ['OS_AUTH_URL'] = os_auth_url

    password = getpass.getpass('INIT INFO: please input user '
                               '%s_admin password: ' % dc_admin_name)

    os.environ['OS_PASSWORD'] = password
    os.environ['NOVA_ENDPOINT_TYPE'] = 'internalURL'
    os.environ['OS_ENDPOINT_TYPE'] = 'internalURL'
    os.environ['CINDER_ENDPOINT_TYPE'] = 'internalURL'
    os.environ['OS_VOLUME_API_VERSION'] = '2'
    os.environ['OS_PROJECT_DOMAIN_NAME'] = 'Default'
    os.environ['OS_USER_DOMAIN_NAME'] = 'Default'
    os.environ['OS_IDENTITY_API_VERSION'] = '3'
    os.environ['OS_CACERT'] = OPENSTACK_CLI_CRT


def _get_neutron_servers():
    _set_env()
    neutron_cmd = ['cps', 'template-instance-list', '--service', 'neutron',
                   'neutron-server']
    neutron_server = Popen(neutron_cmd, stdout=PIPE)
    server_list = []
    for line in neutron_server.stdout.readlines():
        if 'neutron-server' not in line:
            continue
        if isinstance(line, bytes):
            line = line.decode('utf8')
        for item in line.split('|'):
            if re.match(r'^(25[0-5]|2[0-4]\d|[0-1]?\d?\d)(\.(25[0-5]|2[0-4]\d|[0-1]?\d?\d)){3}$', item.strip()):
                server_list.append(item.strip())

    return len(server_list), server_list


def _get_user_config_mode():
    while True:
        config_mode = six.moves.input('INIT INFO: please select config mode '
                                      '[1]extend [2]remove [3]replace: ')
        if re.match(r'^[123]$', config_mode):
            return int(config_mode.strip())
        print('INTI FAILED: input %s out of range ' % config_mode)


def _get_user_input(black_white, config_mode):
    while True:
        user_input = six.moves.input(
            'INIT INFO: please input %s to %s [\\ is not supported]: ' % (
                black_white, config_mode))
        if not user_input.strip():
            return []
        name_list = user_input.strip().split(',')
        name_list = [name.strip() for name in name_list if name.strip()]
        return name_list


def get_list_matching():
    """get list matching"""
    while True:
        list_match = six.moves.input('INIT INFO: please select list matching '
                                     'mode [1]prefix [2]suffix: ')
        if re.match(r'^[12]$', list_match):
            list_match = int(list_match.strip())
            return 'prefix' if list_match == 1 else 'suffix'
        LOG.error('input %s out of range', list_match)


def _get_host_numbers():
    while True:
        host_numbers = six.moves.input('INIT INFO: please input host numbers: ')
        if re.match(r'^\d+$', host_numbers):
            return host_numbers
        LOG.error('%s is not a number', host_numbers)


def _get_host_list():
    while True:
        host_string = six.moves.input('INIT INFO: please input host list: ')
        if not re.match(r'^[\d\s,.]+$', host_string):
            continue
        host_list = host_string.split(',')
        need_retype = False
        for host in host_list:
            if not re.match(r'^(25[0-5]|2[0-4]\d|[0-1]?\d?\d)(\.(25[0-5]|2[0-4]\d|[0-1]?\d?\d)){3}$',
                            host.strip()):
                need_retype = True
        if not need_retype:
            return [host.strip() for host in host_list]
        LOG.error('please input a comma separated neutron server IP addresses')


def _print_black_white_list(print_type='current'):
    local_function = FunctionBase()
    _print_split_line()
    local_black_list = local_function.get_name_list(BLACK_LIST)
    LOG.info('%s black list: %s', print_type, ','.join(local_black_list))
    local_white_list = local_function.get_name_list(WHITE_LIST)
    LOG.info('%s black list: %s', print_type, ','.join(local_white_list))
    local_list_match = local_function.get_list_matching()
    LOG.info('%s list matching: %s', print_type, local_list_match)
    _print_split_line()
    return local_black_list, local_white_list, local_list_match


def _print_split_line(split_num=80):
    LOG.info('=' * split_num)


def _parse_arguments():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '-b', '--black', action='store', dest='black_list', default='',
        help='black list with commas, default is blank string')
    parser.add_argument(
        '-w', '--white', action='store', dest='white_list', default='',
        help='white list with commas, default is blank string')
    parser.add_argument(
        '-m', '--match', action='store', dest='match_mode', default='suffix',
        help='match mode, default is suffix, [prefix, suffix] is available')
    parser.add_argument(
        '-pb', '--print-black', action='store_true', dest='print_black',
        default=False, help='print black list in current node')
    parser.add_argument(
        '-pw', '--print-white', action='store_true', dest='print_white',
        default=False, help='print white list in current node')
    parser.add_argument(
        '-pm', '--print-mode', action='store_true', dest='print_mode',
        default=False, help='print match mode in current node')
    parser.add_argument('--version', action='version', version='%(prog)s 1.0')
    return parser.parse_args()


def _get_black_white_list():
    org_black_list, org_white_list, _ = _print_black_white_list('original')
    config_mode = _get_user_config_mode()
    LOG.info('input config mode: %s', config_mode)
    if config_mode == EXTEND:
        extend_black_list = _get_user_input(BLACK_LIST, 'extend')
        LOG.info('input black list: %s', extend_black_list)
        black_list = org_black_list
        black_list.extend(extend_black_list)
        extend_white_list = _get_user_input(WHITE_LIST, 'extend')
        LOG.info('input white list: %s', extend_white_list)
        white_list = org_white_list
        white_list.extend(extend_white_list)
    elif config_mode == REMOVE:
        remove_black_list = _get_user_input(BLACK_LIST, 'remove')
        LOG.info('input black list: %s', remove_black_list)
        black_list = list(set(org_black_list) - set(remove_black_list))
        remove_white_list = _get_user_input(WHITE_LIST, 'remove')
        LOG.info('input white list: %s', remove_white_list)
        white_list = list(set(org_white_list) - set(remove_white_list))
    elif config_mode == REPLACE:
        black_list = _get_user_input(BLACK_LIST, 'replace')
        LOG.info('input black list: %s', black_list)
        white_list = _get_user_input(WHITE_LIST, 'replace')
        LOG.info('input white list: %s', white_list)
    else:
        print('PROCESS FAILED: config mode out of range')
        LOG.error('config mode out of range')
        _print_split_line()
        raise Exception("[AC]String length error")
    list_match = get_list_matching()
    LOG.info('input list matching: %s', list_match)
    _print_split_line()

    return black_list, white_list, list_match


def _set_black_white_list(function, host, black_list, white_list, list_match):
    if function.is_fsp:
        conn = ConnectBase(host.hostname, 'fsp', host.fsp_password)
    else:
        conn = ConnectBase(host.hostname, host.neutron_user, host.neutron_password)
    if not conn.check_connection():
        LOG.error('modify black white list failed')
        _print_black_white_list('current')
        _print_split_line()
        raise Exception("[AC]String length error")
    LOG.info('connection initialization complete')

    command = "python {} ".format(SCRIPT_FILE)
    if black_list:
        new_black_list = ','.join(black_list)
        check_list = []
        for char in new_black_list:
            if char in SPECIAL_CHARS and char not in check_list:
                new_black_list = new_black_list.replace(
                    char, SPECIAL_CHAR_MAP.get(char))
                check_list.append(char)
        command = "{}-b {} ".format(command, new_black_list)
    if white_list:
        new_white_list = ','.join(white_list)
        check_list = []
        for char in new_white_list:
            if char in SPECIAL_CHARS and char not in check_list:
                new_white_list = new_white_list.replace(
                    char, SPECIAL_CHAR_MAP.get(char))
                check_list.append(char)
        command = "{}-w {} ".format(command, new_white_list)
    command = "{}-m {}".format(command, list_match)
    if function.is_fsp:
        conn.invoke_execute(command, host)
    else:
        conn.invoke_execute(command, host, False)


def main():
    """Main function of black white list modifier.

    :return: None
    """
    function = FunctionBase()
    if len(sys.argv) > 1:
        args = _parse_arguments()
        if args.print_black:
            print(function.get_name_list(BLACK_LIST))
            sys.exit(0)
        if args.print_white:
            print(function.get_name_list(WHITE_LIST))
            sys.exit(0)
        if args.print_mode:
            print(function.get_list_matching())
            sys.exit(0)
        function.set_black_list(args.black_list)
        function.set_white_list(args.white_list)
        function.set_list_matching(args.match_mode)
    else:
        _print_split_line()
        LOG.info('begin to modify black white list')
        if function.is_fsp:
            server_num, server_list = _get_neutron_servers()
        else:
            server_num = _get_host_numbers()
            server_list = _get_host_list()
        host_list = []
        for number in six.moves.range(int(server_num)):
            host = HostBase(number)
            host.get_host_info(server_list[number], function.is_fsp)
            host_list.append(host)
        _print_split_line()
        LOG.info('input server list: %s', server_list)

        black_list, white_list, list_match = _get_black_white_list()

        for host in host_list:
            _set_black_white_list(function, host, black_list, white_list,
                                  list_match)

        LOG.info('modify black white list successfully')
        _print_black_white_list('current')
        _print_split_line()


if __name__ == '__main__':
    main()
