#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""modify controller account"""

from __future__ import print_function
import base64
import getpass
import os
import six
import requests.packages.urllib3
from configobj import ConfigObj
import eventlet
from oslo_serialization import jsonutils
import encryptor

from networking_huawei.drivers.ac.client.https_adapter import HWHTTPSAdapter
from networking_huawei.drivers.ac.encode_convert import convert_to_bytes

requests.packages.urllib3.disable_warnings()
requests = eventlet.import_patched('requests.__init__')

CONFIG_FILE = os.path.realpath("/etc/neutron/huawei_driver_config.ini")
PRODUCT_NAME = "iMaster NCE-Fabric"
CONTROLLER_PORT = '18002'

MAX_PWD_LEN = 128
MAX_TRY_TIMES = 3


def modify_account(username, pwd):
    """modify account or password"""

    while True:
        p_input = six.moves.input("Press [Y] confirm to modify the %s or "
                                  "press [N] to give up #[Y/N]" % (
                                      'account' if username else 'password'))
        if p_input.lower() == "y":
            break
        elif p_input.lower() == "n":
            return
        else:
            continue
    try:
        print("processing... please wait")
        encrypt_pwd = encryptor.encrypt_password(pwd)
        config = ConfigObj(CONFIG_FILE, encoding='UTF8')
        config["huawei_ac_config"]["ac_auth_password"] = \
            base64.b64encode(convert_to_bytes(encrypt_pwd)).decode()
        if username:
            config["huawei_ac_config"]["ac_auth_username"] = username
        config.write()
    except Exception as ex:
        print("Modify %s fail" % (
            '%s account(north account)' % PRODUCT_NAME if username else
            'password'))
        print(str(ex))
        return
    print("[Success]Modify %s success" % (
        '%s account(north account)' % PRODUCT_NAME if username else 'password'))


def get_token_test(username, pwd, port):
    """get token test"""
    auth_result = {}
    try:
        print("processing... please wait")
        config = ConfigObj(CONFIG_FILE, encoding='UTF8')
        host = config["huawei_ac_agent_config"]["rpc_server_ip"]
        if isinstance(host, list):
            host_list = host
        else:
            host_list = [host]
        for host in host_list:
            headers = {"Content-type": "application/json",
                       "Accept": "application/json"}

            auth_url = "%s%s%s%s%s" % ('https://',
                                       host, ":",
                                       port,
                                       "/controller/v2/tokens")
            auth = {"userName": username,
                    "password": pwd}

            auth_data = jsonutils.dumps(auth)
            adapter = HWHTTPSAdapter(
                cert_file=None, key_file=None, password=None,
                pool_connections=100, pool_maxsize=100)
            requests.session().mount("https://", adapter)
            try:
                req = requests.session().request('POST',
                                                 url=auth_url,
                                                 headers=headers,
                                                 data=auth_data,
                                                 verify=False,
                                                 timeout=10)
                auth_result[host] = req.status_code
            except Exception:
                auth_result[host] = requests.codes.bad_request

    except Exception:
        print('get_token_test occur error')
    return auth_result


def get_username():
    """get username from user"""
    print("Please ensure the north account has setted on %s" % PRODUCT_NAME)
    while True:
        username = six.moves.input('Please enter %s username('
                                   'north account): \n' % PRODUCT_NAME)
        if len(username) > MAX_PWD_LEN:
            print("The length of username should not be more than 128 bytes.")
            continue
        else:
            break
    return username


def main():
    """main function"""
    # before motify, need to ensure the account if has
    # setted on iMaster NCE-Fabric
    username = get_username()

    enter_ensure_times = 0
    while True:
        if enter_ensure_times == MAX_TRY_TIMES:
            print("[Error] Ensure new password to max retry times")
            print("Please operate again to modify password")
            break

        password = getpass.getpass('Please enter %s password('
                                   'north account): \n' % PRODUCT_NAME)
        if len(username) > MAX_PWD_LEN:
            print("The length of password should "
                  "not be more than 128 bytes.")
            continue

        password_ensure = getpass.getpass('Please enter %s password('
                                          'north account) again to '
                                          'confirm: \n' % PRODUCT_NAME)
        if password != password_ensure:
            print("Sorry, passwords do not match.")
            enter_ensure_times += 1
            continue

        auth_result = get_token_test(username, password, CONTROLLER_PORT)
        for host in auth_result:
            if auth_result.get(host) == requests.codes.ok:
                print("[success]The north account is authenticated "
                      "on %s(%s)" % (PRODUCT_NAME, host))
            elif auth_result.get(host) == requests.codes.unauthorized:
                print("[Warning]The north account is failed to "
                      "authenticate on %s(%s), Please "
                      "check the account" % (PRODUCT_NAME, host))
            else:
                print("[Warning]The north account can't be "
                      "authenticated on %s(%s)(network "
                      "or other reason)" % (PRODUCT_NAME, host))

        modify_account(username, password)
        break


if __name__ == "__main__":
    main()
