#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""
    show request result
"""

from __future__ import print_function
import re
import os
import datetime
import string
import requests

# -----------user config start------------#
# config the print number of request
PRINT_REQUEST_NUMBER = 20

# set the log path which is need to parse
LOG_PATH = os.path.realpath("/var/log/neutron/server.log")
# -----------user config end------------#

# the msg that parse from log file
REQUEST_SEND_KEY = "Start send the request information"
REQUEST_RESPONSE = "Send the request response"
REQUEST_RESPONSE_TIMEOUT = "Request sent to AC timed out"
TIMEOUT_MSG = "Send reuqest timeout, please check " \
              "the connetcion with AC"

# rest operation
REST_POST = "POST"
REST_DELETE = "DELETE"
REST_UPDATE = "PUT"
REST_GET = "GET"

TTIME_RULE = r'\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}.\d{3}'


class ParseResultProcess(object):
    """parse result process"""

    def __init__(self):
        self.record_file = self._open_record_file()

    @classmethod
    def _open_record_file(cls):
        now_time = datetime.datetime.now()
        file_name = "show_result_%s.log" % \
                    now_time.strftime("%Y-%m-%d_%H:%M:%S")
        directory = os.path.dirname(os.path.realpath(__file__))
        file_path = os.path.join(directory, file_name)
        return file_path

    def show_result(self, request_total, print_records):
        """show result"""
        print_records.sort(key=lambda x: x['time'])
        count = 0
        _msg = "+-----------------------------------------------+"
        request_diff = request_total - PRINT_REQUEST_NUMBER
        flags = os.O_RDWR | os.O_CREAT | os.O_TRUNC
        with os.fdopen(os.open(self.record_file, flags, 0o644), 'w+') as f_out:
            for record in print_records:
                count = self._show_result_record(
                    _msg, count, f_out, record['print_info'], request_diff)

    @staticmethod
    def _show_result_record(msg, count, path_file, record, request_diff):
        """show result:record"""
        if request_diff > 0:
            if "Method:" in record:
                count += 1
            if count > request_diff:
                if "Method:" in record:
                    print_msg = "%s\n%s\n" % (msg, record)
                else:
                    print_msg = record
                print(print_msg)
                path_file.write("%s\n" % record)
        else:
            print(record)
            path_file.write("%s\n" % record)
        return count


class ParseErrorResponse(object):
    """parse error response"""

    def __init__(self):
        pass

    @classmethod
    def get_error_msg(cls, content):
        """get error msg"""
        status_list = re.findall(r'"error-message":(.+?),', content)
        return status_list[0] if status_list else None


class DisplayProcess(object):
    """display process"""

    def __init__(self):
        pass

    @classmethod
    def display_red(cls, msg_str):
        """display red"""
        return "\033[1;31m %s \033[0m" % msg_str


class ParseRequestLog(object):
    """parse request log"""

    def __init__(self):
        self.parse_result = ParseResultProcess()
        self.parse_error_response = ParseErrorResponse()
        self.display_process = DisplayProcess()
        self.request_records = []
        self.response_prints = []
        self.response_records = []
        self.print_records = []

    def read_log_record(self, path):
        """read log record"""
        request_count = 0
        response_count = 0
        request_record_count = 0
        response_record_count = 0
        with open(path, "r") as file_path:
            lines = file_path.readlines()
            request_total = self.statistic_request_total(lines)
            for _, line in enumerate(lines):
                count_diff = request_total - PRINT_REQUEST_NUMBER
                if REQUEST_SEND_KEY in line:
                    request_record_count = self._read_send_key(
                        line, count_diff < 0 or count_diff >= request_count,
                        request_record_count)
                    request_count += 1
                elif REQUEST_RESPONSE in line and request_count != 0:
                    response_record_count = self._read_response(
                        line, count_diff < 0 or count_diff >= response_count,
                        response_record_count)
                    response_count += 1
                elif REQUEST_RESPONSE_TIMEOUT in line and request_count != 0:
                    response_record_count = self._read_response_timeout(
                        line, count_diff < 0 or count_diff >= response_count,
                        response_record_count)
                    response_count += 1

            self.parse_result.show_result(request_total, self.print_records)

    def _read_response_timeout(self, line, flg, response_record_count):
        """read log record:REQUEST_RESPONSE_TIMEOUT"""
        rtn_time = self.get_timestamp(line)
        timeout_ex = self.display_process.display_red(
            self.get_exception_msg(line))
        rtn_print = "TimeStamp: %s, Exception: %s\n" % (rtn_time, timeout_ex)
        rtn_record = "TimeStamp: %s, Exception: %s\n" % (
            rtn_time, self.get_exception_msg(line))
        self.print_records.append(
            {"time": self.change_time_to_stamp_type(rtn_time),
             "print_info": rtn_print, "record_info": rtn_record})
        if flg:
            self.response_prints.insert(response_record_count, rtn_print)
            self.response_records.insert(response_record_count, rtn_record)
            response_record_count += 1
        return response_record_count

    def _read_response(self, line, flg, response_record_count):
        """read log record:REQUEST_RESPONSE"""
        rtn_time = self.get_timestamp(line)
        status = self.get_response_status(line)
        content = self.get_response_content(line)
        if requests.codes.ok <= int(status) < requests.codes.multiple_choices:
            rtn_print = "TimeStamp: %s, Status Code: %s, " \
                        "Response Content: %s" % (rtn_time, status, content)
            rtn_record = rtn_print
        elif int(status) in {requests.codes.bad_gateway,
                             requests.codes.unavailable,
                             requests.codes.gateway_timeout}:
            rtn_time = self.get_timestamp(line)
            rtn_print = "TimeStamp: %s, Exception: %s\n" % (
                rtn_time, self.get_exception_msg(TIMEOUT_MSG))
            rtn_record = rtn_print
        else:
            error_msg = self.parse_error_response.get_error_msg(str(content))
            error_msg_print = self.display_process.display_red(error_msg)
            rtn_print = "TimeStamp: %s, Response Status Code: %s, Response " \
                        "Content: %s\nError Msg: %s" % (
                            rtn_time, self.display_process.display_red(status),
                            content, error_msg_print)
            rtn_record = "TimeStamp: %s, Response Status Code: %s, " \
                         "Response Content: %s\nError Msg: %s" % (
                             rtn_time, status, content, error_msg)
        self.print_records.append(
            {"time": self.change_time_to_stamp_type(rtn_time),
             "print_info": rtn_print, "record_info": rtn_record})
        if flg:
            self.response_prints.insert(response_record_count, rtn_print)
            self.response_records.insert(response_record_count, rtn_record)
            response_record_count += 1
        return response_record_count

    def _read_send_key(self, line, flg, request_record_count):
        """read_log_record:REQUEST_SEND_KEY"""
        request_time = self.get_timestamp(line)
        method = self.get_request_method(line)
        url = self.get_request_url(line)
        data = self.get_request_data(line)
        request = "TimeStamp: %s, Method: %s, Url: %s\nJsonData: " \
                  "%s" % (request_time, method, url, data)
        self.print_records.append(
            {"time": self.change_time_to_stamp_type(request_time),
             "print_info": request,
             "record_info": request})
        if flg:
            self.request_records.insert(request_record_count, request)
            request_record_count += 1
        return request_record_count

    @classmethod
    def statistic_request_total(cls, lines):
        """statistic request total"""
        number = 0
        for line in lines:
            number += 1 if REQUEST_SEND_KEY in line else 0
        return number

    @classmethod
    def get_request_method(cls, line):
        """get request method"""
        method = re.findall(r"method: (.+?),", line)
        return method[0] if method else None

    def get_request_url(self, line):
        """get request url"""
        oper = self.get_request_method(line)
        if oper in [REST_GET, REST_DELETE]:
            url = re.findall(r"url: (.*)", line)
        else:
            url = re.findall(r"url: (.+?),", line)

        return url[0] if url else None

    def get_request_data(self, line):
        """get request data"""
        oper = self.get_request_method(line)
        if oper in [REST_POST, REST_UPDATE]:
            data_list = re.findall(r"data: (.*)", line)
        else:
            data_list = None

        return data_list[0] if data_list else None

    @classmethod
    def get_response_status(cls, line):
        """get response status"""
        status_list = re.findall(r"status_code: (.+?),", line)
        return status_list[0] if status_list else None

    @classmethod
    def get_response_content(cls, line):
        """get response content"""
        content_list = re.findall(r"res_content: (.*)", line)
        return content_list[0] if content_list else None

    @classmethod
    def get_timestamp(cls, line):
        """get timestamp"""
        timestamp = re.search(TTIME_RULE, line).group(0)
        return timestamp if timestamp else None

    @classmethod
    def change_time_to_stamp_type(cls, time):
        """change time to stamp type"""
        time_stamp = string.replace(time, '-', '')
        time_stamp = string.replace(time_stamp, ':', '')
        time_stamp = string.replace(time_stamp, ' ', '')
        return time_stamp

    @classmethod
    def get_exception_msg(cls, exception_msg=None):
        """get exception msg"""
        if exception_msg:
            ex_msg = re.findall(r"traceback: (.*),", exception_msg)
            if ex_msg:
                return ex_msg
        return TIMEOUT_MSG


PARSE_REQUEST = ParseRequestLog()
PARSE_REQUEST.read_log_record(LOG_PATH)
