#!/usr/bin/env python
# coding=utf-8

"""
功 能：cli类，该类主要涉及cli转SND功能
版权信息：华为技术有限公司，版本所有(C) 2010-2021
"""
import os
import xml.dom.minidom
import codecs
from aoc.snd.clisnd import CliSND
from aoc.snd.snd_model_pb2.sysoidinfo_pb2 import SysoidInfo
from aoc.snd.snd_model_pb2.cliDriver_pb2 import CliDriverInfo
from aoc.snd.snd_model_pb2.ecsdriver_pb2 import CommonDriverInfo
from aoc.snd.snd_model_pb2.ecsDbProcess_pb2 import EcsDbConfigOut
from aoc.snd.snd_model_pb2.cliTransform_pb2 import CliCustomTransformOutput
from aoc.snd.snd_model_pb2.connectinfo_pb2 import ConnectInfos, \
    ProtocolEntity, DEFAULT_CONNECT, PRIMARY_CONNECTION
from aoc.snd.snd_model_pb2.channelInfo_pb2 import SINGLE_CHANNEL
from aoc.snd.snd_model_pb2.FeatureCfgs_pb2 import FeatureCfgsMsg, Feature, Function
from aoc.snd.snd_model_pb2.cliTransform_pb2 import OperType
from com.huawei.controller.f5.clitoyang import cli_to_yang_custom


class F5BIGIP(CliSND):
    """
    功能描述：设备纳管定制
    1. 注册设备sysoid信息
    2. 配置设备的连接能力信息
    """

    def getSysoidInfo(self, aoccontext, request=None):
        """
        注册设备sysoid信息，用于纳管此款型设备。
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return: 设备信息
        """
        self.logger.info("[F5BIGIP][getSysoidInfo] request is: %s, context is: %s"
                         % (request, aoccontext))
        sys_oid_info = SysoidInfo()
        dom = xml.dom.minidom.parse(self.resourceDir + "/resources/snmpsysoid_f5.xml")
        root = dom.documentElement
        snmp_ids = root.getElementsByTagName('snmpsysid')
        for snmp_id in snmp_ids:
            self.gen_sysoid(sys_oid_info, snmp_id)
        return sys_oid_info

    @staticmethod
    def gen_sysoid(sys_oid_info, snmp_id):
        """
        设备sysoid
        :param sysoid_info: 设备信息
        :param device_model: 设备类型
        :param sys_oid: sysoid
        """
        sysoid_entity = sys_oid_info.sysoidEntity.add()
        sysoid_entity.deviceType = snmp_id.getAttribute("devicetype")
        sysoid_entity.deviceModel = snmp_id.getAttribute("devicemodel")
        sysoid_entity.sysoid = snmp_id.getAttribute("id")
        sysoid_entity.deviceVendor = snmp_id.getAttribute("devicevendor")

    def getConnectInfo(self, aoccontext, request=None):
        """
        获取连接信息
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return: 设备信息
        """
        self.logger.info("[F5BIGIP][getConnectInfo] request is: %s, context is: %s"
                         % (request, aoccontext))
        connect_infos = ConnectInfos()
        connect_info = connect_infos.connectInfo.add()
        connect_info.protocolEntity.protocolType = ProtocolEntity.cli
        connect_info.connectPolicy = DEFAULT_CONNECT
        connect_info.channelInfo.writeChannel = SINGLE_CHANNEL
        connect_info.channelInfo.is_read_share_write = True
        connect_info.connectionPriority = PRIMARY_CONNECTION
        self.logger.info('getConnectInfo end.')
        return connect_infos

    def getCliDriverInfo(self, aoccontext, request=None):
        """
        获取命令行驱动信息
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return: 设备信息
        """
        self.logger.info("[F5BIGIP][getCliDriverInfo] request is: %s, context is: %s"
                         % (request, aoccontext))
        cli_driver_info = CliDriverInfo()
        self.logger.info(self.get_dict())
        for key, value in self.get_dict().items():
            cli_driver_entity = cli_driver_info.cliDriverEntity.add()
            cli_driver_entity.key = key
            cli_driver_entity.value = value
        self.logger.info('getCliDriverInfo end.')
        return cli_driver_info

    def dbPostProcess(self, aoccontext, request):
        """
        数据库关联删除
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return: 设备信息
        """
        self.logger.info("[F5BIGIP][dbPostProcess] request is: %s, context is: %s"
                         % (request, aoccontext))
        ecs_data = request.data
        ecs_path = request.path
        ecs_op_type = request.opType
        self.logger.info("ecsDbconfig  getData:" + ecs_data)
        self.logger.info("ecsDbconfig  getPath:" + ecs_path)
        self.logger.info("ecsDbconfig  getOpType:" + ecs_op_type)
        ecs_db_config_out = EcsDbConfigOut()
        return ecs_db_config_out

    def get_dict(self):
        """
        获取设备cli属性
        :return:
        """
        path = os.path.join(self.resourceDir, "resources/cli-driver.properties")
        self.logger.info(self.resourceDir)
        self.logger.info(path)
        properties = {}
        with open(path, 'r') as pro_file:
            for line in pro_file:
                if line.find('=') > 0:
                    strs = line.replace('\n', '').split('=')
                    properties[strs[0]] = codecs.getdecoder("unicode_escape")(strs[1])[0]
        self.logger.info(properties)
        return properties

    def getCommonDriverInfo(self, aoccontext, request=None):
        """
        获取驱动信息
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return: 设备信息
        """
        self.logger.info("[F5BIGIP][getCommonDriverInfo] request is: %s, context is: %s"
                         % (request, aoccontext))
        common_driver = CommonDriverInfo()
        common_driver.unsupportedOperations = "create,delete"
        common_driver.deleteStrategy = 1
        sync_to_del = common_driver.para.add()
        sync_to_del.key = "sync-to-del-enable"
        sync_to_del.value = "true"

        self.logger.info('getCommonDriverInfo end.')
        return common_driver

    def getSyncId(self, aoccontext, request):
        """
        获取同步id
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return:
        """
        self.logger.info("[F5BIGIP][getSyncId] request is: %s, context is: %s"
                         % (request, aoccontext))
        self.logger.info('getSyncId end.')

    def preSyncToNe(self, aoccontext, request):
        """
        预处理
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return:
        """
        self.logger.info("[F5BIGIP][preSyncToNe] request is: %s, context is: %s"
                         % (request, aoccontext))
        self.logger.info('preSyncToNe end.')

    def preSyncFromNe(self, aoccontext, request):
        """
        预处理
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return:
        """
        self.logger.info("[F5BIGIP][preSyncFromNe] request is: %s, context is: %s"
                         % (request, aoccontext))
        self.logger.info('preSyncFromNe end.')

    def postSyncFromNe(self, aoccontext, request):
        """
        后置同步
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return:
        """
        self.logger.info("[F5BIGIP][postSyncFromNe] request is: %s, context is: %s"
                         % (request, aoccontext))
        self.logger.info('postSyncFromNe end.')

    def yangToCli(self, aoccontext, request):
        """
        yang转成cli
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        """
        self.logger.info("[CUCLI][yangToCli] request is: %s, context is: %s"
                         % (request, aoccontext))
        self.logger.info('yangToCli end.')

    def cliToYang(self, aoccontext, request):
        """
        命令行转yang
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return:
        """
        self.logger.info("[F5BIGIP][cliToYang] request is: %s, context is: %s"
                         % (request, aoccontext))
        out = None
        if request.path == '/f5-ltm-node:f5-ltm-node':
            out = cli_to_yang_custom.clitoyang_ltm_node(request, self.logger)
        elif request.path == '/f5-ltm-monitor-tcp:f5-ltm-monitor-tcp':
            out = cli_to_yang_custom.clitoyang_ltm_monitor_tcp(request, self.logger)
        elif request.path == '/f5-ltm-pool:f5-ltm-pool':
            out = cli_to_yang_custom.clitoyang_ltm_pool(request, self.logger)
        elif request.path == '/f5-ltm-profile-fastl4:f5-ltm-profile-fastl4':
            out = cli_to_yang_custom.clitoyang_ltm_profile_fastl4(request, self.logger)
        elif request.path == '/f5-ltm-virtual:f5-ltm-virtual':
            out = cli_to_yang_custom.clitoyang_ltm_virtual(request, self.logger)
        elif request.path == '/f5-ltm-virtual-address:f5-ltm-virtual-address':
            out = cli_to_yang_custom.clitoyang_ltm_virtual_address(request, self.logger)
        elif request.path == '/f5-ltm-snatpool:f5-ltm-snatpool':
            out = cli_to_yang_custom.clitoyang_ltm_snatpool(request, self.logger)
        elif request.path == '/f5-net-vlan:f5-net-vlan':
            out = cli_to_yang_custom.clitoyang_net_vlan(request, self.logger, aoccontext)
        elif request.path == '/f5-net-self:f5-net-self':
            out = cli_to_yang_custom.clitoyang_net_self(request, self.logger)
        elif request.path == '/f5-net-route:f5-net-route':
            out = cli_to_yang_custom.clitoyang_net_route(request, self.logger)
        elif request.path == '/f5-net-trunk:f5-net-trunk':
            out = cli_to_yang_custom.clitoyang_net_trunk(request, self.logger)
        self.logger.info('cliToYang end: %s' % out)
        return out

    def getFeatures(self, aoccontext, request=None):
        """
        获取特性
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return:
        """
        self.logger.info("[F5BIGIP][getFeatures] request is: %s, context is: %s"
                         % (request, aoccontext))
        feature_msg = FeatureCfgsMsg()
        feature_msg.replace = True
        feature_msg.features.extend(
            self.build_feature('f5-ltm-node', '',
                               '(urn:ietf:params:xml:ns:yang:f5-ltm-node?'
                               'revision=2021-11-05)f5-ltm-node'))
        feature_msg.features.extend(
            self.build_feature('f5-ltm-monitor-tcp', '',
                               '(urn:ietf:params:xml:ns:yang:f5-ltm-monitor-tcp?'
                               'revision=2021-11-05)f5-ltm-monitor-tcp'))
        feature_msg.features.extend(
            self.build_feature('f5-ltm-pool', '',
                               '(urn:ietf:params:xml:ns:yang:f5-ltm-pool?'
                               'revision=2021-11-05)f5-ltm-pool'))
        feature_msg.features.extend(
            self.build_feature('f5-ltm-profile-fastl4', '',
                               '(urn:ietf:params:xml:ns:yang:f5-ltm-profile-fastl4?'
                               'revision=2021-11-05)f5-ltm-profile-fastl4'))
        feature_msg.features.extend(
            self.build_feature('f5-ltm-virtual', '',
                               '(urn:ietf:params:xml:ns:yang:f5-ltm-virtual?'
                               'revision=2021-11-05)f5-ltm-virtual'))
        feature_msg.features.extend(
            self.build_feature('f5-ltm-virtual-address', '',
                               '(urn:ietf:params:xml:ns:yang:f5-ltm-virtual-address?'
                               'revision=2021-11-05)f5-ltm-virtual-address'))
        feature_msg.features.extend(
            self.build_feature('f5-ltm-snatpool', '',
                               '(urn:ietf:params:xml:ns:yang:f5-ltm-snatpool?'
                               'revision=2021-11-05)f5-ltm-snatpool'))
        feature_msg.features.extend(
            self.build_feature('f5-net-vlan', '',
                               '(urn:ietf:params:xml:ns:yang:f5-net-vlan?'
                               'revision=2021-11-05)f5-net-vlan'))
        feature_msg.features.extend(
            self.build_feature('f5-net-self', '',
                               '(urn:ietf:params:xml:ns:yang:f5-net-self?'
                               'revision=2021-11-05)f5-net-self'))
        feature_msg.features.extend(
            self.build_feature('f5-net-route', '',
                               '(urn:ietf:params:xml:ns:yang:f5-net-route?'
                               'revision=2021-11-05)f5-net-route'))
        feature_msg.features.extend(
            self.build_feature('f5-net-trunk', '',
                               '(urn:ietf:params:xml:ns:yang:f5-net-trunk?'
                               'revision=2021-11-05)f5-net-trunk'))
        self.logger.info('getFeatures end.')
        return feature_msg

    def build_feature(self, name, depends, path):
        """
        构造特性
        :param name: 特性名称
        :param depends: 依赖
        :param path: yang路径
        :return:
        """
        self.logger.info('build_feature start.')
        feature = Feature()
        feature.name = name
        self.logger.info('=====feature.name:%s, feature.operType:%s'
                         % (feature.name, feature.operType))
        feature.operType = Feature.MERGE
        feature.depends.extend([depends])
        function = Function()
        function.value = path
        function.collectPath = path
        function.preSyncToNe = False
        function.preSyncFromNe = False
        function.postSyncFromNe = False
        self.logger.info('function:%s' % function)
        feature.functions.extend([function])
        return [feature]

    def configCliPostProcessor(self, aoccontext, request):
        """
        配置后置处理
        :param aoccontext: 上下文
        :param request: 请求
        :return:处理过后的命令行
        """
        self.logger.info("[F5BIGIP][configCliPostProcessor] request is: %s, context is: %s"
                         % (request, aoccontext))
        out = CliCustomTransformOutput()
        # 此处可对命令行文本进行自定义处理
        self.logger.info('request:%s' % request)
        self.logger.info('request.oper_type:%s' % request.oper_type)
        cli_str = request.data

        # CREATE,MERGE,DELETE
        if request.oper_type == OperType.Value('DELETE'):
            cli_str = self.modDeleteCommand(cli_str)
        elif request.oper_type == OperType.Value('MERGE'):
            cli_str = self.modReplaceCommand(cli_str)
        elif request.oper_type == OperType.Value('CREATE'):
            if request.path.endswith('replace-all-with'):
                cli_str = self.modReplaceCommand(cli_str)
            else:
                cli_str = self.modCreateCommand(cli_str)

        out.data = cli_str
        self.logger.info('out.data=%s' % out.data)
        self.logger.info('default configCliPostProcessor end')
        return out

    def modDeleteCommand(self, cli_cmds):
        """
        delete配置后置处理
        :param cli_cmds: 处理前的命令行
        :return:处理过后的命令行
        """
        new_cmds = []
        cmds = cli_cmds.splitlines()

        for cmd in cmds:
            if cmd.startswith('delete'):
                if cmd.strip().endswith('replace-all-with'):
                    cmd = cmd.replace('replace-all-with', 'none').replace('delete ', 'modify ')
                elif cmd.strip().endswith('traffic-group'):
                    cmd = cmd.replace('traffic-group', 'traffic-group traffic-group-1')\
                        .replace('delete ', 'modify ')
                new_cmds.append('tmsh ' + cmd)
        cli_cmds = '\n'.join(new_cmds)
        self.logger.info('delete commands:%s' % cli_cmds)
        return cli_cmds

    def modReplaceCommand(self, cli_cmds):
        """
        modify配置后置处理
        :param cli_cmds: 处理前的命令行
        :return:处理过后的命令行
        """
        new_cmds = []
        cmds = cli_cmds.splitlines()

        for cmd in cmds:
            if cmd.strip().endswith('replace-all-with'):
                cmd = cmd.replace('replace-all-with', 'none')
            new_cmds.append('tmsh modify ' + cmd)

        cli_cmds = '\n'.join(new_cmds)
        self.logger.info('replace commands:%s' % cli_cmds)
        return cli_cmds

    def modCreateCommand(self, cli_cmds):
        """
        create配置后置处理
        :param cli_cmds: 处理前的命令行
        :return:处理过后的命令行
        """
        new_cmds = []
        cmds = cli_cmds.splitlines()

        for cmd in cmds:
            new_cmds.append('tmsh create %s' % cmd)
        cli_cmds = '\n'.join(new_cmds)
        self.logger.info('create commands:%s' % cli_cmds)
        return cli_cmds
