#!/usr/bin/env python
# coding=utf-8

"""
功 能：cli类，该类主要涉及cli转SND功能
版权信息：华为技术有限公司，版本所有(C) 2010-2021
"""
import os
import codecs
from aoc.snd.clisnd import CliSND
from aoc.snd.snd_model_pb2.discoverinfo_pb2 import DiscoverInfo, STANDARD_SNMP_DISCOVER
from aoc.snd.snd_model_pb2.sysoidinfo_pb2 import SysoidInfo
from aoc.snd.snd_model_pb2.ecsdriver_pb2 import CommonDriverInfo
from aoc.snd.snd_model_pb2.cliDriver_pb2 import CliDriverInfo
from aoc.snd.snd_model_pb2.cliTransform_pb2 import CliCustomTransformOutput
from aoc.snd.snd_model_pb2.connectinfo_pb2 import ConnectInfos, \
    ProtocolEntity, DEFAULT_CONNECT, PRIMARY_CONNECTION
from aoc.snd.snd_model_pb2.FeatureCfgs_pb2 import FeatureCfgsMsg, Feature, Function
from aoc.snd.snd_model_pb2.channelInfo_pb2 import SINGLE_CHANNEL
from com.huawei.controller.cisco.clitoyang import cli_to_yang_custom


class SND(CliSND):
    """
    功能描述：设备纳管定制
    1. 注册设备sysoid信息
    2. 配置设备的连接能力信息
    """

    def getSysoidInfo(self, aoccontext, request=None):
        """
        注册设备sysoid信息，用于纳管此款型设备。
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return: 设备信息
        """
        self.logger.info("[CISCO][getSysoidInfo] request is: %s, context is: %s"
                         % (request, aoccontext))
        sys_oid_info = SysoidInfo()
        sysoid_entity = sys_oid_info.sysoidEntity.add()
        sysoid_entity.sysoid = "1.3.6.1.4.1.9.12.3.1.3.777"
        sysoid_entity.deviceType = "AOC_SWITCH"
        sysoid_entity.deviceModel = "N7018"
        sysoid_entity.deviceVendor = "CISCO"
        return sys_oid_info

    def getConnectInfo(self, aoccontext, request=None):
        """
        获取连接信息
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return: 设备信息
        """
        self.logger.info("[CISCO][getConnectInfo] request is: %s, context is: %s"
                         % (request, aoccontext))
        self.logger.info('getConnectInfo start.')
        connectinfos = ConnectInfos()
        connectinfo = connectinfos.connectInfo.add()
        connectinfo.protocolEntity.protocolType = ProtocolEntity.cli
        connectinfo.connectPolicy = DEFAULT_CONNECT
        connectinfo.channelInfo.writeChannel = SINGLE_CHANNEL
        connectinfo.channelInfo.is_read_share_write = True
        connectinfo.connectionPriority = PRIMARY_CONNECTION
        self.logger.info('getConnectInfo end.')
        return connectinfos

    def getCliDriverInfo(self, aoccontext, request=None):
        """
        获取命令行驱动信息
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return: 设备信息
        """
        self.logger.info("[CISCO][getCliDriverInfo] request is: %s, context is: %s"
                         % (request, aoccontext))
        clidriver_info = CliDriverInfo()
        self.logger.info(self.get_dict())
        for key, value in self.get_dict().items():
            cli_driver_entity = clidriver_info.cliDriverEntity.add()
            cli_driver_entity.key = key
            cli_driver_entity.value = value
        self.logger.info('getCliDriverInfo ends.')
        return clidriver_info

    def get_dict(self):
        """
        获取设备cli属性
        :return:
        """
        path = os.path.join(self.resourceDir, "resources/cli-driver.properties")
        self.logger.info(self.resourceDir)
        self.logger.info(path)
        properties = {}
        with open(path, 'r') as pro_file:
            for line in pro_file:
                if line.find('=') > 0:
                    strs = line.replace('\n', '').split('=')
                    properties[strs[0]] = codecs.getdecoder("unicode_escape")(strs[1])[0]
        self.logger.info(properties)
        return properties

    def getCommonDriverInfo(self, aoccontext, request=None):
        """
        获取驱动信息
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return: 设备信息
        """
        self.logger.info("[CISCO][getCommonDriverInfo] request is: %s, context is: %s"
                         % (request, aoccontext))
        self.logger.info('getCommonDriverInfo start.')
        common_driver = CommonDriverInfo()
        common_driver.unsupportedOperations = "create,delete"
        common_driver.deleteStrategy = 1
        syncToDel = common_driver.para.add()
        syncToDel.key = "sync-to-del-enable"
        syncToDel.value = "true"

        self.logger.info('getCommonDriverInfo end.')
        return common_driver

    def cliToYang(self, aoccontext, request):
        """
        命令行转yang
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return:
        """
        self.logger.info("[CISCO][cliToYang] request is: %s, context is: %s"
                         % (request, aoccontext))
        if request.path == '/cisco-interface:cisco-interface':
            self.logger.info('[CISCO] interface.')
            output = cli_to_yang_custom.clitoyang_interface(request, self.logger)
        elif request.path == '/cisco-global:cisco-global':
            self.logger.info('[CISCO] global.')
            output = cli_to_yang_custom.clitoyang_global(request, self.logger)
        elif request.path == '/cisco-instance:cisco-instance':
            self.logger.info('[CISCO] instance.')
            output = cli_to_yang_custom.clitoyang_instance(request, self.logger)
        return output

    def getFeatures(self, aoccontext, request=None):
        """
        获取特性
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return:
        """
        self.logger.info("[CISCO][getFeatures] request is: %s, context is: %s"
                         % (request, aoccontext))
        feature_msg = FeatureCfgsMsg()
        feature_msg.replace = True
        feature_msg.features.extend(
            self.build_feature('cisco-interface', '',
                               '(urn:ietf:params:xml:ns:yang:cisco-interface?'
                               'revision=2021-02-25)cisco-interface'))
        feature_msg.features.extend(
            self.build_feature('cisco-global', '',
                               '(urn:ietf:params:xml:ns:yang:cisco-global?'
                               'revision=2021-02-25)cisco-global'))
        feature_msg.features.extend(
            self.build_feature('cisco-instance', '',
                               '(urn:ietf:params:xml:ns:yang:cisco-instance?'
                               'revision=2021-02-25)cisco-instance'))
        self.logger.info('getFeatures end.')
        return feature_msg

    def build_feature(self, name, depends, path):
        """
        构造特性
        :param name: 特性名称
        :param depends: 依赖
        :param path: yang路径
        :return:
        """
        self.logger.info('build_feature start.')
        feature = Feature()
        feature.name = name
        self.logger.info('name:%s, operType:%s' % (feature.name, feature.operType))
        feature.depends.extend([depends])
        function = Function()
        function.value = path
        function.preSyncToNe = False
        function.postSyncFromNe = False
        function.preSyncFromNe = False
        function.collectPath = path
        self.logger.info('function:%s' % function)
        feature.functions.extend([function])
        return [feature]

    def configCliPostProcessor(self, aoccontext, request):
        """
        配置后置处理
        :param aoccontext: 上下文
        :param request: 请求
        :return:处理过后的命令行
        """
        self.logger.info("[CISCO][configCliPostProcessor] request is: %s, context is: %s"
                         % (request, aoccontext))
        out = CliCustomTransformOutput()
        # 此处可对命令行文本进行自定义处理
        out.data = request.data
        self.logger.info('out.data=%s' % out.data)
        self.logger.info('default configCliPostProcessor end')
        return out

    def getDiscoverInfo (self, aoccontext, request=None):
        """
        获取设备发现信息。
        :param aoccontext: 上下文环境
        :param request: 方法携带的参数
        :return: 设备发现信息
        """
        self.logger.info("[CISCO][getDiscoverInfo] request is: %s, context is: %s"
                         % (request, aoccontext))
        discover_info = DiscoverInfo()
        discover_entity = discover_info.discoverEntity.add()
        discover_entity.discoverPolicy = STANDARD_SNMP_DISCOVER
        return discover_info
