#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""rpc client"""

import sched
import socket
import time
import threading

from concurrent.futures import ThreadPoolExecutor
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei._i18n import _LI, _LW, _LE
from networking_huawei.common.exceptions import NoRpcClientFromIpException, \
    NoTerminalFromIpException, TimeoutException, SyncConnectionException, \
    CrtFileNotFoundException
from networking_huawei.drivers.ac.common import constants
from networking_huawei.drivers.ac.ac_agent.rpc.rpc_client_handler import \
    RpcClientHandler
from networking_huawei.drivers.ac.ac_agent.rpc.websocket.websocket_client \
    import WebSocketClient
from networking_huawei.drivers.ac.common.fusion_sphere_alarm import \
    ACPluginAlarm

LOG = ncu.ac_log.getLogger(__name__)

WEBSOCKET_ACTIVE = 1
WEBSOCKET_DOWN = 0
REGISTER_COUNT = 6


class RpcClient(object):
    """Rpc Client"""

    def __init__(self, max_workers=constants.RPC_CLIENT_MAX_WORKERS):
        self.executor = ThreadPoolExecutor(max_workers=max_workers)
        self.websocket_clients = {}
        self.locks = {}
        self.futures = {}
        self.monitor_executor = ThreadPoolExecutor(max_workers=max_workers)
        self.schedule = sched.scheduler(time.time, time.sleep)
        self.is_heartbeat = False
        self.is_registered = REGISTER_COUNT
        self.remote_addr = None
        self.origin_state = None
        self.down_count = 0
        self.active_count = 0
        self.first_flag_fsp_alarm = 1
        self.reconnect_time = constants.RECONNECT_TIME

    def add_client(self, remote_address, local_address):
        """Add rpc client
        :param remote_address: the remote ip address is connected by rpc client
        :param local_address: the local host ip address
        """
        self.websocket_clients[remote_address] = \
            WebSocketClient(remote_address,
                            local_address,
                            RpcClientHandler(None, None),
                            None)
        self.remote_addr = remote_address
        self.locks[remote_address] = threading.Lock()

    def _start(self, remote_address):
        LOG.info("[AC]Rpc dispatcher %s begin to start",
                 self.websocket_clients.get(remote_address).local_address)
        self.websocket_clients.get(remote_address).start()
        LOG.info("[AC]Rpc dispatcher %s stop",
                 self.websocket_clients.get(remote_address).local_address)

    def start_client(self, remote_address, local_ip, cloud_name):
        """Rpc client start connecting to rpc server
        :param remote_address: tuple, the rpc server ip address and port
        :param local_ip: local host ip address
        :param cloud_name: openstack name
        """
        LOG.info(_LI("[AC]Begin to start websocket rpc client and try "
                     "to establish connection with %s,using the main thread"),
                 remote_address)

        if self.websocket_clients.get(remote_address):
            websocket_client = self.websocket_clients.get(remote_address)
        else:
            raise NoRpcClientFromIpException()

        try:
            websocket_client.create_connection()
            LOG.info(_LI("[AC]Rpc client established with %s success"),
                     remote_address)
            self.futures[remote_address] = self.executor.submit(self._start,
                                                                remote_address)
            self.websocket_detect(WEBSOCKET_ACTIVE)
        except socket.error as ex:
            LOG.error(_LE("[AC]Rpc client established with %s exception: %s"),
                      remote_address, str(ex))

        except CrtFileNotFoundException as ex:
            LOG.error(_LE("[AC]Rpc client established with %s exception: %s"),
                      remote_address, str(ex))

        finally:
            LOG.info("[AC]Start to send heart message")
            if not self.origin_state:
                self.websocket_detect(WEBSOCKET_DOWN)
            self.start_monitor_heartbeat()
            self.schedule.enter(constants.DEFAULT_PERIOD, 0, self._monitor,
                                (remote_address, local_ip, cloud_name))
            self.monitor_executor.submit(self.schedule.run)

    def _register(self, client, local_ip, cloud_name):
        self.is_registered = 0
        # report plugin version to ac
        client.send_plugin_version(local_ip, cloud_name)
        # report network black and white list to ac
        client.send_network_black_white_list(local_ip, cloud_name)

    def _heartbeat(self, client, remote_address):
        send_times = 0
        while send_times < constants.SEND_HEARTBEAT_TIMES:
            future = self.websocket_clients.get(remote_address).future
            future.event.clear()
            LOG.info("[HEARTBEAT]send ping, times:%d", send_times)
            client.websock.send_ping()
            send_times += 1
            try:
                future.get_result(constants.HEARTBEAT_TIMEOUT)
                self.reconnect_time = constants.RECONNECT_MONITOR_TIME
                self.websocket_detect(WEBSOCKET_ACTIVE)
                break
            except (TimeoutException, SyncConnectionException) as ex:
                LOG.error(_LE("[AC]Rpc client send ping exception: %s"),
                          str(ex))
                self.websocket_detect(WEBSOCKET_DOWN, str(ex))
                if send_times == constants.SEND_HEARTBEAT_TIMES and \
                        self.remote_addr in self.websocket_clients:
                    self.websocket_clients.get(
                        self.remote_addr).close_with_signal()

    def _handle_monitor_exception(self, reason, local_ip, cloud_name):
        self.websocket_detect(WEBSOCKET_DOWN, reason)
        if self.remote_addr in self.websocket_clients:
            self.websocket_clients.get(self.remote_addr).close_with_signal()
        self.schedule.enter(self.reconnect_time, 0, self._monitor,
                            (self.remote_addr, local_ip, cloud_name))

    def _monitor(self, remote_address, local_ip, cloud_name):
        curr_thread = threading.currentThread()
        curr_thread.setName("monitor")
        self.reconnect_time = constants.RECONNECT_TIME
        try:
            client = self.get_client(self.remote_addr)
            self._monitor_core(client, cloud_name, local_ip, remote_address)
        except (NoTerminalFromIpException, NoRpcClientFromIpException) as ex:
            LOG.error(_LE("[AC] No terminal or no rpc client, %s"), str(ex))
            self._handle_monitor_exception(str(ex), local_ip, cloud_name)
        except Exception as ex:
            LOG.error(_LE("[AC] Unknow error, %s"), ex)
            self._handle_monitor_exception(str(ex), local_ip, cloud_name)

    def _monitor_core(self, client, cloud_name, local_ip, remote_address):
        """_monitor call it"""
        if self.futures.get(self.remote_addr).running():
            if self.is_registered == REGISTER_COUNT:
                self._register(client, local_ip, cloud_name)
            self.is_registered += 1
            if self.is_heartbeat:
                self._heartbeat(client, remote_address)
        else:
            try:
                ex = self.futures.get(self.remote_addr).exception(1)
                if ex is not None:
                    LOG.error(_LE("[AC]Rpc client start connection exception:"
                                  "%s"), ex)
                    self.websocket_detect(WEBSOCKET_DOWN, str(ex))
            except Exception as ex:
                LOG.error(_LE("[AC]Rpc client,future.exception catch exception:"
                              "%s"), ex)
                self.websocket_detect(WEBSOCKET_DOWN, str(ex))
            LOG.debug("original address: %s, new addr: %s",
                      remote_address, self.remote_addr)
            if self.remote_addr in self.websocket_clients:
                self.websocket_clients.get(self.remote_addr).close_with_signal()
            # if the thread of I/O operation does not start normally,
            # the threadpool will be setted ad running exception,
            # our process is that stop original threadpool and start
            # a new threadpool
            LOG.error(
                _LE("[AC]Future exception,Start new thread pool executor"))
            self.executor.shutdown(wait=False)
            self.executor = ThreadPoolExecutor(
                max_workers=constants.RPC_CLIENT_MAX_WORKERS)
        self.schedule.enter(self.reconnect_time, 0, self._monitor,
                            (self.remote_addr, local_ip, cloud_name))

    def _websocket_detect_state_down_process(self):
        if self.origin_state == WEBSOCKET_ACTIVE and self.active_count != 0:
            self.active_count = 0
        ws_active_to_down = self.origin_state == WEBSOCKET_ACTIVE and self.down_count == 0
        ws_down_to_down = self.origin_state == WEBSOCKET_DOWN and self.down_count != 0
        if ws_active_to_down or ws_down_to_down:
            self.down_count = self.down_count + 1

    def _websocket_detect_state_active_process(self):
        if self.origin_state == WEBSOCKET_DOWN and self.down_count != 0:
            self.down_count = 0
        ws_down_to_active = self.origin_state == WEBSOCKET_DOWN and self.active_count == 0
        ws_active_to_active = self.origin_state == WEBSOCKET_ACTIVE and self.active_count != 0
        if ws_down_to_active or ws_active_to_active:
            self.active_count = self.active_count + 1

    def _websocket_detect_state_process(self, new_state):
        if new_state == WEBSOCKET_DOWN:
            self._websocket_detect_state_down_process()

        if new_state == WEBSOCKET_ACTIVE:
            self._websocket_detect_state_active_process()

    def websocket_detect(self, new_state, reason=''):
        """websocket_detect"""
        if not ncu.IS_FSP:
            return
        if self.first_flag_fsp_alarm:
            if new_state == WEBSOCKET_DOWN:
                self.down_count = 1
            if new_state == WEBSOCKET_ACTIVE:
                self.active_count = 1
            self.origin_state = new_state
            self.first_flag_fsp_alarm = 0
            return

        self._websocket_detect_state_process(new_state)

        if self.down_count == constants.WEBSOCKET_FAIL_COUNT_MAX:
            alarm_info = ACPluginAlarm. \
                get_websocket_connection_fail_alarm(reason)
            LOG.info("[AC]Websocket connection failed. Send alarm message.")
            ACPluginAlarm.send_alarm(alarm_info)
            self.down_count = 0

        if self.active_count == constants.WEBSOCKET_RECOVER_COUNT_MAX:
            alarm_info = ACPluginAlarm.get_websocket_connection_recovery_alarm()
            LOG.info("[AC]Websocket connection recovered. Send alarm message.")
            ACPluginAlarm.send_alarm(alarm_info)
            self.active_count = 0
        self.origin_state = new_state

    def get_client(self, remote_address):
        """get_client"""
        if not self.websocket_clients.get(remote_address):
            raise NoRpcClientFromIpException()
        with self.locks.get(remote_address):
            client = self.websocket_clients.get(remote_address)
            if client.has_connection():
                return client
            try:
                LOG.info(_LI("[AC]connection is down, reconnect to server"))
                res = client.create_connection()
                if res == -1:
                    LOG.info(
                        _LI("[AC]Rpc client established with %s fail again"),
                        remote_address)
                    return client
                LOG.info(
                    _LI("[AC]Rpc client established with %s success again"),
                    remote_address)
                self.submit_client(remote_address)
            except (socket.error, CrtFileNotFoundException) as ex:
                LOG.error(
                    _LE("[AC]Rpc client established with %s exception %s"),
                    remote_address, ex)
                raise NoTerminalFromIpException()

        return client

    def submit_client(self, remote_address):
        """submit_client"""
        if self.futures.get(remote_address):
            self.submit_client_core(remote_address)
        LOG.debug("[AC]Stop former thread and start new thread")
        self.futures[remote_address] = self.executor.submit(self._start,
                                                            remote_address)
        time.sleep(0.1)
        LOG.info(_LI("[AC]The executor process threads %s"),
                 self.executor._threads)
        return True

    def submit_client_core(self, remote_address):
        """submit_client call core"""
        is_running = self.futures.get(remote_address).running()
        count = 0
        while is_running and count < 3:
            LOG.info(_LI("[AC]The future task is running,stop it first"))
            self.futures.get(remote_address).cancel()
            is_running = self.futures.get(remote_address).running()
            time.sleep(1)
            count += 1

    def start_monitor_heartbeat(self):
        """start_monitor_heartbeat"""
        if self.is_heartbeat:
            LOG.warn(_LW("[AC]rpc client has started heartbeat monitor"))
            return
        self.is_heartbeat = True

    def stop_monitor_heartbeat(self):
        """stop_monitor_heartbeat"""
        self.is_heartbeat = False

    def register_all(self, handler):
        """register_all"""
        for _, websocket_client in self.websocket_clients.items():
            websocket_client.register(handler)
