#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Implementation of the Neutron Data Formatter."""

import copy
import json

import netaddr
import sqlalchemy
from neutron import manager
from neutron.db import db_base_plugin_v2 as neutron_db_ml2
from neutron.db.extraroute_db import ExtraRoute_dbonly_mixin
from neutron.db.l3_db import L3_NAT_dbonly_mixin
from neutron.db.models_v2 import Network, Subnet, Port
from neutron.db.securitygroups_db import SecurityGroupDbMixin
from neutron.plugins.ml2 import models
from neutron.plugins.ml2.models import PortBinding
from oslo_config import cfg

from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.util import ACCommonUtil, DataFilterUtil
from networking_huawei.drivers.ac.db.bgp_route.bgp_route_db import BgpRouteDbMixin
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei.drivers.ac.db.dnat.dnat import DNATDBMixin
from networking_huawei.drivers.ac.db.firewall_rule.firewall_rule_db import ACFirewallruleAddrsDbMixin
from networking_huawei.drivers.ac.db.flow_mirror.flow_mirror_db import FlowMirrorDbMixin
from networking_huawei.drivers.ac.db.l2br.l2br_db import L2brDbMixin
from networking_huawei.drivers.ac.db.schema import ACPortAttributes, ACResourceAttributes
from networking_huawei.drivers.ac.db.snat.snat import SnatDbMixin
from networking_huawei.drivers.ac.db.vpc_connection.vpc_connection_db import VpcConnectionDbMixin
from networking_huawei.drivers.ac.external.ext_if import ACKeyStoneIf
from networking_huawei.drivers.ac.model import bgp_route_model
from networking_huawei.drivers.ac.model import dnat_model
from networking_huawei.drivers.ac.model import exroute_model
from networking_huawei.drivers.ac.model import firewall_model, firewall_policy_model, firewall_rule_model
from networking_huawei.drivers.ac.model import floatingip_model
from networking_huawei.drivers.ac.model import flow_mirror_model
from networking_huawei.drivers.ac.model import ipsecvpn_model
from networking_huawei.drivers.ac.model import l2br_model
from networking_huawei.drivers.ac.model import network_model
from networking_huawei.drivers.ac.model import port_model
from networking_huawei.drivers.ac.model import qos_model
from networking_huawei.drivers.ac.model import router_model
from networking_huawei.drivers.ac.model import security_group_model
from networking_huawei.drivers.ac.model import snat_model
from networking_huawei.drivers.ac.model import subnet_model
from networking_huawei.drivers.ac.model import vpc_connection_model
from networking_huawei.drivers.ac.model.external_policy_model import ACExternalPolicyModel
from networking_huawei.drivers.ac.model.external_segment_model import ACExternalSegmentModel
from networking_huawei.drivers.ac.model.ipv6_ns_qos_policy_model import ACIpv6NsQoSpolicyModel
from networking_huawei.drivers.ac.model.policy_classifier_model import ACPolicyClassifierModel
from networking_huawei.drivers.ac.model.policy_rule_model import ACPolicyRuleModel
from networking_huawei.drivers.ac.model.policy_rule_set_model import ACPolicyRuleSetModel
from networking_huawei.drivers.ac.model.target_group_model import ACTargetGroupModel

try:
    from neutron_lib import exceptions as n_exc
except ImportError:
    from neutron.common import exceptions as n_exc
try:
    from neutron.db.common_db_mixin import CommonDbMixin
except ImportError:
    from networking_huawei.drivers.ac.common.common_db_mixin import \
        CommonDbMixin
try:
    from neutron.services.timestamp.timestamp_db import TimeStamp_db_mixin
except ImportError:
    TimeStamp_db_mixin = None
try:
    from neutron.objects.qos import policy as policy_object
except ImportError:
    pass
try:
    from neutron.common.constants import DEVICE_OWNER_ROUTER_GW
except ImportError:
    from neutron_lib.constants import DEVICE_OWNER_ROUTER_GW
try:
    try:
        from FSSecurity import crypt
    except ImportError:
        from FSComponentUtil import crypt
except ImportError:
    pass
try:
    from neutron_fwaas.db.firewall.firewall_db_poly import \
        FirewallPolyDbMixin as FirewallDBMixin
except ImportError:
    try:
        from neutron_fwaas.db.firewall.firewall_db import \
            Firewall_db_mixin as FirewallDBMixin
    except ImportError:
        try:
            from neutron.db.firewall.firewall_db import \
                Firewall_db_mixin as FirewallDBMixin
        except ImportError:
            pass
try:
    from neutron_fwaas.db.firewall.firewall_router_insertion_db \
        import FirewallRouterAssociation
except ImportError:
    try:
        from neutron.db.firewall.firewall_router_insertion_db \
            import FirewallRouterAssociation
    except ImportError:
        pass
try:
    from neutron_vpnaas.db.vpn import vpn_db
except ImportError:
    try:
        from neutron.db.vpn import vpn_db
    except ImportError:
        pass
try:
    from neutron.db.ipv6_ns_qos_policy_db import Ipv6NsQoSPolicyDbMixin
except ImportError:
    class Ipv6NsQoSPolicyDbMixin(object):
        pass

LOG = ncu.ac_log.getLogger(__name__)

TIME_FORMAT_WHOLE_SECONDS = '%Y-%m-%dT%H:%M:%S'


def _extend_resource_dict_timestamp(resource_res, resource_db):
    if (resource_db and resource_db.created_at and
            resource_db.updated_at):
        resource_res['created_at'] = \
            (resource_db.created_at.strftime(TIME_FORMAT_WHOLE_SECONDS)) + 'Z'
        resource_res['updated_at'] = \
            (resource_db.updated_at.strftime(TIME_FORMAT_WHOLE_SECONDS)) + 'Z'


def query_all_exroutes_with_condition(session, condition=None):
    """查询所有的exroute，包括fps的exroute和huawei_ac_ml2的exroute

    :param session: session
    :param condition: condition
    :return: Iterable[RouterExRoute]
    """
    try:
        from networking_cascading.db.exroutes_db import RouterExRoute
        for elem in session.query(RouterExRoute).filter_by(**condition):
            yield elem
    except ImportError:
        pass

    from networking_huawei.drivers.ac.db.exroute.exroute import RouterExRoute
    for elem in session.query(RouterExRoute).filter_by(**condition):
        yield elem


class SecurityGroupDBManager(SecurityGroupDbMixin, CommonDbMixin):
    """Security group DB manager."""
    pass


class L3DBManager(L3_NAT_dbonly_mixin, CommonDbMixin):
    """L3 DB manager."""

    def _make_router_dict(self, router, fields=None,
                          process_extensions=True):
        router_dict = super(L3DBManager, self). \
            _make_router_dict(router, fields, process_extensions)
        if TimeStamp_db_mixin:
            if ncu.get_ops_version() in {ac_constants.OPS_P, ac_constants.OPS_Q,
                                         ac_constants.OPS_R, ac_constants.OPS_T, ac_constants.OPS_W,
                                         ac_constants.FSP_6_5,
                                         ac_constants.FSP_6_5_1,
                                         ac_constants.FSP_8_0_0,
                                         ac_constants.FSP_8_0_3,
                                         ac_constants.FSP_21_0}:
                _extend_resource_dict_timestamp(router_dict, router)
            else:
                TimeStamp_db_mixin().extend_resource_dict_timestamp(
                    None, router_dict, router)
        return router_dict

    def _make_floatingip_dict(self, floatingip, fields=None,
                              process_extensions=True):
        if ncu.get_ops_version() in \
                [ac_constants.OPS_M, ac_constants.OPS_N,
                 ac_constants.OPS_O, ac_constants.OPS_P,
                 ac_constants.OPS_Q, ac_constants.OPS_R, ac_constants.OPS_W,
                 ac_constants.OPS_T, ac_constants.FSP_6_1,
                 ac_constants.FSP_6_3_0, ac_constants.FSP_6_3_1,
                 ac_constants.FSP_6_5, ac_constants.OPS_EZ_M,
                 ac_constants.FSP_6_5_1, ac_constants.FSP_8_0_0,
                 ac_constants.FSP_8_0_3, ac_constants.FSP_21_0]:
            floatingip_dict = super(L3DBManager, self). \
                _make_floatingip_dict(floatingip, fields, process_extensions)
        else:
            floatingip_dict = super(L3DBManager, self). \
                _make_floatingip_dict(floatingip, fields)
        if TimeStamp_db_mixin:
            if ncu.get_ops_version() in [ac_constants.OPS_P,
                                         ac_constants.OPS_Q,
                                         ac_constants.OPS_R,
                                         ac_constants.OPS_W,
                                         ac_constants.FSP_6_5,
                                         ac_constants.OPS_T,
                                         ac_constants.FSP_21_0]:
                _extend_resource_dict_timestamp(floatingip_dict, floatingip)
            else:
                TimeStamp_db_mixin().extend_resource_dict_timestamp(
                    None, floatingip_dict, floatingip)
        return floatingip_dict


def _get_action_types(ctx, group_policy, policy_rule):
    """set_policy_rule call"""
    action_types = []
    if policy_rule.get('policy_actions') is not None:
        policy_action_ids = policy_rule['policy_actions']
        for action_id in policy_action_ids:
            policy_action = group_policy.get_policy_action(ctx, action_id)
            if policy_action:
                action_types.append(policy_action["action_type"])
    return action_types


def _read_pf_core(admin_ctx, fields, filters, floatingip, pf_plugin):
    """read_pf call"""
    result = []
    port_forwardings = pf_plugin. \
        get_floatingip_port_forwardings(admin_ctx, floatingip['id'],
                                        filters=filters)
    for port_forwarding in port_forwardings:
        if fields:
            result.append({'id': port_forwarding['id']})
        else:
            router_id = port_forwarding.get('router_id') or \
                        pf_plugin._find_a_router_for_fip_port_forwarding(
                            admin_ctx, port_forwarding, floatingip)
            result.append({
                'floating_ip_id': port_forwarding['floatingip_id'],
                'router_id': router_id,
                'fixed_ip_address':
                    port_forwarding['internal_ip_address'],
                'protocol': port_forwarding['protocol'],
                'fixed_ip_port': port_forwarding['internal_port'],
                'tenant_id': floatingip['tenant_id'],
                'floating_ip_port': port_forwarding['external_port'],
                'status': "ACTIVE",
                'port_id': port_forwarding['internal_port_id'],
                'id': port_forwarding['id']
            })
    return result


class ACNeutronFormatterDBMixin(object):
    """ 提供sqlalchemy原生db查询接口替代neutron提供的接口 """

    prefix_matching = False
    suffix_matching = False
    white_black_network_ids_cache = {}
    port_bindings_cache = {}

    @classmethod
    def cleanup(cls):
        cls.white_black_network_ids_cache.clear()
        cls.port_bindings_cache.clear()

    @classmethod
    def _get_network_ids(cls, admin_ctx, network_name_list):
        if not network_name_list:
            return list()

        query = admin_ctx.session.query(Network)
        conds = list()
        for network in network_name_list:
            if cls.prefix_matching:
                conds.append(Network.name.like('{}%'.format(network)))
            elif cls.suffix_matching:
                conds.append(Network.name.like('%{}'.format(network)))
        query = query.filter(sqlalchemy.or_(*conds))
        return [network['id'] for network in query.all()]

    @classmethod
    def _get_network_conditions(cls, admin_ctx, res_type):
        list_matching = cfg.CONF.huawei_ac_config.network_list_matching
        cls.prefix_matching = list_matching == 'prefix'
        cls.suffix_matching = list_matching == 'suffix'

        network_black_list = cfg.CONF.huawei_ac_config.network_black_list
        cls._refresh_network_list(network_black_list)
        network_white_list = cfg.CONF.huawei_ac_config.network_white_list
        cls._refresh_network_list(network_white_list)

        if 'white_networks' not in cls.white_black_network_ids_cache:
            white_network_list = ACNeutronFormatterDBMixin._get_network_ids(admin_ctx, network_white_list)
            cls.white_black_network_ids_cache['white_networks'] = white_network_list
        if 'black_networks' not in cls.white_black_network_ids_cache:
            black_network_list = ACNeutronFormatterDBMixin._get_network_ids(admin_ctx, network_black_list)
            cls.white_black_network_ids_cache['black_networks'] = black_network_list
        white_network_list = cls.white_black_network_ids_cache.get('white_networks')
        black_network_list = cls.white_black_network_ids_cache.get('black_networks')

        cond_id = None
        if res_type == ac_constants.NW_HW_NETWORKS:
            cond_id = Network.id
        if res_type == ac_constants.NW_HW_SUBNETS:
            cond_id = Subnet.network_id
        elif res_type == ac_constants.NW_HW_PORTS:
            cond_id = Port.network_id

        conditions = list()
        if cond_id is None:
            return True, conditions

        if white_network_list:
            for network_id in white_network_list:
                conditions.append(cond_id == network_id)
            return False, conditions

        for network_id in black_network_list:
            conditions.append(cond_id != network_id)
        return True, conditions

    @classmethod
    def _refresh_network_list(cls, network_list):
        for i, _ in enumerate(network_list):
            if "*" not in network_list[i]:
                continue
            if cls.suffix_matching:
                network_list[i] = network_list[i].split("*")[-1]
            elif cls.prefix_matching:
                network_list[i] = network_list[i].split("*")[0]

    @staticmethod
    def _make_network(query_network):
        if query_network:
            return {
                'id': query_network[0],
                'name': query_network[1],
                'tenant_id': query_network[2],
                'status': query_network[3],
                'created_at': query_network[4],
                'updated_at': query_network[5],
            }
        return dict()

    @staticmethod
    def query_networks(admin_ctx, filters=None):
        """用于ACNeutronFormatter.read_networks

        :param admin_ctx: ctx
        :param filters: filters
        :return: list
        """
        attributes = (Network.id, Network.name, Network.tenant_id, Network.status,
                      ACResourceAttributes.create_time, ACResourceAttributes.update_time)
        query = admin_ctx.session.query(*attributes).outerjoin(
            ACResourceAttributes, ACResourceAttributes.uuid == Network.id)

        # 过滤黑白名单
        query = ACNeutronFormatterDBMixin._add_network_filters(admin_ctx, query, ac_constants.NW_HW_NETWORKS)

        # filter过滤
        if filters and 'id' in filters:
            if isinstance(filters['id'], list):
                query = query.filter(Network.id.in_(filters['id']))
            else:
                query = query.filter(Network.id == filters['id'])

        networks = query.all()
        return [ACNeutronFormatterDBMixin._make_network(network) for network in networks]

    @staticmethod
    def _make_subnet(query_subnet):
        if query_subnet:
            return {
                'id': query_subnet[0],
                'name': query_subnet[1],
                'tenant_id': query_subnet[2],
                'created_at': query_subnet[3],
                'updated_at': query_subnet[4]
            }
        return dict()

    @staticmethod
    def query_subnets(admin_ctx, filters=None):
        """用于ACNeutronFormatter.read_subnets

        :param admin_ctx: ctx
        :param filters: filters
        :return: list
        """
        attributes = (Subnet.id, Subnet.name, Subnet.tenant_id,
                      ACResourceAttributes.create_time, ACResourceAttributes.update_time)
        query = admin_ctx.session.query(*attributes).outerjoin(
            ACResourceAttributes, ACResourceAttributes.uuid == Subnet.id)

        # 过滤黑白名单
        query = ACNeutronFormatterDBMixin._add_network_filters(admin_ctx, query, ac_constants.NW_HW_SUBNETS)

        # filter过滤
        if filters and 'id' in filters:
            if isinstance(filters['id'], list):
                query = query.filter(Subnet.id.in_(filters['id']))
            else:
                query = query.filter(Subnet.id == filters['id'])

        subnets = query.all()
        return [ACNeutronFormatterDBMixin._make_subnet(subnet) for subnet in subnets]

    @classmethod
    def query_port_bindings(cls, admin_ctx):
        """用于ACNeutronFormatter.read_ports

        :param admin_ctx: ctx
        :return: list
        """
        if not cls.port_bindings_cache:
            portbindings = admin_ctx.session.query(PortBinding).all()
            data = {binding.get('port_id'): binding for binding in portbindings}
            cls.port_bindings_cache.update(data)
        return cls.port_bindings_cache

    @staticmethod
    def _make_port(query_port, trunk):
        if query_port:
            ret = {
                'id': query_port[0],
                'name': query_port[1],
                'tenant_id': query_port[2],
                'network_id': query_port[3],
                'device_id': query_port[4],
                'device_owner': query_port[5],
                'status': query_port[6],
                'created_at': query_port[7],
                'updated_at': query_port[8],

            }
            if trunk:
                # Just to indicate that this port is a parent port
                ret['trunk_details'] = trunk
            return ret
        return dict()

    @staticmethod
    def query_ports(admin_ctx, filters=None):
        """用于ACNeutronFormatter.read_ports

        :param admin_ctx: ctx
        :param filters: filters
        :return: list
        """
        # 过滤network:router_interface类型的port
        attributes = (Port.id, Port.name, Port.tenant_id, Port.network_id, Port.device_id,
                      Port.device_owner, Port.status,
                      ACPortAttributes.create_time, ACPortAttributes.update_time)
        query = admin_ctx.session.query(*attributes).outerjoin(
            ACPortAttributes, ACPortAttributes.uuid == Port.id).filter(
            Port.device_owner != ncu.DEVICE_OWNER_ROUTER_INTF)

        # 过滤黑白名单
        query = ACNeutronFormatterDBMixin._add_network_filters(admin_ctx, query, ac_constants.NW_HW_PORTS)

        # filter过滤
        if filters and 'network_id' in filters:
            query = query.filter(Port.network_id == filters['network_id'])
        if filters and 'device_owner' in filters:
            query = query.filter(Port.device_owner.in_(filters['device_owner']))
        if filters and 'id' in filters:
            if isinstance(filters['id'], list):
                query = query.filter(Port.id.in_(filters['id']))
            else:
                query = query.filter(Port.id == filters['id'])

        ports = query.all()

        trunk_dict = dict()
        try:
            from neutron.objects.trunk.models import Trunk
            trunks = admin_ctx.session.query(Trunk).all()
            trunk_dict.update({trunk.port_id: trunk for trunk in trunks})
        except ImportError:
            pass
        return [ACNeutronFormatterDBMixin._make_port(port, trunk_dict.get(port[0])) for port in ports]

    @staticmethod
    def query_ports_router_interface(admin_ctx, filters=None):
        """用于ACNeutronFormatter.read_router_if

        :param admin_ctx: ctx
        :param filters: filters
        :return: list
        """
        attributes = (Port.id, Port.name, Port.tenant_id, Port.network_id, Port.device_id,
                      Port.device_owner, Port.status,
                      ACPortAttributes.create_time, ACPortAttributes.update_time)
        query = admin_ctx.session.query(*attributes).outerjoin(
            ACPortAttributes, ACPortAttributes.uuid == Port.id).filter(
            Port.device_owner == ncu.DEVICE_OWNER_ROUTER_INTF)

        # 过滤黑白名单
        query = ACNeutronFormatterDBMixin._add_network_filters(admin_ctx, query, ac_constants.NW_HW_PORTS)

        # filter过滤
        if filters and 'id' in filters:
            if isinstance(filters['id'], list):
                query = query.filter(Port.id.in_(filters['id']))
            else:
                query = query.filter(Port.id == filters['id'])

        ports = query.all()
        return [ACNeutronFormatterDBMixin._make_port(port, None) for port in ports]

    @staticmethod
    def _add_network_filters(admin_ctx, query, res_type):
        _and, conds = ACNeutronFormatterDBMixin._get_network_conditions(admin_ctx, res_type)
        if _and:
            query = query.filter(sqlalchemy.and_(*conds))
        else:
            query = query.filter(sqlalchemy.or_(*conds))
        return query


class ACNeutronDataFormatter(ACNeutronFormatterDBMixin):
    """Implementation of the neutron data formatter."""

    def __init__(self):
        self._db_if = ACdbInterface()
        self.enable_snat = cfg.CONF.huawei_ac_config.enable_snat
        self.data_filter = DataFilterUtil()
        self.firewall_rule_addrs_func = ACFirewallruleAddrsDbMixin()
        if hasattr(Ipv6NsQoSPolicyDbMixin, 'get_ipv6_ns_qos_policies'):
            self.ipv6_ns_qos_policy = Ipv6NsQoSPolicyDbMixin()
        else:
            self.ipv6_ns_qos_policy = None
        self.fm_db = FlowMirrorDbMixin()

    def read_routers(self, admin_ctx, filters=None, fields=None):
        """Read router data."""
        LOG.debug("[AC] Begin to read router data: %s, %s", filters, fields)
        l3_db = L3DBManager()
        resources = l3_db.get_routers(
            admin_ctx, filters=filters, fields=None)
        routers = copy.deepcopy(resources)
        for router in routers:
            if self.data_filter.not_in_white_or_in_black_list(
                    admin_ctx, {'router': router}, ac_constants.NW_HW_ROUTERS,
                    router.get('id')):
                resources.remove(router)
        LOG.debug("Read routers data: %s", resources)
        return resources

    def set_router_data(self, ctx, tenant_name, router_id=None, resource_data=None):
        """Set router data."""
        LOG.debug("[AC] Begin to set router data: %s, %s, %s", tenant_name, router_id, resource_data)
        if router_id:
            resource_data = self._db_if.get_neutron_record(ctx.session, router_id, ac_constants.NW_HW_ROUTERS)

        if not resource_data:
            return {}

        router_extra = self._db_if.get_neutron_record(
            ctx.session, resource_data['id'],
            ac_constants.NW_HW_ROUTER_EXTRA_ATTRIBUTES)

        extra_route = ExtraRoute_dbonly_mixin()
        router_routes = extra_route._get_extra_routes_by_router_id(ctx, resource_data['id'])

        router_dict = resource_data

        if resource_data.gw_port_id:
            ext_fixed_ips = [{'subnet_id': ip["subnet_id"], 'ip_address': ip["ip_address"]}
                             for ip in resource_data.gw_port['fixed_ips']]
            ext_gw_info = {'network_id': resource_data.gw_port['network_id'], 'external_fixed_ips': ext_fixed_ips}

            if not self.enable_snat:
                ext_gw_info['enable_snat'] = False
            else:
                ext_gw_info['enable_snat'] = resource_data.enable_snat
            router_dict['external_gateway_info'] = ext_gw_info

        router_dict['distributed'] = router_extra.distributed
        router_dict['routes'] = router_routes
        if hasattr(router_dict, 'standard_attr'):
            router_dict['created_at'] = router_dict.standard_attr.created_at.strftime(ac_constants.ISO8601_TIME_FORMAT)
            router_dict['updated_at'] = router_dict.standard_attr.updated_at.strftime(ac_constants.ISO8601_TIME_FORMAT)

        ac_router_dict = router_model.ACRouterModel.ac_model_format(router_dict, tenant_name)
        LOG.info(_LI("[AC] Router is dependent, router: %s"), ac_router_dict)
        return ac_router_dict

    def read_networks(self, admin_ctx, filters=None, fields=None):
        """Read network data."""
        LOG.debug("[AC] Begin to read network data, filters: %s, fields: %s", filters, fields)
        resources = self.query_networks(admin_ctx, filters)
        LOG.debug("[AC] Read networks end, len: %s", len(resources))
        return resources

    def set_network_data(self, ctx, tenant_name, network_id=None, resource_data=None):
        """Set network data."""
        LOG.debug("[AC] Begin to set network data: %s, %s, %s", tenant_name, network_id, resource_data)
        if network_id:
            try:
                resource_data = ncu.get_core_plugin().get_network(ctx, network_id)
            except Exception as e:
                LOG.error('[AC] Get network by id %s failed, error: %s', network_id, str(e))
                return {}

        if not resource_data:
            return {}

        LOG.debug("Resource data: %s", resource_data)
        tenant_name = ACKeyStoneIf.get_tenant_name(resource_data, ctx)
        ac_network_dict = network_model.ACNetworkModel.ac_model_format(resource_data, tenant_name)
        LOG.info(_LI("[AC] Network is dependent, network: %s"), ac_network_dict)
        return ac_network_dict

    def read_subnets(self, admin_ctx, filters=None, fields=None):
        """Read subnet data."""
        LOG.debug("[AC] Begin to read subnet data, filters: %s, fields: %s", filters, fields)
        resources = self.query_subnets(admin_ctx, filters)
        LOG.debug("[AC] Read subnets end, len: %s", len(resources))
        return resources

    def set_subnet_data(self, ctx, tenant_name, subnet_id=None,
                        resource_data=None):
        """Set subnet data."""
        LOG.debug("[AC] Begin to set subnet data: %s, %s, %s",
                  tenant_name, subnet_id, resource_data)
        if subnet_id:
            resource_data = ncu.get_core_plugin().get_subnet(ctx, subnet_id)
        if not resource_data:
            return {}

        LOG.debug("Resource data: %s", resource_data)
        tenant_name = ACKeyStoneIf.get_tenant_name(resource_data, ctx)
        ac_subnet_dict = subnet_model.ACSubnetModel. \
            ac_model_format(resource_data, tenant_name)
        LOG.info(_LI("[AC] Subnet is dependent, subnet: %s"), ac_subnet_dict)
        return ac_subnet_dict

    def read_ports(self, admin_ctx, filters=None, fields=None):
        """Read port data."""
        LOG.debug("[AC] Begin to read port data, filters: %s, fields: %s", filters, fields)
        resources = self.query_ports(admin_ctx, filters)

        port_binding_dict = self.query_port_bindings(admin_ctx)

        # 过滤掉相关port
        ports = copy.deepcopy(resources)
        for port in ports:
            port_id = port.get('id')
            port_binding = port_binding_dict.get(port_id)
            if not port_binding:
                port_info = dict()
            else:
                port_info = {
                    'binding:profile': json.loads(port_binding.get('profile')) if port_binding.get(
                        'profile') else dict()
                }
            if self.data_filter.port_contain_anti_affinity(port_info):
                resources.remove(port)
                continue

            # port for dryRun, not compare to controller.
            port_description = port_info.get('description')
            if port_description and port_description.find('__resource_preemption') != -1:
                resources.remove(port)
        LOG.debug("[AC] Read ports end, len: %s", len(resources))
        return resources

    def set_port_data(self, ctx, tenant_name, port_id=None,
                      resource_data=None):
        """Set port data."""
        LOG.debug("[AC] Begin to set port data: %s, %s, %s",
                  tenant_name, port_id, resource_data)
        if port_id:
            resource_data = ncu.get_core_plugin().get_port(ctx, port_id)

        if not resource_data:
            return {}
        LOG.debug("Resource data: %s", resource_data)
        tenant_name = ACKeyStoneIf.get_tenant_name(resource_data, ctx)
        self.get_hw_bond_mode(resource_data, port_id)

        ac_port_dict = port_model.ACPortModel. \
            ac_model_format(resource_data, tenant_name)

        LOG.info(_LI("[AC] Port is dependent, port: %s"), ac_port_dict)
        return ac_port_dict

    def read_router_if(self, admin_ctx, filters=None, fields=None):
        """Read router interface data."""
        LOG.debug("[AC] Begin to read router_if data, filters: %s, fields: %s", filters, fields)
        resources = self.query_ports_router_interface(admin_ctx, filters)
        LOG.debug("[AC] Read router_ifs end, len: %s", len(resources))
        return resources

    def set_router_if(self, ctx, tenant_name, port_id=None, resource_data=None):
        """Set router interface data."""
        LOG.info("[AC] Begin to set router interface data: %s, %s, %s", tenant_name, port_id,
                 resource_data)
        if port_id:
            resource_data = ncu.get_core_plugin().get_port(ctx, port_id)

        if not resource_data:
            return {}
        LOG.info("Resource router interface data: %s", resource_data)
        tenant_name = ACKeyStoneIf.get_tenant_name(resource_data, ctx)

        ac_port_dict = port_model.ACPortModel. \
            ac_model_format(resource_data, tenant_name)

        LOG.info(_LI("[AC] router interface is dependent,  router interface : %s"), ac_port_dict)
        return ac_port_dict

    def read_bgp_route(self, admin_ctx, filters=None, fields=None):
        """Read bgp route data."""
        try:
            LOG.debug("[AC] Begin to read bgp route data: %s, %s", filters, fields)
            bgp_route_db = BgpRouteDbMixin()
            bgp_route_list = bgp_route_db.get_db_bgp_routes(admin_ctx, filters=filters, fields=fields)
            res_list = admin_ctx.session.query(ACResourceAttributes).filter(
                ACResourceAttributes.res_type == ac_constants.NW_HW_BGP_ROUTE).all()
            res_dict = {res.uuid: res for res in res_list}
            for bgp_route in bgp_route_list:
                res = res_dict.get(bgp_route['id'])
                if res:
                    bgp_route['created_at'] = res.create_time
                    bgp_route['updated_at'] = res.update_time
            LOG.debug("[AC] Read bgp route len: %s", len(bgp_route_list))
        except Exception as ex:
            LOG.error("[AC] Failed to read bgp route: %s", ex)
            return []
        return bgp_route_list

    def set_bgp_route(self, ctx, tenant_name, bgp_route_id=None,
                      resource_data=None):
        """Set BGP route data."""
        LOG.debug("[AC] Begin to set BGP route data: %s, %s", tenant_name, bgp_route_id)
        if bgp_route_id:
            bgp_route_db = BgpRouteDbMixin()
            resource_data = bgp_route_db.get_db_bgp_route(ctx, bgp_route_id)
        if not resource_data:
            return {}
        bgp_route_dict = bgp_route_model.ACBgpRouteModel.ac_model_format(resource_data)
        return bgp_route_dict

    def read_exroute(self, admin_ctx, filters=None, fields=None):
        """Read exroute data."""
        LOG.debug("[AC] Begin to read exroute data: %s, %s", filters, fields)
        if filters:
            filter_attr = filters
        else:
            filter_attr = {}
        exroutes_db = list(query_all_exroutes_with_condition(admin_ctx.session, condition=filter_attr))
        exroutes = copy.deepcopy(exroutes_db)
        LOG.debug("Read exroutes: %s", exroutes)
        if not ncu.IS_FSP:
            return exroutes
        exroutes_result = []
        for exroute in exroutes:
            exroute['destination'] = exroute['destination'].upper()
            exroute['nexthop'] = exroute['nexthop'].upper()

            # exroute for dryRun, not compare to controller.
            exroute_description = exroute.get('description')
            if exroute_description and exroute_description.find('__resource_preemption') != -1:
                continue

            exroutes_result.append(exroute)
        return exroutes_result

    def set_exroute(self, ctx, tenant_name, router_id=None, attribute=None):
        """Set exroute data."""
        LOG.debug("[AC] Begin to set exroute data: %s, %s, %s",
                  tenant_name, router_id, attribute)
        resource_data = None
        if router_id and attribute:
            destination_tmp = netaddr.IPNetwork(attribute['destination'])
            destination = '%s/%s' % (destination_tmp.network,
                                     destination_tmp.prefixlen)
            nexthop = str(netaddr.IPAddress(attribute['nexthop']))
            filter_attr = {
                'router_id': router_id,
                'nexthop': nexthop,
                'destination': destination}
            try:
                resource_data = next(query_all_exroutes_with_condition(ctx.session, condition=filter_attr))
            except StopIteration as ex:
                LOG.error("can not find exroute data,%s", ex)

        if resource_data:
            interface_id = ACCommonUtil.get_exroute_interface_id(
                ctx, attribute['nexthop'], attribute['destination'])
            exroute_dict = exroute_model.ACExRouteModel.ac_model_format(resource_data, resource_data.router_id,
                                                                        interface_id=interface_id)
        else:
            exroute_dict = {}
        LOG.info(_LI("[AC] Exroute is dependent, exroute: %s"),
                 exroute_dict)
        return exroute_dict

    def set_policy_classifier(self, ctx, tenant_name, classifier_id=None):
        """Set policy classifier data."""
        try:
            LOG.debug("[AC] Begin to set policy classifier data: %s, %s",
                      tenant_name, classifier_id)
            plugins = manager.NeutronManager.get_service_plugins()
            grouppolicy_plugin = plugins.get("GROUP_POLICY")
            classifier = grouppolicy_plugin.get_policy_classifier(
                ctx, classifier_id)
            classifier_to_ac = ACPolicyClassifierModel.ac_model_format(
                classifier, tenant_name)
            LOG.debug("Resource data: %s", classifier_to_ac)
            return classifier_to_ac
        except Exception as ex:
            LOG.error("when get classifier catch %s", ex)
            return {}

    def set_policy_rule(self, ctx, tenant_name, policy_rule_id=None):
        """Set policy rule data."""
        try:
            LOG.debug("[AC] Begin to set policy rule data: %s, %s",
                      tenant_name, policy_rule_id)
            plugins = manager.NeutronManager.get_service_plugins()
            grouppolicy_plugin = plugins.get("GROUP_POLICY")
            policy_rule = grouppolicy_plugin.get_policy_rule(
                ctx, policy_rule_id)
            action_types = _get_action_types(ctx, grouppolicy_plugin,
                                             policy_rule)
            policy_rule_to_ac = ACPolicyRuleModel.ac_model_format(
                policy_rule, action_types)
            LOG.debug("Resource data: %s", policy_rule_to_ac)
            return policy_rule_to_ac
        except Exception as ex:
            LOG.error("when read policy rules catch %s", ex)
            return {}

    def set_policy_rule_set(self, ctx, tenant_name, policy_rule_set_id=None):
        """Set policy rule set data."""
        try:
            LOG.debug("[AC] Begin to set policy rule set data: %s, %s",
                      tenant_name, policy_rule_set_id)
            plugins = manager.NeutronManager.get_service_plugins()
            grouppolicy_plugin = plugins.get("GROUP_POLICY")
            rule_set = grouppolicy_plugin. \
                get_policy_rule_set(ctx, policy_rule_set_id)
            policy_rule_set_to_ac = ACPolicyRuleSetModel. \
                ac_model_format(rule_set)
            LOG.debug("Resource data: %s", policy_rule_set_to_ac)
            return policy_rule_set_to_ac
        except Exception as ex:
            LOG.error("when read policy rule sets catch %s", ex)
            return {}

    def set_policy_target_group(
            self, ctx, tenant_name, policy_target_group_id=None):
        """Set policy target group data."""
        try:
            LOG.debug("[AC] Begin to set policy target group data: %s, %s",
                      tenant_name, policy_target_group_id)
            plugins = manager.NeutronManager.get_service_plugins()
            grouppolicy_plugin = plugins.get("GROUP_POLICY")
            policy_target_group = grouppolicy_plugin. \
                get_policy_target_group(ctx, policy_target_group_id)
            target_group_to_ac = ACTargetGroupModel.ac_model_format(
                policy_target_group)
            LOG.debug("Resource data: %s", target_group_to_ac)
            return target_group_to_ac
        except Exception as ex:
            LOG.error("when read policy target groups catch %s", ex)
            return {}

    def set_external_segment(self, ctx, tenant_name, external_segment_id=None):
        """Set external segment data."""
        try:
            LOG.debug("[AC] Begin to set external segment data: %s, %s",
                      tenant_name, external_segment_id)
            plugins = manager.NeutronManager.get_service_plugins()
            group_policy_plugin = plugins.get("GROUP_POLICY")
            external_segment = group_policy_plugin. \
                get_external_segment(ctx, external_segment_id)
            external_segment_to_ac = \
                ACExternalSegmentModel.ac_model_format(external_segment)
            LOG.debug("Resource data: %s", external_segment_to_ac)
            return external_segment_to_ac
        except Exception as ex:
            LOG.error(_LE("[AC] Failed to set external segment: %s", ex))
            return {}

    def set_external_policy(self, ctx, tenant_name, external_policy_id=None):
        """Set external policy data."""
        try:
            LOG.debug("[AC] Begin to set external policy data: %s, %s",
                      tenant_name, external_policy_id)
            plugins = manager.NeutronManager.get_service_plugins()
            group_policy_plugin = plugins.get("GROUP_POLICY")
            external_policy = group_policy_plugin. \
                get_external_policy(ctx, external_policy_id)
            external_policy_to_ac = \
                ACExternalPolicyModel.ac_model_format(external_policy)
            LOG.debug("Resource data: %s", external_policy_to_ac)
            return external_policy_to_ac
        except Exception as ex:
            LOG.error(_LE("[AC] Failed to set external policy: %s", ex))
            return {}

    def get_hw_bond_mode(self, port_info, port_id):
        """get bond mode"""
        binding_profile = port_info.get('binding:profile')
        if not binding_profile:
            return None
        local_link = binding_profile.get('local_link_information')
        if not local_link or len(local_link) <= 1:
            return None
        local_group_information = \
            binding_profile.get('local_group_information', {})
        if binding_profile.get('bond_mode'):
            bond_mode = binding_profile.get('bond_mode')
        elif local_group_information.get('bond_mode'):
            bond_mode = local_group_information['bond_mode']
        else:
            bond_mode = self._db_if.get_bm_bond_mode(None, port_id)
        if bond_mode:
            if bond_mode not in ac_constants.BOND_MODE_LIST:
                bond_mode = ac_constants.BOND_MODE_LIST[int(bond_mode)]
            binding_profile['mode'] = bond_mode
        return None

    def set_security_group_data(self, ctx, tenant_name, security_group_id=None,
                                resource_data=None):
        """Set security group data."""
        LOG.debug("[AC] Begin to set security group data: %s, %s, %s",
                  tenant_name, security_group_id, resource_data)
        if security_group_id:
            resource_data = self._db_if.get_neutron_record(
                ctx.session, security_group_id, ac_constants.NW_HW_SEC_GRP)
        if not resource_data or resource_data['tenant_id'] is None:
            return {}

        if hasattr(resource_data, 'standard_attr'):
            resource_data['created_at'] = resource_data.standard_attr. \
                created_at.strftime(ac_constants.ISO8601_TIME_FORMAT)
            resource_data['updated_at'] = resource_data.standard_attr. \
                updated_at.strftime(ac_constants.ISO8601_TIME_FORMAT)

        ac_security_group_dict = security_group_model.ACSecurityGroupModel. \
            ac_model_format(resource_data)

        LOG.info(_LI('[AC] Security group is dependented, '
                     'security group: %s'), ac_security_group_dict)
        return ac_security_group_dict

    def set_security_group_rule_data(self, ctx, tenant_name,
                                     security_group_rule_id=None,
                                     resource_data=None):
        """Set security group rule data."""
        LOG.debug("[AC] Begin to set security group rule data: %s, %s, %s",
                  tenant_name, security_group_rule_id, resource_data)
        if security_group_rule_id:
            resource_data = self._db_if.get_neutron_record(
                ctx.session, security_group_rule_id,
                ac_constants.NW_HW_SEC_GRP_RULE)

        if not resource_data or resource_data['tenant_id'] is None:
            return {}

        security_group_rule_dict = \
            security_group_model.ACSecurityGroupRuleModel.ac_model_format(
                resource_data)

        LOG.info(_LI('[AC] Security group rule dict: %s'),
                 security_group_rule_dict)
        return security_group_rule_dict

    def set_ike_policy(self, ctx, tenant_name, ike_policy_id=None,
                       resource_data=None):
        """Set IKE policy data."""
        LOG.debug("[AC] Begin to set IKE policy data: %s, %s, %s",
                  tenant_name, ike_policy_id, resource_data)
        if ike_policy_id:
            resource_data = self._db_if.get_neutron_record(
                ctx.session, ike_policy_id,
                ac_constants.NW_HW_IKE_POLICY)
        if not resource_data:
            return {}

        ike_policy_info = resource_data
        ike_policy_info['lifetime'] = {
            'units': resource_data['lifetime_units'],
            'value': resource_data['lifetime_value']}

        tenant_name = ACKeyStoneIf.get_tenant_name(resource_data, ctx)
        ac_ike_policy_dict = ipsecvpn_model.ACIKEpolicyModel. \
            ac_model_format(ike_policy_info, tenant_name)
        LOG.info(_LI("[AC] Ike policy is dependent, ike policy: %s"),
                 ac_ike_policy_dict)
        return ac_ike_policy_dict

    def set_ipsec_policy(self, ctx, tenant_name, ipsec_policy_id=None,
                         resource_data=None):
        """Set IPSec policy data."""
        LOG.debug("[AC] Begin to set IPSec policy data: %s, %s, %s",
                  tenant_name, ipsec_policy_id, resource_data)
        if ipsec_policy_id:
            resource_data = self._db_if.get_neutron_record(
                ctx.session, ipsec_policy_id,
                ac_constants.NW_HW_IPSEC_POLICY)
        if not resource_data:
            return {}
        ipsec_policy_info = resource_data
        ipsec_policy_info['lifetime'] = {
            'units': resource_data['lifetime_units'],
            'value': resource_data['lifetime_value']}

        tenant_name = ACKeyStoneIf.get_tenant_name(resource_data, ctx)
        ac_ipsec_policy_dict = ipsecvpn_model.ACIPSecpolicyModel. \
            ac_model_format(ipsec_policy_info, tenant_name)
        LOG.info(_LI("[AC] Ipsec policy is dependent, ipsec policy: %s"),
                 ac_ipsec_policy_dict)
        return ac_ipsec_policy_dict

    def set_vpn_service(self, ctx, tenant_name, vpn_service_id=None,
                        resource_data=None):
        """Set VPN service data."""
        LOG.debug("[AC] Begin to set VPN service data: %s, %s, %s",
                  tenant_name, vpn_service_id, resource_data)
        if vpn_service_id:
            resource_data = self._db_if.get_neutron_record(
                ctx.session, vpn_service_id,
                ac_constants.NW_HW_VPN_SERVICE)
        if not resource_data:
            return {}
        tenant_name = ACKeyStoneIf.get_tenant_name(resource_data, ctx)
        ac_vpn_service_dict = ipsecvpn_model.ACVPNServiceModel. \
            ac_model_format(resource_data, tenant_name)
        LOG.info(_LI("[AC] Vpn service is dependent, vpn service: %s"),
                 ac_vpn_service_dict)
        return ac_vpn_service_dict

    def set_endpoint_group(self, ctx, tenant_name, ep_group_id=None,
                           resource_data=None):
        """Set endpoint group data."""
        LOG.debug("[AC] Begin to set endpoint group data: %s, %s, %s",
                  tenant_name, ep_group_id, resource_data)
        if ep_group_id:
            vpn_interface = vpn_db.VPNPluginDb()
            resource_data = vpn_interface.get_endpoint_group(ctx, ep_group_id)
        if not resource_data:
            return {}
        ac_endpoint_group_dict = ipsecvpn_model.ACEndPointGroupModel. \
            ac_model_format(resource_data, tenant_name)
        LOG.info(_LI("[AC] Endpoint group is dependent, endpoint group: %s"),
                 ac_endpoint_group_dict)
        return ac_endpoint_group_dict

    def set_ipsec_site_connection(self, ctx, tenant_name, site_conn_id=None,
                                  resource_data=None):
        """Set IPSec site connection data."""
        LOG.debug("[AC] Begin to set IPSec site connection data: %s, %s", tenant_name, site_conn_id)
        if site_conn_id:
            resource_data = self._db_if.get_neutron_record(
                ctx.session, site_conn_id, ac_constants.NW_HW_IPSEC_SITE_CONNECTION)
        if not resource_data:
            return {}
        ipsec_site_conn_info = resource_data
        ipsec_site_conn_info['dpd'] = {"action": resource_data.dpd_action, "interval": resource_data.dpd_interval,
                                       "timeout": resource_data.dpd_timeout}
        tenant_name = ACKeyStoneIf.get_tenant_name(resource_data, ctx)
        ac_ipsec_site_conn_dict = ipsecvpn_model.ACIPSecSiteConnModel.ac_model_format(ipsec_site_conn_info, tenant_name)
        # 'peer-cidrs' is not mandatory field of ACIPSecSiteConnModel
        if 'peer-cidrs' in ac_ipsec_site_conn_dict:
            peer_cidrs = ac_ipsec_site_conn_dict.get('peer-cidrs', [])
            ac_ipsec_site_conn_dict['peer-cidrs'] = [pcidr['cidr'] for pcidr in peer_cidrs]

        try:
            if ncu.IS_FSP and 'psk' in ac_ipsec_site_conn_dict:
                ac_ipsec_site_conn_dict['psk'] = crypt.decrypt(ac_ipsec_site_conn_dict.get('psk'))
        except Exception as e:
            LOG.error('set_ipsec_site_connection occur error.%s', e)
        ipsec_conn_log = copy.deepcopy(ac_ipsec_site_conn_dict)
        ipsec_conn_log.pop('psk', None)
        LOG.info(_LI("[AC] Ipsec site connection is dependented, Ipsec site connection: %s"), ipsec_conn_log)
        return ac_ipsec_site_conn_dict

    def set_firewall(self, ctx, tenant_name, firewall_id=None, resource_data=None):
        """Set firewall data."""
        LOG.debug("[AC] Begin to set firewall data: %s, %s, %s", tenant_name, firewall_id, resource_data)
        if firewall_id:
            resource_data = self._db_if.get_neutron_record(ctx.session, firewall_id, ac_constants.NW_HW_FIREWALL)
        if not resource_data:
            return {}

        with ctx.session.begin(subtransactions=True):
            fw_rtr_qry = ctx.session.query(FirewallRouterAssociation.router_id)
            fw_rtr_rows = fw_rtr_qry.filter_by(fw_id=firewall_id)
            fw_rtrs = [entry.router_id for entry in fw_rtr_rows]

        resource_data['router_ids'] = fw_rtrs
        ac_firewall_dict = firewall_model.ACFirewallModel.ac_model_format(resource_data)
        LOG.info(_LI("[AC] Firewall is dependented, firewall: %s"), ac_firewall_dict)
        return ac_firewall_dict

    def set_firewall_rule(self, ctx, tenant_name, firewall_rule_id=None, resource_data=None):
        """Set firewall rule data."""
        LOG.debug("[AC] Begin to set firewall rule data: %s, %s, %s",
                  tenant_name, firewall_rule_id, resource_data)
        if firewall_rule_id:
            resource_data = FirewallDBMixin().get_firewall_rule(
                ctx, firewall_rule_id)
            if 'huawei_ac_firewall_rule' in cfg.CONF.ml2.extension_drivers \
                    and cfg.CONF.huawei_ac_config.OPS_version in \
                    [ac_constants.OPS_M, ac_constants.OPS_K,
                     ac_constants.OPS_P]:
                new_firewall_rule = self.firewall_rule_addrs_func. \
                    get_db_ac_firewall_rule(firewall_rule_id)

                resource_data['source_ports'] = ''
                resource_data['destination_ports'] = ''
                resource_data['source_ip_addresses'] = ''
                resource_data['destination_ip_addresses'] = ''

                if new_firewall_rule:
                    resource_data['source_ports'] = \
                        new_firewall_rule.get('source_ports', '')
                    resource_data['destination_ports'] = \
                        new_firewall_rule.get('destination_ports', '')
                    resource_data['source_ip_addresses'] = \
                        new_firewall_rule.get('source_ip_addresses', '')
                    resource_data['destination_ip_addresses'] = \
                        new_firewall_rule.get('destination_ip_addresses', '')

        if not resource_data:
            return {}
        ac_firewall_rule_dict = firewall_rule_model.ACFirewallRuleModel. \
            ac_model_format(resource_data)
        LOG.info(_LI("[AC] Firewall rule is dependent, firewall rule: %s"),
                 ac_firewall_rule_dict)
        return ac_firewall_rule_dict

    def set_firewall_policy(self, ctx, tenant_name, firewall_policy_id=None,
                            resource_data=None):
        """Set firewall policy data."""
        LOG.debug("[AC] Begin to set firewall policy data: %s, %s, %s",
                  tenant_name, firewall_policy_id, resource_data)
        if firewall_policy_id:
            resource_data = FirewallDBMixin().get_firewall_policy(
                ctx, firewall_policy_id)
        if not resource_data:
            return {}
        ac_firewall_policy_dict = firewall_policy_model.ACFirewallPolicyModel. \
            ac_model_format(resource_data)
        LOG.info(_LI("[AC] Firewall policy is dependent, firewall policy: %s"),
                 ac_firewall_policy_dict)
        return ac_firewall_policy_dict

    def set_qos_policy_data(self, ctx, tenant_name, policy_id=None,
                            resource_data=None):
        """Set QoS policy data."""
        LOG.debug("[AC] Begin to set QoS policy data: %s, %s, %s",
                  tenant_name, policy_id, resource_data)
        if policy_id:
            filters = {'id': policy_id}
            policy_list = policy_object.QosPolicy.get_objects(ctx, **filters)
            if policy_list:
                resource_data = policy_list[0]
            else:
                LOG.debug("[AC] can not find policy info: %s", policy_id)
                return {}
        if resource_data is None:
            raise n_exc.QosPolicyNotFound(policy_id=policy_id)
        policy_data = qos_model.ACQoSpolicyModel. \
            ac_model_format(resource_data)

        LOG.info(_LI("[AC] QoS policy is dependented, policy: %s."),
                 policy_data)
        return policy_data

    def set_cmcc_qos_policy(self, ctx, tenant_name, policy_id=None, resource_data=None):
        """Set CMCC QoS policy data."""
        LOG.debug("[AC] Begin to set CMCC QoS policy data: %s, %s, %s", tenant_name, policy_id, resource_data)
        policy_data = {}
        try:
            if policy_id:
                import neutron.db.qos_db as cmcc_qos_db
                # because cmcc set qos id as qos policy id
                # when sending data to ac.So here the input
                # policy_id is qos id in fact.
                qos = ctx.session.query(cmcc_qos_db.QoS).filter_by(id=policy_id).first()
                qos_policy = ctx.session.query(cmcc_qos_db.QoSPolicy).filter_by(qos_id=policy_id).first()
                name = "qos_%s" % policy_id.replace('-', '')

                policy_data = {"uuid": qos.id, "name": name[0:25], "tenant-id": qos.tenant_id, }
                rules = policy_data['qos-bandwidth-limit-rules'] = {}
                rate_limit = qos_policy.value
                rules['qos-bandwidth-limit-rule'] = [
                    {"uuid": qos.id, "qos-policy-id": qos.id, "max-kbps": rate_limit, "max-burst-kbps": rate_limit}]
        except Exception as ex:
            LOG.error(_LE("[AC]Read qos data error: %s"), str(ex))
            policy_data = {}
        LOG.info(_LI("[AC] QoS policy is dependented, policy: %s."), policy_data)
        return policy_data

    def set_floating_ip(self, ctx, tenant_name, fip_id=None,
                        resource_data=None):
        """Set floating IP data."""
        LOG.debug("[AC] Begin to set floating IP data: %s, %s, %s",
                  tenant_name, fip_id, resource_data)
        if fip_id:
            resource_data = self._db_if.get_fip_neutron_record(
                ctx.session, fip_id)
        if not resource_data:
            return {}
        if ncu.IS_FSP:
            if 'fixed_ip_address' in resource_data \
                    and resource_data['fixed_ip_address'] is not None:
                l3_db = L3DBManager()
                resources = l3_db.get_floatingips(ctx)
                if self._check_is_flexible_snat(resource_data, resources):
                    return {}
        resource_data['port_id'] = resource_data['fixed_port_id']
        self._set_floating_ip_qos_policy(ctx, resource_data)
        ac_fip_dict = floatingip_model.ACFloatingipModel. \
            ac_model_format(resource_data)
        LOG.info(_LI("[AC] Floating IP data: %s"),
                 ac_fip_dict)
        return ac_fip_dict

    @classmethod
    def _set_floating_ip_qos_policy(cls, ctx, resource_data):
        try:
            from neutron.db.qos.models import QosFIPPolicyBinding
        except ImportError:
            return
        qos_policy_binding = ctx.session.query(QosFIPPolicyBinding).filter(
            QosFIPPolicyBinding.fip_id == resource_data['id']).one_or_none()
        if qos_policy_binding:
            resource_data['qos_policy_id'] = qos_policy_binding.policy_id

    def set_dnat(self, ctx, tenant_name, dnat_id=None, resource_data=None):
        """Set DNAT data."""
        LOG.debug("[AC] Begin to set DNAT data: %s, %s, %s",
                  tenant_name, dnat_id, resource_data)
        if not dnat_id:
            return {}
        resource_data = DNATDBMixin().get_dnat_db(ctx, dnat_id)
        if not resource_data:
            return {}
        floating_ip = L3DBManager().get_floatingip(
            ctx, resource_data['floating_ip_id'])
        if not floating_ip:
            return {}
        dnat_info = dnat_model.ACDNATModel.ac_model_format(
            resource_data, floating_ip)
        LOG.info(_LI('[AC] DNAT data: %s'), dnat_info)
        return dnat_info

    def set_pf(self, ctx, tenant_name, pf_id=None, resource_data=None):
        """Set port forwarding data."""
        LOG.debug("[AC] Begin to set port forwarding data: %s, %s, %s",
                  tenant_name, pf_id, resource_data)
        if not pf_id:
            return {}
        resource_data = None
        dnats = self.read_pf(ctx)
        for dnat in dnats:
            if dnat['id'] == pf_id:
                resource_data = dnat
                break
        if not resource_data:
            return {}
        floating_ip = L3DBManager().get_floatingip(
            ctx, resource_data['floating_ip_id'])
        dnat_info = dnat_model.ACDNATModel.ac_model_format(resource_data,
                                                           floating_ip)
        return dnat_info

    def set_snat(self, ctx, tenant_name, snat_id=None, fip=None):
        """Set SNAT data."""
        LOG.debug("[AC] Begin to set SNAT data: %s, %s, %s",
                  tenant_name, snat_id, fip)
        l3_db = L3DBManager()
        snat_cidrs = []
        if snat_id:
            try:
                fip = l3_db.get_floatingip(ctx, snat_id)
            except Exception:
                fip = None
        if not fip:
            return {}
        LOG.debug("FIP data: %s", fip)
        if fip.get('port_id'):
            neutron_db = neutron_db_ml2.NeutronDbPluginV2()
            port_info = \
                neutron_db.get_port(ctx, fip['port_id'])
            LOG.info(_LI('[AC] Begin to get flexible SNAT cidrs from port: '
                         '%s'), port_info)
            for fixed_ip in port_info.get('fixed_ips'):
                subnet_info = neutron_db.get_subnet(
                    ctx, fixed_ip['subnet_id'])
                LOG.info(_LI('[AC] Begin to get flexible SNAT cidrs from '
                             'subnet: %s'), subnet_info)
                snat_cidrs.append(subnet_info['cidr'])
        if not snat_cidrs:
            return {}
        LOG.info(_LI('[AC] Flexible SNAT cidrs: %s'), snat_cidrs)
        snat_info = \
            snat_model.ACSNATModel.ac_model_format(fip, snat_cidrs)
        return snat_info

    def set_snat_v2(self, ctx, snat_id):
        """Set SNAT data."""
        try:
            snat = SnatDbMixin().get_snat_db(ctx, snat_id)
        except Exception:
            return {}
        snat_info = snat_model.ACSNATModel.ac_model_format_v2(snat)
        LOG.info(_LI('[AC] set snat v2: %s'), snat_info)
        return snat_info

    def set_vpc_connection(self, ctx, tenant_name, vpc_conn_id=None,
                           resource_data=None):
        """Set VPC connection data."""
        LOG.debug("[AC] Begin to set VPC connection data: %s, %s, %s", tenant_name, vpc_conn_id, resource_data)
        if vpc_conn_id:
            vpc_conn_db = VpcConnectionDbMixin()
            resource_data = vpc_conn_db.get_db_vpc_connection(ctx, vpc_conn_id)
        if not resource_data:
            return {}
        vpc_conn_dict = vpc_connection_model.ACVpcConnectionModel.ac_model_format(resource_data)

        for vpc_conn_dict_key in ['local-firewall-enable',
                                  'peer-firewall-enable',
                                  'priority', 'local-cidrs',
                                  'peer-cidrs', 'mode']:
            if vpc_conn_dict.get(vpc_conn_dict_key) is None:
                vpc_conn_dict.pop(vpc_conn_dict_key)

        LOG.info(_LI("[AC] Vpc connection is dependent, Vpc connection: %s"), vpc_conn_dict)
        return vpc_conn_dict

    def set_l2br(self, ctx, tenant_name, l2br_id=None, resource_data=None):
        """Set L2BR data."""
        LOG.debug("[AC] Begin to set L2BR data: %s, %s, %s",
                  tenant_name, l2br_id, resource_data)
        if l2br_id:
            l2br_db = L2brDbMixin()
            resource_data = l2br_db.get_db_l2br(
                ctx, l2br_id)
        if not resource_data:
            return {}
        l2br_dict = l2br_model.ACL2BRModel. \
            ac_model_format(resource_data)
        LOG.info(_LI("[AC] L2br is dependent, l2br: %s"),
                 l2br_dict)
        return l2br_dict

    def read_binding_levels(self, admin_ctx):
        """Read binding levels."""
        from neutron.db import models_v2
        black_list = cfg.CONF.huawei_ac_config.network_black_list
        white_list = cfg.CONF.huawei_ac_config.network_white_list

        def _need_append(network_name):
            for name in white_list:
                if network_name.endswith(name):
                    return True
            if not white_list:
                for name in black_list:
                    if network_name.endswith(name):
                        return False
                return True
            return False

        with admin_ctx.session.begin(subtransactions=True):
            network_query = admin_ctx.session.query(models_v2.Network)
            network_vxlan = network_query.join(ncu.NetworkSegment).filter(
                ncu.NetworkSegment.network_type == 'vxlan'
            ).all()
            network_ids = [network.id for network in network_vxlan if
                           _need_append(network.name)]
            port_query = admin_ctx.session.query(models_v2.Port)
            exclude_owner = [ncu.DEVICE_OWNER_ROUTER_INTF,
                             ncu.DEVICE_OWNER_ROUTER_GW,
                             ncu.DEVICE_OWNER_FLOATINGIP]
            port_query = port_query.join(models.PortBinding).filter(
                models_v2.Port.network_id.in_(network_ids),
                models_v2.Port.device_owner.notin_(exclude_owner),
                models.PortBinding.vnic_type.in_(['normal', 'direct'])
            ).all()
            binding_levels = []
            for port in port_query:
                bind_query = admin_ctx.session.query(
                    ncu.NetworkSegment, models.PortBindingLevel)
                bind_query = bind_query.join(models.PortBindingLevel)
                segment_unions = bind_query.filter(
                    models.PortBindingLevel.port_id == port.id,
                    models.PortBindingLevel.level == 1,
                    ncu.NetworkSegment.network_id == port.network_id,
                    ncu.NetworkSegment.network_type == 'vlan',
                ).all()
                if len(segment_unions) > 1:
                    LOG.error(_LE('[AC] More than one segment found for port '
                                  '%s: %s'), port.id, segment_unions)
                elif len(segment_unions) < 1:
                    LOG.info(_LI('[AC] No segment found for port %s'), port.id)
                    port_bind = ncu.get_port_binding(admin_ctx.session, port.id)
                    binding_levels.append(
                        (port.id, port.network_id, port_bind.host, None))
                for segment_union in segment_unions:
                    segment, bind = segment_union
                    binding_levels.append((port.id, port.network_id, bind.host,
                                           int(segment.segmentation_id)))

        return {'bind-ports': binding_levels}

    def read_ike_policy(self, admin_ctx, filters=None, fields=None):
        """Read IKE policy data."""
        LOG.debug("[AC] Begin to read IKE policy data: %s, %s",
                  filters, fields)
        return self._db_if.get_neutron_record(
            admin_ctx.session, None,
            ac_constants.NW_HW_IKE_POLICY)

    def read_ipsec_policy(self, admin_ctx, filters=None, fields=None):
        """Read IPSec policy data."""
        LOG.debug("[AC] Begin to read IPSec policy data: %s, %s",
                  filters, fields)
        return self._db_if.get_neutron_record(
            admin_ctx.session, None,
            ac_constants.NW_HW_IPSEC_POLICY)

    def read_vpn_service(self, admin_ctx, filters=None, fields=None):
        """Read VPN service data."""
        LOG.debug("[AC] Begin to read VPN service data: %s, %s",
                  filters, fields)
        vpn_interface = vpn_db.VPNPluginDb()
        return vpn_interface.get_vpnservices(
            admin_ctx, filters=filters, fields=fields)

    def read_ipsec_site_conn(self, admin_ctx, filters=None, fields=None):
        """Read IPSec site connection data."""
        LOG.debug("[AC] Begin to read IPSec site connection data: %s, %s",
                  filters, fields)
        return self._db_if.get_neutron_record(
            admin_ctx.session, None,
            ac_constants.NW_HW_IPSEC_SITE_CONNECTION)

    def read_firewall_policy(self, admin_ctx, filters=None, fields=None):
        """Read firewall policy data."""
        LOG.debug("[AC] Begin to read firewall policy data: %s, %s",
                  filters, fields)
        return FirewallDBMixin().get_firewall_policies(
            admin_ctx, filters=filters, fields=fields)

    def read_firewall_rule(self, admin_ctx, filters=None, fields=None):
        """Read firewall rule data."""
        LOG.debug("[AC] Begin to read firewall rule data: %s, %s",
                  filters, fields)
        original_firewall_rule_list = FirewallDBMixin().get_firewall_rules(
            admin_ctx, filters=filters, fields=fields)
        temp_set = {ac_constants.OPS_M, ac_constants.OPS_K, ac_constants.OPS_P}
        if 'huawei_ac_firewall_rule' not in cfg.CONF.ml2.extension_drivers or \
                cfg.CONF.huawei_ac_config.OPS_version not in temp_set:
            return original_firewall_rule_list

        new_firewall_rule_list = \
            self.firewall_rule_addrs_func.get_db_ac_firewall_rule_list()
        return_firewall_rule_list = []

        for origin_rule in original_firewall_rule_list:
            origin_rule['source_ports'] = ''
            origin_rule['destination_ports'] = ''
            origin_rule['source_ip_addresses'] = ''
            origin_rule['destination_ip_addresses'] = ''

            return_firewall_rule = copy.deepcopy(origin_rule)
            for new_rule in new_firewall_rule_list:
                if origin_rule['id'] == new_rule['id']:
                    return_firewall_rule['source_ports'] = \
                        new_rule.get('source_ports', '')
                    return_firewall_rule['destination_ports'] = \
                        new_rule.get('destination_ports', '')
                    return_firewall_rule['source_ip_addresses'] = \
                        new_rule.get('source_ip_addresses', '')
                    return_firewall_rule['destination_ip_addresses'] = \
                        new_rule.get('destination_ip_addresses', '')
            return_firewall_rule_list.append(return_firewall_rule)
        return return_firewall_rule_list

    def read_firewall(self, admin_ctx, filters=None, fields=None):
        """Read firewall data."""
        LOG.debug("[AC] Begin to read firewall data: %s, %s", filters, fields)
        return FirewallDBMixin().get_firewalls(
            admin_ctx, filters=filters, fields=fields)

    def read_qos_policy(self, admin_ctx, filters=None, fields=None):
        """Read QoS policy data."""
        LOG.debug("[AC] Begin to read QoS policy data: %s, %s", filters, fields)
        qos_policies = policy_object.QosPolicy.get_objects(admin_ctx)
        result = []
        for qos_policy in qos_policies:
            if self.data_filter.not_in_white_or_in_black_list(
                    admin_ctx, qos_policy, ac_constants.NW_HW_QOS_POLICY,
                    qos_policy.id):
                continue

            # qos_policy for dryRun, not compare to controller.
            qos_policy_description = qos_policy.get('description')
            if qos_policy_description and qos_policy_description.find('__resource_preemption') != -1:
                continue

            result.append(qos_policy)
        LOG.debug("Read qos policies data: %s", result)
        return result

    @staticmethod
    def read_cmcc_qos_policy(admin_ctx, filters=None, fields=None):
        """Read CMCC QoS policy data."""
        LOG.debug("[AC] Begin to read CMCC QoS policy data: %s, %s",
                  filters, fields)
        import neutron.db.qos_db as cmcc_qos_db
        if filters:
            filter_attr = filters
        else:
            filter_attr = {}
        resources = admin_ctx.session.query(cmcc_qos_db.QoSPolicy). \
            filter_by(**filter_attr).all()
        results = []
        for resource in resources:
            qos = admin_ctx.session.query(cmcc_qos_db.QoS). \
                filter_by(id=resource.qos_id).first()
            if qos:
                resource['tenant_id'] = qos.tenant_id
            else:
                resource['tenant_id'] = None
            results.append(resource)
        LOG.debug("Read cmcc qos policies: %s", resources)
        return resources

    def read_security_group(self, admin_ctx, filters=None, fields=None):
        """Read security group data."""
        LOG.debug("[AC] Begin to read security group data: %s, %s",
                  filters, fields)
        sg_db = SecurityGroupDBManager()
        resources = sg_db.get_security_groups(
            admin_ctx, filters=filters, fields=None)
        id_filter = []
        for resource in resources:
            if not resource['tenant_id']:
                id_filter.append(resource['id'])
        result = []
        resources = sg_db.get_security_groups(
            admin_ctx, filters=filters, fields=fields)
        for resource in resources:
            if resource['id'] not in id_filter:
                result.append(resource)
        LOG.debug("Read security group data: %s", result)
        return result

    def read_security_group_rule(self, admin_ctx, filters=None, fields=None):
        """Read security group rule data."""
        LOG.debug("[AC] Begin to read security group rule data: %s, %s",
                  filters, fields)
        resources = self._db_if.get_neutron_record(
            admin_ctx.session, None, ac_constants.NW_HW_SEC_GRP_RULE)
        sg_rules = copy.deepcopy(resources)
        data_list = []
        for sg_rule in sg_rules:
            if sg_rule.get('tenant_id'):
                data_list.append(sg_rule)
        LOG.debug("Read security group rules data: %s", data_list)
        return data_list

    def read_floating_ip(self, admin_ctx, filters=None, fields=None):
        """Read floating IP data."""
        LOG.debug("[AC] Begin to read floating IP data: %s, %s",
                  filters, fields)
        l3_db = L3DBManager()
        res_list = []
        resources = l3_db.get_floatingips(
            admin_ctx, filters=filters, fields=fields)
        LOG.debug("Read floatingips data: %s", resources)
        for res in resources:
            if res.get('fixed_ip_address') is None:
                continue
            if ncu.IS_FSP:
                if not self._check_is_flexible_snat(res, resources):
                    res_list.append(res)
            else:
                res_list.append(res)

        LOG.debug("Sync floatingip data: %s", res_list)
        return res_list

    def read_dnat(self, admin_ctx, filters=None, fields=None):
        """Read DNAT data."""
        LOG.debug("[AC] Begin to read DNAT data: %s, %s", filters, fields)
        dnats = DNATDBMixin().get_dnats_db(admin_ctx, filters, fields)
        for dnat in dnats:
            if 'floating_ip_id' in dnat:
                floating_ip = L3DBManager().get_floatingip(
                    admin_ctx, dnat['floating_ip_id'])
                dnat['tenant_id'] = floating_ip['tenant_id']
        return dnats

    def read_pf(self, admin_ctx, filters=None, fields=None):
        """Read port forwarding data."""
        LOG.debug("[AC] Begin to read port forwarding data: %s, %s",
                  filters, fields)
        from neutron.services.portforwarding.pf_plugin import \
            PortForwardingPlugin
        dnats = []
        l3_db = L3DBManager()
        floatingips = l3_db.get_floatingips(
            admin_ctx, filters=None, fields=fields)
        pf_plugin = PortForwardingPlugin()
        for floatingip in floatingips:
            dnats.extend(_read_pf_core(admin_ctx, fields, filters, floatingip,
                                       pf_plugin))
        return dnats

    @classmethod
    def _check_is_flexible_snat(cls, fip, resources):
        for res in resources:
            if (fip['id'] != res['id'] and fip['floating_ip_address'] ==
                    res['floating_ip_address']):
                return True
        return False

    def read_endpoint_group(self, admin_ctx, filters=None, fields=None):
        """Read endpoint group data."""
        LOG.debug("[AC] Begin to read endpoint group data: %s, %s",
                  filters, fields)
        vpn_interface = vpn_db.VPNPluginDb()
        return vpn_interface.get_endpoint_groups(admin_ctx, filters=filters,
                                                 fields=fields)

    # Supports only in FSP_6_1
    def read_snats(self, admin_ctx, filters=None, fields=None):
        """Read SNAT data."""
        LOG.debug("[AC] Begin to read SNAT data: %s, %s", filters, fields)
        l3_db = L3DBManager()
        res_list = []
        resources = l3_db.get_floatingips(
            admin_ctx, filters=filters, fields=fields)
        for fip in list(resources):
            if fip['fixed_ip_address'] and \
                    ACCommonUtil.is_sub_floatingip(admin_ctx, fip):
                res_list.append(fip)
        LOG.debug("Read snats data: %s", res_list)
        return res_list

    def read_snats_v2(self, context, filters=None, fields=None):
        """Read SNAT data."""
        snats = SnatDbMixin().get_snats_db(context, filters, fields)
        LOG.debug("[AC] read snats v2: %s", snats)
        return snats

    def read_vpc_connection(self, admin_ctx, filters=None, fields=None):
        """Read VPC connection data."""
        LOG.debug("[AC] Begin to read VPC connection data: %s, %s",
                  filters, fields)
        vpc_conn_db = VpcConnectionDbMixin()
        vpc_conns = vpc_conn_db.get_db_vpc_connections(
            admin_ctx, filters=filters, fields=fields)
        LOG.debug("Read vpc connections: %s", vpc_conns)
        return vpc_conns

    def read_bgp_route(self, admin_ctx, filters=None, fields=None):
        """Read BGP route data."""
        LOG.debug("[AC] Begin to read BGP route data: %s, %s", filters, fields)
        bgp_route_db = BgpRouteDbMixin()
        bgp_routes = bgp_route_db.get_db_bgp_routes(
            admin_ctx, filters=filters, fields=fields)
        return bgp_routes

    def read_l2br(self, admin_ctx, filters=None, fields=None):
        """Read L2BR data."""
        LOG.debug("[AC] Begin to read L2BR data: %s, %s", filters, fields)
        l2br_db = L2brDbMixin()
        l2brs = l2br_db.get_db_l2brs(
            admin_ctx, filters=filters, fields=fields)
        LOG.debug("Read l2brs: %s", l2brs)
        return l2brs

    def read_classifier(self, admin_ctx, filters=None, fields=None):
        """Read policy classifier data."""
        try:
            LOG.debug("[AC] Begin to read policy classifier data: %s, %s",
                      filters, fields)
            plugins = manager.NeutronManager.get_service_plugins()
            grouppolicy_plugin = plugins.get("GROUP_POLICY")
            classifiers = grouppolicy_plugin.get_policy_classifiers(
                admin_ctx, fields=fields)
            return classifiers
        except Exception as ex:
            LOG.error("when read classifiers catch %s", ex)
            return []

    def read_policy_rule(self, admin_ctx, filters=None, fields=None):
        """Read policy rule data."""
        try:
            LOG.debug("[AC] Begin to read policy rule data: %s, %s",
                      filters, fields)
            plugins = manager.NeutronManager.get_service_plugins()
            grouppolicy_plugin = plugins.get("GROUP_POLICY")
            policy_rules = grouppolicy_plugin.get_policy_rules(
                admin_ctx, fields=fields)
            return policy_rules
        except Exception as ex:
            LOG.error("when read policy rules catch %s", ex)
            return []

    def read_policy_rule_set(self, admin_ctx, filters=None, fields=None):
        """Read policy rule set data."""
        try:
            LOG.debug("[AC] Begin to read policy rule set data: %s, %s",
                      filters, fields)
            plugins = manager.NeutronManager.get_service_plugins()
            grouppolicy_plugin = plugins.get("GROUP_POLICY")
            rule_sets = grouppolicy_plugin.get_policy_rule_sets(
                admin_ctx, fields=fields)
            return rule_sets
        except Exception as ex:
            LOG.error("when read policy rule sets catch %s", ex)
            return []

    def read_policy_target_group(self, admin_ctx, filters=None, fields=None):
        """Read policy target group data."""
        try:
            LOG.debug("[AC] Begin to read policy target group data: %s, %s",
                      filters, fields)
            plugins = manager.NeutronManager.get_service_plugins()
            grouppolicy_plugin = plugins.get("GROUP_POLICY")
            policy_target_groups = grouppolicy_plugin.get_policy_target_groups(
                admin_ctx, fields=fields)
            return policy_target_groups
        except Exception as ex:
            LOG.error("when read policy target groups catch %s", ex)
            return []

    def read_external_segment(self, admin_ctx, filters=None, fields=None):
        """Read external segment data."""
        try:
            LOG.debug("[AC] Begin to read policy rule data: %s, %s",
                      filters, fields)
            plugins = manager.NeutronManager.get_service_plugins()
            group_policy_plugin = plugins.get("GROUP_POLICY")
            external_segments = group_policy_plugin.get_external_segments(
                admin_ctx, filters=filters, fields=fields)
            return external_segments
        except Exception as ex:
            LOG.error(_LE("[AC] Failed to read external segment: %s"), ex)
            return []

    def read_external_policy(self, admin_ctx, filters=None, fields=None):
        """Read external policy data."""
        try:
            LOG.debug("[AC] Begin to read external policy data: %s, %s",
                      filters, fields)
            plugins = manager.NeutronManager.get_service_plugins()
            group_policy_plugin = plugins.get("GROUP_POLICY")
            external_policies = group_policy_plugin.get_external_policies(
                admin_ctx, filters=filters, fields=fields)
            return external_policies
        except Exception as ex:
            LOG.error(_LE("[AC] Failed to read external policy: %s"), ex)
            return []

    def set_ipv6_ns_qos_policy(self, ctx, tenant_name, policy_id=None):
        """ set ipv6 ns qos policy """
        LOG.info("tenant_name=%s", tenant_name)
        policy_data = {}
        try:
            if policy_id and self.ipv6_ns_qos_policy:
                qos_policy = self.ipv6_ns_qos_policy.get_ipv6_ns_qos_policy(
                    ctx, policy_id)
                policy_data = ACIpv6NsQoSpolicyModel.ac_model_format(qos_policy)
        except Exception as ex:
            LOG.error(_LE("[AC]Set ipv6 ns qos policy error: %s"), str(ex))
        LOG.info(_LI("[AC]Set ipv6 ns qos policy: %s."), policy_data)
        return policy_data

    def read_ipv6_ns_qos_policy(self, admin_ctx, filters=None, fields=None):
        """ read ipv6 ns qos policy """
        qos_policies = []
        if not self.ipv6_ns_qos_policy:
            return qos_policies
        try:
            qos_policies = self.ipv6_ns_qos_policy.get_ipv6_ns_qos_policies(
                admin_ctx, filters=filters, fields=fields)
            for qos_policy in qos_policies:
                uuid = qos_policy.get('id')
                rec = self._db_if.get_res_attribute(
                    session=None, res_id=uuid,
                    res_type=ac_constants.NW_HW_IPV6_NS_QOS_POLICY)
                if rec:
                    qos_policy['updated_at'] = rec.update_time.strftime(
                        ac_constants.ISO8601_TIME_FORMAT)
                    qos_policy['created_at'] = rec.create_time.strftime(
                        ac_constants.ISO8601_TIME_FORMAT)
        except Exception as ex:
            LOG.error("Read ipv6 ns qos policy catch exception: %s", str(ex))
        LOG.info(_LI("[AC] Read ipv6 ns qos policy : %s."), qos_policies)
        return qos_policies

    def set_tap_service(self, ctx, tenant_name, tap_service_id=None,
                        resource_data=None):
        """ set tap service """
        LOG.info("tenant_name=%s", tenant_name)
        if tap_service_id:
            resource_data = self.fm_db.get_tap_service(ctx, tap_service_id)
        if not resource_data:
            return {}
        tap_service_dict = flow_mirror_model.ACFlowMirrorModel. \
            ac_model_format_service(resource_data)
        res = self._db_if.get_res_attribute(
            session=None, res_id=tap_service_id,
            res_type=ac_constants.NW_HW_TAP_SERVICE)
        if res:
            tap_service_dict['updated_at'] = res.update_time.strftime(
                ac_constants.ISO8601_TIME_FORMAT)
            tap_service_dict['created_at'] = res.create_time.strftime(
                ac_constants.ISO8601_TIME_FORMAT)
        LOG.info(_LI("[AC] tap service is dependent, Vpc connection: %s"),
                 tap_service_dict)
        return tap_service_dict

    def set_tap_flow(self, ctx, tenant_name, tap_flow_id=None,
                     resource_data=None):
        """ set tap flow """
        LOG.info("tenant_name=%s", tenant_name)
        if tap_flow_id:
            resource_data = self.fm_db.get_tap_flow(ctx, tap_flow_id)
        if not resource_data:
            return {}
        tap_flow_dict = flow_mirror_model.ACFlowMirrorModel. \
            ac_model_format_flow(resource_data)
        res = self._db_if.get_res_attribute(
            session=None, res_id=tap_flow_id,
            res_type=ac_constants.NW_HW_TAP_FLOW)
        if res:
            tap_flow_dict['updated_at'] = res.update_time.strftime(
                ac_constants.ISO8601_TIME_FORMAT)
            tap_flow_dict['created_at'] = res.create_time.strftime(
                ac_constants.ISO8601_TIME_FORMAT)
        LOG.info(_LI("[AC] tap service is dependent, Vpc connection: %s"),
                 tap_flow_dict)
        return tap_flow_dict

    def read_tap_service(self, admin_ctx, filters=None, fields=None):
        """ read tap service """
        try:
            tap_services = self.fm_db.get_tap_services(admin_ctx,
                                                       filters=filters,
                                                       fields=fields)
            tap_service_result = []
            for tap_service in tap_services:
                # tap_service for dryRun, not compare to controller.
                tap_service_description = tap_service.get('description')
                if tap_service_description and tap_service_description.find(
                        '__resource_preemption') != -1:
                    continue
                uuid = tap_service.get('id')
                res = self._db_if. \
                    get_res_attribute(session=None,
                                      res_id=uuid,
                                      res_type=ac_constants.NW_HW_TAP_SERVICE)
                if res:
                    tap_service['updated_at'] = res.update_time. \
                        strftime(ac_constants.ISO8601_TIME_FORMAT)
                    tap_service['created_at'] = res.create_time. \
                        strftime(ac_constants.ISO8601_TIME_FORMAT)
                tap_service_result.append(tap_service)
            LOG.debug("Read tap services: %s", tap_service_result)
            return tap_service_result
        except Exception as ex:
            LOG.error(_LE("[AC] Failed to read tap services: %s"), ex)
            return []

    def read_tap_flow(self, admin_ctx, filters=None, fields=None):
        """ read tap flow """
        try:
            tap_flows = self.fm_db.get_tap_flows(admin_ctx, filters=filters,
                                                 fields=fields)
            tap_flow_result = []
            for tap_flow in tap_flows:
                # tap_flow for dryRun, not compare to controller.
                tap_flow_description = tap_flow.get('description')
                if tap_flow_description and tap_flow_description.find('__resource_preemption') != -1:
                    continue
                uuid = tap_flow.get('id')
                res = self._db_if. \
                    get_res_attribute(session=None,
                                      res_id=uuid,
                                      res_type=ac_constants.NW_HW_TAP_FLOW)
                if res:
                    tap_flow['updated_at'] = res.update_time. \
                        strftime(ac_constants.ISO8601_TIME_FORMAT)
                    tap_flow['created_at'] = res.create_time. \
                        strftime(ac_constants.ISO8601_TIME_FORMAT)
                tap_flow_result.append(tap_flow)
            LOG.debug("Read tap flows: %s", tap_flows)
            return tap_flow_result
        except Exception as ex:
            LOG.error(_LE("[AC] Failed to read tap flow: %s"), ex)
            return []
