# !/usr/bin/env python
# -*- coding:utf-8 -*-
"""
| 功能：处理AC 配置文件
| 版本：2022-03-31 11:27 创建
"""
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
import os
from ast import literal_eval

import six

from networking_huawei.common.exceptions import NeutronSyncError
from networking_huawei.drivers.ac.common.constants import LIMIT_FILE_SIZE, NEUTRON_SYNC_WHITELIST_FILENAME, \
    ML2_CONFIG_FILE, HUAWEI_CONFIG_FILE, NEUTRON_CONF
from networking_huawei.drivers.ac.common.file_utils import FileMeta
from networking_huawei.drivers.ac.db.config_check.ac_config_schema import ACBasicConfig


class ConfigParas(object):
    """Config params manage,read the params from config file"""

    def __init__(self, config_file):
        """Init function"""
        self.config = six.moves.configparser.ConfigParser()
        self.config.read(config_file)

    def get_config_detail(self, group_name, config_name, default=''):
        """Get config param values from config file"""
        try:
            return self.config.get(group_name, config_name)
        except Exception:
            return default


def read_whitelist_file(filepath=NEUTRON_SYNC_WHITELIST_FILENAME):
    """读取"/etc/neutron/whitelist.ini"文件，返回dict

    :param filepath:str,文件路径，默认NEUTRON_SYNC_WHITELIST_FILENAME
    :return: dict
    """
    with open(filepath, 'r') as whitelist_file:
        f_size = os.path.getsize(filepath)
        # if file size is too big, empty the file
        if f_size > LIMIT_FILE_SIZE:
            raise NeutronSyncError(reason=('Whitelist file-size is greater than %s' % LIMIT_FILE_SIZE))
        return literal_eval(whitelist_file.read())


def read_ac_config_in_ini_file(filepath, whitelist=None):
    """读取标准格式的配置文件

    :param filepath: str,配置文件路径
    :param whitelist: list[tuple(str,str)],如果不为none代表仅读取在该白名单内的配置项
    :return: dict,{section.option:value} 用 '%s.%s'%(section,option)作为键
    """
    result = {}
    parser = ConfigParas(filepath)
    if whitelist:
        for group, option_key in whitelist:
            option_value = parser.get_config_detail(group, option_key)
            result['%s.%s' % (group, option_key)] = option_value

    else:
        for elem in parser.config.sections():
            for (option_key, option_value) in parser.config.items(elem):
                result['%s.%s' % (elem, option_key)] = option_value
    return result


# AC 配置文件清单
AC_CONFIG_FILES = (NEUTRON_SYNC_WHITELIST_FILENAME, NEUTRON_CONF, ML2_CONFIG_FILE, HUAWEI_CONFIG_FILE)


def read_ac_config(filepath, filetype=None):
    """读取配置文件

    :param filepath: str,文件路径
    :param filetype: str,文件类型,用来区分配置文件类型和要读取的配置项清单
    :return: ACBasicConfig
    """
    if filetype is None:
        filetype = filepath
    file_meta = FileMeta(filepath)
    result = ACBasicConfig(config_file=filetype,
                           owner=file_meta.user,
                           user_group=file_meta.user_group,
                           right=file_meta.permission)
    if filetype == NEUTRON_SYNC_WHITELIST_FILENAME:
        # 处理"/etc/neutron/whitelist.ini"，比较特殊，非标准配置文件格式
        config_value = read_whitelist_file(filepath)
    else:
        # 不为None时代表仅需读取该清单里面的配置项
        config_options = None
        # 要分割成列表的配置项清单，{配置项名:分隔符}
        split_dict = {}
        if filetype == NEUTRON_CONF:
            # neutron.conf 检查配置项：service_plugins、api_extensions_path、notification_drivers
            config_options = (('DEFAULT', 'service_plugins'), ('DEFAULT', 'api_extensions_path'),
                              ('qos', 'notification_drivers'))
            split_dict.update({'DEFAULT.service_plugins': ',', 'DEFAULT.api_extensions_path': ':',
                               'qos.notification_drivers': ','})
        elif filetype == ML2_CONFIG_FILE:
            # ml2_conf.ini 检查配置项：mechanism_drivers、extension_drivers
            config_options = (('ml2', 'mechanism_drivers'), ('ml2', 'extension_drivers'))
            split_dict.update({'ml2.mechanism_drivers': ',', 'ml2.extension_drivers': ':'})
        else:
            split_dict.update({'huawei_ac_config.network_black_list': ','})

        config_value = read_ac_config_in_ini_file(filepath, whitelist=config_options)

        # 处理配置项值为列表的情况
        for key, value in split_dict.items():
            if key in config_value:
                # 以 "__list_" 开头的配置项代表原值是列表，后面需要特殊处理
                config_value[ACBasicConfig.get_list_key(key)] = sorted(config_value.get(key).split(value))

    result.config_value = config_value
    return result
