#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""cryptor"""

import os
import base64
import six
try:
    from Crypto.Cipher import AES
except ImportError:
    from Cryptodome.Cipher import AES

try:
    from networking_huawei.drivers.ac.cryptor.pbkdf2 import enable_pbkdf2
except ImportError:
    from pbkdf2 import enable_pbkdf2

try:
    from networking_huawei.drivers.ac.common.neutron_compatible_util import \
        ac_log as logging

    LOG = logging.getLogger(__name__)
except ImportError:
    pass

USER_PASS_KEY = "pw_key"
USER_PASS_IV = "pw_iv"
KEY_IV = "key_iv"
FACTOR = "factor"
ASCII_MAX_VALUE = 256

ACCOUNT_SECURE_KEY = 1
KEYSTONE_AUTH_SECURE_KEY = 2
WEBSOCKET_SECURE_KEY = 3


def get_factor_from_file(file_path):
    """get_websocket_key_factor call: get factor from file"""
    with open(file_path, "r") as file_point:
        factor_value = None
        for line in file_point.readlines():
            line = line.strip().strip("\n")
            if FACTOR in line:
                factor_value = line.split("%s=" % FACTOR)[1]
        return factor_value


class AESCryptor(object):
    """AESCryptor"""

    def __init__(self, pw_key, pw_iv, key_iv):
        self.pw_key = pw_key
        self.pw_iv = pw_iv
        self.key_iv = key_iv

    def decrypt_data(self, text, cfg_factor=None):
        """
        :param text: the text is decrypted
        :param cfg_factor:
        :return:
        """
        cipher_key = self._decrypt_key(self.pw_key, self.key_iv, cfg_factor)
        cryptor = AES.new(cipher_key, AES.MODE_CBC, self.pw_iv)
        # support six.ensure_binary when six >= 1.12.0
        # https://github.com/benjaminp/six/blob/master/CHANGES
        if hasattr(six, "ensure_binary"):
            return six.ensure_binary(cryptor.decrypt(text)).strip(b'\0')
        return cryptor.decrypt(text).strip('\0')

    def _decrypt_key(self, text, key_iv, cfg_factor=None):
        if not cfg_factor:
            cfg_factor = self._get_cfg_factor()
        # support six.ensure_binary when six >= 1.12.0
        # https://github.com/benjaminp/six/blob/master/CHANGES
        if hasattr(six, "ensure_binary"):
            key = enable_pbkdf2(
                base64.b64decode(six.ensure_binary(self._get_init_vector())),
                base64.b64decode(six.ensure_binary(cfg_factor)))
        else:
            key = enable_pbkdf2(base64.b64decode(self._get_init_vector()),
                                base64.b64decode(cfg_factor))
        return AES.new(key, AES.MODE_CBC, key_iv).decrypt(text)

    def decrypt_dependency(self, data_type=ACCOUNT_SECURE_KEY):
        """decrypt_dependency"""
        file_dir = os.path.dirname(os.path.realpath(__file__))
        file_path = os.path.join(file_dir, "decrypt_dependency.txt")
        if data_type == KEYSTONE_AUTH_SECURE_KEY:
            file_path = os.path.join(file_dir,
                                     "decrypt_dependency_keystone.txt")
        if data_type == WEBSOCKET_SECURE_KEY:
            file_path = os.path.join(file_dir,
                                     "decrypt_dependency_websocket.txt")
        with open(file_path, "r") as file_point:
            for line in file_point.readlines():
                line = line.strip().strip("\n")
                if USER_PASS_KEY in line:
                    self.pw_key = self._convert_base64(
                        line.split(USER_PASS_KEY + "=")[1])
                elif USER_PASS_IV in line:
                    self.pw_iv = self._convert_base64(
                        line.split(USER_PASS_IV + "=")[1])
                elif KEY_IV in line:
                    self.key_iv = self._convert_base64(
                        line.split(KEY_IV + "=")[1])

    @classmethod
    def _convert_base64(cls, input_data):
        # support six.ensure_binary when six >= 1.12.0
        # https://github.com/benjaminp/six/blob/master/CHANGES
        if hasattr(six, "ensure_binary"):
            return base64.b64decode(six.ensure_binary(input_data))
        return base64.b64decode(input_data)

    @classmethod
    def _convert(cls, avalue):
        if avalue > ASCII_MAX_VALUE - 1:
            avalue -= ASCII_MAX_VALUE
        elif avalue < 0:
            avalue += ASCII_MAX_VALUE
        return avalue

    def _get_init_vector(self):
        init_vector = [0x69, 0x1D, 0x10, 0x3B,
                       0xDB, 0x1B, 0x14, 0xBF,
                       0xA6, 0x83, 0xEE, 0x2E,
                       0xC9, 0xAD, 0x43, 0x60]

        init_vector[0] = self._convert(init_vector[13] & init_vector[14])
        init_vector[1] = self._convert(init_vector[12] & init_vector[11])
        init_vector[2] = self._convert(init_vector[9] + init_vector[2])
        init_vector[3] = self._convert(init_vector[8] + init_vector[9])
        init_vector[4] = self._convert(init_vector[10] & init_vector[4])
        init_vector[5] = self._convert(init_vector[0] - init_vector[8])
        init_vector[6] = self._convert(init_vector[2] - init_vector[11])
        init_vector[7] = self._convert(init_vector[12] + init_vector[8])
        init_vector[8] = self._convert(init_vector[3] & init_vector[10])
        init_vector[9] = self._convert(init_vector[11] | init_vector[15])
        init_vector[10] = self._convert(init_vector[13] - init_vector[15])
        init_vector[11] = self._convert(init_vector[6] | init_vector[1])
        init_vector[12] = self._convert(init_vector[6] | init_vector[1])
        init_vector[13] = self._convert(init_vector[10] | init_vector[15])
        init_vector[14] = self._convert(init_vector[10] + init_vector[14])
        init_vector[15] = self._convert(init_vector[10] & init_vector[2])

        return base64.b64encode(bytearray(init_vector))

    @classmethod
    def _get_cfg_factor(cls):
        file_dir = os.path.dirname(os.path.realpath(__file__))
        file_path = os.path.join(file_dir, "cfg_factor.txt")
        with open(file_path, "r") as file:
            return file.read().strip("\n").strip(" ")

    @classmethod
    def get_websocket_key_factor(cls):
        """get_websocket_key_factor"""
        dir_path = os.path.dirname(os.path.realpath(__file__))
        file_path = os.path.join(dir_path, "decrypt_dependency_websocket.txt")
        try:
            return get_factor_from_file(file_path)
        except Exception as ex:
            LOG.error('Failed to get factor for websocket key password: %s', ex)
            raise ex
