#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""
Neutron sync class
"""

import copy
import os
import socket
from datetime import datetime

from oslo_config import cfg

try:
    from neutron.common.exceptions import Conflict
except ImportError:
    from neutron_lib.exceptions import Conflict

from networking_huawei._i18n import _
from networking_huawei.drivers.ac.extensions.sync import resource_sync
from networking_huawei.drivers.ac.sync.neutron_sync import ACNeutronSync
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    is_valid_result_filename
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.common.exceptions import NeutronSyncOrCompareError, \
    CompareOverwriteError
from networking_huawei.drivers.ac.common import fusion_sphere_alarm as fsa

LOG = logging.getLogger(__name__)


class NeutronSyncConflict(Conflict):
    """Neutron server already performing neutron sync"""
    message = _("Neutron sync or consistency check in-progress, "
                "please wait until it is over.")


class NeutronSyncComplete(Conflict):
    """Neutron server already performing neutron sync"""
    message = _("No server performs neutron sync.")


class NeutronSyncWhitelistError(Conflict):
    """Neutron server already performing neutron sync"""
    message = _("IO error occured on whitelist file \'%(filename)s\', error: "
                "(%(error)s)")


class AcExtendedSupport(resource_sync.ResourcesyncBase):
    """Implementation of the Huawei AC Neutron sync Service Plugin."""
    supported_extension_aliases = ['resourcesync']

    @classmethod
    def _validate_file(cls, compare_opt):
        """
        :param compare_opt: dict of compare-option
        :return: compare result filename
        """

        # Use the default values for 'overwrite' and 'filename' as below.
        overwrite = True
        filename = ac_const.NEUTRON_SYNC_COMPARE_RESULT_FILE
        if 'overwrite' in compare_opt:
            overwrite = compare_opt['overwrite']
        if 'filename' in compare_opt:
            ret, msg = is_valid_result_filename(compare_opt['filename'])
            if ret:
                filename = os.path.join(ac_const.DEFAULT_LOG_PATH,
                                        compare_opt['filename'])
            else:
                raise NeutronSyncOrCompareError(operation='compare',
                                                reason=msg)

        if filename == ac_const.NEUTRON_SYNC_COMPARE_RESULT_FILE:
            filename = filename.split(".")[0]
            filename += "_" + datetime.utcnow().strftime("%Y%m%d_%H%M%S") + ".csv"

        LOG.debug('compare result filename: %s', filename)
        # If the doesn't exists check the dir have write access and file in
        # .csv format. If file exists, check whether 'overwrite' flag is True,
        #  else raise error.
        if os.path.exists(filename):
            if overwrite:
                if not os.access(filename, os.W_OK):
                    raise NeutronSyncOrCompareError(
                        operation='compare',
                        reason='Permission denied for the file.')
            else:
                raise CompareOverwriteError()
        else:
            if not os.access(ac_const.DEFAULT_LOG_PATH, os.W_OK):
                raise NeutronSyncOrCompareError(
                    operation='compare',
                    reason='Permission denied for the path.')
        return filename

    def get_plugin_type(self):
        """Get plugin type"""
        return 'resourcesync'

    def get_plugin_description(self):
        """Get plugin description"""
        return 'Perform neutron sync'

    def get_resourcesyncs(self, context, filters=None, fields=None):
        """Get resource syncs"""

        db_if = ACdbInterface()
        session = db_if.get_session()
        (status, server) = db_if.check_is_neutron_sync_in_progress(session)
        neutron_sync_record = db_if.get_neutron_sync_record()
        sync_start_time = neutron_sync_record.sync_start_time if neutron_sync_record else ''
        expected_completion_time = neutron_sync_record.expected_completion_time if neutron_sync_record else ''
        hostname = neutron_sync_record.hostname if neutron_sync_record else socket.gethostname()
        if status:
            if server.state == ac_const.CONSISTENCY_CHECK:
                return [{'server': hostname,
                         'state': server.state,
                         'sync_res': server.sync_res,
                         'sync_start_time': sync_start_time,
                         'expected_completion_time': 'computing'}]
            return [{'server': hostname,
                     'state': server.state,
                     'sync_res': server.sync_res,
                     'sync_start_time': sync_start_time,
                     'expected_completion_time': expected_completion_time}]
        raise NeutronSyncComplete()

    @classmethod
    def enable_white_list_file(cls):
        """Check whitelist whether is enabled and file is valid"""
        try:
            if cfg.CONF.huawei_ac_config.neutron_sync_enable_white_list:
                whitelist_file = ac_const.NEUTRON_SYNC_WHITELIST_FILENAME
                with open(whitelist_file, 'r'):
                    LOG.debug('Whitelist is enabled and file is valid.')
        except IOError as ex:
            reason = "IO error occured on whitelist file"
            alarm_info = fsa.ACPluginAlarm.get_neutron_sync_fail_alarm(reason)
            fsa.ACPluginAlarm.send_alarm(alarm_info)
            raise NeutronSyncWhitelistError(
                filename=ac_const.NEUTRON_SYNC_WHITELIST_FILENAME,
                error=ex.strerror)

    def create_resourcesync(self, context, resourcesync):
        """Construct the sync body, invoke the sync class"""
        LOG.debug('Perform resource sync operations: %s', resourcesync)
        need_delete = True
        operation = None
        filename = None
        sync_resources = ''

        db_if = ACdbInterface()
        session = db_if.get_session()
        (status, _) = db_if.check_is_neutron_sync_in_progress(session)
        if status:
            raise NeutronSyncConflict()
        if resourcesync['resourcesync']['sync_data'] and resourcesync['resourcesync']['compare_data']:
            operation = ac_const.SYNC_OP_SYNC_AND_COMPARE_DATA
            if resourcesync['resourcesync']['sync_data'] == 'exclude-delete':
                need_delete = False
            compare_result = resourcesync['resourcesync']['compare_data']
            # Validate compare result file
            filename = self._validate_file(compare_result)
        elif resourcesync['resourcesync']['sync_data']:
            operation = ac_const.SYNC_OP_SYNC_DATA
            if resourcesync.get('resourcesync', {}).get('sync_resources'):
                sync_resources = resourcesync['resourcesync']['sync_resources']
            else:
                sync_resources = str({'params': []})
            if resourcesync['resourcesync']['sync_data'] == 'exclude-delete':
                need_delete = False
        elif resourcesync['resourcesync']['compare_data']:
            operation = ac_const.SYNC_OP_COMPARE_DATA
            compare_result = resourcesync['resourcesync']['compare_data']
            sync_resources = resourcesync['resourcesync']['sync_resources']
            # Validate compare result file
            filename = self._validate_file(compare_result)
        elif resourcesync.get('resourcesync', {}).get('sync_resources'):
            operation = ac_const.SYNC_OP_SYNC_SINGLE_INSTANCE
            sync_resources = resourcesync['resourcesync']['sync_resources']
        sync_resource_type = resourcesync['resourcesync']['resource_type']
        include_bindports = resourcesync['resourcesync']['include_bindports']
        sync_segments = resourcesync['resourcesync'].get('sync_segments')
        if not sync_segments:
            self.enable_white_list_file()
            db_if.create_neutron_sync_record(
                sync_start_time=datetime.utcnow(),
                expected_completion_time=datetime.utcnow(),
                hostname=socket.gethostname(),
                session=session)
        self._start_neutron_sync((
            operation, filename, need_delete, include_bindports, sync_resources, sync_resource_type, sync_segments))
        return {}

    @classmethod
    def _start_neutron_sync(cls, params):
        operation, filename, need_delete, include_bindports, sync_resources, sync_resource_type, sync_segments = params
        sync_param = copy.deepcopy(ac_const.NEUTRON_SYNC_PARAM_DEFAULT)
        sync_param.update({
            "force_trigger": True, "sync_op": operation,
            "compare_data": filename, "need_delete": need_delete,
            "include_bindports": include_bindports, "sync_resource_type": sync_resource_type,
            "sync_resources": sync_resources, "sync_segments": sync_segments})
        ACNeutronSync(sync_param)
