#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""l3 exroutes custom"""

import abc
from neutron.api import extensions
from neutron.api.v2 import base

from neutron.api.v2 import resource as wsgi_resource

try:
    from neutron.api.extensions import ExtensionDescriptor
except ImportError:
    from neutron_lib.api.extensions import ExtensionDescriptor

try:
    from neutron_lib.api.definitions.l3 import RESOURCE_ATTRIBUTE_MAP
except ImportError:
    from neutron.extensions.l3 import RESOURCE_ATTRIBUTE_MAP

try:
    from neutron.api.v2.base import FAULT_MAP as neutronfaultmap
except ImportError:
    from neutron_lib.api.faults import FAULT_MAP as neutronfaultmap

from networking_huawei.drivers.ac.common import constants
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu

EXROUTES = 'list_exroutes'
EXROUTES_DESTINATION = 'destination'
EXROUTES_NEXTHOP = 'nexthop'
EXROUTES_TYPE = 'type'
EXROUTES_SEGMENT_TYPE = 'segment_type'
EXROUTES_SEGMENT_ID = 'segment_id'
EXROUTES_PRIORITY = 'priority'
EXROUTES_PROTOCOL = 'protocol'
EXROUTES_PORT = 'port'
EXROUTES_EXTRA_OPT = 'extra_opt'
EXROUTES_BFD = 'bfd'
EXROUTES_SOURCE_IPS = "source_ips"


def create_resource_exroutes(collection, resource, plugin, params, allow_bulk=False,
                             member_actions=None, parent=None, allow_pagination=False,
                             allow_sorting=False):
    """create_resource_exroutes"""
    controller = base.Controller(plugin, collection, resource, params, allow_bulk,
                                 member_actions=member_actions, parent=parent,
                                 allow_pagination=allow_pagination,
                                 allow_sorting=allow_sorting)
    return wsgi_resource.Resource(controller, neutronfaultmap, None, None,
                                  dict(create=201, delete=204,
                                       create_exroutes=201, delete_exroutes=201))


class L3_exroutes_custom(ExtensionDescriptor):
    """L3_exroutes_custom"""

    @classmethod
    def get_name(cls):
        """get_name"""
        return "Neutron L3 Router Exroutes"

    @classmethod
    def get_alias(cls):
        """get_alias"""
        return "custom_exroutes_ext_alias"

    @classmethod
    def get_description(cls):
        """get_description"""
        return "L3 Router exroutes extension"

    @classmethod
    def get_namespace(cls):
        """get_namespace"""
        return ""

    @classmethod
    def get_updated(cls):
        """get_updated"""
        return "2016-09-19T10:00:00-00:00"

    @classmethod
    def get_resources(cls):
        """get_resources"""
        plugin = ncu.get_service_plugin()['L3_ROUTER_NAT']
        resources = []
        ops_version = ncu.get_ops_version()
        for resource_name in ['router']:
            collection_name = resource_name + 's'
            params = RESOURCE_ATTRIBUTE_MAP.get(collection_name, dict())
            member_actions = {}
            member_actions['create_exroutes'] = 'PUT'
            member_actions['delete_exroutes'] = 'PUT'
            member_actions['list_exroutes'] = 'GET'
            controller = create_resource_exroutes(
                collection_name, resource_name, plugin, params,
                allow_bulk=True,
                member_actions=member_actions,
                allow_pagination=True,
                allow_sorting=True)
            resource = extensions.ResourceExtension(
                collection_name, controller,
                member_actions=member_actions,
                attr_map=params)
            if ops_version in [constants.OPS_P, constants.OPS_O]:
                from neutron.extensions import \
                    l3_ext_gw_mode, l3_ext_ha_mode, dvr, extraroute, \
                    timestamp, standardattrdescription
                resource.attr_map.update(
                    l3_ext_gw_mode.EXTENDED_ATTRIBUTES_2_0.get('routers'))
                resource.attr_map.update(
                    l3_ext_ha_mode.EXTENDED_ATTRIBUTES_2_0.get('routers'))
                resource.attr_map.update(
                    dvr.EXTENDED_ATTRIBUTES_2_0.get('routers'))
                resource.attr_map.update(
                    extraroute.EXTENDED_ATTRIBUTES_2_0.get('routers'))
                resource.attr_map.update(timestamp.TIMESTAMP_BODY)
                resource.attr_map.update(
                    standardattrdescription.DESCRIPTION_BODY)
            if ops_version in [constants.OPS_P]:
                from neutron.extensions import tagging
                resource.attr_map.update(tagging.TAG_ATTRIBUTE_MAP)
            if ops_version in [constants.OPS_O]:
                from neutron.extensions import tag
                resource.attr_map.update(tag.TAG_ATTRIBUTE_MAP)
            resources.append(resource)
        return resources

    def update_attributes_map(self, attributes):
        """update_attributes_map"""
        super(L3_exroutes_custom, self).update_attributes_map(
            attributes, extension_attrs_map=RESOURCE_ATTRIBUTE_MAP)

    def get_extended_resources(self, version):
        """get_extended_resources"""
        return {}

    def get_required_extensions(self):
        """get_required_extensions"""
        return ["router"]


class CustomRouterExroutesBase(object):
    """CustomRouterExroutesBase"""

    @abc.abstractmethod
    def create_exroutes(self, context, router_id, exroutes_info):
        """create_exroutes"""
        pass

    @abc.abstractmethod
    def delete_exroutes(self, context, router_id, exroutes_info):
        """delete_exroutes"""
        pass

    @abc.abstractmethod
    def list_exroutes(self, context, router_id):
        """list_exroutes"""
        pass
