#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""vpc connection"""

import abc

import six
from oslo_config import cfg

try:
    from neutron_lib.plugins import directory
except ImportError:
    pass
from neutron import manager

try:
    from neutron_lib.api import validators
except ImportError:
    validators = None
try:
    from neutron.api.v2 import attributes
except ImportError:
    attributes = None
try:
    from neutron.common.exceptions import NeutronException
except ImportError:
    from neutron_lib.exceptions import NeutronException
from neutron.api.v2 import base
from neutron.api import extensions

try:
    from neutron.api.extensions import ExtensionDescriptor
except ImportError:
    from neutron_lib.api.extensions import ExtensionDescriptor
try:
    from neutron.common.exceptions import NotFound
except ImportError:
    from neutron_lib.exceptions import NotFound
try:
    from neutron.services.service_base import ServicePluginBase
except ImportError:
    from neutron_lib.services.base import ServicePluginBase

import networking_huawei.drivers.ac.common.constants as ac_constants
from networking_huawei.drivers.ac.extensions import vpc_connection \
    as vpc_conn_extension
from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging

LOG = logging.getLogger(__name__)


class VpcConnNotFound(NotFound):
    """Vpc connection not found class"""
    message = "Vpc connetcion %(vpc_conn_id)s could not be found."


extensions.append_api_extensions_path(vpc_conn_extension.__path__)

VPC_CONNECTION = 'vpc-connection'

OPS_VERSION = ncu.get_ops_version()


def _validate_ac_range_or_none(data, valid_values=None):
    if data is None or data == '':
        return
    elif data not in range(1, 256):
        msg = "'%s' is not a valid range" % data
        raise InvalidRangeInput(error_message=msg)


def check_validators(type_name, call):
    """ check validators """
    if validators and hasattr(validators, 'add_validator') and \
            'type:boolean_or_none' not in validators.validators:
        validators.add_validator('boolean_or_none',
                                 validate.validate_boolean_or_none)
    if validators and hasattr(validators, 'add_validator') and \
            'type:list_of_strings' not in validators.validators:
        validators.add_validator('list_of_strings',
                                 validate.validate_list_of_strings)
    if validators and hasattr(validators, 'add_validator') and \
            'type:%s' % type_name not in validators.validators:
        validators.add_validator(type_name, call)


def check_attributes(type_name, call):
    """ check attributes """
    if attributes and hasattr(attributes, 'validators') and \
            'type:boolean_or_none' not in attributes.validators:
        attributes.validators['type:boolean_or_none'] = \
            validate.validate_boolean_or_none
    if attributes and hasattr(attributes, 'validators') and \
            'type:list_of_strings' not in attributes.validators:
        attributes.validators['type:list_of_strings'] = \
            validate.validate_list_of_strings
    if attributes and hasattr(attributes, 'validators') and \
            type_name not in attributes.validators:
        attributes.validators[type_name] = call


check_validators('ac_range_or_none', _validate_ac_range_or_none)
check_attributes('type:ac_range_or_none', _validate_ac_range_or_none)


class InvalidRangeInput(NeutronException):
    """Invalid Range Input"""
    message = "Invalid input for operation: %(error_message)s."


# Attribute Map
RESOURCE_ATTRIBUTE_MAP = {
    "vpc_connections": {
        'id': {
            'allow_post': False, 'allow_put': False,
            'validate': {'type:uuid': None},
            'is_visible': True, 'primary_key': True},
        'name': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:string': 255},
            'is_visible': True, 'default': ''},
        'status': {
            'allow_post': False, 'allow_put': False,
            'is_visible': True},
        'description': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:string': 255},
            'is_visible': True, 'default': ''},
        'tenant_id': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:string': 255},
            'is_visible': True, 'default': None},
        'local_router': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
            'validate': {'type:uuid': None}},
        'local_subnets': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': [],
            'validate': {'type:uuid_list': None}},
        'local_cidrs': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': [],
            'validate': {'type:list_of_strings': None}},
        'local_firewall_enable': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
            'validate': {'type:boolean_or_none': None}},
        'peer_router': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
            'validate': {'type:uuid': None}},
        'peer_subnets': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': [],
            'validate': {'type:uuid_list': None}},
        'peer_cidrs': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': [],
            'validate': {'type:list_of_strings': None}},
        'peer_firewall_enable': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
            'validate': {'type:boolean_or_none': None}},
        'mode': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': 0,
            'validate': {'type:values': [0, 1]}},
        'priority': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
            'validate': {'type:ac_range_or_none': None}},
        'fw_enabled': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
            'validate': {'type:boolean_or_none': None}},
    }
}


class Vpc_connection(ExtensionDescriptor):
    """Consistency report generation"""

    @classmethod
    def get_name(cls):
        """ Name for this extension """
        return "Vpc connection"

    @classmethod
    def get_alias(cls):
        """ core_plugin class to load the extension """
        return 'ac-vpc-connection'

    @classmethod
    def get_description(cls):
        """ A small description about this extension """
        return "Connect one vpc with another"

    @classmethod
    def get_namespace(cls):
        """ The XML namespace for this extension """
        return ""

    @classmethod
    def get_updated(cls):
        """ Specify when was this extension last updated """
        return "2017-03-14T21:10:50-00:00"

    @classmethod
    def get_resources(cls):
        """Returns Ext Resources."""
        ops_version = ncu.get_ops_version()
        if ops_version in ac_constants.OPS_VERSION_O_PQRTW_6_21:
            allow_pagination = True
            allow_sorting = True
            plugin = directory.get_plugin(VPC_CONNECTION)
        else:
            plugin = \
                manager.NeutronManager.get_service_plugins()[
                    VPC_CONNECTION]
            allow_pagination = cfg.CONF.allow_pagination
            allow_sorting = cfg.CONF.allow_sorting
        resources = []
        for resource_name in ["vpc_connection"]:
            collection_name = VPC_CONNECTION + "s"
            params = RESOURCE_ATTRIBUTE_MAP.get("vpc_connections", dict())
            member_actions = {}
            controller = base.create_resource(
                collection_name, resource_name, plugin, params, allow_bulk=True,
                member_actions=member_actions, allow_sorting=allow_sorting,
                allow_pagination=allow_pagination)
            resources.append(extensions.ResourceExtension(
                collection_name, controller, attr_map=params,
                member_actions=member_actions))

        return resources

    @classmethod
    def get_plugin_interface(cls):
        """ get plugin interface """
        return VpcConnectionBase

    def update_attributes_map(self, attributes, extension_attrs_map=None):
        """ update attributes map """
        super(Vpc_connection, self).update_attributes_map(
            attributes, extension_attrs_map=RESOURCE_ATTRIBUTE_MAP)

    def get_extended_resources(self, version):
        """ get extended resources """
        if version != '2.0':
            return {}
        return RESOURCE_ATTRIBUTE_MAP


@six.add_metaclass(abc.ABCMeta)
class VpcConnectionBase(ServicePluginBase):
    """VpcConnection class"""

    def get_plugin_type(self):
        """ get plugin type """
        pass

    def get_plugin_description(self):
        """ get plugin description """
        pass

    def get_plugin_name(self):
        """ get plugin name """
        pass

    @abc.abstractmethod
    def create_vpc_connection(self, context, vpc_connection):
        """the abstractmethod to create vpc connection"""
        pass

    @abc.abstractmethod
    def update_vpc_connection(self, context, vpc_connection_id, vpc_connection):
        """the abstractmethod to update vpc connection"""
        pass

    @abc.abstractmethod
    def delete_vpc_connection(self, context, vpc_connection_id):
        """the abstractmethod to delete vpc connection"""
        pass

    @abc.abstractmethod
    def get_vpc_connection(self, context, vpc_connection_id, fields=None):
        """the abstractmethod to get vpc connection"""
        pass

    @abc.abstractmethod
    def get_vpc_connections(self, context, filters=None, fields=None):
        """the abstractmethod to get vpc connections"""
        pass
