#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""vpc connection custom class"""

import abc

import six
from oslo_config import cfg

from neutron.api import extensions

try:
    from neutron_lib.plugins import directory
except ImportError:
    pass
from neutron import manager
try:
    from neutron.api.extensions import ExtensionDescriptor
except ImportError:
    from neutron_lib.api.extensions import ExtensionDescriptor
try:
    from neutron.common.exceptions import NeutronException
except ImportError:
    from neutron_lib.exceptions import NeutronException
try:
    from neutron.services.service_base import ServicePluginBase
except ImportError:
    from neutron_lib.services.base import ServicePluginBase

import networking_huawei.drivers.ac.common.constants as ac_constants
from networking_huawei.drivers.ac.extensions import vpc_connection_custom \
    as vpc_conn_custom_extension
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei.drivers.ac.extensions.bgpneighbor import bgpneighbor

from networking_huawei.drivers.ac.extensions.vpc_connection.vpc_connection \
    import check_validators, check_attributes

LOG = logging.getLogger(__name__)

extensions.append_api_extensions_path(vpc_conn_custom_extension.__path__)

VPC_CONNECTION = 'vpc_connection'

OPS_VERSION = ncu.get_ops_version()


def _validate_ac_range_or_none_custom(data, valid_values=None):
    """ _validate ac range or none custom """
    if data is None:
        return
    err_msg = "'%s' is not a valid range" % data
    try:
        date_int = int(data)
    except InvalidRangeInput:
        raise InvalidRangeInput(error_message=err_msg)
    if date_int not in range(1, 256):
        raise InvalidRangeInput(error_message=err_msg)


check_validators(
    'ac_range_or_none_custom', _validate_ac_range_or_none_custom)
check_attributes(
    'type:ac_range_or_none_custom', _validate_ac_range_or_none_custom)


class InvalidRangeInput(NeutronException):
    """ InvalidRangeInput """
    message = "Invalid input for operation: %(error_message)s."


# Attribute Map
RESOURCE_ATTRIBUTE_MAP = {
    "vpc_connections_custom": {
        'vpc_connection_id': {
            'allow_post': False, 'allow_put': False,
            'validate': {'type:uuid': None},
            'is_visible': True, 'primary_key': True},
        'name': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:string': 255},
            'is_visible': True},
        'description': {
            'allow_post': True, 'allow_put': True, 'is_visible': True,
            'default': '', 'validate': {'type:string': 255}},
        'tenant_id': {
            'allow_post': True, 'allow_put': True, 'is_visible': True,
            'validate': {'type:string': 255}, 'default': None},
        'local_router': {
            'allow_post': True, 'allow_put': True, 'is_visible': True,
            'default': None, 'validate': {'type:uuid': None}},
        'local_cidrs': {
            'allow_post': True, 'allow_put': True, 'is_visible': True,
            'validate': {'type:list_of_strings': None}},
        'local_firewall_enable': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
            'validate': {'type:boolean_or_none': None}},
        'peer_router': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
            'validate': {'type:uuid': None}},
        'peer_cidrs': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True,
            'validate': {'type:list_of_strings': None}},
        'peer_firewall_enable': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
            'validate': {'type:boolean_or_none': None}},
        'mode': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True,
            'validate': {'type:values': ["0", "1"]}},
        'priority': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
            'validate': {'type:ac_range_or_none_custom': None}},
    }
}


class Vpc_connection_custom(ExtensionDescriptor):
    """Consistency report generation"""

    @classmethod
    def get_name(cls):
        """Name for this extension"""
        return "Vpc connection"

    @classmethod
    def get_alias(cls):
        """core_plugin class to load the extension"""
        return 'vpc-connection-custom'

    @classmethod
    def get_description(cls):
        """A small description about this extension"""
        return "Connect one vpc with another"

    @classmethod
    def get_namespace(cls):
        """The XML namespace for this extension"""
        return ""

    @classmethod
    def get_updated(cls):
        """Specify when was this extension last updated"""
        return "2017-03-14T21:10:50-00:00"

    @classmethod
    def get_resources(cls):
        """Returns Ext Resources."""
        if OPS_VERSION in ac_constants.OPS_VERSION_O_PQRTW_6_21:
            allow_pagination = True
            allow_sorting = True
            plugin = directory.get_plugin(VPC_CONNECTION)
        else:
            plugin = manager.NeutronManager.get_service_plugins()[
                VPC_CONNECTION]
            allow_pagination = cfg.CONF.allow_pagination
            allow_sorting = cfg.CONF.allow_sorting

        resources = []
        for resource_name in [VPC_CONNECTION]:
            collection_name = VPC_CONNECTION + "s"
            params = RESOURCE_ATTRIBUTE_MAP.get("vpc_connections_custom",
                                                dict())
            member_actions = {}
            controller = bgpneighbor.create_resource_bgpneighbor(
                collection_name, resource_name, plugin, params, allow_bulk=True,
                allow_sorting=allow_sorting, member_actions=member_actions,
                allow_pagination=allow_pagination)
            resources.append(extensions.ResourceExtension(
                collection_name, controller, member_actions=member_actions,
                attr_map=params))

        return resources

    @classmethod
    def get_plugin_interface(cls):
        """ get plugin interface """
        return VpcConnectionCustomBase

    def update_attributes_map(self, attributes, extension_attrs_map=None):
        """ update attributes map """
        super(Vpc_connection_custom, self).update_attributes_map(
            attributes, extension_attrs_map=RESOURCE_ATTRIBUTE_MAP)

    def get_extended_resources(self, version):
        """ get extended resources """
        if version != '2.0':
            return {}
        return RESOURCE_ATTRIBUTE_MAP


@six.add_metaclass(abc.ABCMeta)
class VpcConnectionCustomBase(ServicePluginBase):
    """VpcConnection class"""

    def get_plugin_type(self):
        """ get plugin type """
        pass

    def get_plugin_description(self):
        """ get plugin description """
        pass

    def get_plugin_name(self):
        """ get plugin name """
        pass

    @abc.abstractmethod
    def create_vpc_connection(self, context, vpc_connection):
        """the abstractmethod to create vpc connection"""
        pass

    @abc.abstractmethod
    def update_vpc_connection(self, context, vpc_connection_id, vpc_connection):
        """the abstractmethod to update vpc connection"""
        pass

    @abc.abstractmethod
    def delete_vpc_connection(self, context, vpc_connection_id):
        """the abstractmethod to delete vpc connection"""
        pass

    @abc.abstractmethod
    def get_vpc_connection(self, context, vpc_connection_id, fields=None):
        """the abstractmethod to get vpc connection"""
        pass

    @abc.abstractmethod
    def get_vpc_connections(self, context, filters=None, fields=None):
        """the abstractmethod to get vpc connections"""
        pass
