#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Ac exroute model"""

import ast
import six
from oslo_serialization import jsonutils

from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei._i18n import _LE

LOG = logging.getLogger(__name__)


class ACExRouteModel:
    """AC exroute model converter"""

    @staticmethod
    def ac_model_format(exroutes_dict, router_id, interface_id=None):
        """ac model format"""
        try:
            exroutes_info = {
                'router-id': router_id
            }
            if 'nexthop' in exroutes_dict:
                exroutes_info['nexthop'] = exroutes_dict['nexthop']
            if 'destination' in exroutes_dict:
                exroutes_info['destination'] = exroutes_dict['destination']
            if 'description' in exroutes_dict and exroutes_dict['description'] is not None:
                if '__resource_preemption' in exroutes_dict['description']:
                    exroutes_info['description'] = exroutes_dict['description']
            if interface_id:
                exroutes_info['router-interface-id'] = interface_id
            if 'ip_version' in exroutes_dict:
                exroutes_info['ip-version'] = "%s%s%s" % (
                    ac_const.NEUTRON_CONSTANTS, 'ip-version-v',
                    str(exroutes_dict['ip_version'])
                )
            if exroutes_dict.get('type'):
                exroutes_info['type'] = int(exroutes_dict['type'])
                if exroutes_info["type"] == 401:
                    exroutes_info['ecmp-enable'] = True
            if exroutes_dict.get('ports') and exroutes_dict['ports'] != 'null':
                ports = exroutes_dict['ports']
                # to be compatible with sync format
                if isinstance(ports, six.text_type):
                    ports = ast.literal_eval(ports)
                exroutes_info['ports'] = ports
            exroutes_info = ACExRouteModel.get_exroutes_extra_opt(exroutes_dict, exroutes_info)
            if exroutes_dict.get('peer6_fw_enable'):
                exroutes_info['peer6-fw-enable'] = exroutes_dict['peer6_fw_enable']
        except KeyError as ex:
            LOG.error(_LE("[AC]Key Error, doesn't contain all fields %s."), ex)
            raise
        return exroutes_info

    @staticmethod
    def get_exroutes_extra_opt(exroutes_dict, exroutes_info):
        """get exroutes extra opt"""
        if exroutes_dict.get('extra_opt'):
            # to be compatible with sync format
            extra_opt = exroutes_dict['extra_opt']
            if isinstance(extra_opt, six.text_type) or isinstance(extra_opt, str):
                extra_opt = jsonutils.loads(extra_opt)
            if 'remote_ip' in extra_opt:
                exroutes_info['remote-vtepip'] = extra_opt.pop('remote_ip')
            source_ips = exroutes_dict.get('source_ips')
            if source_ips is not None:
                extra_opt['source_ips'] = source_ips
            bfd = exroutes_dict.get('bfd')
            if bfd is not None:
                extra_opt['health_detect'] = bfd
            json_data = (jsonutils.dumps(extra_opt)).replace(
                'true', 'True').replace('false', 'False').replace('null', "''")
            exroutes_info['extra-opt'] = str(ast.literal_eval(json_data))
        return exroutes_info
