#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Huawei network model."""

from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei._i18n import _LE
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.util import \
    ACCommonUtil, DataFilterUtil

LOG = logging.getLogger(__name__)


class ACNetworkModel(object):
    """Huawei network model."""
    @staticmethod
    def ac_model_format(network, tenant_name):
        """Convert specific network data to Huawei model.

        :param network: network data.
        :param tenant_name: tenant name of network.
        :return: Huawei network model.
        """
        try:
            network_info = {'uuid': network['id'],
                            'name': network['name'],
                            'tenant-id': network['tenant_id'],
                            'tenant-name': tenant_name,
                            'admin-state-up': network['admin_state_up'],
                            'shared': network['shared']}

            # description is not supported in kilo
            if 'description' in network:
                network_info['description'] = network['description']

            if 'provider:segmentation_id' in network:
                network_info['segmentation-id'] = \
                    network['provider:segmentation_id']

            if 'provider:physical_network' in network:
                network_info['physical-network'] = \
                    network['provider:physical_network']

            if 'vxlan_physnet_prefix' in network:
                network_info['physical-network'] = \
                    network['vxlan_physnet_prefix']

            ACNetworkModel.get_network_segments(network, network_info)
            ACNetworkModel.get_network_time(network, network_info)
            ACNetworkModel.get_network_type(network, network_info)
            ACNetworkModel.get_network_external(network, network_info)

            if network.get('vlan_transparent') is not None:
                network_info['vlan-transparent'] = \
                    network['vlan_transparent']

            if network.get('qos_policy_id') is not None and \
                    ACCommonUtil.qos_plugin_configured():
                data_filter = DataFilterUtil()
                context = ncu.neutron_context.get_admin_context()
                if not data_filter.not_in_white_or_in_black_list(
                        context, {}, ac_const.NW_HW_QOS_POLICY,
                        network['qos_policy_id']):
                    network_info['qos-policy-id'] = \
                        network['qos_policy_id']

            if 'mtu' in network:
                network_info['mtu'] = network['mtu']

            if 'dns_domain' in network:
                network_info['dns-domain'] = network['dns_domain']

        except KeyError as ex:
            LOG.error(_LE("[AC]Key Error, doesn't contain all fields %s."), ex)
            raise KeyError

        return network_info

    @staticmethod
    def get_network_segments(network, network_info):
        """Query segments that relate to specific network.

        :param network: network data.
        :param network_info: Huawei network model.
        :return: None.
        """
        if 'segments' not in network:
            return
        network_info['segments'] = []
        for segment in network['segments']:
            network_segment = {}
            if 'provider:network_type' in segment:
                network_segment['network-type'] = \
                    'huawei-ac-neutron-networks:network-type-' + \
                    segment['provider:network_type']
                if segment['provider:network_type'] == 'vxlan':
                    network_segment['segmentation-index'] = 1
                elif segment['provider:network_type'] == 'vlan':
                    network_segment['segmentation-index'] = 2
            if 'provider:segmentation_id' in segment:
                network_segment['segmentation-id'] = \
                    segment['provider:segmentation_id']
            if 'provider:physical_network' in segment and \
                    segment['provider:physical_network']:
                network_segment['physical-network'] = \
                    segment['provider:physical_network']
            network_info['segments'].append(network_segment)

    @staticmethod
    def get_network_time(network, network_info):
        """Query create and update time of specific network.

        :param network: network data.
        :param network_info: Huawei network model.
        :return: None.
        """
        if network.get('created_at'):
            network_info['created-at'] = network['created_at']
        if network.get('updated_at'):
            network_info['updated-at'] = network['updated_at']

    @staticmethod
    def get_network_type(network, network_info):
        """Query network type of specific network.

        :param network: network data.
        :param network_info: Huawei network model.
        :return: None.
        """
        if 'provider:network_type' in network \
                and network['provider:network_type'] is not None:
            if network['provider:network_type'] == 'vxlan' \
                    or network['provider:network_type'] == 'flat' \
                    or network['provider:network_type'] == 'vlan' \
                    or network['provider:network_type'] == 'gre':
                network_info['provider:network-type'] = \
                    'huawei-ac-neutron-networks:network-type-' \
                    + network['provider:network_type']
            else:
                network_info['provider:network-type'] = \
                    network['provider:network_type']

    @staticmethod
    def get_network_external(network, network_info):
        """Query router:external property of specific network.

        :param network: network data.
        :param network_info: Huawei network model.
        :return: None.
        """
        if 'router:external' in network \
                and network['router:external']:
            network_info['external'] = bool(True)
            LOG.debug("[AC]The request is for an external network.")
        else:
            network_info['external'] = bool(False)
            LOG.debug("[AC]The request is for an internal network.")
