#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Huawei port model."""

import ast
import copy

from neutron.db import securitygroups_db
from oslo_config import cfg
from oslo_serialization import jsonutils
from oslo_utils import uuidutils

from networking_huawei._i18n import _LE
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.util import ACCommonUtil, DataFilterUtil
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei.drivers.ac.db.lock import lockedobjects_db as lock_db
from networking_huawei.drivers.ac.sync import util

try:
    from neutron_lbaas.db.loadbalancer import models as lb_db
except ImportError:
    try:
        from neutron.db import loadbalancer as lb_db
    except ImportError:
        # ignore LB db import error
        pass

LOG = ncu.ac_log.getLogger(__name__)


class ACPortModel(object):
    """Huawei port model."""

    @staticmethod
    def ac_model_format(port, tenant_name):
        """Convert specific port data to Huawei model.

        :param port: port data.
        :param tenant_name: tenant name of port.
        :return: Huawei port model.
        """
        try:
            port_info = {'uuid': port['id'],
                         'name': port['name'],
                         'network-id': port['network_id'],
                         'tenant-name': tenant_name,
                         'tenant-id': port['tenant_id']}

            # description is not supported in kilo
            if 'description' in port:
                port_info['description'] = port['description']

            ACPortModel.get_port_time(port, port_info)
            ACPortModel.get_port_host_id(port, port_info)
            ACPortModel.get_port_mac_address(port, port_info)
            ACPortModel.get_port_admin_state_up(port, port_info)
            ACPortModel.get_port_device_info(port, port_info)
            ACPortModel.get_port_binding_info(port, port_info)
            ACPortModel.get_port_security_group(port, port_info)
            ACPortModel.get_port_qos_info(port, port_info)
            ACPortModel.get_port_vif_details(port, port_info)
            ACPortModel.get_port_fixed_ips(port, port_info)
            ACPortModel.get_port_allowed_address_pairs(port, port_info)
            ACPortModel.get_port_extra_dhcp_opts(port, port_info)
            ACPortModel.get_port_trunk_details(port, port_info)
            ACPortModel.get_port_dns_info(port, port_info)
            ACPortModel.get_port_security_enabled(port, port_info)

        except KeyError as ex:
            LOG.error(_LE("[AC]Key Error, doesn't contain all fields %s."), ex)
            raise

        return port_info

    @staticmethod
    def format_f5_mappings(conf):
        """Convert F5 mappings config to list.

        :param conf: LB host mappings config.
        :return: LB host mapping list.
        """
        return [mapping.split(":")
                for mapping in conf if hasattr(mapping, "split")] \
            if isinstance(conf, list) else []

    @staticmethod
    def convert_host_id(host_mappings, host_id):
        """Query host id that relate to LB host mappings.

        :param host_mappings: LB host mapping list.
        :param host_id: host id of specific LB port.
        :return: mapped host id of specific LB port.
        """
        if isinstance(host_mappings, list):
            for mapping in host_mappings:
                if host_id in mapping:
                    return mapping[0]
        return host_id

    @staticmethod
    def is_f5_port(port):
        """Check whether the port is F5 port or not.

        :param port: port data.
        :return: is F5 port or not.
        """
        if isinstance(port, dict):
            device_owner = port.get("device_owner", None)
            return device_owner in [ac_const.DEVICE_OWNER_F5_V1,
                                    ac_const.DEVICE_OWNER_F5_V2] or \
                device_owner.startswith('neutron:LOADBALANCER')
        return False

    @staticmethod
    def get_ac_port_profile(port):
        """Convert specific port binding:profile to Huawei model.

        :param port: port data.
        :return: binding:profile of Huawei port model.
        """
        profile = copy.deepcopy(port['binding:profile'])
        if util.ACUtil.is_compute_port(port):
            if "vm_name" in profile:
                vm_name = ACCommonUtil.get_base64_str(
                    profile.get("vm_name", ""))
                profile.pop("vm_name")
            else:
                device_id = port.get('device_id', None)
                vm_name = ACPortModel.get_vm_port_names(device_id)
            profile.update({'vm-name': vm_name})
        network_type = ACPortModel.get_network_type(port)
        if cfg.CONF.huawei_ac_config.physical_network and \
                (util.ACUtil.is_compute_port(port) or
                 util.ACUtil.is_dhcp_port(port)) and network_type == \
                ncu.TYPE_VXLAN and not profile.get("physical_network", None):
            profile.update({
                "physical_network": cfg.CONF.huawei_ac_config.physical_network})
        if profile.get("service_type", None):
            profile.update({"service-type": profile.get("service_type", None)})
            profile.pop("service_type")
        if 'enable_internet' in profile:
            profile.update({
                "enable-internet": profile.get("enable_internet", None)})
            profile.pop("enable_internet")
        return profile

    @staticmethod
    def get_network_type(port):
        """Query network type of specific port.

        :param port: port data.
        :return: network type of specific port.
        """
        context = ncu.neutron_context.get_admin_context()
        network_type = ncu.get_core_plugin().get_network(
            context, port['network_id']).get('provider:network_type')
        return network_type

    @staticmethod
    def get_port_time(port, port_info):
        """Query create and update time of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        if port.get('created_at'):
            port_info['created-at'] = port['created_at']
        if port.get('updated_at'):
            port_info['updated-at'] = port['updated_at']

    @staticmethod
    def get_port_host_id(port, port_info):
        """Query host id of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        if port.get('binding:host_id') is not None:
            port_info['host-id'] = port['binding:host_id']

            if ACPortModel.is_f5_port(port):
                conf = cfg.CONF.huawei_ac_config.LB_host_mappings
                mappings = ACPortModel.format_f5_mappings(conf)
                port_info['host-id'] = ACPortModel.convert_host_id(
                    mappings, port_info['host-id'])
                LOG.debug("port[%s]: convert %s to %s" %
                          (port['id'], port['binding:host_id'],
                           port_info['host-id']))

    @staticmethod
    def get_host_id_from_port(port):
        """Query host id of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        result = None
        if port.get('binding:host_id') is not None:
            result = port['binding:host_id']

            if ACPortModel.is_f5_port(port):
                conf = cfg.CONF.huawei_ac_config.LB_host_mappings
                mappings = ACPortModel.format_f5_mappings(conf)
                result = ACPortModel.convert_host_id(mappings, result)
                LOG.debug("port[%s]: convert %s to %s", port['id'], port['binding:host_id'], result)
        return result

    @staticmethod
    def get_port_mac_address(port, port_info):
        """Query MAC address of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        if port.get('mac_address') is not None:
            port_info['mac-address'] = port['mac_address']

    @staticmethod
    def get_port_admin_state_up(port, port_info):
        """Query admin state up of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        if port.get('admin_state_up') is not None:
            port_info['admin-state-up'] = port['admin_state_up']

    @staticmethod
    def get_port_device_info(port, port_info):
        """Query device owner and device id of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        if port.get('device_owner') is not None:
            port_info['device-owner'] = port['device_owner']
            if ncu.get_ops_version() in [ac_const.OPS_K]:
                name_list = port['name'].split('loadbalancer-')
                if len(name_list) == 2 and name_list[0] == '' and uuidutils. \
                        is_uuid_like(name_list[1]) and ACdbInterface(). \
                        get_session().query(lb_db.LoadBalancer). \
                        filter_by(id=name_list[1]).first():
                    port_info['device-owner'] = "neutron:LOADBALANCERV2"
        if port.get('device_id') is not None:
            port_info['device-id'] = port['device_id']

    @staticmethod
    def get_port_binding_info(port, port_info):
        """Query binding:profile/vnic_type/vif_type of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        if port.get('binding:profile') is not None:
            if 'cvk_host' in port['binding:profile']:
                port_info['host-id'] = \
                    port['binding:profile']['cvk_host']
            elif 'esxi_host' in port['binding:profile']:
                port_info['host-id'] = \
                    port['binding:profile']['esxi_host']
            json_data = (jsonutils.dumps(ACPortModel.get_ac_port_profile(
                port))). \
                replace('true', 'True').replace('false', 'False'). \
                replace('null', '\'\'')
            port_info['profile'] = str(ast.literal_eval(json_data))
        if port.get('binding:vnic_type') is not None:
            port_info['vnic-type'] = port['binding:vnic_type']
        else:
            session = ACdbInterface().get_session()
            rec = ncu.get_port_binding(session, port["id"])
            if rec:
                port_info['vnic-type'] = rec['vnic_type']
        if port.get('binding:vif_type') is not None:
            port_info['vif-type'] = port['binding:vif_type']
        else:
            session = ACdbInterface().get_session()
            rec = ncu.get_port_binding(session, port["id"])
            if rec:
                port_info['vif-type'] = rec['vif_type']
        if port_info.get('vnic-type') == 'baremetal' and \
                port_info['device-owner'].startswith('network:f5'):
            port_info['vnic-type'] = 'normal'

    @staticmethod
    def get_port_security_group(port, port_info):
        """Query security group of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        if cfg.CONF.huawei_ac_config.enable_security_group:
            if port.get('security_groups') is not None:
                port_info['security-groups'] = port['security_groups']
            else:
                session = ACdbInterface().get_session()
                rec_list = session. \
                    query(securitygroups_db.SecurityGroupPortBinding). \
                    filter_by(port_id=port["id"]).all()
                add_list = []
                for rec in rec_list:
                    add_list.append(rec['security_group_id'])
                if add_list:
                    port_info['security-groups'] = add_list

    @staticmethod
    def get_port_qos_info(port, port_info):
        """Query QoS policy id of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        if port.get('qos_policy_id') is not None and \
                ACCommonUtil.qos_plugin_configured():
            # if qos plugin configured, add the qos policy in port
            data_filter = DataFilterUtil()
            context = ncu.neutron_context.get_admin_context()
            if not data_filter.not_in_white_or_in_black_list(
                    context, {}, ac_const.NW_HW_QOS_POLICY,
                    port['qos_policy_id']):
                port_info['qos-policy-id'] = port['qos_policy_id']

        if port.get('qos') is not None:
            port_info['qos-policy-id'] = port['qos']

    @staticmethod
    def get_port_vif_details(port, port_info):
        """Query vif details of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        if port.get('binding:vif_details') is not None:
            detail = {}
            details = []
            if 'port_filter' in port['binding:vif_details'] \
                    and 'ovs_hybrid_plug' in port['binding:vif_details']:
                if 'port_filter' in port['binding:vif_details']:
                    detail['port-filter'] = \
                        port['binding:vif_details']['port_filter']
                if 'ovs_hybrid_plug' in port['binding:vif_details']:
                    detail['ovs-hybrid-plug'] = \
                        port['binding:vif_details'][
                            'ovs_hybrid_plug']
                details.append(detail)
            port_info['vif-details'] = details
        return port_info

    @staticmethod
    def get_port_fixed_ips(port, port_info):
        """Query fixed ips of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        if port.get('fixed_ips') is not None:
            port_info['fixed-ips'] = []
            for fixed_ip in port['fixed_ips']:
                port_info['fixed-ips'].append({
                    'subnet-id': fixed_ip['subnet_id'],
                    'ip-address': fixed_ip['ip_address']
                })

    @staticmethod
    def get_port_allowed_address_pairs(port, port_info):
        """Query allowed address pairs of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        if port.get('allowed_address_pairs') is not None:
            allowedaddresspairs = []
            for item in port['allowed_address_pairs']:
                allowedaddresspair = {}
                allowedaddresspair['mac-address'] = item[
                    'mac_address']
                allowedaddresspair['ip-address'] = item['ip_address']
                allowedaddresspairs.append(allowedaddresspair)
            port_info['allowed-address-pairs'] = allowedaddresspairs

    @staticmethod
    def get_port_extra_dhcp_opts(port, port_info):
        """Query extra DHCP options of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        if port.get('extra_dhcp_opts') is not None:
            extradhcpopts = []
            for item in port['extra_dhcp_opts']:
                extradhcpopt = {'opt-name': item['opt_name'],
                                'opt-value': item['opt_value']}
                if item.get('ip_version') is not None:
                    ip_version = 'huawei-ac-neutron-constants:ip-version-v%s' \
                                 % str(item['ip_version'])
                    extradhcpopt['ip-version'] = ip_version
                extradhcpopts.append(extradhcpopt)
            port_info['extra-dhcp-opts'] = extradhcpopts
        return port_info

    @staticmethod
    def get_port_trunk_details(port, port_info):
        """Query trunk details of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        if port.get('trunk_details') is not None:
            trunk_details = {}
            sub_ports = []
            trunk_details['trunk-id'] = port['trunk_details']['trunk_id']
            for item in port['trunk_details']['sub_ports']:
                sub_port = {}
                sub_port['segmentation-id'] = item['segmentation_id']
                if item['segmentation_type'] == 'vlan':
                    sub_port['segmentation-type'] = \
                        'huawei-ac-neutron-networks:network-type-vlan'
                else:
                    sub_port['segmentation-type'] = \
                        item['segmentation_type']
                sub_port['port-id'] = item['port_id']
                sub_port['mac-address'] = item['mac_address']
                sub_ports.append(sub_port)
            trunk_details['sub-ports'] = sub_ports
            port_info['trunk-details'] = trunk_details
        return port_info

    @staticmethod
    def get_port_dns_info(port, port_info):
        """Query DNS name and DNS assignment of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        if 'dns_name' in port:
            port_info['dns-name'] = port['dns_name']

        if 'dns_assignment' in port:
            port_info['dns-assignment'] = []
            for assignment in port['dns_assignment']:
                port_info['dns-assignment'].append({
                    'hostname': assignment['hostname'],
                    'fqdn': assignment['fqdn'],
                    'ip-address': assignment['ip_address'],
                })

    @staticmethod
    def get_port_security_enabled(port, port_info):
        """Query port security enabled of specific port.

        :param port: port data.
        :param port_info: Huawei port model.
        :return: None.
        """
        if 'port_security_enabled' in port:
            port_info['port-security-enabled'] = \
                port['port_security_enabled']

    @staticmethod
    @lock_db.wrap_db_lock(lock_db.GET_VM_PORT_NAME)
    def get_vm_port_names(device_id):
        """Query VM port name of specific port.

        :param device_id: device id of specific port.
        :return: VM port name.
        """
        return ACCommonUtil.get_vm_port_name(device_id)
