#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Huawei router model."""
import copy

from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei._i18n import _LE

LOG = ncu.ac_log.getLogger(__name__)


class ACRouterModel(object):
    """Huawei router model."""

    @staticmethod
    def ac_model_format(router_dict, tenant_name):
        """Convert specific router data to Huawei model.

        :param router_dict: router data.
        :param tenant_name: tenant name of router.
        :return: Huawei router model.
        """
        try:
            router_dict_info = {
                'uuid': router_dict['id'],
                'name': router_dict['name'],
                'tenant-id': router_dict['tenant_id'],
                'tenant-name': tenant_name,
                'admin-state-up': router_dict['admin_state_up'],
                'routes': copy.deepcopy(router_dict['routes'])
            }

            # description is not supported in kilo
            if 'description' in router_dict:
                router_dict_info['description'] = router_dict['description']

            if 'GROUP_POLICY' in ncu.get_service_plugin():
                router_dict_info.pop('routes', None)

            if router_dict.get('gw_port_id'):
                router_dict_info['gateway-port-id'] = router_dict['gw_port_id']

            for route in router_dict_info.get('routes', []):
                route.pop('route_type', None)

            ACRouterModel.get_router_time(router_dict, router_dict_info)
        except KeyError as value:
            LOG.error(_LE("[AC]Key Error, doesn't contain all fields %s."),
                      value)
            raise KeyError

        if router_dict.get('external_gateway_info'):
            egw_info = router_dict['external_gateway_info']
            try:
                router_dict_info['external-gateway-info'] = {
                    'external-network-id': egw_info['network_id'],
                    'enable-snat': egw_info['enable_snat'],
                    'external-fixed-ips': []
                }
                external_fixed_ips = []
                for fixed_ip in egw_info.get('external_fixed_ips', []):
                    external_fixed_ips.append({
                        'subnet-id': fixed_ip['subnet_id'],
                        'ip-address': fixed_ip['ip_address'],
                    })
                if egw_info.get('external_fixed_ips', None):
                    router_dict_info['external-gateway-info'].update({
                        'external-fixed-ips': external_fixed_ips
                    })
            except KeyError as value:
                LOG.error(_LE("[AC]Key Error, "
                              "doesn't contain all fields %s."), value)
                raise KeyError
        return router_dict_info

    @staticmethod
    def get_router_time(router_dict, router_dict_info):
        """Query create and update time of specific router.

        :param router_dict: router data.
        :param router_dict_info: Huawei router model.
        :return: None.
        """
        if router_dict.get('created_at'):
            router_dict_info['created-at'] = router_dict['created_at']
        if router_dict.get('updated_at'):
            router_dict_info['updated-at'] = router_dict['updated_at']
