# !/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.

import csv
import os
from codecs import BOM_UTF8
from neutronclient.common import extension

# neutron client cann't import "logging" module,whatever direct or indirect
from networking_huawei._i18n import _
from networking_huawei.drivers.ac.plugins.config_check.ac_config_requests import AcConfigCreateReq


class AcConfigClientExt(extension.NeutronClientExtension):
    """some common info of NeutronClientCommand"""
    resource = 'ac_config'  # Ac_config_ext().get_alias()
    resource_plural = '%s' % resource
    object_path = '/%s' % resource_plural
    resource_path = '/%s/%%s' % resource_plural
    versions = ['2.0']
    allow_names = False


class CreateAcConfig(extension.ClientExtensionCreate, AcConfigClientExt):
    """save ac config to db"""
    shell_command = 'acctrl-config-save'

    def add_known_arguments(self, parser):
        """命令行参数"""
        parser.add_argument('--auto_clear', choices=('y', 'n'),
                            help=_('decide clear config information."y":default,should clear;"n":no.'))
        parser.add_argument('--file', help=_('config file path.'))
        return parser

    def args2body(self, parsed_args):
        """Convert command line parameters to request body

        :param parsed_args: Namespace,command line parameters
        :return: dict
        """
        result = AcConfigCreateReq(auto_clear=parsed_args.auto_clear != 'n', file=parsed_args.file)
        return {self.resource: result.to_dict()}


class DeleteAcConfig(extension.ClientExtensionDelete, AcConfigClientExt):
    """delete ac config"""
    # if id is all,will clear all config.
    shell_command = 'acctrl-config-delete'


class ShowAcConfig(extension.ClientExtensionShow, AcConfigClientExt):
    """query ac config"""
    shell_command = 'acctrl-config-show'


class ListAcConfig(extension.ClientExtensionList, AcConfigClientExt):
    """list ac config"""
    shell_command = 'acctrl-config-list'
    list_columns = ['id', 'config_file', 'owner', 'user_group', 'right', 'create_time', 'config_value']


class AcConfigCheckClientExt(extension.NeutronClientExtension):
    """some common info of NeutronClientCommand"""
    resource = 'ac_config_check'  # Config_check_ext().get_alias()
    resource_plural = '%s' % resource
    object_path = '/%s' % resource_plural
    resource_path = '/%s/%%s' % resource_plural
    versions = ['2.0']
    allow_names = False


class ListAcConfigCheck(extension.ClientExtensionList, AcConfigCheckClientExt):
    """check ac config of other node"""
    shell_command = 'acctrl-config-check'
    # see networking_huawei.drivers.ac.plugins.config_check.config_check_plugin.AssertElem.to_dict()
    list_columns = ['filepath', 'name', 'expected', 'actual', 'is_same']

    def add_known_arguments(self, parser):
        """add command line parameters"""
        parser.add_argument("--host", required=True, help=_('host'))
        parser.add_argument("--username", required=True, help=_('the user who can login by ssh.'))
        parser.add_argument("--password", required=True, help=_('the password of login user.'))
        parser.add_argument("--port", default=22, help=_('the port of login by ssh,default 22.'))
        parser.add_argument("--neutron_user", help=_('the user who can read neutron config.'))
        parser.add_argument("--neutron_password", help=_('the password of user who can read config.'))
        parser.add_argument("--output", help=_('filename,save compare result to txt file in "/var/log/neutron".'))
        parser.add_argument("--max_size", type=int, default=50, help=_('the max size of character to show.'))
        parser.add_argument("--show_same", choices=('y', 'n'), default='n',
                            help=_('decide show same config or not."y":show;"n":default,no.'))
        return parser

    def args2search_opts(self, parsed_args):
        """Convert command line parameters to request body

        :param parsed_args:Namespace,command line parameters
        :return: dict
        """
        neutron_user = parsed_args.neutron_user if parsed_args.neutron_user else parsed_args.username
        neutron_password = parsed_args.neutron_password if parsed_args.neutron_password else parsed_args.password
        return {'host': parsed_args.host, 'port': parsed_args.port,
                'username': parsed_args.username, 'password': parsed_args.password,
                'neutron_user': neutron_user, 'neutron_password': neutron_password,
                'should_show_same': parsed_args.show_same == 'y' or (parsed_args.output is not None)}

    def execute(self, parsed_args):
        """execute command"""
        temp = super(ListAcConfigCheck, self).execute(parsed_args)
        # save the results to a file and put them in the “/var/log/neutron” directory
        if parsed_args.output and parsed_args.output.find(os.path.sep) == -1:
            flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
            with os.fdopen(os.open(os.path.join('/var/log/neutron', parsed_args.output), flags, 0o640), 'w') as fw:
                fw.write(BOM_UTF8)
                csv_writer = csv.writer(fw, dialect=csv.excel())
                csv_writer.writerow(temp[0])
                csv_writer.writerows(temp[1])
            return [], []

        max_size = parsed_args.max_size
        result = []
        for elem in temp[1]:
            add_elem = []
            for value in elem:
                if len(value) > max_size:
                    add_elem.append(value[:max_size] + '...')
                else:
                    add_elem.append(value)
            result.append(add_elem)
        return temp[0], result
