# !/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2021 Huawei Technologies Co. Ltd. All rights reserved.
"""
| 功能：HuaweiACL3RouterPlugin的父类，为了解决大文件问题
| 版权信息：华为技术有限公司，版本所有(C) 2010-2017
"""

import six
import contextlib
import inspect

import netaddr
from neutron.db import l3_db
from neutron.db import models_v2
try:
    from neutron.extensions.external_net import EXTERNAL
except ImportError:
    from neutron_lib.api.definitions.external_net import EXTERNAL
from neutron.services.l3_router import l3_router_plugin
from oslo_config import cfg
from oslo_utils import excutils
from oslo_utils import uuidutils
from sqlalchemy.orm import exc

from networking_huawei._i18n import _LI
from networking_huawei._i18n import _LE
# import config register for fusionsphere
from networking_huawei.drivers.ac.common import constants as ac_cnst
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler
from networking_huawei.drivers.ac.common.util import \
    ACCommonUtil
from networking_huawei.drivers.ac.db.exroute.exroute_custom_db import \
    ExroutesDbMixinCustom
from networking_huawei.drivers.ac.db.lock import lockedobjects_db as lock_db
from networking_huawei.drivers.ac.external.ext_if import ACKeyStoneIf
from networking_huawei.drivers.ac.model.port_model import ACPortModel
from networking_huawei.drivers.ac.model.router_model import ACRouterModel
from networking_huawei.drivers.ac.model.exroute_model import ACExRouteModel
from networking_huawei.drivers.ac.model.floatingip_model \
    import ACFloatingipModel
from networking_huawei.drivers.ac.plugins.l3.l3_router_util import \
    generate_floatingip_port_desc, handle_router_exception, HuaweiACL3Util, \
    ExRoutesException, is_router_has_gw_port

try:
    from neutron.extensions.l3 import RouterInterfaceNotFoundForSubnet, RouterInterfaceNotFound
except ImportError:
    from neutron_lib.exceptions.l3 import RouterInterfaceNotFoundForSubnet, RouterInterfaceNotFound
try:
    from neutron.common.exceptions import \
        BadRequest, ExternalIpAddressExhausted, PortNotFound
except ImportError:
    from neutron_lib.exceptions import \
        BadRequest, ExternalIpAddressExhausted, PortNotFound
try:
    from neutron.db.l3_db import RouterPort, FloatingIP
except ImportError:
    from neutron.db.models.l3 import RouterPort, FloatingIP
try:
    from neutron import context as neutron_context
except ImportError:
    from neutron_lib import context as neutron_context
try:
    from neutron.objects import router as l3_obj
except ImportError:
    l3_obj = None
try:
    from networking_cascading.db.exroutes_db import Exroutes_db_mixin
except ImportError:
    from networking_huawei.drivers.ac.db.exroute.exroute import \
        Exroutes_db_mixin

LOG = ncu.ac_log.getLogger(__name__)


class AbstractHwACL3RouterPlugin(l3_router_plugin.L3RouterPlugin,
                                 Exroutes_db_mixin, ExroutesDbMixinCustom,
                                 HuaweiACL3Util):
    """HuaweiACL3RouterPlugin的父类"""

    @lock_db.wrap_db_lock(lock_db.SUBNET_OF_PUBLIC_SERVICE)
    def add_router_interface_public_network(self, context, tenant_id,
                                            router_id, subnet_id):
        """Add router interface for public network."""
        LOG.info(_LI('[AC]Begin to add router interface of public network:'
                     ' %s'), subnet_id)
        ac_osprofiler.record_chain_start("create router port:" + str(router_id))
        router = self._get_router(context, router_id)
        for port in (rp.port for rp in router.attached_ports):
            for fixed_ip in port['fixed_ips']:
                if fixed_ip['subnet_id'] == subnet_id:
                    LOG.info(_LI('Router already has a port on subnet '
                                 '%s'), subnet_id)
                    return

        self._create_ipv6_interface_port(context, tenant_id,
                                         router_id, subnet_id)
        LOG.info(_LI("[AC]Huawei AC "
                     "create public network ipv6 interface port successful."))

        ac_osprofiler.record_chain_end()

    def _create_ipv6_interface_port(self, context, tenant_id,
                                    router_id, subnet_id):
        """create ipv6 interface port"""
        admin_context = neutron_context.get_admin_context()
        subnet = self._core_plugin.get_subnet(admin_context, subnet_id)
        tag = 'service_type=Internet6'
        port_data = {
            'port': {
                'tenant_id': tenant_id, 'network_id': subnet['network_id'],
                'fixed_ips': [{'subnet_id': subnet['id']}],
                'admin_state_up': True, 'device_id': router_id,
                'device_owner': 'network:router_interface', 'name': '',
                'mac_address': ACCommonUtil.get_mac(ncu.get_ops_version()),
                'binding:profile': {
                    'service_type': 'Internet6', 'enable_internet': True}}}
        try:
            LOG.info(_LI('[AC] begin to create public network ipv6 port,'
                         'the port info is: %s'), port_data)
            interface_port = ncu.get_core_plugin().create_port(
                admin_context, port_data)
            LOG.info(_LI('Create port success: %s'), interface_port)
            from neutron.db import tag_db
            with admin_context.session.begin():
                port_db = admin_context.session.query(models_v2.Port).filter_by(
                    id=interface_port['id']).first()
                router_port = RouterPort(
                    port_id=interface_port['id'], router_id=router_id,
                    port_type='network:router_interface')
                admin_context.session.add(router_port)
                tag_info = tag_db.Tag(
                    standard_attr_id=port_db.standard_attr_id, tag=tag)
                admin_context.session.add(tag_info)
                admin_context.session.flush()
        except Exception as ex:
            LOG.error(_LE("[AC]Huawei AC create ipv6 interface port failed. "
                          "Roll back: Delete interface port in Neutron DB. "
                          "catch: %s"), ex)
            super(AbstractHwACL3RouterPlugin, self).remove_router_interface(
                context, router_id, {'subnet_id': subnet_id})
            ac_osprofiler.record_chain_exception("create interface port fail, "
                                                 "delete it")
            raise BadRequest(resource='interface port', msg=ex)

    def _delete_gate_way_port_ops_ez_m(self, context, router, router_id):
        if ncu.get_ops_version() in [ac_cnst.OPS_EZ_M] and router.gw_port:
            gw_port_id = router.get('gw_port_id')
            LOG.info(_LI('[AC] begin to delete gateway port for %s'),
                     router_id)
            self.l3_reliability.update_plugin_record(
                context, gw_port_id, {}, ac_cnst.NW_HW_DELETE_PORT)
            try:
                gw_port = self._core_plugin.get_port(context, gw_port_id)
            except PortNotFound:
                LOG.info(_LI('[AC] gateway port %s could not be found'),
                         gw_port_id)
            else:
                gw_port_info = ACPortModel.ac_model_format(gw_port, None)
                LOG.info(_LI('[AC] begin to create floating ip port for %s'),
                         router_id)
                self.l3_reliability.update_plugin_record(
                    context, gw_port_id, gw_port_info,
                    ac_cnst.NW_HW_CREATE_PORT)

    def _check_router_gw_port_in_use(self, context, router_id):
        if not ncu.IS_FSP:
            super(AbstractHwACL3RouterPlugin, self)._check_router_gw_port_in_use(
                context, router_id)

    def _convert_ez_gateway_port(self, context, router_id, original, router):
        LOG.info(_LI('[AC] begin to convert gateway port for %s'), router_id)
        original_gateway_info = original.get('external_gateway_info')
        current_gateway_info = router['router'].get('external_gateway_info')

        current = super(AbstractHwACL3RouterPlugin, self).update_router(
            context, router_id, router)
        LOG.info(_LI('[AC] current router: %s'), current)

        original_gw_port_id = original.get('gw_port_id')
        current_gw_port_id = current.get('gw_port_id')

        try:
            if not original_gateway_info and current_gateway_info and \
                    current_gw_port_id:
                LOG.info(_LI('[AC] begin to delete floating ip port for %s'),
                         router_id)
                self.l3_reliability.update_plugin_record(
                    context, current_gw_port_id, {},
                    ac_cnst.NW_HW_DELETE_PORT)
                gw_port = self._core_plugin.get_port(
                    context, current_gw_port_id)
                gw_port_info = ACPortModel.ac_model_format(gw_port, None)
                LOG.info(_LI('[AC] begin to create gateway port for %s'),
                         router_id)
                self.l3_reliability.update_plugin_record(
                    context, current_gw_port_id, gw_port_info,
                    ac_cnst.NW_HW_CREATE_PORT)
            elif original_gateway_info and not current_gateway_info and \
                    original_gw_port_id:
                self._delete_gateway_port(
                    context, original_gateway_info, router_id)

            tenant_name = ACKeyStoneIf.get_tenant_name(current, context)
            router_info = ACRouterModel.ac_model_format(current, tenant_name)
            self.l3_reliability.update_plugin_record(
                context, router_id, router_info,
                ac_cnst.NW_HW_UPDATE_ROUTER)
        except Exception as ex:
            LOG.error(_LE('[AC] update router failed: %s'), ex)
            original['status'] = ac_cnst.NEUTRON_STATUS_ERROR
            original.pop('availability_zone_hints', [])
            super(AbstractHwACL3RouterPlugin, self).update_router(
                context, router_id, {'router': original})
            raise

        LOG.info(_LI('[AC] update router %s successfully'), router_id)
        return current

    def _update_router_rollback(self, context, tenant_name,
                                rollback_router_data, exception):
        """Update router rollback."""
        original_router, router_id, original_gw_info, current_gw_info, \
        original_gw_port = rollback_router_data
        if original_gw_info and not current_gw_info:
            msg = "[AC] Failed to delete %s gateway port: %s" % \
                  (router_id, exception)
            LOG.error(msg)
            ac_osprofiler.record_call_chain(msg)
            with context.session.begin(subtransactions=True):
                router = self._get_router(context, router_id)
                router.gw_port_id = original_router['gw_port_id']
                router.enable_snat = original_gw_info.get('enable_snat')
                context.session.merge(router)
                context.session.flush()
        elif original_gw_info and current_gw_info:
            if is_router_has_gw_port(context, router_id):
                msg = "[AC] Failed to delete %s gateway port: %s" % \
                      (router_id, exception)
                LOG.error(msg)
                ac_osprofiler.record_call_chain(msg)
                with context.session.begin(subtransactions=True):
                    router = self._get_router(context, router_id)
                    router.gw_port = original_gw_port
                    context.session.add(router)
            elif original_gw_port:
                msg = "[AC] Failed to create %s gateway port: %s" % \
                      (router_id, exception)
                LOG.error(msg)
                ac_osprofiler.record_call_chain(msg)
                admin_context = neutron_context.get_admin_context()
                router = self._get_router(admin_context, router_id)
                try:
                    self._rollback_gateway_port(
                        admin_context, router, original_gw_port,
                        original_gw_info.get('external_fixed_ips'))
                except Exception as ex:
                    msg = "[AC] Failed to rollback %s gateway port: %s" % \
                          (router_id, ex)
                    handle_router_exception(admin_context, msg, router)
                try:
                    original_router_info = ACRouterModel.ac_model_format(
                        original_router, tenant_name)
                    self.l3_reliability.update_plugin_record(
                        context, router_id, original_router_info,
                        ac_cnst.NW_HW_UPDATE_ROUTER)
                except Exception as ex:
                    msg = "[AC] Failed to update router %s: %s" % \
                          (router_id, ex)
                    handle_router_exception(admin_context, msg, router)

    def _process_to_remove_interface(self, context, port, router_id):
        """Remove router interface."""

        def _check_none(fixed_ips, ip_address, enable_internet):
            return fixed_ips and ip_address and enable_internet

        fixed_ips = port.get('fixed_ips', [{}])
        if fixed_ips:
            ip_address = fixed_ips[0].get('ip_address', None)
            subnet_id = fixed_ips[0].get('subnet_id', None)
            subnet_ids = {'subnet_id': subnet_id}
            profile = port.get('binding:profile', None)
            if profile:
                service_type = profile.get('service_type', None)
                enable_internet = profile.get('enable_internet', None)
                if _check_none(fixed_ips, ip_address, enable_internet) \
                        and netaddr.valid_ipv6(ip_address) \
                        and service_type == 'Internet6':
                    LOG.info(_LI("Remove router ipv6 interface port from : "
                                 "%s"), subnet_ids)
                    super(AbstractHwACL3RouterPlugin, self).remove_router_interface(
                        context, router_id, subnet_ids)
                    return subnet_id
        return None

    def _update_router_postcommit(self, context, router_info, router_id,
                                  tenant_name):
        current_router, current_gw_info, original_router, original_gw_info = \
            router_info
        LOG.info(_LI("[AC] Current router: %s"), current_router)
        current_router_info = ACRouterModel.ac_model_format(
            current_router, tenant_name)

        try:
            self._send_notify(
                ac_cnst.NW_HW_ROUTERS, ac_cnst.BEFORE_AC_UPDATE, current_router)
            self.l3_reliability.update_plugin_record(
                context, router_id, current_router_info,
                ac_cnst.NW_HW_UPDATE_ROUTER)
            self._send_notify(
                ac_cnst.NW_HW_ROUTERS, ac_cnst.AFTER_AC_UPDATE, current_router)
            if ncu.IS_FSP:
                self.process_public_network_ipv6_port(
                    context, original_gw_info, current_gw_info, current_router)
        except Exception as ex:
            msg = "[AC] Failed to update router %s: %s" % (router_id, ex)
            LOG.error(msg)
            ac_osprofiler.record_chain_exception_end(msg)
            self._handle_update_router_failure(
                context, original_router, current_router, tenant_name)
            raise BadRequest(resource='router', msg=ex)

    def _update_router_precommit(self, context, router, router_id):
        self.l3_reliability.check_neutron_sync(
            context, ac_cnst.OPER_UPDATE, ac_cnst.NW_HW_ROUTERS, router_id)
        msg = "[AC] Begin to update router %s: %s" % (router_id, router)
        LOG.info(msg)
        ac_osprofiler.record_chain_start(msg)
        original_router = self.get_router(context, router_id)
        LOG.info(_LI("[AC] Original router: %s"), original_router)
        self.modify_router_info(router)
        original_gw_info = original_router.get('external_gateway_info', {})
        tenant_name = ACKeyStoneIf.get_tenant_name(original_router, context)
        if router['router'].get('name'):
            can_update, error_msg = self.data_filter.name_can_be_updated(
                context, router, {'router': original_router},
                ac_cnst.NW_HW_ROUTERS)
            if not can_update:
                raise BadRequest(resource='router', msg=error_msg)
        return original_gw_info, original_router, tenant_name

    def _handle_update_router_failure(self, context, original_router,
                                      current_router, tenant_name):
        original_router['status'] = ac_cnst.NEUTRON_STATUS_ERROR
        original_router.pop('availability_zone_hints', [])
        original_router.pop('gw_port_id', [])
        original_gw_info = original_router.get('external_gateway_info', {})
        router_id = current_router['id']

        try:
            rollback_router = super(AbstractHwACL3RouterPlugin, self).update_router(
                context, router_id, {'router': original_router})
        except Exception as ex:
            LOG.info(_LI("[AC] Rollback router %s fail"), original_router)
            self._handle_rollback_router_failure(
                context, current_router, original_gw_info, router_id)
            raise BadRequest(resource='router', msg=ex)

        LOG.info(_LI("[AC] Rollback router: %s"), rollback_router)
        rollback_router_info = ACRouterModel.ac_model_format(
            rollback_router, tenant_name)
        self.l3_reliability.update_plugin_record(
            context, router_id, rollback_router_info,
            ac_cnst.NW_HW_UPDATE_ROUTER)

    def _get_fip_networks(self, context):
        """Get floating IP networks."""
        fip_networks = self.get_default_public_service_networks(context)
        if fip_networks:
            LOG.info(_LI('[AC] Get default public service networks: %s'),
                     fip_networks)
            return fip_networks
        return None

    def _get_public_network_id(self, context, router_id):
        router_gateway_query = context.session.query(models_v2.Port)
        router_gateway_query = router_gateway_query.join(RouterPort)
        router_gateway = router_gateway_query.filter(
            RouterPort.router_id == router_id,
            RouterPort.port_type == l3_db.DEVICE_OWNER_ROUTER_GW
        ).first()
        if not router_gateway:
            LOG.error(_LE('[AC] No gateway found for router %s'), router_id)
            return self._get_fip_networks(context)
        external_network_id = router_gateway.network_id
        external_network = self._core_plugin.get_network(
            context, external_network_id)
        if not external_network:
            LOG.error(_LE('[AC] No external network found for id %s'),
                      external_network_id)
            admin_context = neutron_context.get_admin_context()
            return self._get_fip_networks(admin_context)
        for elem in cfg.CONF.huawei_ac_config.public_service_networks:
            external_network_name = external_network.get('name', '')
            if external_network_name.endswith(elem):
                filters = {'name': [elem], EXTERNAL: [True]}
                fields = ['id']
                public_networks = self._core_plugin.get_networks(
                    context, filters=filters, fields=fields)
                LOG.info(_LI('[AC] Public service network found: %s'),
                         public_networks)
                if not public_networks or len(public_networks) > 1:
                    LOG.error(_LE('[AC]No public service network found for '
                                  'name %s' % elem))
                    return self._get_fip_networks(context)
                return public_networks
        return self._get_fip_networks(context)

    def _create_public_service_fip(self, context, port, router_id):
        LOG.info(_LI('[AC]Begin to create public service FIP,port:%s,router id:'
                     '%s'), port, router_id)

        public_networks = self._validate_tenant_id(context, port, router_id)
        if not public_networks:
            ac_osprofiler.record_call_chain("no public service netwrok,return")
            return

        public_service_fip = dict()
        public_service_fip['floatingip'] = {
            'subnet_id': None, 'tenant_id': port['tenant_id'],
            'floating_network_id': public_networks[0]['id'],
            'fixed_ip_address': None, 'floating_ip_address': None,
            'port_id': port['id']}

        self.create_floatingip(context, public_service_fip)

    def _delete_public_service_fip(self, context, port, router_id):
        LOG.info(_LI('[AC]Begin to delete public service FIP,port:%s,router id:'
                     '%s'), port, router_id)

        public_networks = self._validate_tenant_id(context, port, router_id)
        if not public_networks:
            ac_osprofiler.record_call_chain("no public service ip found")
            return

        filters = {'port_id': [port['id']],
                   'floating_network_id': [public_networks[0]['id']]}
        fips = self.get_floatingips(context, filters=filters, fields=['id'])
        if fips:
            LOG.info(_LI('[AC] Begin to delete public service FIP: %s'), fips)
            self.delete_floatingip(context, fips[0]['id'])

    def _confirm_remove_by_subnet(self, context, interface_info, router_id):
        """Confirm router interface remove by subnet."""
        port_id = interface_info.get('port_id')
        subnet_id = interface_info.get('subnet_id')
        LOG.debug(_LI('[AC]Get port id by subnet id: %s'), subnet_id)
        subnet = self._core_plugin.get_subnet(context, subnet_id)
        if not (ncu.IS_FSP and cfg.CONF.huawei_ac_config.vpc_peering):
            subnet_req = subnet if ncu.get_ops_version() in [ac_cnst.OPS_W] else subnet_id
            self._confirm_router_interface_not_in_use(
                context, router_id, subnet_req)
        device_owner = self._get_device_owner(context, router_id)
        try:
            rport_qry = context.session.query(models_v2.Port). \
                join(RouterPort)
            ports = rport_qry.filter(
                RouterPort.router_id == router_id,
                RouterPort.port_type == device_owner,
                models_v2.Port.network_id == subnet['network_id']
            )
            for port in ports:
                port_subnets = [fip['subnet_id'] for fip in port['fixed_ips']]
                LOG.debug(_LI('[AC]Port subnets is %s'), port_subnets)
                if subnet_id in port_subnets:
                    port_id = port['id']
                    LOG.debug(_LI('[AC]Port id is %s'), port_id)
                    break
            if not port_id:
                raise RouterInterfaceNotFoundForSubnet(router_id=router_id,
                                                       subnet_id=subnet_id)
        except exc.NoResultFound:
            ac_osprofiler.record_chain_exception_end("router interface not"
                                                     " found")
            raise RouterInterfaceNotFoundForSubnet(router_id=router_id,
                                                   subnet_id=subnet_id)
        return port_id

    def _confirm_remove_by_port(self, context, interface_info, router_id):
        """Confirm router interface remove by port."""
        port_id = interface_info.get('port_id')
        device_owner = self._get_device_owner(context, router_id)
        qry = context.session.query(RouterPort)
        qry = qry.filter_by(
            port_id=port_id, router_id=router_id, port_type=device_owner)
        try:
            port_db = qry.one().port
        except exc.NoResultFound:
            ac_osprofiler.record_chain_exception_end("router interface not"
                                                     " found")
            raise RouterInterfaceNotFound(router_id=router_id,
                                          port_id=port_id)
        port_subnet_ids = [fixed_ip['subnet_id']
                           for fixed_ip in port_db['fixed_ips']]
        for port_subnet_id in port_subnet_ids:
            if not (ncu.IS_FSP and cfg.CONF.huawei_ac_config.vpc_peering):
                subnet_req = port_subnet_id
                if ncu.get_ops_version() in [ac_cnst.OPS_W]:
                    subnet_req = self._core_plugin.get_subnet(context, port_subnet_id)
                self._confirm_router_interface_not_in_use(
                    context, router_id, subnet_req)
        return port_id

    def _confirm_removing(self, context, router_id, interface_info):
        if six.PY3:
            validate_interface_info_args = inspect.getfullargspec(self._validate_interface_info).args
        else:
            validate_interface_info_args = inspect.getargspec(self._validate_interface_info).args
        if 'for_removal' in validate_interface_info_args:
            _, remove_by_subnet = self._validate_interface_info(
                interface_info, for_removal=True)
        else:
            _, remove_by_subnet = self._validate_interface_info(interface_info)

        if remove_by_subnet:
            port_id = self._confirm_remove_by_subnet(
                context, interface_info, router_id)
        else:
            port_id = self._confirm_remove_by_port(
                context, interface_info, router_id)

        return port_id

    def is_multi_outlet_port(self, context, port_id):
        """ is multi outlet scenario port """
        port = self._core_plugin.get_port(context, port_id)
        profile = port['binding:profile']
        if profile.get("enable_snat") or profile.get("fw_enabled"):
            return True
        return False

    def _validate_floatingip(self, context, floatingip):
        """Validate floating IP."""
        LOG.debug(_LI('[AC] Begin to validate floating IP: %s.'), floatingip)
        floatingip_dict = floatingip['floatingip']
        if floatingip_dict.get('port_id'):
            internal_port = self._core_plugin.get_port(
                context, floatingip_dict['port_id'])
            if internal_port and internal_port.get('device_owner') == \
                    ncu.DEVICE_OWNER_ROUTER_INTF:
                if (not floatingip_dict.get('floating_ip_address')) or \
                        (not floatingip_dict.get('floating_network_id')):
                    msg = 'floating IP address and floating network id is ' \
                          'needed for router interface association'
                    LOG.error(_LE('[AC]Failed to validate floating IP:%s.'), msg)
                    raise BadRequest(resource='floatingip', msg=msg)

                father_floatingip = ACCommonUtil.get_father_floatingip(
                    context, floatingip_dict)
                if not father_floatingip:
                    msg = 'floating IP address of sub floating IP should ' \
                          'be the same as father floating IP for router ' \
                          'interface association'
                    LOG.error(_LE('[AC]Failed to validate floating IP:%s.'), msg)
                    raise BadRequest(resource='floatingip', msg=msg)

    def _get_floatingip_db(self, context, fip, initial_status, dns_integration):
        """Get floating IP from DB."""
        fip_id = uuidutils.generate_uuid()
        floatingip_port = generate_floatingip_port_desc(fip, fip_id)
        if ncu.IS_FSP and ACCommonUtil.is_sub_floatingip(context, fip):
            dns_data, external_port, floatingip_db, floatingip_dict = \
                self._get_sub_floatingip_db(
                    context, initial_status, (fip, fip_id, dns_integration))
        else:
            # 'status' in port dict could not be updated by default, use
            # check_allow_post to stop the verification of system
            versions = [ac_cnst.OPS_N, ac_cnst.OPS_O, ac_cnst.OPS_P,
                        ac_cnst.OPS_Q, ac_cnst.OPS_R, ac_cnst.OPS_T, ac_cnst.OPS_W,
                        ac_cnst.FSP_6_5, ac_cnst.FSP_21_0]
            if ncu.get_ops_version() in versions:
                setattr(context, 'GUARD_TRANSACTION', False)
            if ncu.get_ops_version() in [ac_cnst.OPS_W]:
                db_context = ncu.neutron_context.get_admin_context()
                external_port = self._create_floatingip_port(db_context, floatingip_port)
            else:
                external_port = self._create_floatingip_port(context, floatingip_port)

            with self.delete_port_error(self._core_plugin, context.elevated(),
                                        external_port['id']):
                # Ensure IPv4 addresses are allocated on external port
                external_ipv4_ips = self._port_ipv4_fixed_ips(external_port)
                if not external_ipv4_ips:
                    raise ExternalIpAddressExhausted(
                        net_id=fip['floating_network_id'])

                floating_fixed_ip = external_ipv4_ips[0]
                floating_ip_address = floating_fixed_ip['ip_address']
                if ncu.get_ops_version() in \
                        [ac_cnst.OPS_Q, ac_cnst.OPS_R, ac_cnst.OPS_T, ac_cnst.OPS_W,
                         ac_cnst.FSP_6_5, ac_cnst.FSP_21_0]:
                    floatingip_db = l3_obj.FloatingIP(
                        context, id=fip_id, project_id=fip['tenant_id'],
                        status=initial_status,
                        floating_network_id=fip['floating_network_id'],
                        floating_ip_address=floating_ip_address,
                        floating_port_id=external_port['id'])
                else:
                    floatingip_db = FloatingIP(
                        id=fip_id, tenant_id=fip['tenant_id'],
                        status=initial_status,
                        floating_network_id=fip['floating_network_id'],
                        floating_ip_address=floating_ip_address,
                        floating_port_id=external_port['id'])
                floatingip_dict, dns_data = self._get_floatingip_dict(
                    context, floatingip_db, external_port,
                    (fip_id, fip, dns_integration))
        floatingip_data = {"floatingip_db": floatingip_db,
                           "external_port": external_port,
                           "floatingip_dict": floatingip_dict,
                           "dns_data": dns_data}
        return floatingip_data

    @contextlib.contextmanager
    def delete_port_error(self, core_plugin, context, port_id):
        """Delete port error."""
        try:
            yield
        except Exception:
            with excutils.save_and_reraise_exception():
                try:
                    self.__delete_port(core_plugin, context, port_id)
                except PortNotFound:
                    LOG.debug("Port %s not found", port_id)
                except Exception:
                    LOG.exception("Failed to delete port: %s", port_id)

    def __delete_port(self, core_plugin, context, port_id):
        if ncu.get_ops_version() in [ac_cnst.OPS_W]:
            db_context = ncu.neutron_context.get_admin_context()
            core_plugin.delete_port(db_context, port_id, l3_port_check=False)
        else:
            core_plugin.delete_port(context, port_id, l3_port_check=False)

    def _get_dns_integration(self, context, f_net_id):
        """Get DNS integration."""
        if not self._core_plugin._network_is_external(context, f_net_id):
            msg = "Network %s is not a valid external network" % f_net_id
            raise BadRequest(resource='floatingip', msg=msg)

        if ncu.get_ops_version() in [ac_cnst.OPS_W]:
            self._check_net_for_ops_w(context, f_net_id)
        else:
            if not self._is_ipv4_network(context, f_net_id):
                msg = "Network %s does not contain any IPv4 subnet" % f_net_id
                raise BadRequest(resource='floatingip', msg=msg)
        if ncu.get_ops_version() in [ac_cnst.OPS_R, ac_cnst.OPS_T, ac_cnst.OPS_W, ac_cnst.FSP_21_0]:
            from neutron_lib.api import extensions as utils
        else:
            from neutron.common import utils
        dns_integration = utils.is_extension_supported(self._core_plugin,
                                                       'dns-integration')
        return dns_integration

    def _check_net_for_ops_w(self, context, f_net_id):
        f_net_db = self._core_plugin._get_network(context, f_net_id)
        if not f_net_db.external:
            msg = "Network %s is not a valid external network" % f_net_id
            raise BadRequest(resource='floatingip', msg=msg)
        if not self._is_ipv4_network(context, f_net_db):
            msg = "Network %s does not contain any IPv4 subnet" % f_net_id
            raise BadRequest(resource='floatingip', msg=msg)

    def _get_floatingip_dict(self, context, floatingip_db,
                             external_port, fip_data):
        """Get floating IP dict."""
        fip_id, fip, dns_integration = fip_data
        # Update association with internal port
        # and define external IP address
        self._update_fip_assoc(context, fip, floatingip_db, external_port)
        if ncu.get_ops_version() in \
                [ac_cnst.OPS_Q, ac_cnst.OPS_R, ac_cnst.OPS_T, ac_cnst.OPS_W,
                 ac_cnst.FSP_6_5, ac_cnst.FSP_21_0]:
            floatingip_db.create()
        else:
            context.session.add(floatingip_db)
        if six.PY3:
            make_floatingip_dict_args = inspect.getfullargspec(self._make_floatingip_dict).args
        else:
            make_floatingip_dict_args = inspect.getargspec(self._make_floatingip_dict).args
        if 'process_extensions' in make_floatingip_dict_args:
            floatingip_dict = self._make_floatingip_dict(
                floatingip_db, process_extensions=False)
        else:
            floatingip_dict = self._make_floatingip_dict(floatingip_db)

        if dns_integration and \
                hasattr(self, '_process_dns_floatingip_create_precommit'):
            dns_data = self._process_dns_floatingip_create_precommit(
                context, floatingip_dict, fip)
        else:
            dns_data = None

        if all(hasattr(self, f) for f in ['_is_fip_qos_supported', '_process_extra_fip_qos_create']):
            if self._is_fip_qos_supported:
                self._process_extra_fip_qos_create(context, fip_id, fip)

        return floatingip_dict, dns_data

    def _validate_father_floatinip(self, context, fip_old, fip):
        if ncu.IS_FSP and self.is_father_floatingip(context, fip_old) and \
                set(fip.keys()) & {'port_id', 'fixed_ip_address'}:
            msg = 'father floating IP could not be updated'
            LOG.error(_LE('[AC] Failed to update floating IP: %s'), msg)
            ac_osprofiler.record_chain_exception_end(msg)
            raise BadRequest(resource='floatingip', msg=msg)

    def _disassociate_floatingip_to_ac(self, context, fip_id, fip, fip_old):
        LOG.info(_LI("[AC]The update operation is disassociation."))
        try:
            ac_osprofiler.record_call_chain(
                "delete AC fip, when cloud update fip")
            self._send_notify(ac_cnst.NW_HW_FIP, ac_cnst.BEFORE_AC_DELETE, fip)
            if self._hill_stone_not_config():
                self.l3_reliability.update_plugin_record(
                    context, fip_id, {}, ac_cnst.NW_HW_DELETE_FIP)
            self._send_notify(
                ac_cnst.NW_HW_FIP, ac_cnst.AFTER_AC_DELETE, fip_old)
            self.update_floatingip_status(
                context, fip_id, ncu.FLOATINGIP_STATUS_DOWN)
        except Exception as ex:
            LOG.error(_LE("[AC] AC delete floating IP failed: %s"), ex)
            ac_osprofiler.record_chain_exception_end(
                "delete AC fip failed, when cloud update fip")
            self._update_floatingip_rollback(context, fip_id, fip_old)
            raise BadRequest(resource=ac_cnst.NW_HW_FIP, msg=ex)

    def _associate_floatingip_to_ac(self, context, fip_id, fip, fip_old):
        LOG.info(_LI("[AC]The update operation is association."))
        floating_ip = ACFloatingipModel.ac_model_format(fip)
        try:
            ac_osprofiler.record_call_chain(
                "create AC fip, when cloud update fip")
            self._send_notify(ac_cnst.NW_HW_FIP, ac_cnst.BEFORE_AC_CREATE, fip)
            if self._hill_stone_not_config():
                self.l3_reliability.update_plugin_record(
                    context, fip_id, floating_ip, ac_cnst.NW_HW_CREATE_FIP)
            self._send_notify(ac_cnst.NW_HW_FIP, ac_cnst.AFTER_AC_CREATE, fip)
        except Exception as ex:
            LOG.error(_LE("[AC] AC create floating IP failed: %s"), ex)
            ac_osprofiler.record_call_chain(
                "create AC fip failed, when cloud update fip")
            self._update_floatingip_rollback(context, fip_id, fip_old)
            raise BadRequest(resource=ac_cnst.NW_HW_FIP, msg=ex)

    def _update_floatingip_to_ac(self, context, fip_id, fip, fip_old):
        try:
            floating_ip = ACFloatingipModel.ac_model_format(fip)
            ac_osprofiler.record_call_chain(
                "update AC fip, when cloud update fip")
            if self._hill_stone_not_config():
                self.l3_reliability.update_plugin_record(
                    context, fip_id, floating_ip, ac_cnst.NW_HW_UPDATE_FIP)
        except Exception as ex:
            LOG.error(_LE("[AC] Failed to update Floating IP: %s"), ex)
            ac_osprofiler.record_call_chain(
                "update AC fip failed, when cloud update fip")
            self._update_floatingip_rollback(context, fip_id, fip_old)
            raise BadRequest(resource=ac_cnst.NW_HW_FIP, msg=ex)

    def update_floatingip_info(self, context, fip_id, floatingip):
        """Update floating IP's fixed_ip when port fixed_ips change."""
        LOG.info(_LI("[AC]Begin to update floating IP : %s"), fip_id)
        fip_old = self.get_floatingip(context, fip_id)
        LOG.info(_LI("[AC]Old floating IP is %s"), fip_old)
        ac_osprofiler.record_chain_start("update floatingip start: " + fip_id)

        fip = super(AbstractHwACL3RouterPlugin, self).update_floatingip(
            context, fip_id, floatingip)
        LOG.info(_LI("[AC]New floating IP is %s"), fip)
        floating_ip = ACFloatingipModel.ac_model_format(fip)
        try:
            ac_osprofiler.record_call_chain("update AC fip, when cloud "
                                            "update fip")
            self.l3_reliability.update_plugin_record(
                context, fip['id'], floating_ip, ac_cnst.NW_HW_UPDATE_FIP)
        except Exception as ex:
            LOG.error(_LE("[AC]AC update floating IP failed for %s. "
                          "Update fip status to error."), ex)
            ac_osprofiler.record_call_chain(
                "update AC fip fail, when cloud update fip")
        LOG.info(_LI("[AC]AC update floating IP successful."))
        ac_osprofiler.record_chain_end_with_reason("update fip success")
        return fip

    @classmethod
    def _optisize_exroute_info(cls, exroute):
        key_blacklist = ['segment_type', 'segment_id', 'priority', 'extra_opt']
        for key in key_blacklist:
            exroute.pop(key, None)

    def _generate_exroute_list(self, context, session, exroutes_data, router_id):
        rec_plugin_list = []
        exroute_info_list = []
        exroute_list = []
        for exroute in exroutes_data:
            if ncu.get_ops_version() in [ac_cnst.FSP_6_1, ac_cnst.FSP_6_3_0]:
                self._update_exroute_ip_version(exroute)
            interface_id = ACCommonUtil.get_exroute_interface_id(
                context, exroute.get('nexthop'), exroute.get('destination'))
            exroute_info = ACExRouteModel.ac_model_format(exroute, router_id,
                                                          interface_id)
            rec_plugin_tmp = self.db_if.create_plugin_record(
                session, exroute_info,
                (router_id, ac_cnst.OPER_CREATE, ac_cnst.NW_HW_EXROUTE),
                ac_cnst.IN_PROCESS)
            rec_plugin_list.append(rec_plugin_tmp)
            exroute_info_list.append(exroute_info)
            exroute_list.append(exroute)
        return rec_plugin_list, exroute_info_list, exroute_list

    def _delete_exroute_plugin_record(self, session, rec_plugin_list):
        for plugin in rec_plugin_list:
            try:
                self.db_if.delete_plugin_record(session, plugin.seq_num)
            except Exception as ex:
                LOG.error('Huawei AC delete exroute plugin_record: %s failed: %s',
                          plugin.seq_num, str(ex))

    def _handle_add_exroutes_failure(self, context, router_id, exroutes_failed):
        if exroutes_failed:
            LOG.error(_LE('[AC] Huawei AC add exroutes %s failed.'),
                      exroutes_failed)
            super(AbstractHwACL3RouterPlugin, self).remove_exroutes(
                context, router_id, {'exroutes': exroutes_failed})
            ac_osprofiler.record_chain_exception_end("add exroutes fail")

            raise ExRoutesException(opr='add', ex=exroutes_failed)

    def _handle_create_exroutes_failure(self, context, router_id, exroutes_failed):
        if exroutes_failed:
            LOG.error(_LE('[AC] Huawei AC create exroutes %s failed.'),
                      exroutes_failed)
            super(AbstractHwACL3RouterPlugin, self).delete_exroutes(
                context, router_id, {'exroutes': exroutes_failed})
            ac_osprofiler.record_chain_exception_end("create exroutes fail")

            raise ExRoutesException(opr='create', ex=exroutes_failed)

    def list_router_exroutes(self, context, router_id, type_value=None,
                             nexthop_value=None):
        """List router exroutes."""
        LOG.info(_LI('[AC] Begin to list exroutes for router %(id)s, '
                     'type: %(type)s, nexthop: %(hop)s'),
                 {'id': router_id, 'type': type_value, 'hop': nexthop_value})

        exroutes_dict = super(AbstractHwACL3RouterPlugin, self).\
            list_router_exroutes(context, router_id, type_value, nexthop_value)
        exroutes_dict = self._extend_exroutes_dict(exroutes_dict)
        return exroutes_dict

    def _disassociate_floatingips_ovo(self, context, port_id, do_notify):
        """Disassociate all floating IPs linked to specific port.
        @param context: neutron request context.
        @param port_id: ID of the port to disassociate floating IPs.
        @param do_notify: whether we should notify routers right away.
                          This parameter is ignored.
        @return: set of router-ids that require notification updates
        """
        with context.session.begin(subtransactions=True):
            floating_ip_objs = l3_obj.FloatingIP.get_objects(
                context, fixed_port_id=port_id)
            if not floating_ip_objs:
                floating_ip_objs = l3_obj.FloatingIP.get_objects(
                    context, floating_port_id=port_id)
            router_ids = {fip.router_id for fip in floating_ip_objs}
            self._send_disassociate_fip(context, floating_ip_objs, port_id,
                                        router_ids)

        for fip in floating_ip_objs:
            assoc_result = {'fixed_ip_address': None, 'fixed_port_id': None,
                            'router_id': None, 'router_ids': router_ids,
                            'floating_ip_address': fip.floating_ip_address,
                            'floating_network_id': fip.floating_network_id,
                            'floating_ip_id': fip.id, 'context': context, }
            if ncu.get_ops_version() in [ac_cnst.OPS_T, ac_cnst.OPS_W, ac_cnst.FSP_21_0]:
                assoc_result['association_event'] = False
            ncu.registry.notify(ncu.resources.FLOATING_IP,
                                ncu.events.AFTER_UPDATE, self, **assoc_result)
        if do_notify:
            self.notify_routers_updated(context, router_ids)
            # since caller assumes that we handled notifications on its
            # behalf, return nothing
            return None
        return router_ids

    def _disassociate_floatingips_no_ovo(self, context, port_id,
                                         do_notify=True):
        """Disassociate all floating IPs linked to specific port.
        @param context: neutron request context.
        @param port_id: ID of the port to disassociate floating IPs.
        @param do_notify: whether we should notify routers right away.
        @return: set of router-ids that require notification updates
                 if do_notify is False, otherwise None.
        """
        LOG.info(_LI("[AC]Begin to disassociate floating IPs. "
                     "The port ID is %s"), port_id)
        router_ids = set()
        with context.session.begin(subtransactions=True):
            fip_qry = context.session.query(FloatingIP)
            floating_ips = fip_qry.filter_by(fixed_port_id=port_id).all()

            if not floating_ips:
                floating_ips = fip_qry.filter_by(floating_port_id=port_id).all()

        for floating_ip in floating_ips:
            fip = self._make_floatingip_dict(floating_ip)
            try:
                self._send_notify(ac_cnst.NW_HW_FIP,
                                  ac_cnst.BEFORE_AC_DELETE, fip)
                if self._hill_stone_not_config():
                    self.l3_reliability.update_plugin_record(
                        context, fip['id'], {}, ac_cnst.NW_HW_DELETE_FIP)
                self._send_notify(ac_cnst.NW_HW_FIP,
                                  ac_cnst.AFTER_AC_DELETE, fip)
            except Exception as ex:
                LOG.error(_LE("[AC]AC delete floating IP failed for %s."), ex)
                raise ex
            LOG.info(_LI("[AC]AC disassociate floating IPs successful."))
            router_ids.add(floating_ip['router_id'])
            floating_ip.update({'fixed_port_id': None,
                                'fixed_ip_address': None, 'router_id': None})
            self.update_floatingip_status(
                context, fip['id'], ncu.FLOATINGIP_STATUS_DOWN)
        LOG.info(_LI("[AC]Update floating IP records in Neutron DB."))
        if do_notify:
            self.notify_routers_updated(context, router_ids)
            # since caller assumes that we handled notifications on its
            # behalf, return nothing
            return None
        return router_ids

    def update_eip_status(self, resource, event, trigger, **kwargs):
        """Update floating IP status."""
        try:
            LOG.info("[AC]receive fip notify %s", kwargs)
            status = kwargs.get('floatingip', {}).get('status')
            floatingip_id = kwargs.get('floatingip', {}).get('id')
            LOG.info("[AC] update floatingip status to %s", status)
            ctx = ncu.neutron_context.get_admin_context()
            self.update_floatingip_status(ctx, floatingip_id, status)
        except Exception as ex:
            LOG.info("[AC] update floatingip status fail %s", ex)
