#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Implementation of the Neutron L3 Router Service Plugin."""

import ast
import copy
import threading
import netaddr
from neutron.db import models_v2
from neutron.plugins.ml2 import db
from neutron.services.l3_router import l3_router_plugin
from oslo_config import cfg

from networking_huawei._i18n import _LE
from networking_huawei._i18n import _LI
from networking_huawei.drivers.ac.client.restclient import ACReSTClient
from networking_huawei.drivers.ac.client.service import ACReSTService
# import config register for fusionsphere
from networking_huawei.drivers.ac.common import constants as ac_cnst
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler
from networking_huawei.drivers.ac.common.constants import NW_HW_CREATE_EXROUTE
from networking_huawei.drivers.ac.common.util import \
    ACCommonUtil, DataFilterUtil
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei.drivers.ac.db.lock import lockedobjects_db as lock_db
from networking_huawei.drivers.ac.model.exroute_model import ACExRouteModel
from networking_huawei.drivers.ac.model.floatingip_model \
    import ACFloatingipModel
from networking_huawei.drivers.ac.model.port_model import ACPortModel
from networking_huawei.drivers.ac.model.snat_model import ACSNATModel
from networking_huawei.drivers.ac.plugins.l3 import l3_router_util
from networking_huawei.drivers.ac.plugins.l3.abstract_ac_l3_router_plugin \
    import AbstractHwACL3RouterPlugin
from networking_huawei.drivers.ac.plugins.l3.l3_router_util import \
    get_router_for_fip_fsp, get_router_port
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI

try:
    from neutron.common.exceptions import BadRequest, PortNotFound
except ImportError:
    from neutron_lib.exceptions import BadRequest, PortNotFound
try:
    from neutron.extensions.l3 import ExternalGatewayForFloatingIPNotFound, \
        RouterInUse, RouterExternalGatewayInUseByFloatingIp
except ImportError:
    from neutron_lib.exceptions.l3 import ExternalGatewayForFloatingIPNotFound, \
        RouterInUse, RouterExternalGatewayInUseByFloatingIp

try:
    from neutron.db.l3_db import RouterPort
except ImportError:
    from neutron.db.models.l3 import RouterPort
try:
    from neutron.objects import router as l3_obj
except ImportError:
    l3_obj = None
try:
    from neutron_lib.api.definitions import l3 as l3_apidef
except ImportError:
    l3_apidef = None

LOG = ncu.ac_log.getLogger(__name__)
INTERFACE_TYPE = [ncu.DEVICE_OWNER_ROUTER_INTF,
                  ncu.DEVICE_OWNER_DVR_INTERFACE,
                  ncu.DEVICE_OWNER_ROUTER_HA_INTF]


class HuaweiACL3RouterPlugin(AbstractHwACL3RouterPlugin):
    """Implementation of the neutron L3 router service plugin."""
    __native_pagination_support = True
    __native_sorting_support = True

    if ncu.get_ops_version() in [ac_cnst.OPS_P, ac_cnst.OPS_Q, ac_cnst.OPS_R, ac_cnst.OPS_W,
                                 ac_cnst.OPS_T, ac_cnst.FSP_21_0]:
        @property
        def supported_extension_aliases(self):
            """Get supported extension aliases."""
            if not hasattr(self, '_aliases'):
                aliases = self._supported_extension_aliases[:]
                l3_router_plugin.disable_dvr_extension_by_config(aliases)
                l3_router_util._disable_qos_extension_by_plugins(aliases)
                aliases.append("exroutes_ext_alias")
                if "exroute_custom" in cfg.CONF.api_extensions_path:
                    aliases.append("custom_exroutes_ext_alias")
                self._aliases = aliases
            return self._aliases
    elif ncu.get_ops_version() in [ac_cnst.FSP_6_5, ac_cnst.OPS_EZ_M]:
        supported_extension_aliases = \
            l3_router_plugin.L3RouterPlugin._supported_extension_aliases
    else:
        supported_extension_aliases = \
            l3_router_plugin.L3RouterPlugin.supported_extension_aliases
    if isinstance(supported_extension_aliases, list):
        supported_extension_aliases.append("exroutes_ext_alias")
        if "exroute_custom" in cfg.CONF.api_extensions_path:
            supported_extension_aliases.append("custom_exroutes_ext_alias")

    def __init__(self):
        LOG.info(_LI("[AC]Init Huawei L3 plugin."))
        super(HuaweiACL3RouterPlugin, self).__init__()
        self.router_scheduler = None
        self.l3_reliability = ACReliabilityAPI(ac_cnst.NW_HW_L3)
        self.rest_service = ACReSTService()
        self.client = ACReSTClient()
        self.data_filter = DataFilterUtil()
        self.db_if = ACdbInterface()

        if self._hill_stone_config():
            ncu.registry.subscribe(self.update_eip_status,
                                   ac_cnst.NW_HW_FIP,
                                   ac_cnst.AFTER_HS_CREATE)

            ncu.registry.subscribe(self.update_eip_status,
                                   ac_cnst.NW_HW_FIP,
                                   ac_cnst.AFTER_HS_DELETE)
        LOG.info(_LI("[AC]Initialization finished successfully "
                     "for Huawei L3 plugin."))

    def get_plugin_type(self):
        """Type of the neutron L3 router service plugin."""
        return "L3_ROUTER_NAT"

    def get_plugin_description(self):
        """Description of the neutron L3 router service plugin."""
        return "Huawei L3 Router Service Plugin for basic L3 forwarding" \
               " between (L2) Neutron networks and" \
               " access to external networks via a NAT gateway."

    def create_router(self, context, router):
        """Create router."""
        LOG.info(_LI("[AC]Begin to create router : %s "), router)
        ac_osprofiler.record_chain_start(
            "create router start: %s" % str(router))

        gateway_info = router['router'].get('external_gateway_info')
        self.modify_router_info(router)

        try:
            router_dict = super(HuaweiACL3RouterPlugin, self).create_router(
                context, router)
        except Exception as ex:
            LOG.error('[AC]Create router db failed with reason: %s.', ex)
            alarm_description = {
                'operation': ac_cnst.OPER_CREATE,
                'res_type': ac_cnst.NW_HW_ROUTERS,
                'router_name': router['router'].get('name', ''),
                'reason': "Create router db failed"}
            self.client.send_cloud_alarm(alarm_description, context.session)
            raise BadRequest(resource='router', msg=ex)

        LOG.info(_LI('[AC]Create router db %s.'), router_dict)

        self._create_gateway_port_ez(context, router_dict, gateway_info)

        if self.data_filter.not_in_white_or_in_black_list(
                context, router, ac_cnst.NW_HW_ROUTERS, router_dict['id']):
            return router_dict

        self._notify_router_create(context, router, router_dict, gateway_info)

        LOG.info(_LI("[AC]Huawei AC create router successful."))
        ac_osprofiler.record_chain_end_with_reason("create ruoter success")
        return router_dict

    def notify_router_deleted(self, context, router_id):
        """Notify router deleted."""
        pass

    def delete_router(self, context, router_id):
        """Delete router."""
        LOG.info(_LI("[AC]Begin to delete router : %s"), router_id)
        ac_osprofiler.record_chain_start("delete router start,ID:" + router_id)
        self._check_router_not_in_use(context, router_id)

        router = self._ensure_router_not_in_use(context, router_id)
        if router.gw_port:
            try:
                self._check_router_gw_port_in_use(context, router_id)
            except RouterInUse:
                ac_osprofiler.record_chain_exception_end("router gw is in "
                                                         "using.")
                raise
            except Exception as ex:
                LOG.error(_LE('[AC] Failed to check router gateway port '
                              'in use or not in huawei L3 plugin: %s'), ex)

        self._validate_firewall_routers_not_in_use(context, router_id)

        subnet_id_list = []
        if not self.data_filter.not_in_white_or_in_black_list(
                context, {'router': router}, ac_cnst.NW_HW_ROUTERS,
                router_id):
            try:
                if ncu.IS_FSP:
                    subnet_id_list = self.delete_public_network_ipv6_port(
                        context, router_id)
                self._send_notify(
                    ac_cnst.NW_HW_ROUTERS, ac_cnst.BEFORE_AC_DELETE, router)
                self.l3_reliability.update_plugin_record(
                    context, router_id, {}, ac_cnst.NW_HW_DELETE_ROUTER)
                self._send_notify(
                    ac_cnst.NW_HW_ROUTERS, ac_cnst.AFTER_AC_DELETE, router)

            except Exception as ex:
                LOG.error(_LE("[AC]AC delete router failed for %s."), ex)
                if ncu.IS_FSP:
                    for subnet_id in subnet_id_list:
                        tenant_id = router.get('tenant_id')
                        self.add_router_interface_public_network(
                            context, tenant_id, router_id, subnet_id)
                raise BadRequest(resource='router', msg=ex)

            LOG.info(_LI("[AC]AC delete router successful."))

        super(HuaweiACL3RouterPlugin, self).delete_router(context, router_id)

        self._delete_gate_way_port_ops_ez_m(context, router, router_id)

        LOG.info(_LI("[AC]Delete router record in Neutron DB."))
        ac_osprofiler.record_chain_end_with_reason("delete router success")
        return

    def _delete_current_gw_port(self, context, router_id, router,
                                new_network_id):
        try:
            super(HuaweiACL3RouterPlugin, self)._delete_current_gw_port(
                context, router_id, router, new_network_id)
        except RouterExternalGatewayInUseByFloatingIp:
            if ncu.IS_FSP:
                with context.session.begin(subtransactions=True):
                    gw_port = router.gw_port
                    router.gw_port = None
                    context.session.add(router)
                    context.session.expire(gw_port)
                self._core_plugin.delete_port(
                    context.elevated(), gw_port['id'], l3_port_check=False)
            else:
                raise

    def _delete_gateway_port(self, context, gw_port_id, router_id):
        LOG.info(_LI('[AC] begin to delete gateway port for %s'), router_id)
        self.l3_reliability.update_plugin_record(
            context, gw_port_id, {}, ac_cnst.NW_HW_DELETE_PORT)
        try:
            gw_port = self._core_plugin.get_port(context, gw_port_id)
        except PortNotFound:
            LOG.info(_LI('[AC]gateway port %s could not be found'), gw_port_id)
        else:
            gw_port_info = ACPortModel.ac_model_format(gw_port, None)
            LOG.info(_LI('[AC]begin to create floating ip port for %s'), router_id)
            self.l3_reliability.update_plugin_record(
                context, gw_port_id, gw_port_info, ac_cnst.NW_HW_CREATE_PORT)

    @lock_db.wrap_db_lock(lock_db.RESOURCE_L3_ROUTER)
    def update_router(self, context, router_id, router):
        """Update router."""

        original_gw_info, original_router, tenant_name = \
            self._update_router_precommit(context, router, router_id)
        current_gw_info = router['router'].get('external_gateway_info', {})
        LOG.info(_LI("The current external gateway info: %s"), current_gw_info)

        original_gw_port = self._core_plugin._get_port(
            context.elevated(), original_router.get('gw_port_id')) \
            if original_router.get('gw_port_id') else None

        if self._fixed_subnet_not_ready(current_gw_info, original_gw_info):
            return router['router']

        if ncu.get_ops_version() in [ac_cnst.OPS_EZ_M]:
            return self._convert_ez_gateway_port(
                context, router_id, original_router, router)

        try:
            current_router = super(HuaweiACL3RouterPlugin, self).update_router(
                context, router_id, router)
        except Exception as ex:
            self._update_router_rollback(
                context, tenant_name, (original_router, router_id, original_gw_info,
                                       current_gw_info, original_gw_port), ex)
            alarm_description = {
                'operation': ac_cnst.OPER_UPDATE, 'res_type': ac_cnst.NW_HW_ROUTERS,
                'res_id': router_id, 'reason': "Update router db failed."}
            self.client.send_cloud_alarm(alarm_description, context.session)
            raise BadRequest(resource='router', msg=ex)

        if self.data_filter.not_in_white_or_in_black_list(
                context, {'router': current_router}, ac_cnst.NW_HW_ROUTERS,
                router_id):
            return current_router

        router_info = (current_router, current_gw_info, original_router,
                       original_gw_info)
        self._update_router_postcommit(context, router_info, router_id,
                                       tenant_name)
        LOG.info(_LI("[AC] AC update router successfully."))
        ac_osprofiler.record_chain_end()
        return current_router

    def _update_port(self, context, port):
        LOG.info(_LI('[AC] Begin to update port in L3 plugin: %s'), port)
        port_info = ACPortModel.ac_model_format(port, context.tenant_name)
        network = self._core_plugin.get_network(context, port['network_id'])
        LOG.debug("[AC] Network name is %s", network.get('name', None))
        if not ncu.IS_FSP and network and network.get('router:external'):
            profile_json = ast.literal_eval(port_info["profile"])
            if not profile_json.get('fw_enabled'):
                profile_json.update({'fw_enabled': False})
                port_info["profile"] = str(profile_json)

        if self.data_filter.not_in_white_or_in_black_list(
                context, network, ac_cnst.NW_HW_PORTS, None):
            return False

        try:
            self.l3_reliability.update_plugin_record(
                context, port['id'], port_info, ac_cnst.NW_HW_UPDATE_PORT)
            return True
        except Exception as ex:
            LOG.error(_LE('[AC]Failed to update port in Huawei driver:%s'), ex)
            raise BadRequest(resource='router interface', msg=ex)

    def _delete_port(self, context, port):
        LOG.info(_LI('[AC] Begin to delete port in L3 plugin: %s'), port)
        port_info = ACPortModel.ac_model_format(port, context.tenant_name)
        try:
            self.l3_reliability.update_plugin_record(
                context, port['id'], port_info, ac_cnst.NW_HW_DELETE_PORT)
        except Exception as ex:
            LOG.error(_LE('[AC]Failed to delete port in Huawei driver:%s'), ex)
            raise BadRequest(resource='router interface', msg=ex)

    def add_router_interface(self, context, router_id, interface_info):
        """Add router interface."""
        LOG.info(_LI('[AC]Begin to add router interface: %s'), interface_info)
        ac_osprofiler.record_chain_start("add router interface start,router ID"
                                         ":" + router_id + str(interface_info))
        add_by_port, _ = self._validate_interface_info(interface_info)

        device_owner = self._get_device_owner(context, router_id)
        LOG.debug(_LI('[AC]Router interface device owner: %s'), device_owner)

        interface_added = super(HuaweiACL3RouterPlugin, self).add_router_interface(
            context, router_id, interface_info)
        LOG.info(_LI('[AC]Router interface added: %s'), interface_added)
        self._send_notify(ac_cnst.NW_HW_ROUTER_IF,
                          ac_cnst.AFTER_AC_CREATE, interface_added)

        port_added = self._core_plugin.get_port(context, interface_added['port_id'])
        LOG.info(_LI('[AC]Interface port added: %s'), port_added)
        fixed_ips = port_added.get('fixed_ips', None)
        need_ac_process = True
        if add_by_port:
            try:
                need_ac_process = self._update_port(context, port_added)
            except Exception as ex:
                with context.session.begin(subtransactions=True):
                    port_object = self._core_plugin._get_port(
                        context, port_added['id'])
                    port_object.update({'device_owner': '', 'device_id': ''})
                    qry = context.session.query(RouterPort)
                    qry = qry.filter_by(
                        port_id=port_added['id'], router_id=router_id,
                        port_type=device_owner)
                    context.session.delete(qry.one())
                    context.session.flush()
                LOG.error(_LE('[AC]Interface port update failed for %s'), ex)
                ac_osprofiler.record_chain_exception_end("update router "
                                                         "interface failed")
                raise BadRequest(resource='router interface', msg=ex)

            if ncu.IS_FSP and cfg.CONF.huawei_ac_config.vpc_peering \
                    and need_ac_process and fixed_ips \
                    and ACCommonUtil.is_port_contain_ipv4_address(fixed_ips):
                ac_osprofiler.record_call_chain("add vpc_peer fip")
                self._create_public_service_fip(context, port_added, router_id)

        self._create_public_network_ipv6_port(context, fixed_ips, router_id)
        ac_osprofiler.record_chain_end()
        return interface_added

    @classmethod
    def get_dist_port_bindings(cls, context, port_id):
        """get_dist_port_bindings"""
        if ncu.get_ops_version() \
                in [ac_cnst.OPS_K, ac_cnst.OPS_L,
                    ac_cnst.OPS_M, ac_cnst.OPS_EZ_M] or \
                (ncu.IS_FSP and
                 ncu.get_ops_version() not in [ac_cnst.FSP_21_0]):
            return db.get_dvr_port_bindings(context.session, port_id)
        elif ncu.get_ops_version() in [ac_cnst.OPS_N]:
            return db.get_distributed_port_bindings(context.session, port_id)
        return db.get_distributed_port_bindings(context, port_id)

    def remove_router_interface(self, context, router_id, interface_info):
        """Remove router interface."""
        LOG.info(_LI('[AC]Begin to remove router interface:%s'), interface_info)
        ac_osprofiler.record_chain_start("remove interface start, router ID:" +
                                         router_id + str(interface_info))

        port_id = self._confirm_removing(context, router_id, interface_info)
        port_removed = self._core_plugin.get_port(context, port_id)
        LOG.info(_LI('[AC]Interface port removed: %s'), port_removed)

        if ncu.IS_FSP and cfg.CONF.huawei_ac_config.vpc_peering:
            ac_osprofiler.record_call_chain("delete public service eip")
            self._delete_public_service_fip(context, port_removed, router_id)

        distributed_port_bindings = \
            self.get_dist_port_bindings(context, port_id)
        if port_removed.get('device_owner') == ncu.DEVICE_OWNER_DVR_INTERFACE \
                and not distributed_port_bindings:
            LOG.info(_LI('[AC]Begin to delete DVR interface with no bindings'))
            try:
                self._delete_port(context, port_removed)
            except Exception as ex:
                LOG.error(_LE('[AC]AC delete interface port failed: %s'), ex)
                raise BadRequest(resource='router interface', msg=ex)

        router_dict = super(HuaweiACL3RouterPlugin, self).get_router(
            context, router_id)
        LOG.debug(_LI('[AC]Router info is: %s'), router_dict)

        try:
            interface_removed = super(HuaweiACL3RouterPlugin, self). \
                remove_router_interface(context, router_id, interface_info)
            LOG.info(_LI('[AC]Router interface removed: %s'), interface_removed)
        except Exception as ex:
            LOG.error(_LE("[AC]OPS remove router interface failed: %s."), ex)
            raise BadRequest(resource='router interface', msg=ex)
        ac_osprofiler.record_chain_end_with_reason("remove interface success")
        self._send_notify(ac_cnst.NW_HW_ROUTER_IF,
                          ac_cnst.AFTER_AC_DELETE, interface_removed)
        return interface_removed

    def _get_router_for_floatingip(self, context, port, subnet_id, network_id):
        if ncu.IS_FSP:
            if not cfg.CONF.huawei_ac_config.vpc_peering:
                subnet = self._core_plugin.get_subnet(context, subnet_id)
                if not subnet['gateway_ip']:
                    raise BadRequest(
                        resource='floatingip',
                        msg=('Cannot add floating IP to port on subnet %s which'
                             ' has no gateway_ip' % subnet_id))
            return get_router_for_fip_fsp(context, port, subnet_id)
        else:
            try:
                router_id = super(HuaweiACL3RouterPlugin, self)._get_router_for_floatingip(
                    context, port, subnet_id, network_id)
            except ExternalGatewayForFloatingIPNotFound as ex:
                router_intf_qry = context.session.query(RouterPort)
                router_intf_qry = router_intf_qry.join(models_v2.Port)

                vm_related_router_id = router_id = None
                router_intf_ports_in = router_intf_qry.filter(
                    models_v2.Port.network_id == port['network_id'],
                    RouterPort.port_type == ncu.DEVICE_OWNER_ROUTER_INTF)
                for router_port in get_router_port(router_intf_ports_in,
                                                   subnet_id):
                    vm_related_router_id = router_port.router.id

                router_intf_ports_out = router_intf_qry.filter(
                    models_v2.Port.network_id == network_id,
                    RouterPort.port_type == ncu.DEVICE_OWNER_ROUTER_INTF)

                for router_port in router_intf_ports_out:
                    if self.is_multi_outlet_port(context, router_port.port_id) \
                            and router_port.router.id == vm_related_router_id:
                        router_id = router_port.router.id
                        break
                if not router_id:
                    LOG.error(_LE('[AC]cannot find external way for floating ip'))
                    raise ex
            return router_id

    def _create_floatingip_port(self, context, port):
        if ncu.get_ops_version() in \
                [ac_cnst.OPS_P, ac_cnst.OPS_Q, ac_cnst.FSP_6_5,
                 ac_cnst.FSP_6_5_1, ac_cnst.FSP_8_0_0, ac_cnst.FSP_8_0_3]:
            from neutron.plugins.common import utils as p_utils
        else:
            try:
                from neutron_lib.plugins import utils as p_utils
            except ImportError:
                from neutron.plugins.common import utils as p_utils

        return p_utils.create_port(self._core_plugin, context.elevated(),
                                   {'port': port}, check_allow_post=False)

    def _create_floatingip(self, context, floatingip,
                           initial_status=ncu.FLOATINGIP_STATUS_ACTIVE):
        fip = floatingip['floatingip']
        f_net_id = fip['floating_network_id']
        dns_integration = self._get_dns_integration(context, f_net_id)
        with context.session.begin(subtransactions=True):
            # This external port is never exposed to the tenant.
            # it is used purely for internal system and admin use when
            # managing floating IPs.

            floatingip_data = self._get_floatingip_db(context, fip, initial_status, dns_integration)
            floatingip_db = floatingip_data.get("floatingip_db")
            floatingip_dict = floatingip_data.get("floatingip_dict")
            dns_data = floatingip_data.get("dns_data")

        if dns_integration and \
                hasattr(self, '_process_dns_floatingip_create_postcommit'):
            self._process_dns_floatingip_create_postcommit(context,
                                                           floatingip_dict,
                                                           dns_data)
        try:
            from neutron.extensions import l3
        except ImportError:
            l3 = None
        if hasattr(self, '_apply_dict_extend_functions') and \
                hasattr(l3, 'FLOATINGIPS'):
            self._apply_dict_extend_functions(l3.FLOATINGIPS, floatingip_dict,
                                              floatingip_db)

        if ncu.get_ops_version() in [ac_cnst.OPS_P]:
            from neutron.extensions import l3
            from neutron.db import _resource_extend as resource_extend
            resource_extend.apply_funcs(l3.FLOATINGIPS, floatingip_dict,
                                        floatingip_db)
        elif ncu.get_ops_version() in [ac_cnst.OPS_Q]:
            from neutron.extensions import l3
            from neutron.db import _resource_extend as resource_extend
            resource_extend.apply_funcs(l3.FLOATINGIPS, floatingip_dict,
                                        floatingip_db.db_obj)
        elif ncu.get_ops_version() in [ac_cnst.OPS_R]:
            from neutron.db import _resource_extend as resource_extend
            resource_extend.apply_funcs(l3_apidef.FLOATINGIPS, floatingip_dict,
                                        floatingip_db.db_obj)
        elif ncu.get_ops_version() in [ac_cnst.OPS_T, ac_cnst.OPS_W, ac_cnst.FSP_21_0]:
            from neutron_lib.db import resource_extend
            resource_extend.apply_funcs(l3_apidef.FLOATINGIPS, floatingip_dict,
                                        floatingip_db.db_obj)

        return floatingip_dict

    def create_floatingip(self, context, floatingip):
        """Create floating IP.

        :param context: Neutron request context
        :param floatingip: data for the floating IP being created
        :returns: A floating IP object on success

        As the l3 router plugin asynchronously creates floating IPs
        leveraging the l3 agent, the initial status for the floating`
        IP object will be DOWN.
        """
        LOG.info(_LI("[AC]Begin to create floating IP : %s "), floatingip)
        ac_osprofiler.record_chain_start("create fip start:" + str(floatingip))
        if ncu.IS_FSP and not cfg.CONF.huawei_ac_config.vpc_peering:
            self._validate_floatingip(context, floatingip)
        floatingip_dict = super(HuaweiACL3RouterPlugin, self).create_floatingip(
            context, floatingip)
        LOG.info(_LI('[AC]Create floating IP db %s.'), floatingip_dict)
        if floatingip_dict.get('fixed_ip_address') is not None:
            if ncu.IS_FSP and \
                    ACCommonUtil.is_sub_floatingip(context, floatingip_dict):
                self._create_snat(context, floatingip_dict)
                return floatingip_dict
            floating_ip = ACFloatingipModel.ac_model_format(floatingip_dict)
            try:
                self._send_notify(ac_cnst.NW_HW_FIP, ac_cnst.BEFORE_AC_CREATE,
                                  floatingip_dict)
                if self._hill_stone_not_config():
                    self.l3_reliability.update_plugin_record(
                        context, floatingip_dict['id'], floating_ip,
                        ac_cnst.NW_HW_CREATE_FIP)
                self._send_notify(ac_cnst.NW_HW_FIP,
                                  ac_cnst.AFTER_AC_CREATE, floatingip_dict)

            except Exception as ex:
                msg = "[AC] create floating IP failed: %s" % ex
                LOG.error(msg)
                ac_osprofiler.record_chain_exception(msg)
                super(HuaweiACL3RouterPlugin, self).delete_floatingip(
                    context, floatingip_dict['id'])
                raise BadRequest(resource='floatingip', msg=ex)
            LOG.info(_LI("[AC]Huawei AC create floating IP successful."))
        else:
            ac_osprofiler.record_call_chain("fip has no fixed_ip_address, "
                                            "no need to send to AC")
            LOG.info(_LI("[AC]Create floating IP successful."))
        ac_osprofiler.record_chain_end_with_reason("create fip success")
        return floatingip_dict

    def _create_snat(self, context, floatingip_dict):
        snat_cidrs = self._get_snat_cidrs(context, floatingip_dict)
        snat_info = ACSNATModel.ac_model_format(floatingip_dict, snat_cidrs)
        try:
            ac_osprofiler.record_call_chain("create snat")
            self.l3_reliability.update_plugin_record(
                context, snat_info['uuid'], snat_info,
                ac_cnst.NW_HW_CREATE_SNAT)
        except Exception as ex:
            LOG.error(_LE("[AC]Failed to create flexible SNAT:%s"), ex)
            super(HuaweiACL3RouterPlugin, self).delete_floatingip(
                context, snat_info['uuid'])
            ac_osprofiler.record_chain_exception("create snat fail")
            raise BadRequest(resource='floatingip', msg=ex)
        LOG.info(_LI('[AC]Huawei AC create flexible SNAT successfully.'))
        ac_osprofiler.record_call_chain("create snat success")
        ac_osprofiler.record_chain_end_with_reason("create fip success")

    def _update_fip_assoc(self, context, fip, floatingip_db,
                          external_port=None):
        LOG.info(_LI("[AC] Begin to update fip association: %s, %s, %s"),
                 fip, floatingip_db, external_port)
        previous_router_id = floatingip_db.router_id
        port_id, internal_ip_address, router_id = (
            self._check_and_get_fip_assoc(context, fip, floatingip_db))
        association_event = None
        if floatingip_db.fixed_port_id != port_id:
            association_event = bool(port_id)
        if ncu.get_ops_version() in \
                [ac_cnst.OPS_Q, ac_cnst.OPS_R, ac_cnst.OPS_T, ac_cnst.OPS_W,
                 ac_cnst.FSP_6_5, ac_cnst.FSP_21_0]:
            floatingip_db.fixed_ip_address = internal_ip_address \
                if internal_ip_address else None
            floatingip_db.fixed_port_id = port_id
            floatingip_db.router_id = router_id
            floatingip_db.last_known_router_id = previous_router_id
            if 'description' in fip:
                floatingip_db.description = fip['description']
        else:
            update = {'fixed_ip_address': internal_ip_address,
                      'fixed_port_id': port_id,
                      'router_id': router_id,
                      'last_known_router_id': previous_router_id}
            if 'description' in fip:
                update['description'] = fip['description']
            floatingip_db.update(update)
        floating_ip_address = (str(floatingip_db.floating_ip_address)
                               if floatingip_db.floating_ip_address else None)
        floatingip_dict_update = {'fixed_ip_address': internal_ip_address,
                                  'fixed_port_id': port_id,
                                  'router_id': router_id,
                                  'last_known_router_id': previous_router_id,
                                  'floating_ip_address': floating_ip_address,
                                  'floating_network_id':
                                      floatingip_db.floating_network_id,
                                  'floating_ip_id': floatingip_db.id,
                                  'context': context}
        if ncu.get_ops_version() in \
                [ac_cnst.OPS_O, ac_cnst.OPS_P, ac_cnst.OPS_Q,
                 ac_cnst.OPS_R, ac_cnst.FSP_6_5]:
            return floatingip_dict_update
        if ncu.get_ops_version() in [ac_cnst.OPS_T, ac_cnst.OPS_W, ac_cnst.FSP_21_0]:
            floatingip_dict_update['association_event'] = association_event
            return floatingip_dict_update
        return None

    def update_floatingip(self, context, fid, floatingip):
        """Update floating IP.
        :param context: Neutron request context
        :param fid: Id for the floating IP being updated
        :param floatingip: data for the floating IP being updated
        :returns: A floating IP object on success

        The update operation is either associate or disassociate.
        When the operation fail, the rollback will be done.
        """
        LOG.info(_LI("[AC]Begin to update floating IP:%s,%s"), fid, floatingip)
        fip_old = self.get_floatingip(context, fid)
        LOG.info(_LI("[AC]Old floating IP is %s"), fip_old)
        ac_osprofiler.record_chain_start("update floatingip start,ID:" + fid)

        self._validate_father_floatinip(context, fip_old,
                                        floatingip['floatingip'])
        self._validate_sub_floatingip(context, fip_old,
                                      floatingip['floatingip'])
        self._validate_dnat(context, fid)

        with context.session.begin(subtransactions=True):
            floatingip_db_old = self._get_floatingip(context, fid)
            fip_old['last_known_router_id'] = floatingip_db_old.last_known_router_id
        LOG.info(_LI("[AC]The last known router of old floating IP is %s"),
                 fip_old['last_known_router_id'])
        fip = super(HuaweiACL3RouterPlugin, self).update_floatingip(
            context, fid, floatingip)
        LOG.info(_LI("[AC]New floating IP is %s"), fip)

        if ncu.IS_FSP and self.is_father_floatingip(context, fip_old):
            LOG.info(_LI("[AC] No need to update father floating IP: %s"), fid)
            return fip

        if ncu.IS_FSP and ACCommonUtil.is_sub_floatingip(context, fip_old):
            LOG.info(_LI("[AC] No need to update sub floating IP: %s"), fid)
            return fip

        if fip_old['fixed_ip_address'] and not fip['fixed_ip_address']:
            self._disassociate_floatingip_to_ac(context, fid, fip, fip_old)
        elif not fip_old['fixed_ip_address'] and fip['fixed_ip_address']:
            self._associate_floatingip_to_ac(context, fid, fip, fip_old)
        elif fip['fixed_ip_address']:
            self._update_floatingip_to_ac(context, fid, fip, fip_old)

        LOG.info(_LI("[AC]AC update floating IP successful."))
        ac_osprofiler.record_chain_end_with_reason("update fip success")
        return fip

    def disassociate_floatingips(self, context, port_id, do_notify=True):
        """Disassociate all floating IPs linked to specific port.
        @param context: neutron request context.
        @param port_id: ID of the port to disassociate floating IPs.
        @param do_notify: whether we should notify routers right away.
        @return: set of router-ids that require notification updates
                 if do_notify is False, otherwise None.
        """
        if ncu.get_ops_version() in \
                [ac_cnst.OPS_Q, ac_cnst.OPS_R, ac_cnst.OPS_T, ac_cnst.OPS_W,
                 ac_cnst.FSP_6_5, ac_cnst.FSP_21_0]:
            return self._disassociate_floatingips_ovo(context, port_id,
                                                      do_notify)

        return self._disassociate_floatingips_no_ovo(context, port_id,
                                                     do_notify)

    def delete_floatingip(self, context, id):
        """Delete floating IP."""
        LOG.info(_LI('[AC] Begin to delete floating IP: %s'), id)
        ac_osprofiler.record_chain_start("delete floating start,ID:" + id)
        floatingip = self._get_floatingip(context, id)
        floatingip_dict = self._make_floatingip_dict(floatingip)

        if ncu.IS_FSP and self.is_father_floatingip(context, floatingip_dict):
            msg = 'father floating IP should not be deleted'
            LOG.error(_LI('[AC] Failed to delete floating IP: %s.'), msg)
            ac_osprofiler.record_chain_exception_end(
                "father fip should not be deleted")
            raise BadRequest(resource='floatingip', msg=msg)

        self.check_dnat_from_db(context, id)

        if ncu.IS_FSP and \
                ACCommonUtil.is_sub_floatingip(context, floatingip_dict):
            try:
                ac_osprofiler.record_call_chain("delete snat")
                self.l3_reliability.update_plugin_record(
                    context, floatingip_dict['id'], {},
                    ac_cnst.NW_HW_DELETE_SNAT)
                ac_osprofiler.record_call_chain("delete snat success")
            except Exception as ex:
                ac_osprofiler.record_call_chain("delete snat fail")
                LOG.error(_LE('[AC] Failed to delete sub floating IP: %s'), ex)
                raise BadRequest(resource='floatingip', msg=ex)
        super(HuaweiACL3RouterPlugin, self).delete_floatingip(context, id)
        ac_osprofiler.record_chain_end_with_reason("delete fip success")

    def _delete_floatingip(self, context, fip_id):
        floatingip = self._get_floatingip(context, fip_id)
        floatingip_dict = self._make_floatingip_dict(floatingip)
        if ncu.get_ops_version() in \
                [ac_cnst.OPS_R, ac_cnst.OPS_T, ac_cnst.OPS_W, ac_cnst.FSP_21_0]:
            from neutron_lib.api import extensions as utils
        else:
            from neutron.common import utils
        if utils.is_extension_supported(self._core_plugin, 'dns-integration') \
                and hasattr(self, '_process_dns_floatingip_delete'):
            self._process_dns_floatingip_delete(context, floatingip_dict)

        if ncu.IS_FSP and \
                ACCommonUtil.is_sub_floatingip(context, floatingip_dict):
            if ncu.get_ops_version() == ac_cnst.FSP_6_5:
                l3_obj.FloatingIP.delete_objects(context, id=fip_id)
            else:
                with context.session.begin(subtransactions=True):
                    context.session.delete(floatingip)
            LOG.debug(_LI('[AC] Sub floating IP port should not be deleted.'))
            return floatingip_dict
        else:
            # Foreign key cascade will take care of the removal of the
            # floating IP record once the port is deleted. We can't start
            # a transaction first to remove it ourselves because the
            # delete_port method will yield in its post-commit activities.
            return super(HuaweiACL3RouterPlugin, self)._delete_floatingip(
                context, fip_id)

    def add_exroutes(self, context, router_id, exroutes_info):
        """Add exroutes."""
        LOG.info(_LI('[AC] Begin to add exroutes %(ex)s for router %(id)s'),
                 {'ex': exroutes_info, 'id': router_id})
        ac_osprofiler.record_chain_start("add exroutes start,router id:" +
                                         router_id + str(exroutes_info))
        exroutes_failed = []
        for exroute in exroutes_info.get('exroutes'):
            if ncu.get_ops_version() not in [ac_cnst.FSP_6_1, ac_cnst.FSP_6_3_0]:
                self._update_exroute_ip_version(exroute)
            if exroute.get('destination'):
                exroute['destination'] = self._get_destination(exroute)
            if exroute.get('nexthop'):
                exroute['nexthop'] = str(netaddr.IPAddress(exroute['nexthop']))

        exroutes_dict = super(HuaweiACL3RouterPlugin, self).add_exroutes(
            context, router_id, exroutes_info)

        session = self._get_session(context)
        rec_plugin_list, exroute_info_list, exroute_list = \
            self._generate_exroute_list(
                context, session, exroutes_dict.get('exroutes'), router_id)

        for flag_number, _ in enumerate(exroute_info_list):
            exroute = exroute_list[flag_number]
            try:
                self.l3_reliability.update_plugin_record(
                    context, router_id, exroute_info_list[flag_number],
                    NW_HW_CREATE_EXROUTE, rec_plugin=rec_plugin_list[flag_number])
            except Exception as ex:
                LOG.error(_LE('[AC] Huawei AC add exroute %(route)s '
                              'failed: %(ex)s'), {'route': exroute, 'ex': ex})
                self._optisize_exroute_info(exroute)
                exroutes_failed.append(self._get_exroute_format(exroute))

        self._delete_exroute_plugin_record(session, rec_plugin_list)
        self._handle_add_exroutes_failure(context, router_id, exroutes_failed)

        exroutes_dict = self._extend_exroutes_dict(exroutes_dict)
        LOG.info(_LI('[AC] Huawei AC add exroutes successfully.'))
        ac_osprofiler.record_chain_end_with_reason("add exroute success")
        return exroutes_dict

    def remove_exroutes(self, context, router_id, exroutes_info):
        """Remove exroutes."""
        LOG.info(_LI('[AC] Begin to remove exroutes %(ex)s for router %(id)s'),
                 {'ex': exroutes_info, 'id': router_id})

        exroutes = exroutes_info.get('exroutes')
        for exroute in exroutes:
            if exroute.get('type') != exroutes[0].get('type'):
                raise l3_router_util.ExRoutesTypeException(
                    ex1=str(exroute), ex2=str(exroutes[0]))

        ac_osprofiler.record_chain_start("remove exroutes start,router id:" +
                                         router_id + str(exroutes_info))
        exroutes_removed = []
        exroutes_failed = []

        for exroute in exroutes_info.get('exroutes'):
            exroute_to_format = self._get_exroute_format(exroute)
            exroute_to_format_copy = copy.deepcopy(exroute_to_format)
            if 'description' in exroute:
                exroute_to_format_copy['description'] = exroute['description']
            exroute_info = ACExRouteModel.ac_model_format(exroute_to_format_copy, router_id)
            exroute_info.pop('ecmp-enable', None)
            try:
                self.l3_reliability.update_plugin_record(
                    context, router_id, exroute_info,
                    ac_cnst.NW_HW_DELETE_EXROUTE)
                exroutes_removed.append(exroute_to_format)
            except Exception as ex:
                LOG.error(_LE('[AC] Huawei AC remove exroute %(route)s '
                              'failed: %(ex)s'), {'route': exroute, 'ex': ex})
                exroutes_failed.append(exroute)

        exroutes_dict = super(HuaweiACL3RouterPlugin, self).remove_exroutes(
            context, router_id, {'exroutes': exroutes_removed})

        if exroutes_failed:
            LOG.error(_LE('[AC] Huawei AC remove exroutes %s failed.'),
                      exroutes_failed)
            ac_osprofiler.record_chain_exception_end("remove exroutes fail")
            raise l3_router_util.ExRoutesException(
                opr='remove', ex=exroutes_failed)

        exroutes_dict = self._extend_exroutes_dict(exroutes_dict)

        LOG.info(_LI('[AC] Huawei AC remove exroutes successfully.'))
        ac_osprofiler.record_chain_end_with_reason("remove exroute success")
        return exroutes_dict

    def create_exroutes(self, context, router_id, exroutes_info):
        """Create exroutes."""
        LOG.info(_LI('[AC] Begin to create exroutes %(ex)s for router %(id)s'),
                 {'ex': exroutes_info, 'id': router_id})
        ac_osprofiler.record_chain_start("create exroutes start,router id:" +
                                         router_id + str(exroutes_info))
        exroutes_dict = super(HuaweiACL3RouterPlugin, self).create_exroutes(
            context, router_id, exroutes_info)

        self._create_exroutes_after(context, router_id, exroutes_dict.pop('exroutes_list', []))

        exroutes_dict.pop('id', None)
        LOG.info(_LI('[AC] Huawei AC create exroutes successfully.'))
        ac_osprofiler.record_chain_end_with_reason("create exroute success")
        return exroutes_dict

    def _create_exroutes_after(self, context, router_id, ex_routes):
        exroutes_failed = []
        session = self._get_session(context)
        rec_plugin_list, exroute_info_list, exroute_list = self._generate_exroute_list(
            context, session, ex_routes, router_id)
        for flag_number, _ in enumerate(exroute_info_list):
            exroute = exroute_list[flag_number]
            try:
                self.l3_reliability.update_plugin_record(
                    context, router_id, exroute_info_list[flag_number],
                    NW_HW_CREATE_EXROUTE, rec_plugin=rec_plugin_list[flag_number])
            except Exception as ex:
                LOG.error(_LE('[AC] Huawei AC create exroute %(route)s '
                              'failed: %(ex)s'), {'route': exroute, 'ex': ex})
                self._optisize_exroute_info(exroute)
                exroutes_failed.append(self._get_exroute_format(exroute))
        self._delete_exroute_plugin_record(session, rec_plugin_list)
        self._handle_create_exroutes_failure(context, router_id, exroutes_failed)

    def delete_exroutes(self, context, router_id, exroutes_info):
        """Delete exroutes."""
        LOG.info(_LI('[AC] Begin to delete exroutes %(ex)s for router %(id)s'),
                 {'ex': exroutes_info, 'id': router_id})
        ac_osprofiler.record_chain_start("delete exroutes start,router id:" +
                                         router_id + str(exroutes_info))
        exroutes_deleted = []
        exroutes_failed = []
        exroutes_dict_del = super(HuaweiACL3RouterPlugin, self).pre_delete_exroutes(
            context, router_id, exroutes_info)

        for exroute in exroutes_dict_del.get('exroutes_del'):
            exroute_to_format = self._get_exroute_format(exroute)
            try:
                exroute_info = ACExRouteModel.ac_model_format(
                    exroute_to_format, router_id)
                exroute_info.pop('ecmp-enable', None)
                self.l3_reliability.update_plugin_record(
                    context, router_id, exroute_info,
                    ac_cnst.NW_HW_DELETE_EXROUTE)
                exroutes_deleted.append(exroute_to_format)
            except Exception as ex:
                LOG.error(_LE('[AC] Huawei AC delete exroute %(route)s '
                              'failed: %(ex)s'), {'route': exroute, 'ex': ex})
                exroutes_failed.append(exroute)

                LOG.error(_LE('[AC] Huawei AC delete exroutes %s failed.'),
                          exroutes_failed)
                ac_osprofiler.record_chain_exception_end("delete exroutes fail")

        exroutes_dict = super(HuaweiACL3RouterPlugin, self).delete_exroutes(
            context, router_id, {'exroutes': exroutes_deleted})
        if exroutes_failed:
            LOG.error(_LE('[AC] Huawei AC remove exroutes %s failed.'),
                      exroutes_failed)
            ac_osprofiler.record_chain_exception_end("remove exroutes fail")
            raise BadRequest(resource='exroutes', msg=exroutes_failed)

        exroutes_dict.pop('exroutes_list', None)
        exroutes_dict.pop('id', None)

        LOG.info(_LI('[AC] Huawei AC delete exroutes successfully.'))
        ac_osprofiler.record_chain_end_with_reason("delete exroute success")
        return exroutes_dict

    def list_exroutes(self, context, router_id, type_value=None,
                      nexthop_value=None):
        """List exroutes."""
        LOG.info(_LI('[AC]Begin to list exroutes for router %s,type:%s,nexthop:'
                     '%s'), router_id, type_value, nexthop_value)

        exroutes_dict = super(HuaweiACL3RouterPlugin, self).list_exroutes(
            context, router_id, type_value, nexthop_value)

        exroutes_dict.pop('exroutes_list')
        exroutes_dict.pop('id', None)

        return exroutes_dict

    def dvr_deletens_if_no_port(self, context, port_id, port_host=None):
        """Delete the DVR namespace if no dvr serviced port exists."""
        return []

    def _send_notify(self, res_type, event, payload):
        """Send notify to hill stone plugin."""
        if self._hill_stone_config():
            if event != ac_cnst.AFTER_AC_DELETE:
                ncu.registry.notify(res_type, event, self, **payload)
            else:
                thd = threading.Thread(
                    target=self._send_notify_of_delete_event,
                    args=(res_type, event, payload,))
                thd.start()
