#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""ml2 driver"""

import ast
import os
import re
import time
from datetime import datetime

from neutron import manager
from neutron.db import db_base_plugin_v2
from neutron.db import models_v2
from neutron.db.allowedaddresspairs_db import AllowedAddressPairsMixin
from neutron.plugins.ml2 import db as ml2db
from oslo_config import cfg
from oslo_serialization import jsonutils

from networking_huawei._i18n import _LE
from networking_huawei._i18n import _LI
from networking_huawei.common import constants
from networking_huawei.common import exceptions as ml2_exc
from networking_huawei.drivers.ac.client.service import ACReSTService
from networking_huawei.drivers.ac.common import constants as ac_cnst, neutron_version_util
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import osprofiler_warp as ac_osprofiler
from networking_huawei.drivers.ac.common.neutron_compatible_util import ac_ml2_api as api
from networking_huawei.drivers.ac.common.util import ACCommonUtil, DataFilterUtil
from networking_huawei.drivers.ac.db import dbif
from networking_huawei.drivers.ac.db.lock import lockedobjects_db as lock_db
from networking_huawei.drivers.ac.model.port_model import ACPortModel
from networking_huawei.drivers.ac.plugins.ml2.pubservice_driver import PublicService
from networking_huawei.drivers.ac.sync import util
from networking_huawei.drivers.ac.sync.message_reliability_api import ACReliabilityAPI

try:
    from neutron.objects.trunk import models as trunk

    TRUNK_SUBPORT = True
except ImportError:
    TRUNK_SUBPORT = False
try:
    from neutron.services.trunk import constants as trunkport_const

    TRUNK_CALLBACK = True
except ImportError:
    try:
        from neutron_lib.callbacks import resources as trunkport_const

        TRUNK_CALLBACK = True
    except ImportError:
        TRUNK_CALLBACK = False
try:
    from neutron.db.common_db_mixin import CommonDbMixin
except ImportError:
    from networking_huawei.drivers.ac.common.common_db_mixin import CommonDbMixin
try:
    from neutron_lib import constants as neutron_const
except ImportError:
    pass

LOG = ncu.ac_log.getLogger(__name__)

VHOST_USER_MODE = 'vhostuser_mode'
# - server: socket created by hypervisor
VHOST_USER_MODE_SERVER = 'server'
# - client: socket created by vswitch
VHOST_USER_MODE_CLIENT = 'client'
VHOST_USER_OVS_PLUG = 'vhostuser_ovs_plug'
VHOST_USER_SOCKET = 'vhostuser_socket'
PRIMARY_INTERFACE = 'primary_interface'
REQUEST_SHIELD = True
CPS_IP_INI = os.path.realpath('/etc/huawei/fusionsphere/cfg/serverIp.ini')
CPS_SYS_INI = os.path.realpath(
    '/opt/fusionplatform/data/fusionsphere/cfg/sys.ini')
NEUTRON_CONF = os.path.realpath('/etc/neutron/neutron.conf')


class AAPMixin(AllowedAddressPairsMixin, CommonDbMixin):
    """Mixin class for allowed address pairs."""
    pass


class MechanismDriverPortUtil(object):
    """Port Util"""
    ac_reliability_api = ACReliabilityAPI()

    def __init__(self):
        pass

    @classmethod
    def _need_to_hierarchical_port_binding(cls, network, port_context,
                                           host_of_agent):
        """need to hierarchical port binding"""
        try:
            if 'vxlan_physnet_prefix' in network:
                return False
            for network_name in cfg.CONF.huawei_ac_config.network_black_list:
                if network["name"].endswith(network_name):
                    return False
            if network.get('provider:network_type', "") != "vxlan" or host_of_agent == port_context.host:
                return False
            if not port_context.host or port_context.host in cfg.CONF.huawei_ac_config.logical_hostnames:
                return False
            LOG.debug('[AC] Need to notify migrating, the network type is %s, host in db is %s, agent host is %s',
                      network.get('provider:network_type', ""), port_context.host, host_of_agent)
            return True
        except Exception as ex:
            LOG.error(_LE('[AC] judge need to bind port when vm migrating, catch %s.'), ex)
            return False

    @classmethod
    def _check_bind_port_output(cls, output, context=None):
        """check bind port output"""
        if not output:
            cls._handle_bind_level_exception("output in bind port level response is None.")

        next_segment = output.get('next-segment', {})
        network_type = next_segment.get('network-type')
        network_type_prefix = ac_cnst.NETWORK_TYPE_PREFFIX
        if not network_type or not network_type.startswith(network_type_prefix):
            cls._handle_bind_level_exception("network type in bind port level response is None.")

        segmentation_id = next_segment.get('segmentation-id')
        if not segmentation_id:
            cls._handle_bind_level_exception("segmentation id in bind port level response is None.")

        high_priority_physnet = cls.get_high_priority_physnet(context) if context else None
        phy_net = high_priority_physnet if high_priority_physnet else cfg.CONF.huawei_ac_config.physical_network
        dynamic_segment = {api.PHYSICAL_NETWORK: phy_net, api.NETWORK_TYPE: network_type[40:],
                           api.SEGMENTATION_ID: segmentation_id}
        return dynamic_segment

    @classmethod
    def _handle_bind_level_exception(cls, error_msg):
        LOG.error(_LE(error_msg))
        bind_port_url = ACCommonUtil.get_bind_url()
        raise ml2_exc.MechanismDriverError(method=constants.REST_POST, url=bind_port_url, error=error_msg)

    @classmethod
    def _get_bind_port_level_util(cls, network_id, host_id, context):
        """get bind port level"""
        physical_network = cfg.CONF.huawei_ac_config.physical_network
        if context.current['binding:profile'].get("physical_network", None):
            physical_network = context.current['binding:profile'].get("physical_network", None)
        high_priority_physnet = cls.get_high_priority_physnet(context)
        if high_priority_physnet:
            physical_network = high_priority_physnet
        bind_port_info = {'input': {'host-id': host_id, 'physical-network': physical_network,
                                    'network-id': network_id, 'port-id': context.current['id']}}
        bind_port_url = ACCommonUtil.get_bind_url()
        rest_service = ACReSTService()
        response = rest_service.send_bind_port_request(bind_port_info)

        if response:
            output = response.get('huawei-ac-neutron-bind-port-level:output', None)
            LOG.debug(_LI('[AC] Bind port level response get from AC: %s'), output)
            return cls._check_bind_port_output(output, context)

        LOG.error(_LE('[AC] Failed to get bind port level response from AC.'))
        error = 'bind port level response from AC is None.'
        raise ml2_exc.MechanismDriverError(method=constants.REST_POST, url=bind_port_url, error=error)

    @classmethod
    def _need_to_migrate_port(cls, host, port_context):
        """need to migrate port"""
        illegal_host = port_context.host and port_context.host != "" and host != port_context.host
        if illegal_host and port_context.host not in cfg.CONF.huawei_ac_config.logical_hostnames:
            return True
        return False

    @classmethod
    def _get_port_host(cls, context, port_id):
        """get port host"""
        if ncu.get_ops_version() in ac_cnst.OPS_VERSION_O_PQRTW_6_21:
            port_host = ml2db.get_port_binding_host(context, port_id)
        else:
            port_host = ml2db.get_port_binding_host(context.session, port_id)
        return port_host

    @classmethod
    def _filter_port_process(cls, context, port, port_id):
        """filter port process"""
        if not port:
            LOG.debug("No Port match for: %s", port_id)
            return True
        data_filter = DataFilterUtil()
        return data_filter.not_in_white_or_in_black_list(context, port, ac_cnst.NW_HW_PORTS, port_id)

    @classmethod
    def _ac_trunk_sub(cls, parent_host_id, parent_migrating_host, ac_trunk_details, sub_port_ready):
        """ac migrate trunk"""
        ac_sub_ports = ac_trunk_details.get('sub-ports', [])
        for sub_port in ac_sub_ports:
            sub_port_id = sub_port.get('port-id', '')
            if sub_port_id in sub_port_ready:
                continue
            ac_sub_info = cls._get_ac_port_info(sub_port_id)
            ac_host_id = ac_sub_info[0]
            ac_migrating_host = ac_sub_info[1]
            if parent_host_id != ac_host_id or parent_migrating_host != ac_migrating_host:
                return False
            LOG.info(_LI('[AC] ac migrating sub port success xjz_test'), sub_port_ready)
            sub_port_ready.append(sub_port_id)
        return True

    @classmethod
    def _get_ac_port_info(cls, port_id):
        """get ac port"""
        ac_port_info = ACReSTService().get_ac_port_info(port_id)
        ac_host_id = ac_port_info.get('huawei-ac-neutron-binding:host-id', '')
        ac_profile = ac_port_info.get('huawei-ac-neutron-binding:profile', {})
        ac_trunk_details = ac_port_info.get('trunk-details', {})
        ac_migrating_host = jsonutils.loads(
            ac_profile.replace("'", '"').replace("False", "false").replace("True", "true")).get('migrating-host', '')
        return (ac_host_id, ac_migrating_host, ac_trunk_details)

    @classmethod
    def _fsp_migrate(cls, port_id, port_host, host, start_time, sub_port_ready):
        """fsp migrate"""
        ac_host_id, ac_migrating_host, ac_trunk_details = cls._get_ac_port_info(port_id)
        LOG.info(_LI('[AC] %s ac host id:%s, ops host id:%s'), port_id, ac_host_id, port_host)
        LOG.info(_LI('[AC] %s ac migrating host:%s,ops migrating host:%s'), port_id, ac_migrating_host, host)
        ac_migrate_success = ac_host_id == port_host and ac_migrating_host == host
        if ac_migrate_success and not ac_trunk_details:
            LOG.info(_LI('[AC] %s migrate successfully.'), port_id)
            return True
        if ac_migrate_success and ac_trunk_details and cls._ac_trunk_sub(
                ac_host_id, ac_migrating_host, ac_trunk_details, sub_port_ready):
            LOG.info(_LI('[AC] %s migrate successfully.'), port_id)
            return True
        if (datetime.utcnow() - start_time).seconds > ac_cnst.PORT_MIGRATE_CONTROLLER_TIMEOUT:
            LOG.info(_LI('[AC] %s migrate timeout'), port_id)
            return True
        time.sleep(ac_cnst.CHECK_PORT_MIGRATED_INTERVAL)
        return False

    @classmethod
    def set_segment_and_notify_migrate(cls, resource, event, set_segment, **kwargs):
        """set segment and notify migrate"""
        LOG.info(_LI('[AC] Received the event %s for setting segment and notify migrating, %s, resource: %s'), event,
                 kwargs, resource)
        port_context = kwargs.get('port_context')
        host = kwargs.get('host')
        segment = None

        if cls._need_to_migrate_port(host, port_context):
            LOG.info(_LI('[AC] Begin to migrate port to new host: %s'), host)
            port_id = port_context.current['id']
            port = db_base_plugin_v2.NeutronDbPluginV2().get_port(port_context._plugin_context, port_id)
            allowed_address_pairs = AAPMixin().get_allowed_address_pairs(port_context._plugin_context, port_id)
            if allowed_address_pairs:
                port['allowed_address_pairs'] = allowed_address_pairs
            LOG.info(_LI('[AC] port get from ml2 interface %s'), port)
            if cls._filter_port_process(port_context._plugin_context, port, port_id):
                set_segment(segment)
                return

            port_host = cls._migrate_port_to_new_host(host, port_context, port_id, port)
            start_time = datetime.utcnow()
            LOG.info(_LI('[AC] %s migrate start time: %s'), port_id, start_time)
            sub_port_ready = []
            while ncu.IS_FSP:
                if cls._fsp_migrate(port_id, port_host, host, start_time, sub_port_ready):
                    break

        network_id = port_context.current["network_id"]
        network = kwargs.get('plugin').get_network(port_context._plugin_context, network_id)
        if cls._need_to_hierarchical_port_binding(network, port_context, host):
            segment = cls._get_bind_port_level_util(network_id, host, port_context)
            segment['segmentation_id'] = int(segment.get('segmentation_id'))
            LOG.debug("[AC]Need to set segment %s from AC", segment)
        set_segment(segment)

    @classmethod
    def _migrate_port_to_new_host(cls, host, port_context, port_id, port):
        """ migrate port to new host"""
        port_info = ACPortModel.ac_model_format(port, str(port['tenant_id']))
        if port_info.get('security-groups'):
            port_info.pop('security-groups')

        # u'{"trusted": false}' need change to u'{"trusted": False}'
        port_context._binding.profile = port_context._binding.profile.replace("false", "False").replace("true", "True")
        profile = ast.literal_eval(port_context._binding.profile or u'{}')
        profile.update({'migrating_host': host})
        port_context._binding.profile = jsonutils.dumps(profile)
        physical_network = cfg.CONF.huawei_ac_config.physical_network
        high_priority_physnet = cls.get_high_priority_physnet(port_context)
        if high_priority_physnet:
            physical_network = high_priority_physnet
        port_info['profile'] = str(
            ast.literal_eval(jsonutils.dumps({'migrating-host': host, 'physical_network': physical_network})))
        port_host = cls._get_port_host(port_context._plugin_context, port_id)
        port_info['host-id'] = port_host

        if port_host != host and not cls.is_edge_dc_vm_port(port_context):
            try:
                cls.ac_reliability_api.update_plugin_record(
                    port_context._plugin_context, port_id, port_info, ac_cnst.NW_HW_UPDATE_PORT)
            except Exception as ex:
                LOG.error(_LE('[AC] Failed to migrate VM port in huawei driver: %s'), ex)
                raise
        return port_host

    @classmethod
    def _get_parent_port_id(cls, payload):
        """get parent port id"""
        if payload.original_trunk:
            return payload.original_trunk.port_id
        return payload.current_trunk.port_id

    @classmethod
    def _need_to_update_subport(cls, context, event, subports, parent_port):
        """need to update subnet port"""
        router_external = vlan_transparent = False
        network_id = parent_port.get('network_id', None)
        vnic_type = parent_port.get('binding:vnic_type')
        vhost_user = cfg.CONF.huawei_ac_config.vhost_user
        vhostuser_evs_plug = parent_port.get('binding:vif_details', {}).get('vhostuser_evs_plug')
        if network_id:
            network = ncu.get_core_plugin().get_network(context, network_id)
            LOG.info(_LI('Get the network info :%s'), network)
            if network:
                router_external = network.get('router:external', None)
                vlan_transparent = network.get('vlan_transparent', None)
        if vnic_type != 'direct' and not vhost_user and not vhostuser_evs_plug:
            return False
        if not subports or event not in ['precommit_create', 'precommit_delete']:
            return False
        if router_external and vlan_transparent:
            return False
        return True

    @classmethod
    def _update_hw_subport(cls, context, event, subport_info):
        """update hw subnet port"""
        core_plugin = ncu.get_core_plugin()
        data_filter = DataFilterUtil()
        subport_id = subport_info.get('subport_id')
        subport_host_id = subport_info.get('subport_host_id')
        if cfg.CONF.huawei_ac_config.vhost_user and not ncu.IS_FSP:
            subport = core_plugin.get_port(context, subport_id)
            subport_info = ACPortModel.ac_model_format(subport, context.tenant_name)
            if event == 'precommit_create':
                subport_info['device-owner'] = 'trunk:subport'
                subport_info['device-id'] = subport_info.get('trunk_id')
                subport_device_owner = 'trunk:subport'
                subport_device_id = subport_info.get('trunk_id')
            if event == 'precommit_delete':
                subport_info['device-owner'] = ''
                subport_info['device-id'] = ''
                subport_device_owner = ''
                subport_device_id = ''
            subport_info['host-id'] = subport_host_id
            if data_filter.not_in_white_or_in_black_list(context, subport, ac_cnst.NW_HW_PORTS, subport_id):
                return
            if data_filter.port_contain_anti_affinity(subport):
                return
            LOG.info(_LI("Start to send request of subport: %s"), subport_info)
            cls.ac_reliability_api.update_plugin_record(
                context, subport_info.get('uuid'), subport_info, ac_cnst.NW_HW_UPDATE_PORT)
            LOG.info(_LI("update subport in db start: %s."), subport_id)
            port = context.session.query(models_v2.Port).filter_by(id=subport_id).first()
            if port:
                port.device_id = subport_device_id
                port.device_owner = subport_device_owner
                context.session.merge(port)
            LOG.info(_LI("update subport in db end: %s."), subport_id)
        with context.session.begin(subtransactions=True):
            LOG.info(_LI("update subport in db start: %s."), subport_id)
            bindingport = ncu.get_port_binding(context.session, subport_id)
            if bindingport:
                bindingport.host = subport_host_id
                context.session.merge(bindingport)
                context.session.flush()
            LOG.info(_LI("update_subport_in_db end: %s"), subport_id)

    @classmethod
    def _process_fs_nfvi_subport(cls, trunk_details, subports, parent_port, event):
        """process fs nfvi subnet port"""
        current_ports = []
        mac_address = parent_port.get('mac_address', None)
        current_sub_ports = trunk_details.get('sub_ports', None)
        for subport in current_sub_ports:
            if subport.get('port_id'):
                current_ports.append(subport.get('port_id'))

        LOG.info(_LI("Current subport info: %s"), current_ports)
        for item in subports:
            if event == 'precommit_create' and item['port_id'] not in current_ports and mac_address:
                add_subport = {'segmentation_id': item['segmentation_id'], 'port_id': item['port_id'],
                               'segmentation_type': item['segmentation_type'], 'mac_address': mac_address}
                LOG.info(_LI("Add subport info: %s"), add_subport)
                current_sub_ports.append(add_subport)
            if event == 'precommit_delete' and item['port_id'] in current_ports and mac_address:
                delete_subport = {'segmentation_id': item['segmentation_id'], 'port_id': item['port_id'],
                                  'segmentation_type': item['segmentation_type'], 'mac_address': mac_address}
                LOG.info(_LI("Delete subport info: %s"), delete_subport)
                sub_port_flag = current_sub_ports.index(delete_subport)
                current_sub_ports.pop(sub_port_flag)

    @classmethod
    @lock_db.wrap_db_lock(lock_db.UPDATE_PARENT_PORT)
    def _send_ac_update_parent_port(cls, parent_port, context, event, trunk_des):
        """send ac update parent port.

        :param context: DB context for the parent port update
        :param event: type of operation.
        :param trunk_des: information of the trunk
        :return: None
        """
        subports = trunk_des.get("subports")
        parent_port_info = ACPortModel.ac_model_format(parent_port, context.tenant_name)
        LOG.debug("[AC] The parent port info: %s", parent_port_info)
        cls.ac_reliability_api.update_plugin_record(
            context, parent_port_info.get('uuid'), parent_port_info, ac_cnst.NW_HW_UPDATE_PORT)
        parent_port_data = {"parent_port": parent_port, "subports": subports, "trunk_id": trunk_des.get("trunk_id")}
        cls._process_subnet_port_date(context, event, parent_port_data)

    @classmethod
    def update_parent_port(cls, resource, event, trigger, **kwargs):
        """update parent port"""
        payload = kwargs.get('payload')
        context = payload.context
        parent_port_id = cls._get_parent_port_id(payload)
        data_filter = DataFilterUtil()
        LOG.debug('[AC] Begin to update parent port: %s, resource: %s, trigger: %s', parent_port_id, resource, trigger)
        try:
            parent_port = ncu.get_core_plugin().get_port(context, parent_port_id)
            trunk_details = parent_port.get('trunk_details')
            trunk_id = subports = ''

            if trunk_details:
                trunk_id = trunk_details.get('trunk_id')
            if payload.subports:
                subports = payload.subports
                if ncu.get_ops_version() in (
                        ac_cnst.FSP_6_5, ac_cnst.FSP_6_5_1, ac_cnst.OPS_Q, ac_cnst.OPS_R, ac_cnst.OPS_W, ac_cnst.OPS_T,
                        ac_cnst.FSP_8_0_0, ac_cnst.FSP_8_0_3, ac_cnst.FSP_21_0):
                    cls._process_fs_nfvi_subport(trunk_details, subports, parent_port, event)

            LOG.debug("[AC] The parent port id: %s ,The parent port info: %s", parent_port_id, parent_port)
            if data_filter.not_in_white_or_in_black_list(context, parent_port, ac_cnst.NW_HW_PORTS, parent_port_id):
                return
            if data_filter.port_contain_anti_affinity(parent_port):
                return
            trunk_des = {"subports": subports, "trunk_id": trunk_id}
            cls._send_ac_update_parent_port(parent_port, context, event, trunk_des)
        except Exception as ex:
            ac_osprofiler.record_chain_exception_end("Failed to update parent port:%s" % ex)
            LOG.error('[AC] Failed to update parent port in huawei driver: %s', ex)
            raise
        ac_osprofiler.record_chain_end()

    @classmethod
    def _process_subnet_port_date(cls, context, event, parent_port_data):
        """process subnet port date"""
        parent_port = parent_port_data.get("parent_port")
        subports = parent_port_data.get("subports")
        if not subports:
            return
        if cls._need_to_update_subport(context, event, subports, parent_port):
            sub_ports = []
            for item in subports:
                sub_ports.append(item['port_id'])
            subport_host_id = parent_port.get('binding:host_id')
            if event == 'precommit_delete':
                subport_host_id = ''
            for subport_id in sub_ports:
                subport_info = {"subport_id": subport_id, "trunk_id": parent_port_data.get("trunk_id"),
                                "subport_host_id": subport_host_id}
                cls._update_hw_subport(context, event, subport_info)

    @classmethod
    def get_high_priority_physnet(cls, context):
        """Get high priority physnet from ovs-anget.

        :param context: context of port
        :return: physical_network or None
        """
        if ncu.get_ops_version() not in (ac_cnst.FSP_6_5_1, ac_cnst.FSP_8_0_0, ac_cnst.FSP_8_0_3):
            return None
        current_profile = context.current.get('binding:profile', {})
        host_id = current_profile.get('migrating_to') if current_profile.get('migrating_to') else context.current.get(
            'binding:host_id')
        if not host_id:
            LOG.debug('[AC]Not get host_id of the port: %s', context.current['id'])
            return None
        core_plugin = manager.NeutronManager.get_plugin()
        agents = core_plugin.get_agents(
            context._plugin_context, filters={'agent_type': [neutron_const.AGENT_TYPE_OVS], 'host': [host_id]})
        if not agents:
            LOG.debug('[AC]Not get ovs_agent configurations of the host: %s', host_id)
            return None
        LOG.debug('[AC]Get the ovs_agent configurations: %s of host: %s', agents[0]['configurations'], host_id)
        tunnel_physnet = agents[0]['configurations'].get('tunnel_physnet')
        if tunnel_physnet:
            return tunnel_physnet
        virtio_vf_physnet = ncu.get_neutron_ext_param('neutron.DEFAULT.virtio_vf_physnet')
        if virtio_vf_physnet:
            caps = agents[0]['configurations'].get('ovs_capabilities', {})
            if 'virtio_vf' in caps.get('iface_types', []):
                return virtio_vf_physnet
        return None

    @classmethod
    def is_edge_dc_vm_port(cls, port_context):
        """ is edge dc vm port """
        if not ncu.IS_FSP or not cfg.CONF.huawei_ac_config.edge_dc:
            return False
        flg = port_context.network.network_segments[0]['network_type'] == 'vlan'
        if util.ACUtil.is_compute_port(port_context.current) and flg:
            return True
        return False

    @classmethod
    def compute_port_first_up(cls, port_context):
        """ whether compute port first up"""
        return not port_context.original.get('binding:host_id') and port_context.current.get('binding:host_id')

    @classmethod
    def _port_context_comparator(cls, original, current):
        """port context comparator"""
        LOG.info(_LI('[AC] original port: %s'), original)
        LOG.info(_LI('[AC] current port: %s'), current)
        if original['status'] != ac_cnst.NEUTRON_STATUS_BUILD:
            return False
        return cls._check_port_status(original, current, ac_cnst.NEUTRON_STATUS_ACTIVE)

    @classmethod
    def _check_port_status(cls, original, current, status):
        """check port down"""
        if current['status'] != status:
            return False
        original_keys = set(original.keys())
        if original_keys != set(current.keys()):
            return False
        for key in original_keys - {'revision_number', 'created_at', 'updated_at', 'status'}:
            if original[key] != current[key]:
                return False
        return True

    @classmethod
    def check_port_no_change(cls, original, current):
        """check port no change"""
        for key in set(original.keys()) - {'revision_number', 'created_at', 'updated_at', 'status', 'dns_assignment'}:
            if original[key] != current[key]:
                return False
        return True

    @classmethod
    def _need_update_port_context(cls, context, fixed_ips, device_owner, network_id):
        """need update port context"""
        if not ncu.IS_FSP:
            return False
        if not fixed_ips or not ACCommonUtil.is_port_contain_ipv6_address(fixed_ips):
            return False
        if device_owner != 'network:router_interface':
            return False
        if not network_id:
            return False
        if 'binding:profile' not in context.current:
            return False
        return True

    @classmethod
    def check_subport_no_change(cls, original, current):
        """check the subnet port is change or not"""
        for key in set(original.keys()) - {'revision_number', 'created_at', 'updated_at', 'status'}:
            if original[key] != current[key]:
                LOG.debug("[AC]subport has changes, port_id: %s, key: %s, original_value: %s, current_value: %s",
                          current['id'], key, original[key], current[key])
                return False
        return True

    @classmethod
    def _is_port_add_ipv4_address(cls, context):
        """is port add ipv4 address"""
        current_fixed_ips = context.current.get('fixed_ips', None)
        original_fixed_ips = context.original.get('fixed_ips', None)
        if not ACCommonUtil.is_port_contain_ipv4_address(current_fixed_ips):
            return False
        if ACCommonUtil.is_port_contain_ipv4_address(original_fixed_ips):
            return False
        if not ACCommonUtil.is_port_contain_ipv6_address(original_fixed_ips):
            return False
        return True

    @classmethod
    def _is_port_remove_ipv4_address(cls, context):
        """is port remove ipv4 address"""
        current_fixed_ips = context.current.get('fixed_ips', None)
        original_fixed_ips = context.original.get('fixed_ips', None)
        if ACCommonUtil.is_port_contain_ipv4_address(current_fixed_ips):
            return False
        if not ACCommonUtil.is_port_contain_ipv4_address(original_fixed_ips):
            return False
        if not ACCommonUtil.is_port_contain_ipv6_address(current_fixed_ips):
            return False
        return True

    @classmethod
    def _get_vm_port_count(cls, context, device_id):
        """get vm_port count"""
        LOG.info(_LI('[AC]Begin to get vm port count, port id: %s,device id: %s'), context.current['id'], device_id)
        session = context._plugin_context.session
        vm_ports = session.query(models_v2.Port).filter(models_v2.Port.device_id == device_id).all()
        vm_ports_with_ip = [port for port in vm_ports if port.fixed_ips]
        LOG.info(_LI('[AC]VM ports get from %s,vm ports:%s,vm ports with ip:%s'), device_id, vm_ports, vm_ports_with_ip)
        return len(vm_ports_with_ip)

    @staticmethod
    def is_baremetal_port(port):
        """check is baremetal port"""
        return port["device_owner"].startswith("baremetal:") or (port['binding:vnic_type'] == "baremetal" and
                                                                 not port['device_owner'].startswith('network:f5'))

    @classmethod
    def is_primary_port(cls, context):
        """check the port is primary or not"""
        LOG.info(_LI('[AC] Begin to confirm primary port %s'), context.current)
        LOG.info(_LI('[AC] Original primary port %s'), context.original)
        if (not util.ACUtil.is_compute_port(context.current)) and (not cls.is_baremetal_port(context.current)):
            LOG.info(_LI('[AC] %s is not compute and baremetal port'), context.current['id'])
            return False
        # When VM shelve or unshelve, primary interface's
        # vif_details already exists
        if context.original and context.original.get(ncu.VIF_DETAILS, {}).get(PRIMARY_INTERFACE, False):
            LOG.info(_LI('[AC] %s original is primary interface'), context.current['id'])
            return True
        # When VM migrating, primary interface's vif_details already exists
        if context.vif_details.get(PRIMARY_INTERFACE, False):
            LOG.info(_LI('[AC] %s current is primary interface'), context.current['id'])
            return True
        if cls._get_vm_port_count(context, context.current['device_id']) == 1:
            LOG.info(_LI('[AC] %s get vm port for %s is 1'), context.current['id'], context.current['device_id'])
            return True
        return False

    @classmethod
    def create_or_delete_fip_when_port_update(cls, context):
        """create or delete fip when port update"""
        if not (ncu.IS_FSP or util.ACUtil.is_compute_port(context.current)
                and cls.is_baremetal_port(context.current)):
            LOG.debug("[AC] It is not FS vm,no need create or delete 100.64 EIP")
            return
        fixed_ips = context.current.get('fixed_ips', None)
        if not context.original.get('device_owner') and ACCommonUtil.is_port_contain_ipv4_address(fixed_ips):
            LOG.info(_LI('[AC] Device owner of original is None, current is %s. Try to create public service FIP.'),
                     context.current.get('device_owner'))
            PublicService().create_fip(context)
        elif cls._is_port_add_ipv4_address(context):
            LOG.info(_LI('[AC] port has ipv6 address and now add a ipv4 address.Try to create public service FIP.'))
            PublicService().create_fip(context)
        elif cls._is_port_remove_ipv4_address(context):
            LOG.info(_LI('[AC] port has ipv6 address and now remove a ipv4 address.Try to delete public service FIP.'))
            PublicService().delete_fip(context, by_update=True)
        else:
            LOG.debug("[AC] FS vm no need create or delete 100.64 EIP")

    @staticmethod
    def _check_segment(segment):
        """Check whether segment is valid for the AC MechanismDriver."""
        return segment[api.NETWORK_TYPE] in {ncu.TYPE_LOCAL, ncu.TYPE_GRE, ncu.TYPE_VXLAN, ncu.TYPE_VLAN, ncu.TYPE_FLAT}

    @classmethod
    def bind_port_once(cls, context, segment):
        """bind port once"""
        if cls._check_segment(segment):
            if not cfg.CONF.huawei_ac_config.vhost_user:
                vif_details = {ncu.OVS_HYBRID_PLUG: True, ncu.CAP_PORT_FILTER: True}
                context.set_binding(segment[api.ID], 'ovs', vif_details)
            elif cls.is_baremetal_port(context.current):
                vif_details = {}
                context.set_binding(segment[api.ID], 'other', vif_details)
            else:
                vif_details = {ncu.CAP_PORT_FILTER: False, ncu.OVS_HYBRID_PLUG: False,
                               VHOST_USER_MODE: VHOST_USER_MODE_SERVER, VHOST_USER_OVS_PLUG: True}
                if ncu.IS_FSP:
                    vif_details[VHOST_USER_MODE] = VHOST_USER_MODE_CLIENT
                sock_name = ('vhu' + context.current['id'])[:14]
                vif_details[VHOST_USER_SOCKET] = os.path.join('/usr/ce1800v/vhost-user', sock_name)
                context.set_binding(segment[api.ID], 'vhostuser', vif_details)
                ac_osprofiler.record_call_chain("set ce1800v bind")
        else:
            ac_osprofiler.record_call_chain("no need to bind")
            LOG.info(_LI('[AC]No binding requirements for segment ID %s, segment %s, phys net %s, and network type %s'),
                     segment[api.ID], segment[api.SEGMENTATION_ID], segment[api.PHYSICAL_NETWORK],
                     segment[api.NETWORK_TYPE])

    @classmethod
    def set_port_status(cls, context, port_id, status):
        """set port status"""
        LOG.info(_LI('[AC]Begin to set port %s status %s'), port_id, status)
        session = context._plugin_context.session
        port_rec = session.query(models_v2.Port).filter_by(id=port_id).first()
        if port_rec:
            port_rec.status = status
            session.merge(port_rec)
            session.flush()

    @classmethod
    def _handle_subport_upgrade(cls, context, subport_id, trunk_id, port_info):
        """Handle subport upgrade"""
        LOG.info(_LI("Handle subport upgrade start :%s."), trunk_id)
        if not trunk_id:
            return None
        trunk_info = (context._plugin_context.session.query(trunk.Trunk).filter_by(id=trunk_id).first())
        parent_port_id = trunk_info.get('port_id', None)
        LOG.info(_LI("The trunk info :%s."), trunk_info)
        if not parent_port_id:
            return None
        parent_port_info = ncu.get_port_binding(context._plugin_context.session, parent_port_id)
        LOG.info(_LI("The parent port info: %s."), parent_port_info)
        if not parent_port_info:
            return None
        parent_host_id = parent_port_info.get('host')
        LOG.info(_LI("The parent host id is: %s."), parent_host_id)
        with context._plugin_context.session.begin(subtransactions=True):
            port_binding = ncu.get_port_binding(context._plugin_context.session, subport_id)
            if port_binding:
                port_binding.host = parent_host_id
                # _handle_subport_restore方法会将该处的设置复原
                if parent_port_info.vnic_type == ac_cnst.VNIC_TYPE_BAREMETAL \
                        and ncu.get_ops_version() in [ac_cnst.FSP_21_0]:
                    port_binding.profile = cls.process_profile_for_db(parent_port_info.profile)
                    port_binding.vnic_type = ac_cnst.VNIC_TYPE_SUB_BAREMETAL
                context._plugin_context.session.merge(port_binding)
                context._plugin_context.session.flush()
            subport = context._plugin_context.session.query(models_v2.Port).filter_by(id=subport_id).first()
            if subport:
                subport.device_id = trunk_id
                subport.device_owner = 'trunk:subport'
                context._plugin_context.session.merge(subport)
                context._plugin_context.session.flush()
        port_info['host-id'] = parent_host_id
        port_info['device-id'] = trunk_id
        port_info['device-owner'] = 'trunk:subport'
        # baremetal场景下，使用父port的binding信息更新subport
        if parent_port_info.vnic_type == ac_cnst.VNIC_TYPE_BAREMETAL and ncu.get_ops_version() in [ac_cnst.FSP_21_0]:
            LOG.info('[AC] Update profile for subport: %s', subport_id)
            port_info['profile'] = cls.get_ac_port_profile(context, parent_port_id, parent_port_info.profile)
            if not port_info['profile']:
                json_str = jsonutils.dumps(port_binding.profile) \
                    .replace('true', 'True').replace('false', 'False').replace('null', "''")
                port_info['profile'] = str(ast.literal_eval(json_str))
            if parent_port_info.vnic_type:
                port_info['vnic-type'] = ac_cnst.VNIC_TYPE_SUB_BAREMETAL
        return port_info

    @classmethod
    @neutron_version_util.restrict_version(ac_cnst.FSP_21_0)
    def _handle_subport_restore(cls, context, subport_id):
        LOG.info('[AC] Handle subport restore, subport_id: %s', subport_id)
        original_trunk_id = context.original.get('device_id', '')
        original_trunk_info = context._plugin_context.session.query(trunk.Trunk).filter_by(
            id=original_trunk_id).first()
        if not original_trunk_info or not original_trunk_info.get('port_id', None):
            return
        original_parent_port_id = original_trunk_info.get('port_id')
        original_parent_port_info = ncu.get_port_binding(context._plugin_context.session, original_parent_port_id)
        if original_parent_port_info and original_parent_port_info.vnic_type == ac_cnst.VNIC_TYPE_BAREMETAL:
            LOG.info('[AC] Restore profile for subport: %s', subport_id)
            with context._plugin_context.session.begin(subtransactions=True):
                port_binding = ncu.get_port_binding(context._plugin_context.session, subport_id)
                if port_binding:
                    port_binding.profile = "{}"
                    port_binding.vnic_type = ac_cnst.VNIC_TYPE_NORMAL
                    context._plugin_context.session.merge(port_binding)
                    context._plugin_context.session.flush()

    @classmethod
    def get_ac_port_profile(cls, context, port_id, binding_profile):
        """获取传到ac的port的profile信息

        :param context: context
        :param port_id: parent port id
        :param binding_profile: parent port profile
        :return: profile
        """
        if not binding_profile:
            return ''
        if type(binding_profile) is str:
            binding_profile = jsonutils.loads(binding_profile)
        local_link = binding_profile.get('local_link_information', None)
        if local_link and len(local_link) > 1:
            bond_mode = cls.get_bond_mode(context, port_id, binding_profile)
            binding_profile['mode'] = bond_mode
            json_str = jsonutils.dumps(binding_profile) \
                .replace('true', 'True').replace('false', 'False').replace('null', "''")
            return str(ast.literal_eval(json_str))
        return ''

    @classmethod
    def filter_port_update_status(cls, context, migrating_host):
        """filter port and update port status"""
        if context.current['status'] == ac_cnst.NEUTRON_STATUS_BUILD:
            LOG.info(_LI("[AC] port %s status BUILD, no need send to AC"), context.current['id'])
            if context.original['status'] == ac_cnst.NEUTRON_STATUS_ERROR:
                cls.set_port_status(context, context.current['id'], ac_cnst.NEUTRON_STATUS_ERROR)
            ac_osprofiler.record_chain_end_with_reason("port is BUILD, return")
            return True
        if REQUEST_SHIELD and util.ACUtil.is_compute_port(context.current):
            if cls._port_context_comparator(context.original, context.current) and not migrating_host:
                LOG.info(_LI('[AC] port %s status BUILD -> ACTIVE, no need send to AC'), context.current['id'])
                ac_osprofiler.record_chain_end_with_reason("port status BUILD -> ACTIVE, return")
                return True
            if cls._check_port_status(context.original, context.current, ac_cnst.NEUTRON_STATUS_DOWN):
                LOG.info(_LI('[AC] port %s status DOWN, no need send to AC'), context.current['id'])
                ac_osprofiler.record_chain_end_with_reason("port status DOWN, return")
                return True
        return False

    @classmethod
    def update_port_context(cls, context):
        """update port context"""
        fixed_ips = context.current.get('fixed_ips', [{}])
        device_owner = context.current.get('device_owner', None)
        network_id = context.current.get('network_id', None)
        if cls._need_update_port_context(context, fixed_ips, device_owner, network_id):
            interface_network = ncu.get_core_plugin().get_network(context._plugin_context, network_id)
            LOG.info(_LI("[AC]The network info of ipv6 interface port is: %s."), interface_network)
            network_tags = interface_network.get('tags', None)
            external_network = interface_network.get('router:external', None)
            LOG.info(_LI("[AC]Ipv6 interface port add enable_internet: %s."), context.current)
            if context._binding.profile:
                profile = jsonutils.loads(context._binding.profile)
            else:
                profile = {}

            if 'enable_internet' not in profile:
                profile.update({'enable_internet': False, 'profile_updated': True})

            if network_tags and external_network and 'service_type=Internet6' in network_tags:
                LOG.info(_LI("[AC]Ipv6 interface port add service_type: %s."), context.current)
                if 'service_type' not in profile:
                    profile.update({'service_type': 'Internet6', 'profile_updated': True})

            if 'profile_updated' in profile:
                profile.pop('profile_updated')
                context._binding.profile = jsonutils.dumps(profile)
                context.current[ncu.PROFILE].update(profile)

    @classmethod
    def register_port(cls):
        """register port"""
        try:
            ncu.registry.subscribe(cls.set_segment_and_notify_migrate, ac_cnst.NW_HW_PORTS,
                                   ac_cnst.PORT_MIGRATION_EVENT)
        except Exception as ex:
            LOG.error('register_port occur error:%s', ex)
        if TRUNK_CALLBACK:
            ncu.registry.subscribe(cls.update_parent_port, trunkport_const.SUBPORTS, ncu.events.PRECOMMIT_CREATE)
            ncu.registry.subscribe(cls.update_parent_port, trunkport_const.SUBPORTS, ncu.events.PRECOMMIT_DELETE)
            ncu.registry.subscribe(cls.update_parent_port, trunkport_const.TRUNK, ncu.events.AFTER_CREATE)
            ncu.registry.subscribe(cls.update_parent_port, trunkport_const.TRUNK, ncu.events.AFTER_DELETE)

    @classmethod
    def process_subport_upgrade(cls, context, subport_id, port_info):
        """process subport upgrade"""
        LOG.info(_LI("[AC] Process subport upgrade original:%s."), context.original)
        if not TRUNK_SUBPORT:
            return port_info
        trunk_id = cls._check_subport_upgrade(context, subport_id)
        if trunk_id:
            port_info = cls._handle_subport_upgrade(context, context.current['id'], trunk_id, port_info)
        if context.original.get('device_owner') == 'trunk:subport' \
                and context.current.get('device_owner') == '' \
                and context.current.get('binding:vnic_type') == ac_cnst.VNIC_TYPE_SUB_BAREMETAL:
            cls._handle_subport_restore(context, subport_id)
        if context.current.get('device_owner') == 'trunk:subport' \
                and not context.original.get('device_owner', ''):
            port_info = cls._handle_subport_online(context, trunk_id, port_info)
        return port_info

    @classmethod
    def _handle_subport_online(cls, context, trunk_id, port_info):
        """Handle subport upgrade"""
        trunk_info = context._plugin_context.session.query(trunk.Trunk)\
            .filter_by(id=trunk_id).first()
        parent_port_id = trunk_info.get('port_id', None)
        if parent_port_id:
            profile = jsonutils.loads(port_info.get('profile'))
            profile.update({"port-parent-id": parent_port_id})
            port_info['profile'] = jsonutils.dumps(profile)
            return port_info
        return None

    @classmethod
    def _check_subport_upgrade(cls, context, subport_id):
        """check subport upgrade"""
        LOG.info(_LI("Check subport upgrade :%s."), subport_id)
        subport_info = (context._plugin_context.session.query(trunk.SubPort).filter_by(port_id=subport_id).first())
        if subport_info:
            return subport_info.get('trunk_id', None)
        return None

    @classmethod
    def check_port(cls, port_info):
        """This function is to check whether the port is vm port or
         baremetal port or trunk port

        :param port_info: port info for the port update
        :return: Boolean
        """
        return util.ACUtil.is_compute_port(port_info) or cls.is_baremetal_port(port_info) or port_info.get(
            'device_owner', '').startswith('trunk:subport')

    @classmethod
    def set_fw_enabled(cls, context):
        """
        only allow external network port can set fw_enabled
        fsp: do not handle
        ops: default is false
        """
        network_id = context.current.get('network_id')
        if not network_id:
            return
        network = ncu.get_core_plugin().get_network(context._plugin_context, network_id)
        if not network:
            return
        if not network.get('router:external'):
            context.current['binding:profile'].pop("fw_enabled", None)
        elif not ncu.IS_FSP and "fw_enabled" not in context.current['binding:profile']:
            context.current['binding:profile'].update({"fw_enabled": False})

    @classmethod
    def get_bond_mode(cls, context, port_id, binding_profile):
        """从port_info中获取bond_mode

        :param context: context
        :param port_id: parent port id
        :param binding_profile: binding:profile
        :return: bond_mode
        """
        db_if = dbif.ACdbInterface()
        bond_mode = binding_profile.get('bond_mode', None)
        local_group_information = binding_profile.get('local_group_information', {})
        if bond_mode and re.match(r'^[0-6]$', bond_mode):
            bond_mode = ac_cnst.BOND_MODE_LIST[int(bond_mode)]
        elif local_group_information.get('bond_mode'):
            bond_mode = local_group_information['bond_mode']
            if bond_mode not in ac_cnst.BOND_MODE_LIST:
                bond_mode = ac_cnst.BOND_MODE_LIST[int(bond_mode)]
        else:
            bond_mode = db_if.get_bm_bond_mode(context._plugin_context.session, port_id)
            if not bond_mode:
                bond_mode = ac_cnst.BOND_MODE_LIST[cfg.CONF.huawei_ac_config.bare_metal_bond_mode]
                db_if.create_bm_bond_mode_record(context._plugin_context.session, port_id, bond_mode)
        LOG.info('[AC] Get bond mode from parent port: %s, binding_profile: %s, bond_mode: %s', port_id,
                 binding_profile, bond_mode)
        return bond_mode

    @staticmethod
    def process_profile_for_db(str_profile):
        """Process profile for db

        :param str_profile: binding:profile
        :return: profile
        """
        if not str_profile:
            return "{}"
        return str_profile.replace("'", '"').replace("False", "false").replace("True", "true").replace("null", '""')
