#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""ml2 driver"""

import os
import base64
import six
import netaddr
from oslo_config import cfg
from oslo_serialization import jsonutils
from neutron.plugins.ml2 import db as ml2db
from neutron.db import db_base_plugin_v2

from networking_huawei._i18n import _LI
from networking_huawei.drivers.ac.common import constants as ac_cnst
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.drivers.ac.encode_convert import convert_to_bytes

LOG = ncu.ac_log.getLogger(__name__)

VHOST_USER_MODE = 'vhostuser_mode'
# - server: socket created by hypervisor
VHOST_USER_MODE_SERVER = 'server'
# - client: socket created by vswitch
VHOST_USER_MODE_CLIENT = 'client'
VHOST_USER_OVS_PLUG = 'vhostuser_ovs_plug'
VHOST_USER_SOCKET = 'vhostuser_socket'
DEVICE_OWNER_LB = 'neutron:LOADBALANCER'
CPS_IP_INI = os.path.realpath('/etc/huawei/fusionsphere/cfg/serverIp.ini')
CPS_SYS_INI = os.path.realpath(
    '/opt/fusionplatform/data/fusionsphere/cfg/sys.ini')

NEUTRON_CONF = os.path.realpath('/etc/neutron/neutron.conf')


class MechanismDriverUtil(object):
    """ml2 common Util"""
    def __init__(self):
        pass

    @classmethod
    def get_cps_info(cls):
        """get cps info"""
        config_paser = six.moves.configparser.ConfigParser()
        config_paser.read(CPS_IP_INI)
        cps_server_ip = config_paser.get('IP', 'serverip')

        config_paser.read(CPS_SYS_INI)
        cps_user = config_paser.get('system_account', 'internal_cps_user')
        cps_pwd = config_paser.get('system_account', 'internal_cps_password')

        ncu.CPS_INFO = {'cps_user': cps_user,
                        'cps_pwd': cps_pwd,
                        'cps_server_ip': cps_server_ip}

    @classmethod
    def get_fs_ssl_cfg(cls):
        """ get fs ssl cfg """
        config_paser = six.moves.configparser.ConfigParser()
        config_paser.read(NEUTRON_CONF)
        ssl_cfgs = {'cert_file': None,
                    'key_file': None,
                    'private_key_password': None}
        for ssl_cfg_key in ssl_cfgs:
            if config_paser.has_option('ssl', ssl_cfg_key):
                ssl_cfgs[ssl_cfg_key] = config_paser.get('ssl', ssl_cfg_key)
        ncu.FS_CERT_FILE = ssl_cfgs.get('cert_file')
        ncu.FS_KEY_FILE = ssl_cfgs.get('key_file')
        ncu.FS_PRIVATE_KEY_PASSWORD = ssl_cfgs.get('private_key_password')

    @classmethod
    def parse_vcenter_host_mappings(cls, vcenter_host_mappings):
        """parse vcenter host mappings"""
        vcenter_host_info = {}
        for mapping_item in vcenter_host_mappings:
            info = mapping_item.split(":")
            for logical_host in info[1:]:
                vcenter_host_info[logical_host] = info[0]
        LOG.info(_LI('[AC] vcenter host info: %s'), vcenter_host_info)
        return vcenter_host_info

    @classmethod
    def parse_vcenter_network_mappings(cls, vcenter_network_mappings):
        """parse vcenter network mappings"""
        vcenter_network_info = {}
        for mapping_item in vcenter_network_mappings:
            info = mapping_item.split(":")
            vcenter_network_info[info[0]] = info[1:]
        LOG.info(_LI('[AC] vcenter network info: %s'), vcenter_network_info)
        return vcenter_network_info

    @classmethod
    def validate_ip_address_in_subnet(cls, context, subnet_id, ip_address):
        """validate ip address in subnet"""
        subnet_info = ncu.get_core_plugin().get_subnet(context, subnet_id)
        if not subnet_info:
            return False
        cidr = subnet_info.get('cidr')
        if (not cidr) or (not ip_address):
            return False
        ip_info = netaddr.IPAddress(ip_address)
        net = netaddr.IPNetwork(cidr)
        if ip_info == net.network:
            return False
        if net.version != 6 and ip_info == net[-1]:
            return False
        if net.netmask & ip_info != net.network:
            return False
        return True

    @staticmethod
    def check_l2br_vxlan(l2br, network):
        """check l2br vxlan"""
        if not network.get('provider:network_type') == 'vxlan':
            return False
        if l2br['service_info'].get('network_id') == network['id'] or \
                l2br['service_info'].get('vni') == \
                network.get('provider:segmentation_id'):
            return True
        return False

    @staticmethod
    def check_l2br_vlan(l2br, network):
        """check l2br vlan"""
        if network.get('provider:network_type') == 'vlan' \
                and l2br['service_info'].get('network_id') == network['id']:
            return True
        return False

    @staticmethod
    def update_binding_profile_sdn_status(context, sdn_bind_status):
        """update binding profile with sdn bind status.

        :param context: DB context for the port update
        :param sdn_bind_status: success or failed
        :return: None
        """
        if context._binding.profile:
            profile_binding_status = jsonutils.loads(context._binding.profile)
        else:
            profile_binding_status = {}
        profile_binding_status.update({'sdn_binding_status': sdn_bind_status})
        context._binding.profile = jsonutils.dumps(profile_binding_status)

    @classmethod
    def parse_vlan_ranges(cls, allocated):
        """parse vlan ranges"""
        vlan_set = set()
        for vlan_range in cfg.CONF.huawei_ac_config.allocate_vlan_ranges:
            start = int(vlan_range.split(':')[0])
            end = int(vlan_range.split(':')[1]) + 1
            vlan_set = vlan_set | set(range(start, end))
        if not vlan_set:
            vlan_set = set(range(2, 4095))
        return vlan_set - allocated

    @staticmethod
    def hw_get_segment(context, network_id):
        """hw get segment"""
        if ncu.get_ops_version() in ac_cnst.OPS_VERSION_O_PQRTW_6_21:
            from neutron.db import segments_db
            return segments_db.get_network_segments(context._plugin_context,
                                                    network_id)
        return ml2db.get_network_segments(
            context._plugin_context.session, network_id)

    @staticmethod
    def is_compute_external(context):
        """if contain vlan"""
        profile = context.current.get('binding:profile', {})
        if context.current.get('device_owner') == 'compute:external' \
                and profile.get('external_vlan'):
            return True
        return False

    @classmethod
    def get_h3c_physical_network(cls, context, vxlan_physnet_prefix):
        """get h3c physical network"""
        binary_list = ['neutron-openvswitch-agent', 'neutron-cas-ovs-agent',
                       'neutron-vmware-ovs-agent', 'f5-oslbaasv2-agent']
        filters = {'host': [context._binding.host], 'binary': binary_list}
        agents = context._plugin.get_agent(context._plugin_context, filters)
        physical_networks = []
        for agent in agents:
            configurations = agent.get('configuration', {})
            bridge_mappings = configurations.get('bridge_mappings', {})
            physical_networks.extend(list(bridge_mappings.keys()))

        if context.current.get('device_owner', '').startswith('network:f5'):
            physical_networks = \
                [physical_network for physical_network in physical_networks
                 if not physical_network.startswith('default')]
        LOG.info(_LI('[AC] h3c physical networks: %s'), physical_networks)
        for network in physical_networks:
            decoded_prefix = base64.b64decode(
                convert_to_bytes(vxlan_physnet_prefix))
            if base64.b64decode(
                    convert_to_bytes(network)).startswith(decoded_prefix):
                LOG.info(_LI('[AC] got h3c physical network: %s'), network)
                return network
        return None

    @staticmethod
    def filter_device_owner_process(device_owner):
        """filter device owner process"""
        compute = ncu.DEVICE_OWNER_COMPUTE_PREFIX
        if device_owner.startswith(compute) or \
                        device_owner in [ncu.DEVICE_OWNER_DHCP,
                                         ac_cnst.DEVICE_OWNER_F5_V1,
                                         ac_cnst.DEVICE_OWNER_F5_V2, 'trunk:subport'] or \
                device_owner.startswith(DEVICE_OWNER_LB):
            return True
        return False

    @classmethod
    def is_segment_ac_vxlan(cls, segment):
        """is segment ac vxlan"""
        network_type = segment.get('network_type')
        if network_type == ncu.TYPE_VXLAN:
            LOG.debug(_LI('[AC] Bind port network type is vxlan.'))
            return True
        LOG.debug(_LI('[AC] Bind port network type is not vxlan.'))
        return False

    @classmethod
    def is_segment_ac_vlan(cls, segment):
        """is segment ac vlan"""
        network_type = segment.get('network_type')
        if network_type == ncu.TYPE_VLAN:
            LOG.debug(_LI('[AC] Bind port network type is vlan.'))
            return True
        LOG.debug(_LI('[AC] Bind port network type is not vlan.'))
        return False

    @classmethod
    def get_port_subnet_and_ip(cls, port_fixed_ips):
        """get the port subnet and ip"""
        subnet_ips = {}
        for fixed_ip in port_fixed_ips:
            subnet_id = fixed_ip.get('subnet_id')
            ip_address = fixed_ip.get('ip_address')
            if subnet_id:
                subnet_ips[subnet_id] = ip_address
        return subnet_ips

    @staticmethod
    def if_vhost_compute(context):
        """is vhost vm port or not"""
        device_owner = context.current.get('device_owner', '')
        compute = ncu.DEVICE_OWNER_COMPUTE_PREFIX
        if device_owner.startswith(compute) and \
                cfg.CONF.huawei_ac_config.vhost_user and \
                context._binding.vnic_type not in \
                [ncu.VNIC_DIRECT, 'direct-physical']:
            return True
        return False

    @classmethod
    def check_migrating_host(cls, context):
        """check migrating host"""
        current_profile = context.current['binding:profile']
        migrating_host = current_profile.get('migrating_host')
        if migrating_host and migrating_host != context.host:
            current_profile.pop('migrating_host', None)
            context._binding.profile = jsonutils.dumps(current_profile)
            LOG.info(_LI('[AC] %s is migrating to host %s, no need to '
                         'deal with host %s update request'),
                     context.current['id'], migrating_host,
                     context.host)
            ac_osprofiler.record_chain_end_with_reason(
                "no need to deal whit source host triger port update "
                "when vm migrating,return")
            return True
        return False

    @classmethod
    def update_gateway_ports_when_subnet_create(cls, context):
        """update gateway ports when subnet create"""
        if ncu.get_ops_version() in [ac_cnst.OPS_K]:
            db_plugin = db_base_plugin_v2.NeutronDbPluginV2()
        else:
            from neutron.db import db_base_plugin_common
            db_plugin = db_base_plugin_common.DbBasePluginCommon()
        network_id = context.current['network_id']
        network_db = db_plugin._get_network(context._plugin_context,
                                            network_id)
        LOG.info(_LI('The network info of the subnet is : %s'), network_db)
        plugin = db_base_plugin_v2.NeutronDbPluginV2()
        plugin._update_router_gw_ports(context._plugin_context,
                                       network_db, context.current)

    @classmethod
    def network_qos_policy_changed(cls, context):
        """network qos policy changed"""
        if not ACCommonUtil.qos_plugin_configured():
            return False
        cur_qos_policy = context.current['qos_policy_id']
        org_qos_policy = context.original['qos_policy_id']
        return org_qos_policy and cur_qos_policy != org_qos_policy
