#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""pubservice driver"""

from neutron.db import models_v2
try:
    from neutron.extensions.external_net import EXTERNAL
except ImportError:
    from neutron_lib.api.definitions.external_net import EXTERNAL

try:
    from neutron.db.l3_db import RouterPort, FloatingIP
except ImportError:
    from neutron.db.models.l3 import RouterPort, FloatingIP
from neutron import manager

try:
    from neutron import context as new_context
except ImportError:
    from neutron_lib import context as new_context

try:
    from neutron_lib.plugins import directory
except ImportError:
    pass
from oslo_config import cfg

from networking_huawei._i18n import _LE
from networking_huawei._i18n import _LI
from networking_huawei.common import exceptions as ml2_exc
from networking_huawei.drivers.ac.common import constants
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.drivers.ac.sync.util import ACUtil


LOG = logging.getLogger(__name__)


class PublicService(object):
    """Maintain EIP for Public Service"""

    def __init__(self):
        self.public_service_networks = \
            cfg.CONF.huawei_ac_config.public_service_networks
        self.default_pub_ser_net = \
            cfg.CONF.huawei_ac_config.default_public_service_network
        self.ops_version = ncu.get_ops_version()
        self.admin_context = new_context.get_admin_context()

    @property
    def _core_plugin(self):
        if self.ops_version in constants.OPS_VERSION_O_PQRTW_6_21:
            return directory.get_plugin()
        return manager.NeutronManager.get_plugin()

    @classmethod
    def check_duplicate_pubservice(cls, session, port_id, network_id):
        """check duplicate pubservice"""
        LOG.info(_LI("[AC] begin to check duplicate public service for port "
                     "%(port)s, network id %(network_id)s"),
                 {'port': port_id, 'network_id': network_id})
        floatingip_query = session.query(FloatingIP)
        floatingips = floatingip_query.filter(
            FloatingIP.fixed_port_id == port_id,
            FloatingIP.floating_network_id == network_id
        ).all()
        if floatingips:
            LOG.info(_LI("[AC] %(port)s already has public service "
                         "floatingips: %(floatingip)s"),
                     {'port': port_id, 'floatingip': floatingips})
            return True
        return None

    def create_fip(self, context):
        """Create 100.64 EIP for the public service
        :param context: DB context for the port creation
        :return: None
        """
        try:
            ac_osprofiler.record_call_chain("begin create public service fip")
            port = context.current
            ori_context = new_context.get_admin_context()
            fip_networks = self.get_public_service_networks(ori_context, port)
            if not fip_networks:
                # 100.64 network get failed, no deal
                LOG.debug(_LI("[AC] public service network does not exist "
                              "for port %s in creating fip."), port['id'])
                ac_osprofiler.record_call_chain(
                    "public service network don‘t exist,no need to create fip")
                return

            LOG.info(_LI("[AC] public service process create port: %s."), port)

            is_baremetal_port = (port['binding:vnic_type'] == "baremetal") or \
                                port["device_owner"].startswith("baremetal:")
            if not (ACUtil.is_compute_port(port) or is_baremetal_port):
                # Non-vm port or vlan network does not handle
                LOG.debug(_LI("[AC] port %s is not vm port in creating fip,"
                              " no deal."), port['id'])
                ac_osprofiler.record_call_chain("it is not vm or bare metal "
                                                "port no need to create fip")
                return

            if not port.get('fixed_ips'):
                LOG.info(_LI("[AC] No fixed ips in port %s"), port['id'])
                ac_osprofiler.record_call_chain("port no ip no need to create "
                                                "fip")
                return

            if self.check_duplicate_pubservice(context._plugin_context.session,
                                               context.current['id'],
                                               fip_networks[0]['id']):
                ac_osprofiler.record_call_chain("port already has fip"
                                                " no need to create fip")
                return

            pub_service_fip = {'floatingip': {
                'subnet_id': None, 'tenant_id': port['tenant_id'],
                'floating_network_id': fip_networks[0]['id'],
                'fixed_ip_address': None, 'floating_ip_address': None,
                'port_id': port['id']}}
            l3plugin = ncu.get_service_plugin()['L3_ROUTER_NAT']
            if l3plugin:
                LOG.debug(_LI("[AC]public service create 100.64 fip for port "
                              "%s."), port['id'])
                l3plugin.create_floatingip(ori_context, pub_service_fip)
        except Exception as ex:
            LOG.error(_LE(
                "[AC]public service process create port: %s failed. raise "
                "exception to plugin to rollback."), context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            ac_osprofiler.record_chain_exception("create public service fip "
                                                 "fail")
            raise ml2_exc.MechanismDriverError()

    def delete_fip(self, context, by_update=False):
        """ Delete 100.64 EIP for the public service
        :param context: DB context for the port delete
        :return: None
        """
        try:
            ac_osprofiler.record_call_chain("delete public service fip")
            if not by_update:
                port = context.current
            else:
                port = context.original
            ori_context = context._plugin_context
            fip_networks = self.get_public_service_networks(ori_context, port)
            if not fip_networks:
                # 100.64 network get failed
                LOG.debug(_LI("[AC] public service network does not exist "
                              "for port %s in deleting fip."), port['id'])
                ac_osprofiler.record_call_chain("no public service fip to "
                                                "delete")
                return

            LOG.info(_LI("[AC]Public service to delete port: %s."),
                     port)

            l3plugin = ncu.get_service_plugin()['L3_ROUTER_NAT']
            if l3plugin:
                filters = {'port_id': [port['id']],
                           'floating_network_id': [fip_networks[0]['id']]}
                fields = ['id']
                fips = l3plugin.get_floatingips(ori_context, filters, fields)
                if fips:
                    LOG.debug(_LI("[AC] public service delete 100.64 fip "
                                  "for port %s."), port['id'])
                    l3plugin.delete_floatingip(ori_context, fips[0]['id'])
        except Exception as ex:
            LOG.error(_LE(
                "[AC]Public service delete port: %s failed. raise exception"
                " to plugin to rollback."), context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            ac_osprofiler.record_chain_exception("delete fip fail")
            raise ml2_exc.MechanismDriverError()

    def get_public_service_networks(self, context, port):
        """get public service networks"""
        LOG.info(_LI('[AC] Begin to get public service network for %s'), port)
        if not port.get('fixed_ips'):
            LOG.error(_LE('[AC] %s has no fixed ips'), port.get('id'))
            return None
        subnet_id = ACCommonUtil. \
            get_port_first_ipv4_subnet(port.get('fixed_ips'))
        interface_query = context.session.query(models_v2.Port)
        interface_query = interface_query.join(models_v2.IPAllocation)
        router_interface = interface_query.filter(
            models_v2.IPAllocation.subnet_id == subnet_id,
            models_v2.Port.device_owner == ncu.DEVICE_OWNER_ROUTER_INTF
        ).first()
        if not router_interface:
            LOG.error(_LE('[AC] No interface found for subnet %s'), subnet_id)
            return None
        device_id = router_interface.device_id
        router_gateway_query = context.session.query(models_v2.Port)
        router_gateway_query = router_gateway_query.join(RouterPort)
        router_gateway = router_gateway_query.filter(
            RouterPort.router_id == device_id,
            RouterPort.port_type == ncu.DEVICE_OWNER_ROUTER_GW
        ).first()
        if not router_gateway:
            return self.get_default_public_service_networks(
                '[AC]No gateway found for router %s' % device_id)
        external_network_id = router_gateway.network_id
        external_network = self._core_plugin.get_network(
            self.admin_context, external_network_id)
        if not external_network:
            return self.get_default_public_service_networks(
                '[AC]No external network found for id %s' % external_network_id)
        ex_network_name = external_network.get('name', '')
        for elem in self.public_service_networks:
            if not ex_network_name.endswith(elem):
                continue
            filters = {'name': [elem], EXTERNAL: [True]}
            LOG.debug("[AC] get networks context: %s", context.__dict__)
            public_networks = self._core_plugin.get_networks(
                self.admin_context, filters=filters, fields=['id'])
            LOG.info(_LI('[AC]Public service network found:%s'),
                     public_networks)
            if not public_networks or len(public_networks) > 1:
                return self.get_default_public_service_networks(
                    '[AC]No public service network found for name %s' % elem)
            net = self._core_plugin._get_network(self.admin_context,
                                                 public_networks[0]['id'])
            if any(s.ip_version == 4 for s in net.subnets):
                return [{'id': net.id}]
            return None
        return self.get_default_public_service_networks(None)

    def get_default_public_service_networks(self, message):
        """get default public service networks"""
        if message:
            LOG.error(_LE(message))
        if not self.default_pub_ser_net:
            return None
        filters = {
            'name': [self.default_pub_ser_net],
            EXTERNAL: [True]
        }
        fields = ['id']
        fip_networks = self._core_plugin.get_networks(
            self.admin_context, filters=filters, fields=fields)
        if not fip_networks or len(fip_networks) > 1:
            return None
        net = self._core_plugin._get_network(self.admin_context,
                                             fip_networks[0]['id'])
        if any(s.ip_version == 4 for s in net.subnets):
            result = [{'id': net.id}]
            LOG.info(_LI('[AC]Get default public service networks: %s'), result)
            return result
        return None
