#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""ac qos extension driver"""

import ast

from neutron.db import securitygroups_db
from oslo_log import log as logging
from oslo_serialization import jsonutils

from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei.drivers.ac.model.port_model import ACPortModel
from networking_huawei.drivers.ac.sync.message_reliability_api import ACReliabilityAPI

LOG = logging.getLogger(__name__)


class HuaweiQosServiceNotificationExtensionDriver(object):
    """Huawei AC service notification Extension driver for QoS."""

    def __init__(self):
        LOG.info(_LI("[AC] Init Huawei QoS Extension driver."))
        self.ac_driver = ACReliabilityAPI(ac_const.NW_HW_CMCC_QOS)
        self.dbif = ACdbInterface()
        LOG.info(_LI("[AC] Initialization finished successfully "
                     "for Huawei QoS Extension driver."))

    def get_description(self):
        """get description"""
        return "Notify Huawei AC to create/update/delete/query/list/" \
               "binding/unbinding/update binding QoS policy."

    @classmethod
    def _ac_model_format(cls, policy_db):
        try:
            name = 'qos_' + (policy_db['id']).replace('-', '')
            policy_info = {
                "uuid": policy_db['id'],
                "name": name[0:25],
                "tenant-id": policy_db['tenant_id'],
                "cloud-name": ac_const.CLOUD_NAME_PARAM
            }
            rules = policy_info['qos-bandwidth-limit-rules'] = {}
            rules['qos-bandwidth-limit-rule'] = []
            rate_limit = policy_db['policies']['ratelimit']
            rules['qos-bandwidth-limit-rule'].append({
                "uuid": policy_db['id'],
                "qos-policy-id": policy_db['id'],
                "max-kbps": rate_limit,
                "max-burst-kbps": rate_limit
            })
        except KeyError as value:
            LOG.error(_LE("[AC] Construct QoS policy info error, "
                          "doesn't contain all fields : %s."), value)
            raise
        return policy_info

    @classmethod
    def _get_port_info_a(cls, port, port_info):
        if port.get('created_at'):
            port_info['created-at'] = port['created_at']
        if port.get('updated_at'):
            port_info['updated-at'] = port['updated_at']

        if 'binding:host_id' in port \
                and port['binding:host_id'] is not None:
            port_info['host-id'] = port['binding:host_id']
        if 'mac_address' in port \
                and port['mac_address'] is not None:
            port_info['mac-address'] = port['mac_address']
        if 'admin_state_up' in port \
                and port['admin_state_up'] is not None:
            port_info['admin-state-up'] = port['admin_state_up']
        return port_info

    @classmethod
    def _get_port_info_device(cls, port, port_info):
        if 'device_owner' in port \
                and port['device_owner'] is not None:
            port_info['device-owner'] = port['device_owner']
        if 'device_id' in port \
                and port['device_id'] is not None:
            port_info['device-id'] = port['device_id']
        if 'binding:profile' in port \
                and port['binding:profile'] is not None:
            json_data = (jsonutils.dumps(port['binding:profile'])). \
                replace('true', 'True').replace('false', 'False')
            port_info['profile'] = str(ast.literal_eval(json_data))
        return port_info

    def _get_port_info_binding(self, port, port_info):
        if 'binding:vnic_type' in port \
                and port['binding:vnic_type'] is not None:
            port_info['vnic-type'] = port['binding:vnic_type']
        else:
            session = self.dbif.get_session()
            rec = ncu.get_port_binding(session, port["id"])
            if rec:
                port_info['vnic-type'] = rec['vnic_type']
        if 'binding:vif_type' in port \
                and port['binding:vif_type'] is not None:
            port_info['vif-type'] = port['binding:vif_type']
        else:
            session = self.dbif.get_session()
            rec = ncu.get_port_binding(session, port["id"])
            if rec:
                port_info['vif-type'] = rec['vif_type']
        return port_info

    def _get_port_info_sg(self, port, port_info):
        if 'security_groups' in port \
                and port['security_groups'] is not None:
            port_info['security-groups'] = \
                port['security_groups']
        else:
            session = self.dbif.get_session()
            rec_list = session. \
                query(securitygroups_db.SecurityGroupPortBinding). \
                filter_by(port_id=port["id"]).all()
            if rec_list:
                port_info['security-groups'] = []
                for rec in rec_list:
                    port_info['security-groups']. \
                        append(rec['security_group_id'])
        return port_info

    @classmethod
    def _get_port_info_ip(cls, port, port_info):
        if 'fixed_ips' in port \
                and port['fixed_ips'] is not None:
            fixed_ips = []
            for item in port['fixed_ips']:
                fixed_ip = {}
                fixed_ip['subnet-id'] = item['subnet_id']
                fixed_ip['ip-address'] = item['ip_address']
                fixed_ips.append(fixed_ip)
            port_info['fixed-ips'] = fixed_ips
        if 'allowed_address_pairs' in port \
                and port['allowed_address_pairs'] is not None:
            allowedaddresspairs = []
            for item in port['allowed_address_pairs']:
                allowedaddresspair = {}
                allowedaddresspair['mac-address'] = item[
                    'mac_address']
                allowedaddresspair['ip-address'] = item['ip_address']
                allowedaddresspairs.append(allowedaddresspair)
            port_info['allowed-address-pairs'] = allowedaddresspairs
        return port_info

    def _make_port_body(self, port, qos_id):
        """Convert data to AC model"""
        try:
            port_info = {'uuid': port['id'],
                         'name': port['name'],
                         'network-id': port['network_id'],
                         'tenant-id': port['tenant_id']}

            port_info = self._get_port_info_a(port, port_info)
            port_info = self._get_port_info_device(port, port_info)
            port_info = self._get_port_info_binding(port, port_info)
            port_info = self._get_port_info_sg(port, port_info)
            if qos_id:
                port_info['qos-policy-id'] = qos_id
            port_info = ACPortModel.get_port_vif_details(port, port_info)
            port_info = self._get_port_info_ip(port, port_info)
            port_info = ACPortModel.get_port_extra_dhcp_opts(port, port_info)
            port_info = ACPortModel.get_port_trunk_details(port, port_info)

        except KeyError as ex:
            LOG.error(_LE("[AC]Key Error, doesn't contain all fields %s."), ex)
            raise

        return port_info

    def create_qos(self, context, policy):
        """create qos"""
        LOG.info(_LI("[AC] Begin to create QoS policy: %s."), policy)
        try:
            # 1. convert model
            policy_info = self._ac_model_format(policy)
            LOG.info(_LI("policy info is %s"), policy_info)
            self.ac_driver.update_plugin_record(
                context,
                policy_info.get('uuid'),
                policy_info,
                ac_const.NW_HW_CREATE_QOS_POLICY)
        except Exception as except_msg:
            LOG.error(_LE("[AC] Failed to create QoS policy in huawei driver: "
                          "%s."), except_msg)
            raise
        LOG.info(_LI("[AC] Huawei AC create QoS policy successfully."))

    def binding_qos(self, context, port_id, qos_id):
        """the function is to bind qos to vmm port or EIP port"""
        LOG.info(_LI("[AC]Binding QoS policy: %s in port: %s,context=%s."),
                 qos_id, port_id, context)

    def unbinding_qos(self, context, port_id):
        """the function is to unbind qos from vmm port or eip port"""
        LOG.info(_LI("[AC] Unbinding QoS policy in port: %s."), port_id)

    def delete_qos(self, context, policy_id):
        """delete qos"""
        LOG.info(_LI("[AC] Begin to delete QoS policy: %s."), policy_id)

        try:
            self.ac_driver.update_plugin_record(
                context,
                policy_id,
                {},
                ac_const.NW_HW_DELETE_QOS_POLICY)
        except Exception as except_msg:
            LOG.error(_LE("[AC] Failed to delete QoS policy in huawei driver: "
                          "%s."), except_msg)
            raise

        LOG.info(_LI("[AC] Huawei AC delete QoS policy successfully."))


def check_init_cmcc_huawei_qos_driver():
    """check init cmcc huawei qos driver"""
    try:
        from neutron.db import qos_rpc_base as qos_db_rpc
        if hasattr(qos_db_rpc.QoSServerRpcMixin, "huawei_qos"):
            # if config cmss qos plugin, need to init
            # cmcc huawei qos driver
            HuaweiQosServiceNotificationExtensionDriver()
    except Exception as ex:
        LOG.error('check_init_cmcc_huawei_qos_driver occur error:%s', ex)
    return
