#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""failed resources sync"""

import threading
import time
from oslo_config import cfg
from oslo_db import api as oslo_db_api
try:
    from neutron.common.exceptions import PortNotFound
except ImportError:
    from neutron_lib.exceptions import PortNotFound

from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.drivers.ac.db import dbif
from networking_huawei.drivers.ac.db import schema as dbschema
from networking_huawei.drivers.ac.sync import worker
from networking_huawei.drivers.ac.sync import http_heart_beat
from networking_huawei.drivers.ac.client.formatter import \
    ACNeutronDataFormatter
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI

LOG = ncu.ac_log.getLogger(__name__)
THREAD_EVENT = threading.Event()
FIRST_DELAY_MINITES = 2
SECOND_DELAY_MINITES = 5
DELETE_DELAY_SECONDS = 20
FAILD_RESOURCE_THREAD_POOL_COUNT = 3


class ACFailedResourceSync(object):
    """Failed Resource Sync"""

    def __init__(self):
        self._db_if = dbif.ACdbInterface()
        self.reliability_api = ACReliabilityAPI(ac_const.NW_HW_ML2)
        self.neutron_formatter = ACNeutronDataFormatter()
        self.http_heart_beat = http_heart_beat.HttpHeatBeat2AC()
        self._sync_thread_spawn_state = ac_const.SYNC_THREAD_POOL_NO_SPAWN
        self._thread_pool = worker.ACSyncThreadPool(
            FAILD_RESOURCE_THREAD_POOL_COUNT)
        self._thread_pool.spawn_threads(
            FAILD_RESOURCE_THREAD_POOL_COUNT)
        timer_resource_sync = threading.Timer(
            0, self.failed_resource_sync_process)
        timer_resource_sync.start()
        LOG.info("Failed resource sync thread initialization done.")

    def failed_resource_sync_process(self):
        """Basic processing of the failed_resource_sync thread.

        :return: None
        """
        try:
            if not self.http_heart_beat.is_ac_alive():
                LOG.info("Controller is not reached, Not sync Failed "
                         "resources to Controller.")
                return
            self._handle_failed_resources()
        except Exception as ex:
            LOG.error("[AC] Handle Failed resources with exception: %s", ex)
        finally:
            timer = threading.Timer(ac_const.FAILED_RESOURCE_SYNC_THREAD_WAIT,
                                    self.failed_resource_sync_process)
            timer.start()

    @staticmethod
    def _check_invalid_failed_resource(res_fail):
        """Check the failed resource need to be delete.

        :param res_fail: Failed resource accord
        :return: Bool
        """
        failed_time = res_fail.get("failed_time")
        failed_time_sec = int(time.mktime(failed_time.timetuple()))
        time_diff = int(time.time()) - failed_time_sec
        # third_retry_time (the last retry time), equal to FIRST_DELAY_MINITES
        # and SECOND_DELAY_MINITES plus three times of error_retry_interval.
        third_retry_time = cfg.CONF.huawei_ac_config.error_retry_interval*3 \
                           + (FIRST_DELAY_MINITES + SECOND_DELAY_MINITES)*60
        # delete failed resource after third_retry_time plus ac_response_time
        # and DELETE_DELAY_SECONDS
        timeout_sec = cfg.CONF.huawei_ac_config.ac_response_time \
                      + third_retry_time + DELETE_DELAY_SECONDS
        return res_fail.get("retry_count") == 0 and time_diff >= timeout_sec

    @staticmethod
    def _need_handle_failed_resource(res_fail):
        """Check the failed resource need to be handled now or not.

        :param res_fail: Failed resource accord
        :return: Bool
        """
        if res_fail.get("retry_count") == 0:
            return False
        failed_time = res_fail.get("failed_time")
        failed_time_sec = int(time.mktime(failed_time.timetuple()))
        time_diff = int(time.time()) - failed_time_sec
        error_retry_times = abs(cfg.CONF.huawei_ac_config.error_retry_count -
                                res_fail.get("retry_count")) + 1
        first_retry_time = cfg.CONF.huawei_ac_config.error_retry_interval
        second_retry_time = first_retry_time + \
                            FIRST_DELAY_MINITES*ac_const.SECONDS_EVRY_MINUTE
        third_retry_time = second_retry_time + \
                           SECOND_DELAY_MINITES*ac_const.SECONDS_EVRY_MINUTE
        return (error_retry_times == 1 and time_diff >= first_retry_time) or \
               (error_retry_times == 2 and time_diff >= second_retry_time) or \
               (error_retry_times == 3 and time_diff >= third_retry_time)

    def _send_data_to_ac(self, thread_name, entry_info, operation):
        LOG.debug("%s: Send data to AC initialized.", thread_name)
        context = ncu.neutron_context.get_admin_context()
        self.reliability_api.update_plugin_record(context,
                                                  entry_info['uuid'],
                                                  entry_info,
                                                  operation)

    def _check_and_join_or_spawn_threads(self, failed_resources):
        """check and join or spawn threads.

        :param failed_resources:  failed resources
        :return: None
        """
        if not failed_resources:
            if self._sync_thread_spawn_state != \
                    ac_const.SYNC_THREAD_POOL_NO_SPAWN:
                self._sync_thread_spawn_state += 1
                LOG.debug("New failed resources sync thread spawn counter "
                          "is %d.", self._sync_thread_spawn_state)
                if self._sync_thread_spawn_state == \
                        ac_const.SYNC_THREAD_POOL_MAX_TRY:
                    self._sync_thread_spawn_state = \
                        ac_const.SYNC_THREAD_POOL_NO_SPAWN
                    LOG.debug("Joining failed resources sync threads after "
                              "job finished.")
                    self._thread_pool.clear_task()
        else:
            if self._sync_thread_spawn_state == \
                    ac_const.SYNC_THREAD_POOL_NO_SPAWN:
                LOG.debug("Spawning sync threads for the new job.")
                self._thread_pool.spawn_threads(
                    FAILD_RESOURCE_THREAD_POOL_COUNT)
            self._sync_thread_spawn_state = 1

    def _handle_failed_resources(self):
        """Handle failed resource.

        :return: None
        """
        admin_ctx = ncu.neutron_context.get_admin_context()
        failed_resources = self.get_failed_resources(admin_ctx.session)
        for res_fail in failed_resources:
            if self._check_invalid_failed_resource(res_fail):
                self._db_if.delete_failed_resource(res_fail.get("id"),
                                                   admin_ctx.session)
                continue
            if not self._need_handle_failed_resource(res_fail):
                continue
            LOG.debug("[AC] Begin to deal with failed resources: %s",
                      res_fail)
            self.reduce_count_failed_resource(res_fail.get("id"),
                                              admin_ctx.session)
            try:
                entry_info = self.neutron_formatter.set_port_data(
                    admin_ctx, "", res_fail.get("id"))
            except PortNotFound:
                LOG.error("[AC] Not found failed resources: %s", res_fail)
                self._db_if.delete_failed_resource(res_fail.get("id"),
                                                   admin_ctx.session)
                continue

            rec = self._db_if.get_port_attribute(session=admin_ctx.session,
                                                 res_id=res_fail.get("id"))
            if rec:
                entry_info['updated-at'] = rec.update_time.\
                    strftime(ac_const.ISO8601_TIME_FORMAT)
                entry_info['created-at'] = rec.create_time.\
                    strftime(ac_const.ISO8601_TIME_FORMAT)
            else:
                updated_at = ACCommonUtil.get_standard_current_time()
                entry_info['updated-at'] = updated_at
                entry_info['created-at'] = updated_at
            if ncu.IS_FSP \
                    and res_fail.get("operation") == ac_const.OPER_CREATE \
                    and entry_info.get("trunk-details") \
                    and entry_info.get("host-id"):
                entry_info['host-id'] = ""

            LOG.debug("[AC] Begin to send failed resource: %s to AC",
                      entry_info)

            while True:
                if not self._thread_pool.task_queue.full():
                    # Need the correct state in the worker before in process
                    self._thread_pool.add_task(
                        self._send_data_to_ac, entry_info,
                        res_fail.get("operation"))
                    break
                THREAD_EVENT.wait(.1)

    def get_failed_resources(self, session=None):
        """Get failed resource record.

        :param session: Current session info where the operation is
                        performed. Used to access db.
        :return: Failed resources
        """

        if not session:
            session = self._db_if.get_session('read')
        return session.query(dbschema.ACFailedResources).all()

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES,
                               retry_interval=0,
                               inc_retry_interval=0,
                               max_retry_interval=0)
    def reduce_count_failed_resource(self, res_id, session=None):
        """Reduce count of failed resource.

        :param res_id: Id of failed resource
        :param session: Current session info where the operation is
                        performed. Used to access db.
        :return: None
        """

        if not session:
            session = self._db_if.get_session('write')
        with session.begin(subtransactions=True):
            failed_resource = session.query(dbschema.ACFailedResources).\
                filter_by(id=res_id).first()
            if failed_resource and failed_resource.retry_count >= 1:
                failed_resource.retry_count = failed_resource.retry_count - 1
