#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""neutron sync tool"""

import ast
import codecs
import copy
import os
import stat
import threading
import time
from datetime import datetime, timedelta

import eventlet
import six
from oslo_config import cfg
from oslo_db import exception as db_exc
from oslo_serialization import jsonutils
from six.moves import filter
from six.moves import range

from networking_huawei._i18n import _LI, _LE, _LW
from networking_huawei.common import constants
from networking_huawei.common.exceptions import ContinueTimeoutException, MechanismDriverError
from networking_huawei.common.exceptions import NeutronSyncError
from networking_huawei.drivers.ac.client import restclient as rest
from networking_huawei.drivers.ac.client.formatter import ACNeutronDataFormatter
from networking_huawei.drivers.ac.client.service import ACRestUtils, ACReSTService
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import fusion_sphere_alarm as fsa
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.neutron_compatible_util import ac_ml2_api as api
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.drivers.ac.db import dbif
from networking_huawei.drivers.ac.db.compare_result.compare_results_db import CompareDbMixin
from networking_huawei.drivers.ac.db.schema import ACNeutronStateSchema, ACPluginSchema
from networking_huawei.drivers.ac.db.sync_result.sync_results_db import SyncDbMixin
from networking_huawei.drivers.ac.extensions.dnat import dnat as dnat_extension
from networking_huawei.drivers.ac.external.ext_if import ACKeyStoneIf
from networking_huawei.drivers.ac.sync import util
from networking_huawei.drivers.ac.sync import worker
from networking_huawei.drivers.ac.sync.neutron_sync_router_util import NeutronSyncRouterUtil
from networking_huawei.drivers.ac.sync.neutron_sync_util import ACNeutronSyncUtil

try:
    from oslo_service import loopingcall
except ImportError:
    from neutron.openstack.common import loopingcall
try:
    from neutron import context
except ImportError:
    from neutron_lib import context

requests = eventlet.import_patched('requests.__init__')
LOG = ncu.ac_log.getLogger(__name__)

try:
    if six.PY3:
        import csv as ucsv
    else:
        import unicodecsv as ucsv
except ImportError:
    import csv as ucsv

    LOG.error("[AC]python-unicodecsv is needed, please install it.")


class ACNeutronSync(ACNeutronSyncUtil, NeutronSyncRouterUtil):
    """ AC neutron sync tool """

    def __init__(self, sync_param=None):
        super(ACNeutronSyncUtil, self).__init__()
        if sync_param is None:
            sync_param = ac_const.NEUTRON_SYNC_PARAM_DEFAULT

        if cfg.CONF.huawei_ac_config.neutron_sync_type == \
                ac_const.NEUTRON_SYNC_NONE and not sync_param.get("force_trigger"):
            LOG.info(_LI('Skip initialization of neutron sync, '
                         'as current neutron server '
                         'does not configured for neutron sync.'))
            return
        self._db_if = dbif.ACdbInterface()
        if sync_param.get("sync_segments"):
            self._db_if.handle_segments_records()
            return
        self._init_neutron_sync_config()
        self._ac_rest_service = ACReSTService()
        self._is_looping = sync_param.get("looping")
        self._sync_op = sync_param.get("sync_op")
        self._compare_result = sync_param.get("compare_data")
        self._need_delete = sync_param.get("need_delete")
        self._include_bindports = sync_param.get("include_bindports")
        self._rest = rest.ACReSTClient()
        self._lock = threading.Lock()
        self._thread_event = threading.Event()
        self._util = util.ACUtil()
        self.neutron_formatter = ACNeutronDataFormatter()
        self._neutron_sync_status = ac_const.NW_HW_NEUTRON_SYNC_INIT
        self._thread_pool = worker.ACSyncThreadPool(ac_const.SYNC_THREAD_POOL_DEFAULT_COUNT)
        self.timer = loopingcall.FixedIntervalLoopingCall(self.perform_neutron_sync)
        self.start(sync_param.get("seconds"))
        self.neutron_name = ACCommonUtil.get_neutron_server_name()
        self._resource_list = copy.deepcopy(ac_const.RESOURCE_LIST)
        self.help_init(sync_param.get("sync_resource_type"))
        self._start_time = time.time()
        self.compare_db = CompareDbMixin()
        self.sync_db = SyncDbMixin()
        self._sync_id = self._get_sync_id(sync_param)
        if self._sync_op == ac_const.SYNC_OP_SYNC_SINGLE_INSTANCE or self._sync_op == ac_const.SYNC_OP_SYNC_DATA:
            self.sync_resources = ast.literal_eval(sync_param.get("sync_resources"))
        else:
            self.sync_resources = dict()
        if 'params' not in self.sync_resources:
            return

    def _init_neutron_sync_config(self):
        self._data_list = []
        self._create_data_list = []
        self._update_data_list = []
        self._delete_data_list = []
        self._update_router_list = []
        self._create_router_list = []
        self._create_after_update_type_list = [ac_const.NW_HW_TAP_FLOW]
        self._create_after_update_list = []
        self._delete_data_records = []
        self._neutron_sync_result = {
            ac_const.OPER_CREATE: {},
            ac_const.OPER_UPDATE: {},
            ac_const.OPER_DELETE: {}
        }
        # port resource with id:update_time for comparison
        self._port_attr_dict = {}
        # other resources with id:update_time for comparison
        self._res_attr_dict = {}
        # compare-data related cache for time-saving
        self.cache_network_port_attr_dict = {}
        self.cache_res_attr_dict = {}
        self._tenant_list = []
        self._continuous_timeout = 0
        self._total_records_transfer = 0
        self._white_list = {}
        # set a switch to check if making a report
        self._enable_make_report = True
        self.sync_wait_flag = False
        self._continuous_timeout_type_times = 0
        self._continuous_timeout_type = ''
        self._continuous_sync_next_type_flag = False
        self._continuous_timeout_res_list = []
        self._processed_rec_count = 0
        self.single_res_ids = []
        self.single_create_lists = []
        self.single_update_lists = []
        self.single_delete_lists = []
        self.quota_task_list = []
        self._ops_ver = ncu.get_ops_version()

    def help_init(self, sync_resource_type):
        """ help ACNeutronSync init """
        host_list = cfg.CONF.huawei_ac_agent_config.rpc_server_ip. \
            replace(' ', '').split(',')
        host_list = [x.lower() for x in host_list]
        if len(host_list) == 1:
            self._url = '%s%s%s%s' % (
                ac_const.HTTPS_HEADER, host_list[0], ":",
                str(ac_const.rest_server_port))
        else:
            self._url = '%s%s%s%s' % (
                ac_const.HTTPS_HEADER, ac_const.DEFAULT_AC_IP, ":",
                str(ac_const.rest_server_port))
        if sync_resource_type:
            sync_resource_type_list = []
            for sync_resource_type_single in sync_resource_type:
                if sync_resource_type_single == \
                        ac_const.NW_HW_SECURITY_GROUP_RULE:
                    sync_resource_type_list.append(ac_const.NW_HW_SEC_GRP_RULE)
                else:
                    sync_resource_type_list.append(sync_resource_type_single)
            for res_type_tmp in ac_const.RESOURCE_LIST:
                if res_type_tmp not in sync_resource_type_list:
                    self._resource_list.pop(res_type_tmp, None)
        self.enable_security_group = \
            cfg.CONF.huawei_ac_config.enable_security_group
        if not self.enable_security_group:
            self._resource_list.pop(ac_const.NW_HW_SEC_GRP, None)
            self._resource_list.pop(ac_const.NW_HW_SEC_GRP_RULE, None)
        if 'group_policy' in cfg.CONF and \
                'policy_drivers' in cfg.CONF.group_policy and \
                ac_const.NW_HW_GBP_EXTERNAL \
                not in cfg.CONF.group_policy.policy_drivers:
            self._resource_list.pop(ac_const.NW_HW_EXTERNAL_SEGMENT, None)
            self._resource_list.pop(ac_const.NW_HW_EXTERNAL_POLICY, None)

    def start(self, seconds=None):
        """ Schedule the neutron sync for the provided time interval. """
        if seconds:
            LOG.info(_LI("Scheduling non looping neutron-sync for an initial "
                         "delay of %d seconds."), seconds)
            interval_time = util.ACUtil.find_neutron_sync_interval()
            self.timer.start(interval_time, initial_delay=seconds)
        elif self._is_looping:
            start_delay, interval_time = \
                util.ACUtil.calculate_neutron_sync_interval()
            LOG.info(_LI("Scheduling neutron-sync for an initial delay of %d "
                         "seconds and loop interval of %d seconds."),
                     start_delay, interval_time)
            if ncu.get_ops_version() in [ac_const.OPS_K]:
                self.timer.start(interval_time, initial_delay=start_delay)
            else:
                self.timer.start(interval_time, initial_delay=start_delay,
                                 stop_on_exception=False)

    def perform_neutron_sync(self):
        """Function responsible to take care of the end to end processing of the
        neutron sync operation. Will be triggered by the neutron sync timer.
        """
        LOG.info(_LI("Start the neutron sync processing."))
        self._neutron_sync_status = ac_const.NW_HW_STOP_PROCESSING
        self._total_records_transfer = 0
        self._start_time = time.time()

        session = self._db_if.get_session('write')

        if not self._is_looping:
            LOG.debug("Stopping the loop timer as caller configured as no loop timer.")
            self.timer.stop()
        elif not self._db_if.check_is_leader(session):
            LOG.info(_LI('Skipping neutron sync as it is not leader server.'))
            return
        self._populate_res_list()
        try:
            self._neutron_sync()
        finally:
            self.neutron_formatter.cleanup()
            self._rollback_neutron_sync(session)
            LOG.info('Step 16. Neutron sync process finished')

    def _neutron_sync(self):
        session = self._db_if.get_session('write')
        admin_context = context.get_admin_context()
        try:
            LOG.info("sync op is: %s, sync id is: %s", self._sync_op, self._sync_id)
            self.clear_sync_result(admin_context)
            LOG.info("Start to sync neutron info to ac:%s", self.sync_resources)
            if cfg.CONF.huawei_ac_config.neutron_sync_enable_white_list:
                self._populate_white_list()
            # 1. Check if any other neutron server is performing the neutron sync
            LOG.info('Step 1. Check neutron sync state.')
            (status, server) = self._db_if.check_is_neutron_sync_in_progress(session)
            if status:
                LOG.info(_LI("Neutron sync by neutron server %s failed, "
                             "already in process on neutron server %s."),
                         self.neutron_name, server.neutron_name)
                raise NeutronSyncError(reason='Neutron sync already in-progress')

            # 2. Set the server record as NEUTRON_SYNC so that no one else
            #    will perform the neutron sync at this time.
            LOG.info('Step 2. Update neutron sync state as NEUTRON_SYNC.')
            update_state = ac_const.CONSISTENCY_CHECK \
                if self._sync_op == ac_const.SYNC_OP_COMPARE_DATA \
                else ac_const.NEUTRON_SYNC
            self._db_if.update_server_record(ACNeutronStateSchema(
                state=update_state, sync_res=''), session=session)
            self._neutron_sync_status = ac_const.NW_HW_NEUTRON_SYNC_START

            # 3. Updating the tenant list from keystone
            LOG.info('Step 3. Get tenant list from keystone.')
            self._tenant_list = ACKeyStoneIf.get_tenant_list_from_keystone()
            self._update_sync_start_time()

            # 4-12. CREATE order and UPDATE order are same, so create will handle for update also.
            self._make_system_consistent(session)
            if self._sync_op == ac_const.SYNC_OP_COMPARE_DATA:
                if self._enable_make_report:
                    LOG.info('Step Bonus. Make a report in SYNC_OP_COMPARE_DATA')
                    self._make_consistency_report()
                LOG.info(_LI('Not syncing the inconsistent data if present and terminating neutron-sync'))
                return
            # 13-15. Recover inconsistent data
            self._recover_inconsistent_data(session)
        except Exception as ex:
            self._handle_neutron_sync_except(admin_context, session, ex)

    def _record_neutron_sync_result(self, record_list, sync_result_records,
                                    total_count_info, start_time=None):
        """ record neutron sync result """
        try:
            total_num, total_success = total_count_info
            start_t = start_time if start_time else \
                self._neutron_sync_result['start_time']
            timestamp = time.strftime("%Y%m%d_%H%M%S", time.localtime(start_t))
            sync_result_file = ac_const.NEUTRON_SYNC_RESULT_FILE % timestamp
            with codecs.open(sync_result_file, 'w', 'utf-8') as result_file:
                _msg = "\n==========================================\n"
                record_list.append(_msg)
                sync_stat = \
                    "Neutron sync begin at %s, end at %s. " \
                    "Total time: %s\nTotal create resources: " \
                    "%s\nTotal update resources: %s\nTotal delete " \
                    "resources: %s\nTotal sync resources: %s, " \
                    "success %s, fail %s" % (
                        time.strftime(
                            ac_const.ISO8601_TIME_FORMAT,
                            time.localtime(start_t)),
                        time.strftime(ac_const.ISO8601_TIME_FORMAT),
                        time.time() - start_t,
                        sum(len(value) for key, value in
                            sync_result_records[
                                ac_const.OPER_CREATE].items()),
                        sum(len(value) for key, value in
                            sync_result_records[
                                ac_const.OPER_UPDATE].items()),
                        sum(len(value) for key, value in
                            sync_result_records[
                                ac_const.OPER_DELETE].items()),
                        total_num, total_success, total_num - total_success)
                LOG.info(_LI("%s%s%s" % (_msg, sync_stat, _msg)))
                sync_end_time = time.strftime(ac_const.ISO8601_TIME_FORMAT)
                try:
                    self._db_if.update_neutron_sync_record(
                        sync_end_time=sync_end_time,
                        session=context.get_admin_context().session)
                except AssertionError:
                    LOG.error(
                        _LI('[AC] Record neutron sync results to db failed.'))
                record_list.append(sync_stat)
                record_list.append(_msg)
                result_file.writelines(record_list)
            os.chmod(sync_result_file,
                     stat.S_IRUSR + stat.S_IWUSR + stat.S_IRGRP)
        except IOError as ex:
            LOG.error(_LE(str(ex)))

    def _recover_inconsistent_data(self, session):
        LOG.info('Step 13. Sync the create records.')
        self._sync_create_update_records(
            session, ac_const.OPER_CREATE,
            self._create_data_list)
        LOG.info('Step 14. Sync the update records.')
        self._sync_create_update_records(
            session, ac_const.OPER_UPDATE,
            self._update_data_list)
        if self._create_after_update_list:
            LOG.info('Step 15. Sync the create_after_update records.')
            self._sync_create_update_records(
                session, ac_const.OPER_CREATE,
                self._create_after_update_list)
        fsa.ACPluginAlarm.send_neutron_sync_recovery_alarm()

    def _sync_bindports_process(self, data_list):
        """ sync bind port process """
        LOG.debug('[AC] Begin to sync bindports process: %s', data_list)
        ml2_plugin = ncu.get_core_plugin()
        admin_ctx = context.get_admin_context()
        for data in data_list:
            if data['res'] != ac_const.NW_HW_BINDPORTS:
                continue
            port_info = ml2_plugin.get_port(admin_ctx, data['port_id'])
            network_info = ml2_plugin.get_network(admin_ctx, data['network_id'])
            if ACNeutronSync._bindport_filter(port_info, network_info):
                LOG.debug('the port does not need bindport: %s', data)
                continue
            if data['segmentation_id']:
                dynamic_segment = {
                    api.NETWORK_TYPE: 'vlan',
                    api.SEGMENTATION_ID: data['segmentation_id']
                }
                try:
                    self._set_bind_port_level(
                        data['network_id'], data['host_id'], dynamic_segment,
                        data['port_id'])
                except MechanismDriverError as ex:
                    LOG.error(_LE('[AC] Failed to bind port level: %s'),
                              str(ex))
            else:
                try:
                    ml2_plugin.update_port(
                        admin_ctx, data['port_id'], {'port': port_info})
                except Exception as ex:
                    LOG.error(_LE('[AC] Failed to update port: %s'), str(ex))

    def _sync_create_update_records(self, session, operation, data_list):
        """ sync create and upadte record """
        resource_list = []
        for data in data_list:
            if data['res'] not in resource_list:
                resource_list.append(data['res'])
        LOG.debug("Resource list: %s", resource_list)
        sorted_res_list = []
        for resource_type in ac_const.NEUTRON_SYNC_DEPENDENCY_ORDER:
            if resource_type in resource_list:
                sorted_res_list.append(resource_type)
        LOG.debug("sorted_res_list list: %s", sorted_res_list)

        for resource_type in list(sorted_res_list):
            self._continuous_sync_next_type_flag = False
            self._continuous_timeout_type_times = 0
            self._continuous_timeout_type = ''
            self._continuous_timeout_res_list = []
            res_list = list(filter(lambda res, res_type=resource_type:
                                   res['res'] == res_type, data_list))
            total_record = len(res_list)
            LOG.info('sync_create_update_records Sync the %s of %s '
                     'with record count %d , '
                     '_continuous_timeout_type_times ;%s ,'
                     ' _continuous_sync_next_type_flag : %s , '
                     '_continuous_timeout_type : %s ', operation, resource_type,
                     total_record, self._continuous_timeout_type_times,
                     self._continuous_sync_next_type_flag,
                     self._continuous_timeout_type)
            if total_record:
                self._db_if.update_server_record(
                    ACNeutronStateSchema(state=ac_const.NEUTRON_SYNC,
                                         sync_res=resource_type),
                    session=session)
                if resource_type == ac_const.NW_HW_PORTS:
                    res_list = ACNeutronSync._process_resource_type_hw_ports(
                        res_list, total_record)
                self._data_list = res_list
                if resource_type == ac_const.NW_HW_BINDPORTS:
                    if operation == ac_const.OPER_CREATE:
                        self._sync_bindports_process(data_list)
                else:
                    self.sync_data_process(operation, resource_type,
                                           session, sorted_res_list)
                LOG.info(
                    "sync_create_update_records, resource_type:%s is finish",
                    resource_type)

        # There is no need to block the normal sync during this period. So
        # update the sync_res to 'None'
        self._db_if.update_server_record(
            ACNeutronStateSchema(state=ac_const.NEUTRON_SYNC, sync_res=''),
            session=session)
        self._wait_sync_finished(session)

    def _wait_sync_finished(self, session):
        # Wait for the status response to be updated for the resources which
        #  are created or updated neutron-sync.
        ac_response_time = cfg.CONF.huawei_ac_config.ac_response_time + \
                           ac_const.NEUTRON_FINAL_RSP_CHECK_INTRVL
        for _ in range(0, ac_response_time,
                       ac_const.NEUTRON_FINAL_RSP_CHECK_INTRVL):
            res_list = self._db_if.get_plugin_record_list(
                session, ACPluginSchema(state=ac_const.NEUTRON_SYNC))
            if not res_list:
                return
            LOG.debug('Final response has not yet received for all resources, '
                      'waiting some more time.')
            time.sleep(ac_const.NEUTRON_FINAL_RSP_CHECK_INTRVL)

    def sync_data_process(self, operation, resource_type, session,
                          sorted_res_list):
        """ sync data process """
        if resource_type == ac_const.NW_HW_ROUTER_IF:
            self._sync_data_process(
                session, ac_const.NW_HW_PORTS, operation,
                sorted_res_list[:sorted_res_list.index(resource_type)])
        else:
            self._sync_data_process(
                session, resource_type, operation,
                sorted_res_list[:sorted_res_list.index(resource_type)])

    def _make_system_consistent(self, session):
        """
        Performs generate list of inconsitent data and sync data to be deleted
        from controller.
        """
        # 4. Consistency check
        LOG.info('Step 4. Performing consistency check for delete order.')
        inconsistency_list = self._consistency_check(session)
        original_list = copy.deepcopy(inconsistency_list)

        inconsistency_list.reverse()
        for inconsistent_res in inconsistency_list:
            res_type = inconsistent_res
            LOG.info(_LI("Performing neutron sync for resource %s."), res_type)

            # 5. Update the state for blocking current resource
            LOG.info('Step 5. Set state as CONSISTENCY_CHECK for resource %s.', inconsistent_res)
            self._db_if.update_server_record(
                ACNeutronStateSchema(state=ac_const.CONSISTENCY_CHECK,
                                     sync_res=inconsistent_res),
                session=session)

            # 6. Clear plugin records by deleting or making suspended
            LOG.info('Step 6. Clearing plugin records before sync.')
            self._clear_plugin_records_before_neutron_sync(session, res_type)

            # fix: 去掉ACUtil.get_current_neutron_sync_res 的校验可能引起的数据不一致
            # 7. Read the data from AC for the particular resource type
            LOG.info('Step 7. Get resource data from ac.')
            ac_data = self._get_resource_data_from_ac(res_type)

            # 8. Read the neutron db data for a particular record which
            #    was in the inconsistency list to make it consistent
            LOG.info('Step 8. Read and format neutron db data.')
            neutron_data = self._get_resource_data_from_neutron_db(res_type)

            # 9. Compare both neutron and ac data and insert records
            LOG.info('Step 9. Compare neutron db and ac, then generate difference requests.')
            total_record = self._compare_and_generate_diff_requests(
                session, neutron_data, ac_data, res_type)

            self._generate_sync_record(session)
            # Don't sync data if current sync operation is SYNC_OP_COMPARE_DATA
            if self._sync_op == ac_const.SYNC_OP_COMPARE_DATA:
                self._delete_data_list.extend(self._data_list)
                continue

            if total_record == 0:
                LOG.info(_LI('Data consistent for %s, continue with next '
                             'resource for delete order.'), res_type)
                continue

            # 10. Clear suspend state plugin records by deleting it
            LOG.info('Step 10. Clear suspended resources.')
            self._util.clear_dependent_suspended_records(session, res_type)

            # 11.Start the sync process by opening the lock
            LOG.info('Step 11. Open the lock to NEUTRON_SYNC.')
            self._db_if.update_server_record(
                ACNeutronStateSchema(state=ac_const.NEUTRON_SYNC),
                session=session)

            # 12.Send the data one by one to AC from plugin-db
            LOG.info('Step 12. Start the sync data process.')
            self._sync_data_process(
                session, res_type, ac_const.OPER_DELETE,
                inconsistency_list[:inconsistency_list.index(inconsistent_res)])
        return original_list

    def _generate_sync_record(self, session):
        neutron_sync_record = self._db_if.get_neutron_sync_record()
        sync_start_time = neutron_sync_record.sync_start_time if neutron_sync_record else datetime.utcnow()
        sync_task_count = len(self._data_list) + len(self._create_data_list) + len(self._update_data_list)
        sync_task_count += len(self._delete_data_list) if self._need_delete else 0
        expected_completion_time = sync_start_time
        if self._sync_op == ac_const.SYNC_OP_COMPARE_DATA:
            expected_completion_time += timedelta(seconds=sync_task_count)
        else:
            expected_completion_time += timedelta(seconds=sync_task_count) * 30
        self._db_if.update_neutron_sync_record(
            expected_completion_time, session=session)

    def _make_consistency_report(self):
        """ Performs generate list of inconsitent data and store in csv file """
        LOG.debug('[AC][RPT] Make consistency report')
        compare_result_file = self._compare_result
        if compare_result_file == ac_const.NEUTRON_SYNC_COMPARE_RESULT_FILE:
            compare_result_file = '%s_%s.csv' % (compare_result_file.split(".")[0],
                                                 datetime.utcnow().strftime("%Y%m%d_%H%M%S"))

        # csv file with BOM. used for excel with Chinese in windows.
        try:
            flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
            with os.fdopen(os.open(compare_result_file, flags, 0o640), 'wb') as out:
                out.write(codecs.BOM_UTF8)
        except Exception as ex:
            LOG.error("[AC][RPT] Write BOM_UTF8 head in compare_result_file failed: %s", str(ex))

        have_set = set()
        inconsistent_list = []
        original_rtn = (('delete', self._delete_data_list, 'Missing in neutron'),
                        ('create', self._create_data_list + self._create_after_update_list, 'Missing in controller'),
                        ('update', self._update_data_list, 'Data is not same'))

        for (oper_type, res_list, log_info) in original_rtn:
            LOG.debug('[AC][RPT] operation: %s, inconsistent_res_list_len: %s, log_info: %s', oper_type, len(res_list),
                      log_info)
            for res in res_list:
                add_elem = self._format_inconsistent_data(res, log_info)
                add_elem_key = '%s#%s' % (add_elem.get('Resource'), add_elem.get('Id'))
                if add_elem_key not in have_set:
                    inconsistent_list.append(add_elem)
                    have_set.add(add_elem_key)

        # write to csv
        flags = os.O_WRONLY | os.O_CREAT | os.O_APPEND
        with os.fdopen(os.open(compare_result_file, flags, 0o640), 'a+') as out:
            headers = ['Resource', 'Id', 'Name', 'TenantName', 'CreateTime', 'UpdateTime', 'Status']
            dict_writer = ucsv.DictWriter(out, lineterminator='\n', delimiter=',', fieldnames=headers)
            dict_writer.writeheader()
            if inconsistent_list:
                LOG.debug('[AC][RPT] Begin to write %s records to csv file', len(inconsistent_list))
                dict_writer.writerows(inconsistent_list)

        # write to compare db
        if not inconsistent_list:
            inconsistent_list.append({'Id': ac_const.SYNC_NO_DIFFERENCE, 'Status': 'success'})
        LOG.debug('[AC][RPT] Begin to write %s records to compare db', len(inconsistent_list))
        self.compare_db.create_db_inconsistents(context.get_admin_context(), inconsistent_list)

    def _format_inconsistent_data(self, res, log_info):
        id_msg, name, tenant_name, created_at, updated_at = self._util.get_inconsistent_rec_info(res)
        res_type = res['res']
        if not created_at or not updated_at:
            session = self._db_if.get_session('write')
            if res_type in [ac_const.NW_HW_PORTS]:
                resource = self._db_if.get_port_attribute(session, id_msg)
            elif res_type not in [ac_const.NW_HW_EXROUTE]:
                if res_type not in self.cache_res_attr_dict:
                    res_attr_list = self._db_if.get_res_attribute(session, res_type=res_type)
                    res_attr_dict = {resource.uuid: resource for resource in res_attr_list}
                    self.cache_res_attr_dict[res_type] = res_attr_dict
                res_attr_dict = self.cache_res_attr_dict.get(res_type, dict())
                if id_msg not in res_attr_dict:
                    resource = self._db_if.get_res_attribute(session, id_msg, res_type)
                else:
                    resource = res_attr_dict.get(id_msg)
            else:
                resource = None
            if resource:
                created_at = resource.create_time.strftime(ac_const.ISO8601_TIME_FORMAT)
                updated_at = resource.update_time.strftime(ac_const.ISO8601_TIME_FORMAT)
        return {
            'Resource': res_type,
            'Id': id_msg,
            'Name': name,
            'TenantName': tenant_name,
            'CreateTime': created_at,
            'UpdateTime': updated_at,
            'Status': log_info
        }

    # 1. Set the neutron state table state as CONSISTENCY_CHECK and the
    #    sync_res as 'complete'.
    # 2. Read id alone of resources types required for neutron sync from
    #    neutron-db.
    # 3. Form a consistency query message and send it to AC to get the hash.
    # 4. If failure from AC stop processing and log it. Else,
    # 5. Call the method to generate hash for the neutron db data.
    # 6. Pass the two hash to another method to compare and get the difference
    #    resource types.
    def _consistency_check(self, session):
        """ consistency check """
        self._db_if.update_server_record(
            ACNeutronStateSchema(state=ac_const.CONSISTENCY_CHECK, sync_res=ac_const.NW_HW_COMPLETE), session=session)
        if self._sync_op == ac_const.SYNC_OP_SYNC_DATA:
            sync_info = self.sync_resources.get('params', [])
            if sync_info:
                in_consist_list = self.get_in_consist_list_single_ins()
                return in_consist_list
        if self._sync_op == ac_const.SYNC_OP_SYNC_SINGLE_INSTANCE:
            in_consist_list = self.get_in_consist_list_single_ins()
        else:
            in_consist_list = self.get_in_consist_list_sync(session)

        if in_consist_list:
            LOG.info(_LI('Neutron sync found the data in-consistent in '
                         'controller for resources %s.'), in_consist_list)
        else:
            LOG.info(_LI('Neutron sync found the data consistent in '
                         'controller.'))

        return in_consist_list

    def get_in_consist_list_sync(self, session):
        """获取不一致数据的类型的列表

        :param session: session
        :return: inconsist_type_list
        """
        neutron_hash_list = self._read_neutron_db_hash()
        self._wait_in_process_record_to_finish_sync(session)
        ac_hash_list = self._get_ac_hash()
        inconsist_type_list = self._compare_hash(neutron_hash_list, ac_hash_list)
        return inconsist_type_list

    def get_in_consist_list_single_ins(self):
        """Get inconsistent list for single instance recover"""
        LOG.info('Get sync type for single instance sync :%s',
                 self.sync_resources)
        sync_type = []
        sync_info = self.sync_resources.get('params', [])
        for sync_info_type in sync_info:
            attribute = sync_info_type.get('attribute', None)
            if attribute:
                sync_type.append(attribute)
        return sync_type

    def update_neutron_sync_attribute_list(self, res_type, network_id=None):
        """更新_res_attr_dict和_port_attr_dict中的数据，dict中的数据格式为uuid - update_time

        :param res_type: res_type
        :param network_id: network_id
        """
        if res_type == ac_const.NW_HW_EXROUTE:
            pass
        elif res_type in [ac_const.NW_HW_PORTS, ac_const.NW_HW_ROUTER_IF]:
            # 优先使用内存中的数据，降低循环查库耗费的时间
            if self.cache_network_port_attr_dict and network_id:
                rec_list = self.cache_network_port_attr_dict.get(network_id, [])
            else:
                rec_list = self._db_if.get_port_attribute(network_id=network_id)
            for rec in rec_list:
                self._port_attr_dict[rec.uuid] = rec.update_time.strftime(ac_const.ISO8601_TIME_FORMAT)
        else:
            if res_type not in self._res_attr_dict:
                self._res_attr_dict[res_type] = {}
            rec_list = self._db_if.get_res_attribute(res_type=res_type)
            for rec in rec_list:
                self._res_attr_dict.get(res_type)[rec.uuid] = rec.update_time.strftime(ac_const.ISO8601_TIME_FORMAT)

    def _read_neutron_db_hash(self):
        """更新_res_attr_dict和_port_attr_dict中的数据，返回hash_list

        :return: hash_list
        """
        admin_ctx = context.get_admin_context()
        networks = []
        hash_list = {}

        for res_type in self._resource_list.keys():
            LOG.debug("[AC] Get neutron data for %s", res_type)
            fields = ['id']
            # in case of port we need to segregate with network id
            if res_type == ac_const.NW_HW_PORTS:
                self._update_port_hash_list(admin_ctx, hash_list, networks, res_type)
            elif res_type == ac_const.NW_HW_SNATS:
                hash_list[res_type] = self._calc_neutron_snat_hash(admin_ctx, fields)
            elif res_type == ac_const.NW_HW_SEC_GRP_RULE:
                # security-group rule doesnt support update. So no need to use update fields for hash generation.
                res_list = getattr(self.neutron_formatter, self._resource_list.get(res_type))(admin_ctx, fields=fields)
                id_list = [res['id'] for res in res_list]
                md5_hash = ACNeutronSync._calculate_hash(id_list)
                hash_list[res_type] = md5_hash
            elif res_type == ac_const.NW_HW_FIP:
                fields.extend(['fixed_ip_address', 'floating_network_id',
                               'floating_ip_address', 'tenant_id', 'port_id'])
                res_list = getattr(self.neutron_formatter, self._resource_list.get(res_type))(admin_ctx, fields=fields)
                id_list = self._create_id_list_hash_format(admin_ctx.session, res_type, res_list) if res_list else []
                md5_hash = ACNeutronSync._calculate_hash(id_list)
                hash_list[res_type] = md5_hash
            elif res_type == ac_const.NW_HW_EXROUTE:
                res_list = getattr(self.neutron_formatter, self._resource_list.get(res_type))(admin_ctx, fields=fields)
                id_list = []
                # exroute. caculate hash with id+nexthop+destination
                # exroute don't support update, so don't record updated_time so caculate hash don't use updated_time
                for res in res_list:
                    id_list.append("%s_%s_%s" % (res.router_id, res.nexthop, res.destination))
                md5_hash = ACNeutronSync._calculate_hash(id_list)
                hash_list[res_type] = md5_hash
            else:
                # router, network, subnet, router_interface等都使用_update_hash_list更新hash_list
                networks = self._update_hash_list(admin_ctx, hash_list, networks, res_type)
            LOG.debug("[AC] End of getting neutron data for %s", res_type)
        return hash_list

    def _update_port_hash_list(self, admin_ctx, hash_list, networks, res_type):
        # If network already read in previous loop, use it
        if not networks:
            networks = self.neutron_formatter.read_networks(admin_ctx, fields=['id'])

        # 一次性读取port_attrs
        port_attr_list = self._db_if.get_port_attribute()
        self.cache_network_port_attr_dict.clear()
        for rec in port_attr_list:
            if rec['network_id'] not in self.cache_network_port_attr_dict:
                self.cache_network_port_attr_dict[rec['network_id']] = [rec]
            else:
                self.cache_network_port_attr_dict.get(rec['network_id']).append(rec)

        # 一次性读取ports
        res_list = getattr(self.neutron_formatter, self._resource_list.get(res_type))(admin_ctx, fields=['id'])
        network_port_dict = {}
        for rec in res_list:
            if rec['network_id'] not in network_port_dict:
                network_port_dict[rec['network_id']] = [rec]
            else:
                network_port_dict.get(rec['network_id']).append(rec)

        for network in networks:
            port_list = network_port_dict.get(network['id'], list())
            port_res = '%s_%s' % (ac_const.NW_HW_PORTS, network['id'])
            id_list = self._create_id_list_hash_format(admin_ctx.session, res_type, port_list, network['id'])
            md5_hash = ACNeutronSync._calculate_hash(id_list)
            hash_list[port_res] = md5_hash

    def _calc_neutron_snat_hash(self, admin_ctx, fields):
        res_list = []
        if ncu.IS_FSP:
            fields.extend(
                ['fixed_ip_address', 'floating_network_id',
                 'floating_ip_address', 'tenant_id', 'port_id'])
            res_list = \
                getattr(self.neutron_formatter, self._resource_list.get(
                    ac_const.NW_HW_SNATS))(admin_ctx, fields=fields)
        res_list_ext = self.neutron_formatter.read_snats_v2(admin_ctx)
        res_list.extend(res_list_ext)
        id_list = []
        if res_list:
            id_list = self._create_id_list_hash_format(
                admin_ctx.session, ac_const.NW_HW_SNATS, res_list)
        return ACNeutronSync._calculate_hash(id_list)

    def _update_hash_list(self, admin_ctx, hash_list, networks, res_type):
        res_list = getattr(self.neutron_formatter, self._resource_list.get(res_type))(admin_ctx, fields=['id'])
        # if it is cmcc qos policy, process it as normal qos policy
        check_type = res_type
        if check_type == ac_const.NW_HW_CMCC_QOS_POLICY:
            res_type = ac_const.NW_HW_QOS_POLICY
        if res_type == ac_const.NW_HW_FIREWALL_POLICY:
            self.update_neutron_sync_attribute_list(res_type=ac_const.NW_HW_FIREWALL_POLICY)
            id_list = [self._spell_fw_policy_id_with_rule_id(admin_ctx.session, res['id']) for res in res_list]
        else:
            if res_type == ac_const.NW_HW_NETWORKS:
                networks = res_list
            id_list = self._create_id_list_hash_format(admin_ctx.session, res_type, res_list) if res_list else []
        md5_hash = ACNeutronSync._calculate_hash(id_list)
        if res_type == ac_const.NW_HW_CMCC_QOS_POLICY:
            hash_list[ac_const.NW_HW_QOS_POLICY] = md5_hash
        else:
            hash_list[res_type] = md5_hash
        return networks

    def _spell_fw_policy_id_with_rule_id(self, session, fw_policy_id):
        """spell fw policy id with rule id"""
        fw_rules_list = self._db_if.get_fw_rule_record_by_policy(
            session, fw_policy_id)
        spell_result = fw_policy_id
        for fw_rule in fw_rules_list:
            spell_result = "%s_%s" % (spell_result, fw_rule.id)
        attr_list = self._res_attr_dict.get(ac_const.NW_HW_FIREWALL_POLICY)
        if fw_policy_id in attr_list:
            spell_result = "%s_%s" % (spell_result,
                                      attr_list[fw_policy_id])
        else:
            spell_result = "%s_%s" % (spell_result, '0')
            self._db_if.update_attribute_record(session, fw_policy_id,
                                                ac_const.NW_HW_FIREWALL_POLICY)
        LOG.debug("Get firewall policy is: %s", spell_result)
        return spell_result

    def _create_id_list_hash_format(self, session, res_type, res_list, network_id=None):
        """调用update_neutron_sync_attribute_list更新_res_attr_dict和_port_attr_dict中的数据，返回的id_list

        :param session: session
        :param res_type: res_type
        :param res_list: res_list
        :param network_id: network_id
        :return: id_list，id_list中的item为id+update_time
        """
        id_list = []
        res_container = ac_const.NEUTRON_RESOURCES_YANG.get(res_type, {}).get('container_name')
        if cfg.CONF.huawei_ac_config.neutron_sync_enable_white_list and \
                self._white_list.get(res_container, list()):
            res_list = [res for res in res_list if res['id'] not
                        in self._white_list.get(res_container, list())]
            LOG.debug('Filtered resource list : %s', res_list)
        if res_type in [ac_const.NW_HW_PORTS, ac_const.NW_HW_ROUTER_IF]:
            self.update_neutron_sync_attribute_list(res_type=res_type, network_id=network_id)
            for res in res_list:
                # _port_attr_dict存放所有类型port的数据，包括router_interface
                if res['id'] in self._port_attr_dict:
                    id_list.append('%s_%s' % (res['id'], self._port_attr_dict.get(res['id'])))
                else:
                    network_id = res.get('network_id') if res_type == ac_const.NW_HW_ROUTER_IF and network_id is None \
                        else network_id
                    res_type = ac_const.NW_HW_PORTS
                    self._db_if.update_attribute_record(session, res['id'], res_type, network_id=network_id)
                    id_list.append('%s_%s' % (res['id'], '0'))
        else:
            self.update_neutron_sync_attribute_list(res_type=res_type)
            for res in res_list:
                if res['id'] in self._res_attr_dict.get(res_type):
                    id_list.append('%s_%s' % (res['id'], self._res_attr_dict.get(res_type).get(res['id'])))
                else:
                    self._db_if.update_attribute_record(session, res['id'], res_type)
                    id_list.extend('%s_%s' % (res['id'], '0'))
        return id_list

    # Send consist query, get and parse result
    def _get_ac_hash(self):
        """Send consist query, get and parse result"""
        try:
            rest_result = self._get_resource_data_from_ac("consist")
        except requests.Timeout:
            LOG.error(_LE("Neutron sync by neutron server %s failed, "
                          "controller timed out to process the consistency "
                          "query."), self.neutron_name)
            raise

        output = "huawei-ac-neutron-consist:output"
        if rest_result:
            ac_hash_list = ACNeutronSync._get_ac_hash_list(output, rest_result)
            return ac_hash_list
        else:
            raise NeutronSyncError(reason='No consistency data in controller response')

    @classmethod
    def _get_ac_hash_list(cls, output, rest_result):
        """get ac hash list"""
        pf_plugin = ncu.get_service_plugin().get('PORTFORWARDING')
        res_json = rest_result
        if not res_json or output \
                not in res_json:
            raise NeutronSyncError(
                reason='No consistency data in controller response')
        ac_hash_list = {}
        for res_hash_data in res_json[output].get('consist-result', {}):
            # to compate portforwarding with dnat.
            if res_hash_data['name'] == ac_const.NW_HW_DNAT and \
                    pf_plugin:
                ac_hash_list[ac_const.NW_HW_PF] = res_hash_data['hash']
            else:
                ac_hash_list[res_hash_data['name']] = \
                    res_hash_data['hash']
        for res_hash_data in res_json[output].get('port-result', {}):
            port_res_type = 'port_%s' % (res_hash_data['network-id'])
            ac_hash_list[port_res_type] = res_hash_data['hash']
        return ac_hash_list

    # Compare the hash generated from neutron-db and has got from AC.
    def _compare_hash(self, neutron_hash, ac_hash):
        """Compare the hash generated from neutron-db and has got from AC."""
        ACNeutronSync._compare_hash_pre_process(ac_hash, neutron_hash)
        compare_dict_result = util.ACUtil.compare_dict(neutron_hash, ac_hash)
        added = compare_dict_result[0]
        removed = compare_dict_result[1]
        modified = compare_dict_result[2]
        res_list = added + removed + modified
        # This code will ensure the res_type is returned as per dependency list
        sorted_res_list = []
        for res_type in ac_const.NEUTRON_SYNC_DEPENDENCY_ORDER:
            if res_type in res_list:
                sorted_res_list.append(res_type)
            elif res_type == ac_const.NW_HW_PORTS:
                port_list = [res for res in res_list if res.startswith(res_type)]
                if port_list:
                    sorted_res_list.append(res_type)
            elif res_type == ac_const.NW_HW_BINDPORTS and \
                    self._include_bindports:
                sorted_res_list.append(res_type)
        return sorted_res_list

    def get_single_sync_lists(self, res_type):
        """Get or ids for sync while single instance type"""
        res_info = self.sync_resources.get('params', None)
        for res_type_info in res_info:
            if res_type_info.get('attribute', None) == res_type:
                self.single_create_lists = res_type_info.get('create_lists', [])
                self.single_update_lists = res_type_info.get('update_lists', [])
                self.single_delete_lists = res_type_info.get('delete_lists', [])
                self.single_res_ids = \
                    self.single_create_lists + self.single_update_lists + \
                    self.single_delete_lists

    def _get_resource_data_from_neutron_db(self, res_type):
        """
        Read the data from neutron db for a particular type and format it to
        make same as the one from AC
        :param res_type: resource type
        :return: resourc edata
        """
        LOG.debug('[AC] Begin to get %s data from neutron db', res_type)
        admin_ctx = context.get_admin_context()

        filters = None
        if self._sync_op == ac_const.SYNC_OP_SYNC_SINGLE_INSTANCE:
            self.get_single_sync_lists(res_type)
            filters = {'id': self.single_res_ids}

        if res_type == ac_const.NW_HW_BINDPORTS:
            return self.neutron_formatter.read_binding_levels(admin_ctx)

        res_container = ac_const.NEUTRON_RESOURCES_YANG.get(
            res_type).get('container_name')
        if res_type == ac_const.NW_HW_DNAT:
            res_container = ac_const.NW_HW_DNATS
        data = {res_container: []}

        if res_type == ac_const.NW_HW_QOS_POLICY and \
                ac_const.NW_HW_CMCC_QOS_POLICY in ac_const.NW_HW_NEUTRON_SYNC_SUPPORT_RES:
            resources = getattr(self.neutron_formatter, self._resource_list.get(ac_const.NW_HW_CMCC_QOS_POLICY))(
                admin_ctx, filters=filters)
        elif res_type == ac_const.NW_HW_SNATS:
            resources = self.neutron_formatter.read_snats(admin_ctx, filters)
            resources_ext = self.neutron_formatter.read_snats_v2(admin_ctx)
            resources.extend(resources_ext)
        elif res_type == ac_const.NW_HW_PORTS:
            resources = self._get_port_from_neutron_db(admin_ctx, res_type, filters)
        elif res_type == ac_const.NW_HW_ROUTERS:
            resources = self._get_router_from_neutron_db(admin_ctx, res_type, filters)
        elif res_type == ac_const.NW_HW_EXROUTE:
            resources = self._get_exrouter_from_neutron_db(admin_ctx, res_type)
        else:
            resources = getattr(self.neutron_formatter, self._resource_list.get(res_type))(
                admin_ctx, filters=filters)
        resources_ct = len(resources) if resources else 0
        LOG.debug('Got %d records from neutron db for %s', resources_ct, res_type)

        # No log is allowed in this for-loop
        for resource in resources:
            res_info = self._get_res_info(admin_ctx, res_type, resource)
            self._neutron_sync_update_info(res_info, resource)
            data.get(res_container).append(res_info)
        return data

    def _get_exrouter_from_neutron_db(self, admin_ctx, res_type):
        filters = None
        if self.single_res_ids:
            resources = []
            for reqItem in self.single_res_ids:
                ex_rt = ast.literal_eval(reqItem)
                filters = {
                    'router_id': ex_rt.get('router_id'),
                    'nexthop': ex_rt.get('nexthop'),
                    'destination': ex_rt.get('destination')
                }
                exroute_res = getattr(self.neutron_formatter, self._resource_list.get(res_type))(
                    admin_ctx, filters=filters)
                if exroute_res:
                    resources.extend(exroute_res)
            return resources
        else:
            return getattr(self.neutron_formatter, self._resource_list.get(res_type))(
                admin_ctx, filters=filters)

    @classmethod
    def _get_res_info_port(cls, resource, res_info):
        """get res info port"""
        if ncu.is_trunk_parent_port(resource):
            res_info['trunk_parent_port'] = True
        if resource['device_owner'] == ncu.DEVICE_OWNER_ROUTER_GW:
            res_info['gw'] = True
            res_info['nw_id'] = resource['network_id']
            res_info['device_owner'] = resource['device_owner']
            res_info['device_id'] = resource['device_id']
        elif resource['device_owner'] == ncu.DEVICE_OWNER_DHCP:
            res_info['dhcp_port'] = True
            res_info['device_owner'] = resource['device_owner']
        elif resource['device_owner'] == ncu.DEVICE_OWNER_FLOATINGIP:
            res_info['fip_port'] = True
            res_info['device_owner'] = resource['device_owner']
        elif resource['device_owner'] == \
                ncu.DEVICE_OWNER_ROUTER_INTF:
            res_info['router_intf'] = True
        return res_info

    def _get_res_info(self, admin_ctx, res_type, resource):
        """get res info"""
        if res_type == ac_const.NW_HW_EXROUTE:
            # exroute, compare data with router-id, nethop, destination
            res_info = {
                'router-id': resource['router_id'],
                'nexthop': resource['nexthop'],
                'destination': resource['destination'],
                'type': resource['type'],
                'tenant-name': ""
            }
        else:
            res_info = {
                'uuid': resource['id'],
                'tenant-name': self._get_tenant_name(resource['tenant_id'])
            }
            if res_type == ac_const.NW_HW_PORTS:
                res_info = ACNeutronSync._get_res_info_port(resource, res_info)
            elif res_type == ac_const.NW_HW_FIREWALL_POLICY:
                fw_rules_list = self._db_if.get_fw_rule_record_by_policy(admin_ctx.session, resource['id'])
                res_info['firewall_rules'] = []
                for fw_rule in fw_rules_list:
                    res_info.get('firewall_rules').append(fw_rule.id)
            elif res_type == ac_const.NW_HW_QOS_POLICY and \
                    ac_const.NW_HW_CMCC_QOS_POLICY in ac_const.NW_HW_NEUTRON_SYNC_SUPPORT_RES:
                res_info['uuid'] = resource['qos_id']
        return res_info

    def _get_port_from_neutron_db(self, admin_ctx, res_type, filters=None):
        """get port from neutron db"""
        LOG.debug('[AC] Get ports from db')
        ports = getattr(self.neutron_formatter, self._resource_list.get(res_type))(admin_ctx, filters=filters)
        LOG.debug('[AC] End of getting ports from db, len: %s', len(ports))
        return ports

    def _get_resource_data_from_ac(self, res_type):
        """
        Form the url, condition if segregate is present and send a query request
        to AC and get the result based on the condition provided.
        :param res_type: resource type
        :return: resource data
        """
        LOG.debug('[AC] Begin to get %s data from AC', res_type)
        max_wait_count = ac_const.NW_HW_MAX_GATHER_STATUS_RETRY_COUNT
        result = ac_const.NW_HW_ERROR
        file_list = None
        module_name, container_name = \
            self._get_module_name_and_container_name(res_type)
        uuid = self._gather_resource_data(res_type, module_name)
        while max_wait_count > 0:
            max_wait_count -= 1
            result, file_list = self._gather_resource_status(
                uuid, res_type)
            if result == ac_const.NW_HW_SUCCESS:
                break
            time.sleep(5)

        if result != ac_const.NW_HW_SUCCESS or not file_list:
            LOG.error(_LE("Neutron sync by neutron server %s failed, "
                          "controller timed out while getting gather "
                          "status for %s."), self.neutron_name, res_type)
            raise NeutronSyncError(
                reason='Controller timed out in neutron sync data query')

        return self._process_file_list(file_list, container_name, res_type)

    def _get_formatted_data_for_neutron_sync(
            self, admin_ctx, rec_uuid, record_info):
        """get formmatted data from neutron sync

        :param admin_ctx: context
        :param rec_uuid: uuid
        :param record_info:
        :return: formatted data
        """
        tenant, method, res_type = record_info
        if res_type not in ac_const.FORMATTER_FUNC:
            return {}

        LOG.debug('Processing resource id: %s', rec_uuid)
        if method != constants.REST_DELETE:
            if (res_type == ac_const.NW_HW_QOS_POLICY and
                    ac_const.NW_HW_CMCC_QOS_POLICY in
                    ac_const.NW_HW_NEUTRON_SYNC_SUPPORT_RES):
                data = getattr(
                    self.neutron_formatter, ac_const.FORMATTER_FUNC.get(
                        ac_const.NW_HW_CMCC_QOS_POLICY))(admin_ctx, tenant, rec_uuid)
            elif res_type == ac_const.NW_HW_SNATS:
                data = self.neutron_formatter.set_snat(
                    admin_ctx, tenant, rec_uuid)
                if not data:
                    data = self.neutron_formatter.set_snat_v2(
                        admin_ctx, rec_uuid)
            elif res_type == ac_const.NW_HW_DNAT:
                try:
                    data = getattr(self.neutron_formatter, ac_const.FORMATTER_FUNC.get(res_type))(
                        admin_ctx, tenant, rec_uuid)
                except dnat_extension.DNATNotFound:
                    LOG.info('Not get data from dnat, try get it from '
                             'portforwarding uuid: %s', rec_uuid)
                    res_type = ac_const.NW_HW_PF
                    data = getattr(self.neutron_formatter, ac_const.FORMATTER_FUNC.get(res_type))(
                        admin_ctx, tenant, rec_uuid)
            else:
                data = getattr(self.neutron_formatter, ac_const.FORMATTER_FUNC.get(res_type))(
                    admin_ctx, tenant, rec_uuid)
            self._update_entry_update_time(data, res_type, rec_uuid)
            return data
        return {}

    def _compare_and_generate_diff_requests_for_bindports(
            self, ops_data_list, ac_data_list, res_type):
        """compore and generate diff request for bindingports"""
        LOG.debug('[AC] Begin to compare and generate diff requests for '
                  'bindports, openstack data list: %s, ac data list: %s',
                  ops_data_list, ac_data_list)
        ac_network_list = [x['network-id'] for x in ac_data_list]
        ac_network_host_list = \
            [(x['network-id'], x['host-id']) for x in ac_data_list]
        ac_segmentation_list = [x['segmentation-id'] for x in ac_data_list]

        for ops_bindport in ops_data_list:
            port_id, network_id, host_id, segmentation_id = ops_bindport
            entry_info = {'res': res_type,
                          'port_id': port_id,
                          'network_id': network_id,
                          'host_id': host_id,
                          'segmentation_id': segmentation_id}
            if (network_id, host_id) in ac_network_host_list:
                index = ac_network_host_list.index((network_id, host_id))
                entry_info['ac_segment'] = ac_segmentation_list[index]
                if not segmentation_id:
                    self._data_list.append(entry_info)
                elif segmentation_id != entry_info.get('ac_segment'):
                    self._update_data_list.append(entry_info)
            elif network_id in ac_network_list and not host_id:
                index = ac_network_list.index(network_id)
                entry_info['ac_segment'] = ac_segmentation_list[index]
                self._data_list.append(entry_info)
            else:
                entry_info['ac_segment'] = None
                self._create_data_list.append(entry_info)

    def _compare_and_generate_diff_requests(self, session, neutron_data,
                                            ac_data, res_type):
        """
        Compare the records from AC and neutron db and generate a request and
        insert into plugin db by checking the dependency.
        :param session: session
        :param neutron_data: neutron data
        :param ac_data: ac data
        :param res_type: resource type
        :return: total record count
        """
        LOG.info('[AC] Compare and generate different request, res_type: %s', res_type)
        total_record_count = 0
        res_container = ac_const.NEUTRON_RESOURCES_YANG.get(res_type).get('container_name')
        if res_type == ac_const.NW_HW_DNAT:
            res_container = ac_const.NW_HW_DNATS
        neutron_data_list = neutron_data[res_container]
        ac_data_list = ac_data[res_container]
        self.update_neutron_sync_attribute_list(res_type=res_type)
        self._data_list = []

        if res_type == ac_const.NW_HW_BINDPORTS:
            self._compare_and_generate_diff_requests_for_bindports(neutron_data_list, ac_data_list, res_type)
            return total_record_count
        if res_type == ac_const.NW_HW_EXROUTE:
            total_record_count = \
                self._compare_and_generate_diff_requests_for_exroute(neutron_data_list, ac_data_list, res_type)
            return total_record_count

        neutron_id_list = [n_list['uuid'] for n_list in neutron_data_list]
        ac_id_list = [ac_list['uuid'] for ac_list in ac_data_list]

        if self._sync_op == ac_const.SYNC_OP_SYNC_SINGLE_INSTANCE:
            ac_id_list = self.single_delete_lists
            ac_data_list = [ac_res for ac_res in ac_data_list if ac_res['uuid'] in ac_id_list]
        ac_data_list, ac_id_list = self._process_whitelist_controller(
            ac_data_list, ac_id_list, res_container)
        neutron_data_list, neutron_id_list = self._process_whitelist_neutron(
            neutron_data_list, neutron_id_list, res_container)

        # Don't sync data if current sync operation has sync, need_delete is False.
        if self._sync_op != ac_const.SYNC_OP_SYNC_SINGLE_INSTANCE:
            total_record_count = self._neutron_sync_prepare_data_list(
                res_type, session, total_record_count,
                (ac_data_list, neutron_data_list, neutron_id_list, ac_id_list))
        else:
            total_record_count = self.single_ins_sync_prepare_data_list(
                session, ac_data_list, neutron_data_list,
                (res_type, total_record_count))

        self._delete_data_records.extend(copy.deepcopy(self._data_list))
        # print compare result
        LOG.info("[AC] Compare result, delete list: %s, create list: %s, update list: %s",
                 self._data_list,
                 self._create_data_list + self._create_after_update_list,
                 self._update_data_list)
        return total_record_count

    def _neutron_sync_prepare_data_list(self, res_type, session,
                                        total_record_count, sync_data_lists):
        """neutron sync prepare data list"""
        LOG.info('[AC] Prepare data list')
        ac_data_list, neutron_data_list, neutron_id_list, ac_id_list = sync_data_lists
        neutron_dict = {data['uuid']: data for data in neutron_data_list}
        ac_dict = {data['uuid']: data for data in ac_data_list}

        # Only in controller
        ac_unique_id_list = set(ac_id_list) - set(neutron_id_list)
        deleted_entry_list = [ac_dict[_id] for _id in ac_unique_id_list]
        if self._sync_op == ac_const.SYNC_OP_COMPARE_DATA or (
                self._need_delete and self._sync_op in [ac_const.SYNC_OP_SYNC_DATA,
                                                        ac_const.SYNC_OP_SYNC_AND_COMPARE_DATA]):
            for entry_info in deleted_entry_list:
                delete_data = ACNeutronSync._get_delete_data(entry_info, res_type)
                self._neutron_sync_delete_info(delete_data, entry_info, res_type)
                self._data_list.append(delete_data)
                total_record_count += 1
            if res_type == ac_const.NW_HW_PORTS:
                self._reorder_dhcp_ports()

        # Only in neutron
        neutron_unique_id_list = set(neutron_id_list) - set(ac_id_list)
        created_entry_list = [neutron_dict[_id] for _id in neutron_unique_id_list]
        for entry_info in created_entry_list:
            entry_info['res'] = res_type
            entry_info['status'] = ac_const.STAT_WAIT

            if res_type == ac_const.NW_HW_PORTS:
                self._neutron_sync_create_port(entry_info, session)

            if res_type in self._create_after_update_type_list:
                self._create_after_update_list.append(entry_info)
            else:
                self._create_data_list.append(entry_info)

        # In both controller and neutron
        update_id_list = set(neutron_id_list).intersection(ac_id_list)
        updated_entry_list = [neutron_dict[_id] for _id in update_id_list]
        attr_list = self._res_attr_dict.get(res_type) \
            if (res_type != ac_const.NW_HW_PORTS and res_type != ac_const.NW_HW_ROUTER_IF) else self._port_attr_dict
        for entry_info in updated_entry_list:
            # SEC_GRP_RULE don't support update
            if res_type == ac_const.NW_HW_SEC_GRP_RULE:
                continue
            self._neutron_sync_update(attr_list, ac_dict.get(entry_info['uuid'], None), entry_info, res_type)
        LOG.info('[AC] Prepare data list finished')
        return total_record_count

    def _neutron_sync_update(self, attr_list, ac_data, entry_info, res_type):
        """ neutron sync update"""
        # for firewall policy, calculate ths hash with uuid and
        # fw rule list, so if hash is different, need to update fw
        # policy
        try:
            if res_type == ac_const.NW_HW_FIREWALL_POLICY:
                self._neutron_sync_update_firewall_policy(attr_list, ac_data, entry_info, res_type)
            else:
                self._neutron_sync_update_resource(attr_list, ac_data, entry_info, res_type)
        except KeyError:
            LOG.warning(_LW('Skip update neutron sync for %s'), entry_info['uuid'])

    def _neutron_sync_create_port(self, entry_info, session):
        """neutron sync create port"""
        if 'gw' in entry_info:
            if not entry_info['tenant-name'] and 'nw_id' in entry_info:
                network_res = self._db_if.get_neutron_record(
                    session, entry_info['nw_id'],
                    ac_const.NW_HW_NETWORKS)
                tenant_name = self._get_tenant_name(
                    network_res['tenant_id'])
                LOG.debug('Gateway port tenant name from network: %s',
                          tenant_name)
                entry_info['tenant-name'] = tenant_name

        elif 'router_intf' in entry_info:
            entry_info['res'] = ac_const.NW_HW_ROUTER_IF
        elif "trunk_parent_port" in entry_info:
            # for trunk port,if sync to create trunk parent port,
            # need to add update for this port, because ac just
            # set_check the config when updating trunk port
            trunk_parent_port = {
                'res': ac_const.NW_HW_PORTS,
                'uuid': entry_info['uuid'],
                'tenant-name': entry_info['tenant-name'],
                'status': ac_const.STAT_WAIT}
            LOG.debug('[AC]Sync trunk parent port,'
                      'adding entry to plugin for update: %s',
                      trunk_parent_port)
            self._update_data_list.append(trunk_parent_port)

    def _process_data_list(self, res_type, operation, session):
        for record in list(self._data_list):
            self._statistic_neutron_sync_request(
                res_type, operation, record,
                ac_const.NW_HW_TIMEOUT)
            self._data_list.remove(record)
            if operation == ac_const.OPER_CREATE:
                if record in self._create_data_list:
                    self._create_data_list.remove(record)
                elif record in self._create_after_update_list:
                    self._create_after_update_list.remove(record)
            elif operation == ac_const.OPER_UPDATE and record \
                    in self._update_data_list:
                self._update_data_list.remove(record)
            if self._util.check_resource_have_state(record['res']) and \
                    operation != ac_const.OPER_DELETE:
                self._db_if.update_neutron_db_state(
                    session, record['uuid'], record['res'],
                    ac_const.NEUTRON_STATUS_ERROR)

    def _sync_data_process(
            self, session, res_type, operation, pending_res_list):
        """Read the records one by one from db and send to AC"""
        self._continuous_timeout = 0
        start_time = time.time()
        if res_type == ac_const.NW_HW_PF:
            res_type = ac_const.NW_HW_DNAT
        LOG.debug('Performing sync for operation: %s and res_type: %s',
                  operation, res_type)
        max_count = ac_const.MAX_MSG_IN_BULK if \
            operation == ac_const.OPER_CREATE else 1

        if res_type in ac_const.NW_HW_L47_LIST:
            self._thread_pool = worker.ACSyncThreadPool(
                ac_const.NEUTRON_SYNC_THREAD_POOL_L47_COUNT)
            self._thread_pool.spawn_threads(
                ac_const.NEUTRON_SYNC_THREAD_POOL_L47_COUNT)
        else:
            self._thread_pool.spawn_threads(
                ac_const.NEUTRON_SYNC_THREAD_POOL_DEFAULT_COUNT)
        try:
            self._continuous_timeout_res_list = []
            self._loop_process_data(start_time, max_count, session,
                                    (pending_res_list, res_type, operation))
        except ContinueTimeoutException as ex:
            LOG.error(_LE("Loop process same restype continue timeout "
                          "max times, %s"), str(ex))
        except Exception as ex:
            LOG.error(_LE("Loop process data error, %s"), str(ex))
            self._process_data_list(res_type, operation, session)
            self._update_neutron_sync_result()
            raise
        finally:
            if self._continuous_timeout_type == res_type and \
                    self._continuous_sync_next_type_flag:
                max_timeout_times = min(
                    ac_const.SYNC_TIMEOUT_TIMES_SINGLE_RES_TYPE,
                    len(self._data_list))
                LOG.info("sync res_type %s has continue timeout for %s times,"
                         "step to sync next res_type",
                         res_type, max_timeout_times)
            else:
                LOG.info(
                    'Neutron sync completed for res_type %s for operation : %s',
                    res_type, operation)
            self._thread_pool.wait_completion()

    def _loop_process_data(self, start_time, max_count, session, sync_res_data):
        """loop process data"""
        pending_res_list, res_type, operation = sync_res_data
        # clear self.quota_task_list, speed limit only used
        # in same res_type and operation.
        self.quota_task_list = []
        while True:
            max_rec_count = len(self._data_list)
            if not max_rec_count:
                LOG.debug('No records are pending to send to controller.')
                break
            LOG.debug('Number of records pending to transfer '
                      'to controller: %s.', max_rec_count)
            self._processed_rec_count = 0
            for start_index in range(0, max_rec_count, max_count):
                end_index = min(start_index + max_count, max_rec_count)
                rec_list = list(range(start_index, end_index))
                wait_ac_delete_plugin_db_count = 0
                # if process port, other port need to wait for
                # gateway port
                self._process_wait_for_gw_port(operation, rec_list, res_type)

                self._check_speed_limit_last_minute()

                self._check_plugin_record_count(
                    session, res_type,
                    wait_ac_delete_plugin_db_count)

                self._check_neutron_sync_thread(operation, rec_list, res_type)

            # Following loop waits for all the records to sent to
            # controller or continuous timeout.
            self._check_continuous_timeout(max_rec_count, pending_res_list,
                                           res_type, start_time)
            self._thread_pool.wait_completion()
            self._process_record_in_data_list(max_rec_count, operation,
                                              res_type, session)

    def _process_record_in_data_list(self, max_rec_count,
                                     operation, res_type, session):
        """process record in data list"""
        error_count = 0
        timeout_count = 0
        for record in list(self._data_list):
            error_count, timeout_count = \
                self._process_one_record(session, error_count, timeout_count,
                                         (operation, record, res_type))
        LOG.debug('Pending data-list: %s', self._data_list)
        if error_count:
            LOG.error(_LE('Neutron sync failed to transfer %d '
                          'resources due to the error reported '
                          'from controller.'), error_count)
        if timeout_count:
            if timeout_count == max_rec_count:
                LOG.error(
                    _LE(
                        'Neutron sync failed to transfer %d resource '
                        'due to %s, system might be inconsistent now. '
                        'Please check the controller and perform '
                        'neutron sync manually.'),
                    timeout_count, 'controller not responding')
                raise NeutronSyncError(
                    reason='Requests to controller timed out for'
                           ' %s in a retry cycle' % res_type)

            LOG.error(_LE('Neutron sync failed to transfer %d '
                          'resources due to message timeout, '
                          'retrying again now.'), timeout_count)
        LOG.debug(
            'Transfer statistics for %s, operation:%s.\nTotal '
            'records supposed to be transferred:%d\nTotal success '
            'count:%d\nTotal error count:%d\n', res_type,
            operation, max_rec_count, self._total_records_transfer,
            error_count)

    def _check_continuous_timeout(self, max_rec_count, pending_res_list,
                                  res_type, start_time):
        """check continuous timeout"""
        while True:
            self._thread_event.wait(.1)

            LOG.debug("check_continuous_timeout: %s ,res_type: %s",
                      self._continuous_timeout_type_times, res_type)
            max_timeout_times = min(
                ac_const.SYNC_TIMEOUT_TIMES_SINGLE_RES_TYPE,
                len(self._data_list))
            if self._continuous_timeout_type_times >= max_timeout_times:
                self._continuous_sync_next_type_flag = True
                LOG.info(" sync res_type: %s continue timeout for %s times,"
                         "set flag to True", res_type, max_timeout_times)
                raise ContinueTimeoutException(
                    "Sync continue timeout for %s times" % max_timeout_times)

            if self._processed_rec_count == max_rec_count:
                LOG.debug('Processed all assigned records.')
                break

            end_time = time.time()
            if (end_time - start_time) > \
                    ac_const.NEUTRON_SYNC_LOG_INTERVAL:
                start_time = end_time
                log_string = ('Neutron sync is in progress on neutron server '
                              '%s. Current resource in sync is %s(Total '
                              'transferred: %d, Pending: %d).') % \
                             (self.neutron_name, res_type,
                              self._total_records_transfer,
                              (max_rec_count - self._total_records_transfer))
                if pending_res_list:
                    log_string = '%s Pending resources is/are %s.' % (
                        log_string, pending_res_list)
                # the records are fetched at once.
                LOG.info(_LI(log_string))

    def _check_neutron_sync_thread(self, operation, rec_list, res_type):
        """check neutron sync thread"""
        while True:
            if self._neutron_sync_status != \
                    ac_const.NW_HW_NEUTRON_SYNC_ERROR:
                if not self._thread_pool.task_queue.full():
                    self._thread_pool.add_task(
                        self._process_data_in_queue, res_type,
                        operation, rec_list)
                    break
                else:
                    self._thread_event.wait(.1)
            else:
                raise NeutronSyncError('Terminating neutron sync due '
                                       'to internal error occured.')

    def _check_plugin_record_count(self, session, res_type,
                                   wait_ac_delete_plugin_db_count):
        """check plugin record count"""
        while True:
            wait_ac_delete_plugin_db_count += 1
            ac_processing_request_l47_count = \
                self._db_if.get_plugin_record_l47_count(session)
            if res_type in ac_const.NW_HW_L47_LIST and \
                    ac_processing_request_l47_count >= \
                    ac_const.AC_MAX_PROCESSING_REQUEST_L47_COUNT:
                if wait_ac_delete_plugin_db_count > \
                        ac_const.MAX_LOOP_TIMES_TO_WAIT_AC_PROCESS:
                    raise NeutronSyncError(
                        'Terminating neutron sync due to plugin db '
                        'has record for long time.')
                LOG.info("[AC]the ac now process %s records of l47, need to "
                         "wait.", ac_processing_request_l47_count)
                time.sleep(10)
                continue
            ac_processing_request_count = \
                self._db_if.get_plugin_record_count(session)
            if ac_processing_request_count > \
                    ac_const.AC_MAX_PROCESSING_REQUEST_COUNT:
                if wait_ac_delete_plugin_db_count > \
                        ac_const.MAX_LOOP_TIMES_TO_WAIT_AC_PROCESS:
                    raise NeutronSyncError(
                        'Terminating neutron sync due to plugin db '
                        'has record for long time.')
                LOG.info("[AC]the ac now process %s records, need to "
                         "wait.", ac_processing_request_count)
                time.sleep(10)
                continue
            break

    def _process_one_record(self, session, error_count,
                            timeout_count, record_info):
        """process one record"""
        operation, record, res_type = record_info
        if record['status'] == ac_const.STAT_ERR:
            error_count += 1
            if operation == ac_const.OPER_CREATE:
                if res_type in self._create_after_update_type_list:
                    self._create_after_update_list.remove(record)
                else:
                    self._create_data_list.remove(record)
                self._check_and_update_neutron_db(
                    session, res_type, record,
                    ac_const.NEUTRON_STATUS_ERROR)
            elif operation == ac_const.OPER_UPDATE:
                self._update_data_list.remove(record)
                self._check_and_update_neutron_db(
                    session, res_type, record,
                    ac_const.NEUTRON_STATUS_ERROR)
            self._data_list.remove(record)
        elif record['status'] == ac_const.STAT_SENT:
            if operation == ac_const.OPER_CREATE:
                self._create_data_list.remove(record)
                self._check_and_update_neutron_db(
                    session, res_type,
                    record, ac_const.NEUTRON_STATUS_ACTIVE)
            elif operation == ac_const.OPER_UPDATE:
                self._update_data_list.remove(record)
                self._check_and_update_neutron_db(
                    session, res_type,
                    record, ac_const.NEUTRON_STATUS_ACTIVE)
            self._data_list.remove(record)
        elif record['status'] == ac_const.STAT_TIME_OUT:
            timeout_count += 1
            record['status'] = ac_const.STAT_WAIT
        return error_count, timeout_count

    def _create_neutron_sync_plugin_record(self, session, res_info, operation,
                                           res_type):

        """ Function to insert the plugin record for neutron sync o wait for
        status update

        :param session: DB session
        :param operation: Operation type
        :param res_type: Type of resource
        :return: None
        """
        if operation in [ac_const.OPER_CREATE, ac_const.OPER_UPDATE] and \
                res_type == ac_const.NW_HW_PORTS:
            device_owner = res_info.get('device_owner')
            if device_owner in \
                    [ncu.DEVICE_OWNER_ROUTER_GW,
                     ncu.DEVICE_OWNER_FLOATINGIP]:
                return
        uuid = res_info['uuid']
        if res_type in ac_const.NW_HW_REPORT_STATUS_RESOURCES and operation \
                in [ac_const.OPER_CREATE, ac_const.OPER_UPDATE]:
            self._db_if.create_plugin_record(session, {},
                                             (uuid, operation, res_type),
                                             ac_const.NEUTRON_SYNC)
        return

    def get_response(self, thread_name, session, need_sync_data):
        """ get response and session """
        admin_ctx = context.get_admin_context()
        res_type, operation, rec_list = need_sync_data
        if res_type == ac_const.NW_HW_ROUTER_IF:
            res_type = ac_const.NW_HW_PORTS
        (method, resource_url) = util.ACUtil.get_method_and_resource_url(
            '%s_%s' % (operation, res_type))

        rest_data, rest_info, ret = self._get_rest_data_and_rest_info(
            admin_ctx, thread_name, rec_list, (method, operation, res_type))
        if ret:
            return None
        result, response = self._send_request_to_ac(
            rest_data, resource_url, thread_name,
            (operation, rec_list, res_type, method, rest_info))
        if result == ac_const.NW_HW_TIMEOUT:
            self._process_result_timeout(rec_list, res_type,
                                         operation, thread_name)
        elif result == ac_const.NW_HW_ERROR:
            self._process_result_error((operation, rec_list, res_type),
                                       response, session, thread_name)
        elif result == ac_const.NW_HW_SUCCESS:
            self._process_result_success(admin_ctx, session, thread_name,
                                         (operation, rec_list, res_type))
        return response

    # If any records are in in-process state wait for it to complete
    def _wait_in_process_record_to_finish_sync(
            self, session, res_type=None):
        """If any records are in in-process state wait for it to complete"""
        max_inprocess_wait = 1
        while ac_const.MAX_IN_PROCESS_CLEAR_RETRY_COUNT >= max_inprocess_wait:
            resources = self._db_if.get_plugin_record_list(
                session, ACPluginSchema(neutron_id=-1, res_type=res_type,
                                        state=ac_const.IN_PROCESS))
            if resources:
                LOG.debug('Wait 2 sec for the in-process records to '
                          'finish')
                # Wait for 2 seconds before recheck again
                time.sleep(2)
                max_inprocess_wait += 1
            else:
                return
        LOG.error(_LE('Terminating neutron sync as in_process '
                      'records are not processed for more than '
                      'accepted timeout limit. System might be '
                      'inconsistent. Please check and perform '
                      'neutron sync manually.'))
        raise NeutronSyncError(
            reason='Pending in_process records has not cleared within '
                   'default timeout limit.')

    def _send_request_to_ac(self, rest_data, resource_url,
                            thread_name, sync_res_info):
        operation, rec_list, res_type, method, rest_info = sync_res_info
        res_id = '' if operation == ac_const.OPER_CREATE else \
            self._data_list[rec_list[0]]['uuid']
        url = self._get_url_value(resource_url, rest_info, thread_name,
                                  (method, operation, res_id, res_type))
        result, response = self._send_data_to_ac(rest_data, url, len(rec_list),
                                                 (res_type, method, res_id))
        return result, response

    # Read the records one by one from db and send to AC
    def _process_data_in_queue(self, thread_name, res_type,
                               operation, rec_list):
        """ Read the records one by one from db and send to AC"""
        LOG.debug("%s: Process neutron sync data in queue initialized. "
                  "records:%s", thread_name, rec_list)
        count = len(rec_list)
        session = self._db_if.get_session()
        LOG.debug(" res_type: %s, count: %s , rec_list: %s", res_type, count,
                  rec_list)
        response = None
        try:
            response = self.get_response(
                thread_name, session, (res_type, operation, rec_list))
        except db_exc.RetryRequest:
            self._process_result_error((operation, rec_list, res_type),
                                       response, session, thread_name)
        except Exception as ex:
            LOG.error(_LE("%s, Neutron sync process data exception: %s"),
                      thread_name, ex)
            self._neutron_sync_status = ac_const.NW_HW_NEUTRON_SYNC_ERROR
            raise
        finally:
            with self._lock:
                self._processed_rec_count += count
            if self._continuous_timeout_type == res_type and \
                    self._continuous_sync_next_type_flag:
                LOG.info(" %s: Process neutron sync data in queue return "
                         "for continue 10 timeout.res_type: %s",
                         thread_name, res_type)
            else:
                LOG.info(" %s: Process neutron sync data in queue finished.",
                         thread_name)

        # after process gateway port, release the flag
        if res_type == ac_const.NW_HW_PORTS \
                and operation == ac_const.OPER_CREATE:
            if ncu.is_gw_port(self._data_list[rec_list[0]]):
                self.sync_wait_flag = False

    @classmethod
    def _get_url_value(cls, resource_url, rest_info, thread_name, sync_res_info):
        """get url value"""
        pool_id = None
        method, operation, res_id, res_type = sync_res_info
        if res_type == ac_const.NW_HW_MEMBER and rest_info:
            pool_id = rest_info[-1].get(ac_const.URL_POOL_ID)
        url = util.ACUtil.form_resource_url(
            ac_const.NW_HW_URL, method, resource_url,
            res_id, pool_id=pool_id)
        LOG.debug('%s: Send neutron sync data to controller.', thread_name)
        # process the url of exroute
        if res_type == ac_const.NW_HW_EXROUTE:
            if operation == ac_const.OPER_CREATE:
                url = ac_const.ADD_EXROUTES_URL
            elif operation == ac_const.OPER_DELETE:
                url = ac_const.REMOVE_EXROUTES_URL
        if res_type in [ac_const.NW_HW_TAP_SERVICE, ac_const.NW_HW_TAP_FLOW]:
            if operation == ac_const.OPER_CREATE:
                url = ac_const.NW_HW_FLOW_MIRROR_URL_PREFIX + res_type + 's'
            elif operation == ac_const.OPER_DELETE:
                url = ac_const.NW_HW_FLOW_MIRROR_URL_PREFIX + res_type + 's/' \
                      + res_id
        return url

    def _get_rest_data_and_rest_info(self, admin_ctx, thread_name,
                                     rec_list, sync_res_info):
        """get rest data and rest info"""
        ret = False
        method, operation, res_type = sync_res_info
        if operation != ac_const.OPER_DELETE:
            if res_type == ac_const.NW_HW_EXROUTE:
                (rest_data, rest_info) = \
                    self._form_bulk_rest_body_for_exroute(
                        admin_ctx, rec_list, operation)
            else:
                (rest_data, rest_info) = self._form_bulk_rest_body(
                    admin_ctx, rec_list, res_type, method)
            if rest_data is None:
                for rec_index in rec_list:
                    self._data_list[rec_index]['status'] = \
                        ac_const.STAT_ERR
                    self._statistic_neutron_sync_request(
                        res_type, operation, self._data_list[
                            rec_index],
                        ac_const.NW_HW_ERROR)
                ret = True
            data_info = jsonutils.loads(rest_data).get(res_type, [{}])[0]
            if data_info.get('device-owner') == 'trunk:subport' and \
                    operation == ac_const.OPER_CREATE:
                data_info['device-owner'] = ''
                if data_info.get('device-id'):
                    data_info['device-id'] = ''
                if data_info.get('host-id'):
                    data_info['host-id'] = ''
                trunk_port_info = {'res': res_type,
                                   'uuid': data_info.get('uuid'),
                                   'tenant-name': data_info.get('tenant-name'),
                                   'status': ac_const.STAT_WAIT}
                data_info = ACRestUtils.wrap_entry_info('POST', res_type,
                                                        [data_info])
                rest_data = ACRestUtils.fix_json(jsonutils.dumps(data_info))
                LOG.debug("%s: Convert data is: %s.", thread_name, rest_data)
                self._db_if.update_attribute_record(None,
                                                    trunk_port_info.get('uuid'),
                                                    res_type)
                self._update_data_list.append(trunk_port_info)
        else:
            if res_type == ac_const.NW_HW_EXROUTE:
                (rest_data, rest_info) = \
                    self._form_bulk_rest_body_for_exroute(admin_ctx, rec_list,
                                                          operation)
            else:
                rest_data = None
                rest_info = []
        return rest_data, rest_info, ret

    def _process_result_timeout(self, rec_list, res_type,
                                operation, thread_name):
        """process result timeout"""
        LOG.info('res_type: %s, thread_name: %s', res_type, thread_name)
        with self._lock:
            if res_type == self._continuous_timeout_type:
                if not rec_list[0] in self._continuous_timeout_res_list:
                    self._continuous_timeout_res_list.append(rec_list[0])
                    self._continuous_timeout_type_times += 1
                LOG.info(
                    'same res_type: %s, _continuous_timeout_type_times: %s',
                    res_type,
                    self._continuous_timeout_type_times)
                max_timeout_times = min(
                    ac_const.SYNC_TIMEOUT_TIMES_SINGLE_RES_TYPE,
                    len(self._data_list))
                if self._continuous_timeout_type_times >= max_timeout_times:
                    self._continuous_sync_next_type_flag = True
                    LOG.info('same res_type: %s, '
                             '_continuous_timeout_type_times: %s '
                             'has more than %s times, set Falg True',
                             res_type, self._continuous_timeout_type_times,
                             max_timeout_times)
                    return
            else:
                self._continuous_timeout_type = res_type
                self._continuous_timeout_type_times = 1
                LOG.info('new res_type: %s, _continuous_timeout_type_times: %s',
                         res_type, self._continuous_timeout_type_times)
        for rec in rec_list:
            self._statistic_neutron_sync_request(res_type, operation,
                                                 self._data_list[rec],
                                                 ac_const.NW_HW_TIMEOUT)

    def _process_result_success(self, admin_ctx, session,
                                thread_name, sync_res_data):
        """process result success"""
        operation, rec_list, res_type = sync_res_data
        with self._lock:
            device_id = None
            if operation == ac_const.OPER_CREATE and \
                    res_type == ac_const.NW_HW_PORTS:
                port_info = self._data_list[rec_list[0]]
                device_owner = port_info.get('device_owner')
                if device_owner == ncu.DEVICE_OWNER_ROUTER_GW:
                    device_id = port_info.get('device_id')

            if (operation == ac_const.OPER_DELETE and
                    res_type == ac_const.NW_HW_ROUTERS):
                router_from_ac = self._data_list[rec_list[0]]
                self._process_router_related_port(router_from_ac)

            self._continuous_timeout = 0
            for rec_index in rec_list:
                self._total_records_transfer += 1
                self._data_list[rec_index][
                    'status'] = ac_const.STAT_SENT
                try:
                    self._create_neutron_sync_plugin_record(
                        session, self._data_list[rec_index],
                        operation, res_type)
                except Exception as ex:
                    LOG.error(_LE("%s: Neutron sync create plugin "
                                  "data cause exception: %s"),
                              thread_name, ex)
                    self._statistic_neutron_sync_request(
                        res_type, operation,
                        self._data_list[rec_index],
                        ac_const.NW_HW_ERROR)
                    raise
                self._statistic_neutron_sync_request(
                    res_type, operation,
                    self._data_list[rec_index],
                    ac_const.NW_HW_SUCCESS)
        if not self._create_router_list:
            self._create_router_list = [r for r in self._create_data_list
                                        if r['res'] == ac_const.NW_HW_ROUTERS]
            LOG.debug('[AC] create router list: %s', self._create_router_list)
        if not self._update_router_list:
            self._update_router_list = [r for r in self._update_data_list
                                        if r['res'] == ac_const.NW_HW_ROUTERS]
            LOG.debug('[AC] update router list: %s', self._update_router_list)
        if device_id:
            self._process_with_device_id(admin_ctx, device_id, rec_list, session)

    def _process_result_error(self, sync_res_data, response,
                              session, thread_name):
        """process result error"""
        operation, rec_list, res_type = sync_res_data
        with self._lock:
            self._continuous_timeout = 0
        if not response:
            LOG.error(_LE('Neutron-sync request returns error '
                          'from controller and return content is empty.'))
            self._process_error_res_list(rec_list, res_type, operation)
            return

        ac_error_list = []
        try:
            ac_error_list = response['errors']['error']
        except KeyError:
            LOG.error(_LE('controller returns invalid format '
                          'data as sync_data response.'))
            self._process_error_res_list(rec_list, res_type, operation)
        for error in ac_error_list:
            if int(error['error-info']['error-code'], 16) != 0:
                _, _, uuid = error['error-path']. \
                    rpartition('/')
                LOG.error(_LE('Processing sync_data request in'
                              ' controller failed for uuid %s,'
                              ' error-code %s.'), uuid,
                          error['error-info']['error-code'])
                error_res_list = [rec_index for rec_index in rec_list if
                                  self._data_list[rec_index]['uuid'] == uuid]
                self._process_error_res_list(error_res_list, res_type, operation)
        for rec_index in rec_list:
            if self._data_list[rec_index]['status'] != ac_const.STAT_ERR:
                with self._lock:
                    self._total_records_transfer += 1
                    self._data_list[rec_index]['status'] = ac_const.STAT_SENT
                try:
                    self._create_neutron_sync_plugin_record(
                        session, self._data_list[rec_index],
                        operation, res_type)
                except Exception as ex:
                    LOG.error(_LE("%s: Neutron sync create plugin data cause "
                                  "exception: %s"), thread_name, ex)
                    self._statistic_neutron_sync_request(
                        res_type, operation, self._data_list[rec_index],
                        ac_const.NW_HW_ERROR)
                    raise
                self._statistic_neutron_sync_request(
                    res_type, operation, self._data_list[rec_index],
                    ac_const.NW_HW_SUCCESS)

    def _statistic_neutron_sync_request(self, res_type_input, method,
                                        res, result):
        """stastic neutron sync request"""
        res_type = res_type_input
        if res_type == ac_const.NW_HW_EXROUTE:
            exroute_id = "{'router_id':'%s', 'nexthop':'%s', 'destination':'%s','type':'%s'}" % (
                res['uuid'], res['nexthop'], res['destination'], res.get('type'))
            data_info = {'id': exroute_id,
                         'name': res.get('name', ac_const.EMPTY_STR),
                         'tenant_name': res['tenant-name'],
                         'created_at': res.get('created_at',
                                               ac_const.EMPTY_STR),
                         'updated_at': res.get('updated_at',
                                               ac_const.EMPTY_STR),
                         'result': result}
        else:
            data_info = {'id': res['uuid'],
                         'name': res.get('name', ac_const.EMPTY_STR),
                         'tenant_name': res['tenant-name'],
                         'created_at': res.get('created_at',
                                               ac_const.EMPTY_STR),
                         'updated_at': res.get('updated_at',
                                               ac_const.EMPTY_STR),
                         'result': result}

        if res_type == ac_const.NW_HW_PORTS and res.get('router_intf'):
            res_type = ac_const.NW_HW_ROUTER_IF

        if res_type == ac_const.NW_HW_DNAT and res.get('res') and \
                res.get('res') == ac_const.NW_HW_PF:
            res_type = ac_const.NW_HW_PF

        if res_type not in self._neutron_sync_result[method]:
            self._neutron_sync_result[method][res_type] = []

        # If the data exist in _neutron_sync_result, do not append it.
        for data_info_item in self._neutron_sync_result[method][res_type]:
            if data_info['id'] == data_info_item['id']:
                return

        self._neutron_sync_result[method][res_type].append(data_info)

    # Generate the bulk message body to send the data in bulk to AC, which
    # reduce the messages transferred across the AC and OpenStack. This data
    # must include sync:data option also to notify AC about the neutron-sync
    # data.
    def _form_bulk_rest_body(self, admin_ctx, rec_list, res_type, method):
        """Generate the bulk message body to send the data in bulk to AC for
        bulk-create
        :param rec_list: List of plugin data records to be merged.
        :param res_type: type of resource
        :return: rest body
        """
        rest_data = []
        length = len(self._data_list)
        for index in rec_list:
            if index >= length:
                LOG.error(_LE('index error in func _form_bulk_rest_body, '
                              'index %d, length %d.'), index, length)
                continue
            record = self._data_list[index]
            entry_info = self._get_formatted_data_for_neutron_sync(
                admin_ctx, record['uuid'],
                (record['tenant-name'], method, res_type))
            if entry_info:
                self.process_entry_info(entry_info, method, res_type)

                rest_data.append(entry_info)
        if not rest_data:
            return None, None
        rest_info = ACRestUtils.wrap_entry_info(method, res_type, rest_data)
        rest_data_copy = copy.deepcopy(rest_data)
        if res_type == ac_const.NW_HW_MEMBER:
            for member in rest_info['member']:
                del member['pool-id']

        return ACRestUtils.fix_json(jsonutils.dumps(rest_info)), rest_data_copy

    def _rollback_neutron_sync(self, session):
        """Finish the neutron-sync process by sending sync:end message to AC."""
        try:
            if self._sync_op == ac_const.SYNC_OP_COMPARE_DATA:
                return
            if not self._neutron_sync_status:
                return
            sync_result_records = copy.deepcopy(self._neutron_sync_result)

            self._cleanup_non_synced_records(session)
            neutron_id = self._db_if.get_local_neutron_id(session)
            self._util.clear_neutron_sync_records(session, neutron_id)
            self._util.clear_dependent_suspended_records(session,
                                                         res_type=None)
            LOG.info(_LI('Neutron sync completed by neutron server %s, records '
                         'transferred to AC is %d.'),
                     self.neutron_name, self._total_records_transfer)

            # if the operation is 'neutron_sync' or
            # 'neutron_sync_and_compare', need to compare the
            # data after sync. for delete, need to check if delete success.
            # after sync, compare the data, if the delete
            # data still exsit in AC. set the delete result as failuer.
            del_data_before_sync = copy.deepcopy(self._delete_data_records)
            if self._sync_op in [ac_const.SYNC_OP_SYNC_AND_COMPARE_DATA,
                                 ac_const.SYNC_OP_SYNC_DATA,
                                 ac_const.SYNC_OP_SYNC_SINGLE_INSTANCE]:

                # if the operation is sync,compare the data after sync
                # but not make report
                self._delete_data_records = []
                start_time = self._start_time
                if self._sync_op in [ac_const.SYNC_OP_SYNC_DATA,
                                     ac_const.SYNC_OP_SYNC_SINGLE_INSTANCE]:
                    self._enable_make_report = False

                delete_error_list = ACNeutronSync._check_delete_error_list(
                    del_data_before_sync, [])
                LOG.debug("Sync data error for delete request,%s",
                          delete_error_list)

                record_list = ["\n==========================================\n"]
                total_num, total_success = self._count_neutron_sync_result(
                    record_list, sync_result_records,
                    del_error_list=delete_error_list)
                self._db_if.update_server_record(
                    ACNeutronStateSchema(state=ac_const.NORMAL, sync_res=''),
                    session=session)
                if self._sync_op == ac_const.SYNC_OP_SYNC_AND_COMPARE_DATA:
                    self._sync_op = ac_const.SYNC_OP_COMPARE_DATA
                    self.perform_neutron_sync()

                # record the result in log file neutron_sync.log
                self._record_neutron_sync_result(
                    record_list, sync_result_records,
                    (total_num, total_success), start_time=start_time)
        except Exception as ex:
            LOG.error(_LE('Roll back neutron sync failed: %s.'), ex)
            fsa.ACPluginAlarm.send_neutron_sync_alarm(str(ex))
            raise
        finally:
            LOG.debug('Setting back the neutron sync state to NORMAL.')
            self._db_if.update_server_record(
                ACNeutronStateSchema(state=ac_const.NORMAL, sync_res=''),
                session)

    def _send_data_to_ac(self, rest_data, path, count, send_res_info):
        """send data to ac"""
        res_type, method, res_id = send_res_info
        time_out = cfg.CONF.huawei_ac_config.request_timeout - \
                   ac_const.REQUEST_TIMEOUT_DECREASE
        if count:
            time_out = count * time_out

        try:
            LOG.debug("[AC] res_type: %s,res_id: %s, method: %s",
                      res_type, res_id, method)
            if res_type == self._continuous_timeout_type and \
                    self._continuous_sync_next_type_flag:
                LOG.info("[AC] same res_type continue timeout. so do not "
                         "send to ac and  raise TimeoutException")
                raise requests.Timeout()
            ret = self._rest.send(method, '%s%s' % (self._url, path),
                                  res_id, rest_data, 0, time_out)
            res = {'res_type': res_type}
            if rest_data is None:
                rest_info = None
            else:
                rest_info = jsonutils.loads(rest_data).get(
                    res.get('res_type'), [{}])[0]
            return self.update_db_state(method, res, ret, rest_info)
        except requests.Timeout:
            LOG.error(_LE('AC neutron sync request timeout error,'
                          ' method: %(method)s, url: %(url)s'),
                      {'method': method,
                       'url': path})
            return ac_const.NW_HW_TIMEOUT, None
        except Exception as ex:
            LOG.exception(_LE('AC neutron sync request exception. %s'), str(ex))
        return ac_const.NW_HW_ERROR, None

    def update_db_state(self, method, res, ret, rest_info):
        """update db state"""
        res_code = ret.status_code
        res_json = jsonutils.loads(ret.content) if ret.content else None
        if requests.codes.ok <= res_code < requests.codes.multiple_choices:
            LOG.info(_LI('Send neutron sync request successfully and '
                         'AC process ok, response: %s'), res_json)
            if rest_info and not self._ac_rest_service. \
                    is_need_short_feedback(res, rest_info, method):
                LOG.debug("[AC] Long feedback need to update status "
                          " to down or active")
                if util.ACUtil.check_resource_have_state(res['res_type']):
                    status = self.get_res_status(res, rest_info)

                    session = self._db_if.get_session()
                    self._db_if.update_neutron_db_state(
                        session,
                        rest_info['uuid'],
                        res['res_type'],
                        status)

            return ac_const.NW_HW_SUCCESS, res_json
        else:
            LOG.error(_LE("Neutron sync data failed, controller "
                          "returns status code %s."), res_code)
            if rest_info and not self._ac_rest_service. \
                    is_need_short_feedback(res, rest_info, method):
                LOG.debug("[AC] Long feedback need to update status to error")
                if util.ACUtil.check_resource_have_state(res['res_type']):
                    status = ac_const.NEUTRON_STATUS_ERROR

                    session = self._db_if.get_session()
                    self._db_if.update_neutron_db_state(
                        session,
                        rest_info['uuid'],
                        res['res_type'],
                        status)
            return ac_const.NW_HW_ERROR, res_json

    def _send_gather_request_to_ac(self, sync_res_info,
                                   gather_type, header_data=None,
                                   need_log=True):
        """send gather request to ac"""
        rest_data, method, path, res_type = sync_res_info
        try:
            ret = self._rest.send(method, '%s%s' % (self._url, path),
                                  '', rest_data, max_times=0,
                                  headers=header_data, log_response=need_log)
            res_code = int(ret.status_code)

            if requests.codes.ok <= res_code < requests.codes.multiple_choices:
                LOG.info(_LI('Send neutron sync request successfully.'))
                return ret
            else:
                LOG.error(_LE('Neutron sync by neutron server %s failed, '
                              'controller failed to %s for %s.'),
                          self.neutron_name, gather_type, res_type)
                raise NeutronSyncError(reason='Controller failed in neutron '
                                              'sync %s.' % gather_type)
        except requests.Timeout:
            LOG.error(_LE('Neutron sync by neutron server %s failed, '
                          'controller timed out while %s for %s.'),
                      self.neutron_name, gather_type, res_type)
            raise NeutronSyncError(
                reason='Controller timed out in neutron sync %s.' % gather_type)
        return None

    def _clear_plugin_records_before_neutron_sync(
            self, session, res_type):
        """clear plugin record before neutron sync"""
        delete_count = 0
        update_count = 0
        rec_list = self._db_if.get_plugin_record_list(
            session, ACPluginSchema(neutron_id=-1, res_type=res_type))
        for record in rec_list:
            if res_type in ac_const.RESOURCE_IS_ASYNCHRONOUS:
                update_count += 1
                LOG.debug('Updating plugin record, seq_num:%d to suspend.',
                          record.seq_num)
                self._db_if.update_plugin_record(
                    session, ACPluginSchema(seq_num=record.seq_num,
                                            state=ac_const.SUSPEND))
            else:
                dependency = self._db_if.read_dependency(session,
                                                         record.seq_num)
                if dependency:
                    update_count += 1
                    LOG.debug('Updating plugin record, seq_num:%d to suspend.',
                              record.seq_num)
                    self._db_if.update_plugin_record(
                        session, ACPluginSchema(seq_num=record.seq_num,
                                                state=ac_const.SUSPEND))
                else:
                    dependency = self._db_if.read_dependency(
                        session, dep_seq_num=record.seq_num)
                    if dependency:
                        update_count += 1
                        LOG.debug('Updating plugin record, seq_num:%d to '
                                  'suspend.', record.seq_num)
                        self._db_if.update_plugin_record(
                            session, ACPluginSchema(seq_num=record.seq_num,
                                                    state=ac_const.SUSPEND))
                    else:
                        delete_count += 1
                        LOG.debug('deleting plugin record, seq_num:%d',
                                  record.seq_num)
                        self._db_if.delete_plugin_record(
                            session, seq_num=record.seq_num)
                        self._db_if.delete_dependency(record.seq_num)

    def single_ins_sync_prepare_data_list(self, session, ac_data_list,
                                          neutron_data_list, sync_res_info):
        """single instance sync prepare data list"""
        res_type, total_record_count = sync_res_info
        for entry_info in ac_data_list:
            LOG.debug('Adding entry to plugin for DELETE: %s', entry_info)
            # add name, create_time, update_time, tenant_name for
            # neutron sync output info
            delete_data = self._get_delete_data(entry_info, res_type)
            self._neutron_sync_delete_info(delete_data, entry_info, res_type)
            self._data_list.append(delete_data)
            total_record_count += 1

        for entry_info in neutron_data_list:
            if entry_info.get('uuid', None) in self.single_create_lists:
                entry_info['res'] = res_type
                entry_info['status'] = ac_const.STAT_WAIT
                LOG.debug('Adding entry to plugin for CREATE: %s', entry_info)

                if res_type == ac_const.NW_HW_PORTS:
                    self._neutron_sync_create_port(entry_info, session)

                self._create_data_list.append(entry_info)
            if entry_info.get('uuid', None) in self.single_update_lists:
                if res_type == ac_const.NW_HW_SEC_GRP_RULE:
                    continue
                self.single_ins_sync_update(entry_info, res_type)
        return total_record_count
