#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""normal sync"""

import ast
import threading
import time
import sched

from networking_huawei.drivers.ac.db.schema import ACPluginSchema, ACFailedResources
from oslo_config import cfg

try:
    from neutron import context
except ImportError:
    from neutron_lib import context

from networking_huawei._i18n import _LE
from networking_huawei._i18n import _LI
from networking_huawei.drivers.ac.client import service
from networking_huawei.drivers.ac.client.restclient import ACReSTClient
from networking_huawei.drivers.ac.common import fusion_sphere_alarm as fsa
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei.drivers.ac.db import dbif
from networking_huawei.drivers.ac.db.utcnow import utcnow
from networking_huawei.drivers.ac.sync import util
from networking_huawei.drivers.ac.sync import worker
from networking_huawei.drivers.ac.sync.validation import ACValidation
from networking_huawei.drivers.ac.sync import http_heart_beat
from networking_huawei.drivers.ac.db.dbif import ACdbInterface

LOG = logging.getLogger(__name__)


class ACNormalSync(object):
    """ACNormalSync"""

    def __new__(cls):
        """In case if the neutron state db is not ready, then we need to
        return error for the create request, so check it in the __new__
        """
        _db_if = dbif.ACdbInterface()
        _neutron_id = _db_if.get_local_neutron_id()
        if _neutron_id == -1:
            LOG.error(_LE("Neutron server is not initialized. Please "
                          "initialize it first."))
            return None
        _instance = super(ACNormalSync, cls).__new__(cls)
        _instance._db_if = _db_if
        _instance._neutron_id = _neutron_id
        return _instance

    def __init__(self):
        self.validation = ACValidation()
        self._rest_service = service.ACReSTService()
        self._rest_client = ACReSTClient()
        self._error_retry_count = cfg.CONF.huawei_ac_config.error_retry_count
        self._util = util.ACUtil()
        # Thread event to control the normal sync thread
        self._thread_access_time = time.time()
        self._monitor_thread_log_time = 0
        self._thread_event = threading.Event()
        self.http_heart_beat = http_heart_beat.HttpHeatBeat2AC()
        self._sync_thread_spawn_state = ac_const.SYNC_THREAD_POOL_NO_SPAWN
        self._thread_pool = worker.ACSyncThreadPool(
            ac_const.SYNC_THREAD_POOL_DEFAULT_COUNT)
        self._sync_thread = self.init_normal_sync()
        schedule = sched.scheduler(time.time, time.sleep)
        schedule.enter(0, 0, self._monitor_normal_sync_thread,
                       ())
        schedule.run()
        self.request_timeout_moment = {}
        LOG.debug("Normal sync thread initialization done.")

    def init_normal_sync(self):
        """Initialize the normal sync thread"""
        normal_sync = threading.Thread(target=self.normal_sync_process,
                                       name='normal_sync')
        LOG.info(_LI("Thread object is initialized."))
        normal_sync.start()
        return normal_sync

    def _monitor_normal_sync_thread(self):
        count_num = 0
        thread_stopped = False
        request_timeout = cfg.CONF.huawei_ac_config.request_timeout
        timeout_retry = cfg.CONF.huawei_ac_config.timeout_retry
        thread_monitor_timeout = request_timeout * (timeout_retry + 1) * 3
        try:
            if time.time() - self._monitor_thread_log_time > \
                    ac_const.SECONDS_EVRY_MINUTE * 5:
                self._monitor_thread_log_time = time.time()
                LOG.info("[AC] normal sync monitor time stamp:%s",
                         time.time())
            # check normal sync thread if running for two times
            # here check for two times, just to ensure the
            # thread stopped. if stopped, restart a new thread
            while count_num < 2:
                time_delta = time.time() - self._thread_access_time
                if self._sync_thread.is_alive() and \
                        time_delta < thread_monitor_timeout:
                    thread_stopped = False
                else:
                    status = 'alive' if self._sync_thread.is_alive() else 'dead'
                    LOG.error(_LE('[AC] Normal sync thread %s stop %ss, '
                                  'status: %s'),
                              self._sync_thread.ident,
                              thread_monitor_timeout,
                              status)
                    thread_stopped = True
                    time.sleep(ac_const.NORMAL_SYNC_THREAD_WAIT)
                count_num += 1

            if thread_stopped:
                self._sync_thread = threading.Thread(
                    target=self.normal_sync_process,
                    name='normal_sync')
                self._sync_thread.start()
                LOG.info(_LI("[AC] Normal sync thread %s is started"),
                         self._sync_thread.ident)
            # Timer for check the normal sync thread,
            # check every 5 seconds

        except Exception as ex:
            LOG.error(_LE("[AC] Monitor normal sync thread exception %s"),
                      ex)
        finally:
            timer = threading.Timer(ac_const.NORMAL_SYNC_THREAD_WAIT,
                                    self._monitor_normal_sync_thread)
            timer.start()

    def _check_heartbeat(self, force_sync):
        while True:
            LOG.info("[AC] normal sync heart beat time stamp: %s", time.time())
            if not self.http_heart_beat.is_ac_alive():
                LOG.error("[AC] controller is not reachable, wait for %s",
                          ac_const.SECONDS_EVRY_MINUTE)
                time.sleep(ac_const.SECONDS_EVRY_MINUTE)
                self._thread_access_time = time.time()
                force_sync = True
            else:
                LOG.info(_LI("Normal sync is running and the connection is ok"))
                sync_time = time.time()
                break
        return force_sync, sync_time

    def normal_sync_process(self, need_loop_run=None):
        """Basic processing of the normal sync thread"""
        LOG.debug("Inside thread operation.")
        self._thread_access_time = time.time()
        sync_time = 0
        force_sync = False
        exception_msg = None
        while True:
            try:
                if time.time() - sync_time > ac_const.SECONDS_EVRY_MINUTE:
                    force_sync, sync_time = self._check_heartbeat(force_sync)

                start_time = time.time()
                records_pending = self._sync_data_to_ac(force_sync)
                force_sync = False
                if records_pending:
                    # if any records are pending to be processed, then we just
                    # wait for a short time and retry the operation again.
                    time_diff = .5
                else:
                    end_time = time.time()
                    time_taken = end_time - start_time
                    # In case if the time is calculated as -ve
                    time_taken = 0 if time_taken < 0 else time_taken
                    # Calculate the time take for the current sync process
                    time_diff = ac_const.NORMAL_SYNC_THREAD_WAIT - time_taken
                if time_diff > 0:
                    LOG.debug("Normal sync thread wake-up for sync operation,"
                              "thread waiting for %f sec.", time_diff)
                    # Wait for the next trigger
                    self._thread_event.wait(time_diff)
                    self._thread_event.clear()
            except Exception as ex:
                # add this check
                # if db connection error,will throw exception, so here
                # the exception msg will be printed all the times, so
                # avoid this situation, add this check, fi the error msg is
                # the same, don't need to print all the time ,
                # just print it for a time every 60s
                if str(ex) != exception_msg or time.time() - sync_time \
                        > ac_const.SECONDS_EVRY_MINUTE:
                    LOG.exception(_LE("Exception while running normal "
                                      "sync thread: %s"), ex)
                    exception_msg = str(ex)
            # Check need loop run
            if need_loop_run:
                break

    def _check_dependency_with_optimise(self, plugin_data, session):
        """检查plugin_data是否有依赖，如果有依赖，并且等待超240s，解除依赖

        :param plugin_data: plugin_data
        :param session: session
        :return: 如果plugin_data有依赖，返回True
        """
        dependencies = self._db_if.read_dependency(session, plugin_data.seq_num)
        if dependencies:
            LOG.debug("Normal sync thread: Resource (Resource ID: %s, Resource Type: %s) is dependent, skip it now.",
                      plugin_data.res_uuid, plugin_data.res_type)
            now_time = session.execute(utcnow()).scalar()
            time_diff = (now_time - plugin_data.update_time).total_seconds()
            if time_diff >= ac_const.NORMAL_SYNC_DEPENDENCY_WAITING_TIME:
                for dependency in dependencies:
                    LOG.info("[AC] delete dependency if waiting for up to 240s, res_id: %s,"
                        " res_type: %s, seq_num:%s, dep_num:%s", plugin_data.res_uuid,
                             plugin_data.res_type, dependency.res_seq_num, dependency.dep_seq_num)
                    self._db_if.delete_dependency(rec=dependency)
            return True
        return False

    def _check_plugin_data_state(self, plugin_data, session, force_sync):
        """_sync_data_to_ac call:plugin_data.state"""
        if plugin_data.state == ac_const.ERROR_RETRY:
            now_time = session.execute(utcnow()).scalar()

            retry_cost = cfg.CONF.huawei_ac_config.error_retry_interval * \
                         ac_const.SECONDS_EVRY_MINUTE
            total_cost = (now_time - plugin_data.update_time).total_seconds()
            if (total_cost < retry_cost) and not force_sync:
                return True
            # In-case if user changed the retry count to higher
            # value, we will continue it and in the finally section
            # it will be caught and deleted.
            if plugin_data.retry_count >= self._error_retry_count:
                self._handle_max_retried_record(plugin_data, session)
                return True
            # Increment the retry count for error_retry case
            if force_sync:
                LOG.info("The ac is recovery,need to sync data to AC")
            plugin_data.retry_count += 1

        LOG.info(_LI("Queuing the record(Resource ID: %s, Resource Type"
                     ": %s, Times Retried: %d, Pending Retry: %d."),
                 plugin_data.res_uuid, plugin_data.res_type,
                 plugin_data.retry_count,
                 (self._error_retry_count - plugin_data.retry_count))
        return False

    def _update_state_and_retry_count(self, plugin_data, session):
        """Update the state and retry count"""
        rec = self._db_if.update_plugin_record(
            session, ACPluginSchema(seq_num=plugin_data.seq_num,
                                    state=ac_const.IN_PROCESS,
                                    retry_count=plugin_data.retry_count))
        if not rec:
            self._handle_max_retried_record(plugin_data, session)
            return True
        while True:
            if not self._thread_pool.task_queue.full():
                # Need the correct state in the worker beofre in process
                self._thread_pool.add_task(
                    self._send_data_to_ac, rec, plugin_data.state)
                break
            self._thread_event.wait(.1)
        return False

    def _sync_data_to_ac(self, force_sync=False):
        session = ACdbInterface().get_session('write')
        records_pending = False

        self._thread_access_time = time.time()
        result = self._check_and_join_or_spawn_threads(session)
        if result == ac_const.NW_HW_STOP_PROCESSING:
            LOG.debug("[AC] No record found in plugin db, return")
            return records_pending

        self._db_if.optimise_dependent_record()

        local_seq_num = ac_const.PLUGIN_INVALID_SEQ_NUM
        seq_len = self._hand_request_timeout_moment(session)
        while True:
            plugin_data = self._db_if.get_next_plugin_record(
                session, ACPluginSchema(seq_num=local_seq_num))
            if not plugin_data:
                LOG.debug("[AC] No record found in plugin db, break")
                break
            LOG.debug("[AC] sync data to ac, res_uuid:%s, seq_num:%s, res_type:%s, user_oper: %s, state: %s",
                      plugin_data.res_uuid, plugin_data.seq_num, plugin_data.res_type, plugin_data.user_oper,
                      plugin_data.state)
            # Updating the local sequence number to get next record
            local_seq_num = plugin_data.seq_num

            sync_res = self._util.get_current_neutron_sync_res(session)
            if sync_res:
                LOG.debug('get_current_neutron_sync_res returns res_type: %s',
                          sync_res)
            if sync_res == ac_const.NW_HW_COMPLETE:
                LOG.debug('Skipping normal-sync as consistency-check on '
                          'complete data in progress.')
                records_pending = True
                break

            # Sync applicable only for the WAIT and ERROR_RETRY states
            if plugin_data.state not in [ac_const.WAIT, ac_const.ERROR_RETRY]:
                if plugin_data.state in [ac_const.COMPLETE, ac_const.SUSPEND,
                                         ac_const.NEUTRON_SYNC]:
                    self._handle_ac_response_timeout(
                        session, plugin_data, seq_len)
                continue

            if sync_res == plugin_data.res_type:
                records_pending = True
                LOG.debug('Skipping normal-sync as neutron-sync '
                          'in-progress on res_type: %s', sync_res)
                continue

            # Check dependency and skip if dependent
            if self._check_dependency_with_optimise(plugin_data, session):
                records_pending = True
                continue

            if self._check_plugin_data_state(plugin_data, session, force_sync):
                continue

            # Update the state and retry count
            if self._update_state_and_retry_count(plugin_data, session):
                continue

        LOG.debug("Normal sync thread processing finished now.")
        return records_pending

    def _send_data_to_ac(self, thread_name, plugin_data, previous_state):
        LOG.debug("%s: Send data to AC initialized.", thread_name)
        self._thread_access_time = time.time()
        session = ACdbInterface().get_session('write')
        try:
            # In WAIT state the retry count is same as configured and for
            # ERROR_RETRY it is 1
            retry_value = cfg.CONF.huawei_ac_config.timeout_retry
            if previous_state == ac_const.ERROR_RETRY:
                # In case of error_retry only once need to retry
                LOG.error(_LE("state ERROR_RETRY, set retry_value 0."))
                retry_value = 0

            (method, resource_url) = \
                util.ACUtil.get_method_and_resource_url(
                    '%s_%s' % (plugin_data.user_oper,
                               plugin_data.res_type))
            pool_id = None
            if plugin_data.res_type == ac_const.NW_HW_MEMBER:
                pool_id = plugin_data.data.pop(ac_const.URL_POOL_ID, None)
            if plugin_data.res_type in [ac_const.NW_HW_INSERT_RULE,
                                        ac_const.NW_HW_REMOVE_RULE]:
                base_url = ac_const.NW_HW_RESTFUL_V3
                url = util.ACUtil.form_restful_url(
                    base_url, method, resource_url, plugin_data.res_uuid)
                plugin_data.data.pop('created-at')
                plugin_data.data.pop('updated-at')
            else:
                url = util.ACUtil.form_resource_url(ac_const.NW_HW_URL,
                                                    method,
                                                    resource_url,
                                                    plugin_data.res_uuid,
                                                    pool_id=pool_id)
            if plugin_data.res_type == ac_const.NW_HW_EXROUTE:
                if plugin_data.user_oper == 'create':
                    url = ac_const.ADD_EXROUTES_URL
                elif plugin_data.user_oper == 'delete':
                    url = ac_const.REMOVE_EXROUTES_URL

            self.update_fw_enabled(plugin_data)

            rest_paras = service.RequestServiceParas(
                session, plugin_data.seq_num, plugin_data.res_type,
                method, url, plugin_data.res_uuid, plugin_data.data)
            # send using the rest service
            self._rest_service. \
                request_send_service(rest_paras, max_times=retry_value)
            LOG.debug("%s : Finished sending data to AC.", thread_name)
        # Catching exception here to avoid, not updating the db due to
        # exceptions
        except Exception as ex:
            LOG.error(_LE('Failed to send data to ac: %s'), ex)
            raise ex
        finally:
            self._handle_max_retried_record(plugin_data, session)

    @staticmethod
    def update_fw_enabled(plugin_data):
        """ update fw enabled in profile """
        if plugin_data.res_type == ac_const.NW_HW_PORTS and \
                plugin_data.data.get("device-owner") == \
                "network:router_interface" and not ncu.IS_FSP:
            network = ncu.get_core_plugin().get_network(
                context.get_admin_context(),
                plugin_data.data.get("network-id"))
            if network and network.get('router:external'):
                profile_json = ast.literal_eval(plugin_data.data["profile"])
                if not profile_json.get('fw_enabled'):
                    profile_json.update({'fw_enabled': False})
                    plugin_data.data["profile"] = str(profile_json)

    def _handle_max_retried_record(self, plugin_data, session):
        """If maximum retry is reached, then delete the record
        and log it
        """
        if int(plugin_data.retry_count) and \
                int(plugin_data.retry_count) >= self._error_retry_count:
            LOG.error(_LE("Maximum retry reached for the record. "
                          "Deleting the resource(Resource ID: %s)."),
                      plugin_data.res_uuid)
            # If maximum retry is reached then delete the resource
            # and set the state in neutron-db as ERROR
            self._util.delete_one_record_in_plugin_db(
                session, plugin_data.seq_num, on_error=True, new_session=True)

    def _check_and_join_or_spawn_threads(self, session):
        """check and join or spawn threads"""
        plugin_data = self._db_if.get_next_plugin_record(
            session, ACPluginSchema(seq_num=ac_const.PLUGIN_INVALID_SEQ_NUM))
        if not plugin_data:
            # If there is no records for 2 times, then we will kill the
            # sync threads spawned to release the resource.
            if self._sync_thread_spawn_state != \
                    ac_const.SYNC_THREAD_POOL_NO_SPAWN:
                # Increment the spawn counter each time to reach max, so that
                # the threads will be killed
                self._sync_thread_spawn_state += 1
                LOG.debug("New sync thread spawn counter is %d.",
                          self._sync_thread_spawn_state)
                # If reached max retry, kill the threads
                if self._sync_thread_spawn_state == \
                        ac_const.SYNC_THREAD_POOL_MAX_TRY:
                    self._sync_thread_spawn_state = \
                        ac_const.SYNC_THREAD_POOL_NO_SPAWN
                    LOG.debug("Joining sync threads after job finished.")
                    self._thread_pool.clear_task()
            return ac_const.NW_HW_STOP_PROCESSING

        # If threads are not spawned, spawn it
        if self._sync_thread_spawn_state == \
                ac_const.SYNC_THREAD_POOL_NO_SPAWN:
            # Increment the state count before spawning the threads
            self._sync_thread_spawn_state += 1
            # There is some data to sync, so spawn threads to do the job
            LOG.debug("Spawning sync threads for the new job.")
            self._thread_pool.spawn_threads(
                ac_const.SYNC_THREAD_POOL_DEFAULT_COUNT)
        else:
            # Always re-initialise the counter if some record is found.
            self._sync_thread_spawn_state = 1
        return ac_const.NW_HW_SUCCESS

    def _handle_ac_response_timeout(self, session, plugin_data, seq_len):
        LOG.debug("[AC] Begin to handle ac response timeout")
        if plugin_data.state in [ac_const.SUSPEND]:
            (status, _) = self._db_if.check_is_neutron_sync_in_progress(
                session)
            if status:
                return
        now_time = session.execute(utcnow()).scalar()
        time_diff = (now_time - plugin_data.update_time).total_seconds()

        if plugin_data.seq_num not in self.request_timeout_moment:
            self.request_timeout_moment[plugin_data.seq_num] = {
                'timeout_increase': seq_len * ac_const.TIME_NEED_PER_SEQ, 'update_time': plugin_data.update_time}

        adjust_resp_time = cfg.CONF.huawei_ac_config.ac_response_time + \
                           self.request_timeout_moment[plugin_data.seq_num]['timeout_increase']

        if time_diff > adjust_resp_time or time_diff > ac_const.SECONDS_EVRY_MINUTE * 60:
            LOG.error(_LE("Don't receive %s status from AC "
                          "%s s,id is %s, delete the data in plugin "
                          "db and set the status as error in neutron db"),
                      plugin_data.res_type,
                      time_diff, plugin_data.res_uuid)

            if plugin_data.res_type == ac_const.NW_HW_PORTS:
                self._db_if.create_or_update_failed_resource(
                    ACFailedResources(id=plugin_data.res_uuid, res_type=ac_const.NW_HW_PORTS,
                                      operation='%s_%s' % (plugin_data.user_oper, ac_const.NW_HW_PORTS)),
                    False, session)
                self.deal_dependencies(session, plugin_data.seq_num)
                self.deal_port_alarm(plugin_data.res_uuid)
            self._util.delete_one_record_in_plugin_db(
                session, plugin_data.seq_num, on_error=True, new_session=True)

    @classmethod
    def deal_port_alarm(cls, port_id):
        """ deal port alarm """
        error_reason = "Websocket DisConnected Or TimeOut"
        port_alarm = fsa.ACPluginAlarm.get_port_alarm_info(
            fsa.ALARM_TYPE_GENERATE, port_id, error_reason)
        LOG.error(_LE("deal_port_alarm: %s"), port_alarm)
        fsa.ACPluginAlarm.send_alarm(port_alarm)

    def deal_dependencies(self, session, seq_num):
        """deal dependencies"""
        dependencies = self._db_if.read_dependency(
            session, dep_seq_num=seq_num)
        if not dependencies:
            return
        for dependency in dependencies:
            depend = self._db_if.get_plugin_record(
                session, dependency.res_seq_num)
            if not depend:
                continue
            self._db_if.create_or_update_failed_resource(
                ACFailedResources(id=depend.get('res_uuid'), res_type=depend.get('res_type'),
                                  operation='%s_%s' % (depend.get('user_oper'), depend.get('res_type'))),
                False, session)

    def _hand_request_timeout_moment(self, session):
        LOG.debug("[AC] Begin to handle request timeout moment")
        ac_process_seq = self._db_if.get_ac_process_req()
        now_time = session.execute(utcnow()).scalar()

        for k in ac_process_seq:
            if k not in self.request_timeout_moment:
                self.request_timeout_moment[k] = ac_process_seq[k]
        new_request_timeout_moment = {}
        for key, value in self.request_timeout_moment.items():
            time_diff = (now_time - self.request_timeout_moment.get(key, {}).get('update_time')).total_seconds()
            if time_diff > ac_const.SECONDS_EVRY_MINUTE * 60:
                LOG.debug("[AC]seq %s is more than 3600 in mem,del it", key)
                continue
            new_request_timeout_moment.update({key: value})
        self.request_timeout_moment = new_request_timeout_moment
        return len(ac_process_seq)
