#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""AC db Interface"""

import copy
import datetime

import sqlalchemy as sa
from sqlalchemy import asc
from sqlalchemy import func

from neutron.db import models_v2
from neutron.db import securitygroups_db
from oslo_config import cfg

try:
    # api for fusionsphere
    from oslo.db import api as oslo_db_api
except ImportError:
    # api for openstack
    from oslo_db import api as oslo_db_api
try:
    from neutron.common import rpc as n_rpc
except ImportError:
    from neutron_lib import rpc as n_rpc

from neutron.plugins.ml2 import models
try:
    from neutron.db import api as db_api
except ImportError:
    from neutron_lib.db import api as db_api

try:
    from neutron.db.l3_db import Router, FloatingIP
except ImportError:
    from neutron.db.models.l3 import Router, FloatingIP

try:
    from neutron import context
except ImportError:
    from neutron_lib import context

try:
    from neutron_fwaas.db.firewall import firewall_db as fw_db
except ImportError:
    try:
        from neutron.db.firewall import firewall_db as fw_db
    except ImportError:
        # ignore firewall db import error
        pass
try:
    from neutron_vpnaas.db.vpn import vpn_models as vpn_db
except ImportError:
    try:
        from neutron_vpnaas.db.vpn import vpn_db as vpn_db
    except ImportError:
        try:
            from neutron.db.vpn import vpn_db as vpn_db
        except ImportError:
            # ignore VPN db import error
            pass

from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.common.neutron_compatible_util import ac_log as logging
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.drivers.ac.db import schema as dbschema
from networking_huawei.drivers.ac.db.dnat.dnat import DNAT
from networking_huawei.drivers.ac.db.l2br.l2br_db import L2brDbMixin
from networking_huawei.drivers.ac.db.utcnow import utcnow
from networking_huawei.drivers.ac.db.vpc_connection.vpc_connection_db import VpcConnectionDbMixin

LOG = logging.getLogger(__name__)


class ACdbInterfaceCommon(object):
    """AC db Interface common util"""

    @classmethod
    def get_session(cls, operation='read'):
        """get session"""
        if ncu.get_ops_version() in [ac_const.OPS_R, ac_const.OPS_T, ac_const.OPS_W, ac_const.FSP_21_0]:
            from neutron_lib.db import api as neutron_db_api
        else:
            from neutron.db import api as neutron_db_api
        if ncu.get_ops_version() in [ac_const.OPS_Q, ac_const.OPS_R, ac_const.OPS_T, ac_const.OPS_W, ac_const.FSP_6_5,
                                     ac_const.FSP_21_0]:
            if operation == 'read':
                return neutron_db_api.get_reader_session()
            elif operation == 'write':
                return neutron_db_api.get_writer_session()
            LOG.error(_LE("[AC]Get session failed. Can not identify parameter operation: %s"), operation)
            raise TypeError("Wrong parameter income to get_session")
        return neutron_db_api.get_session()

    # Get the plugin db record, Mostly used to query one record to compare with some other record.
    def get_plugin_record(self, session, seq_num=None):
        """Query a single plugin record from db.

        :param session:     Current session info where the operation is performed. Used to access db.
        :param seq_num:     Sequence number of a record in the plugin table to query the record.

        :return: Single plugin record
        """

        if not session:
            session = self.get_session()
        if seq_num:
            return session.query(dbschema.ACPluginSchema).filter_by(seq_num=seq_num).first()

        return None

    # Get the plugin db record count
    def get_plugin_record_count(self, session):
        """Query a single plugin record from db.

        :param session:     Current session info where the operation is performed. Used to access db.
        :return: ACPluginSchema record count
        """

        if not session:
            session = self.get_session()
        return session.query(func.count(dbschema.ACPluginSchema.seq_num)).filter(
            dbschema.ACPluginSchema.state.in_([ac_const.COMPLETE, ac_const.NEUTRON_SYNC, ac_const.IN_PROCESS])).scalar()

    # Get the plugin db record of l47 count
    def get_plugin_record_l47_count(self, session):
        """Query a single plugin record of l47 from db.

        :param session:     Current session info where the operation is performed. Used to access db.
        :return: ACPluginSchema record count
        """

        if not session:
            session = self.get_session()
        return session.query(func.count(dbschema.ACPluginSchema.seq_num)).filter(
            dbschema.ACPluginSchema.state.in_([ac_const.COMPLETE, ac_const.NEUTRON_SYNC, ac_const.IN_PROCESS])).filter(
            dbschema.ACPluginSchema.res_type.in_(ac_const.NW_HW_L47_LIST)).scalar()

    def get_neutron_sync_record(self, sync_id=None, session=None):
        """get neutron sync record"""
        if not session:
            session = self.get_session()

        if sync_id:
            return session.query(dbschema.ACNeutronSyncSchema).filter_by(sync_id=sync_id).first()

        records = session.query(dbschema.ACNeutronSyncSchema).all()
        if records:
            return records[-1]
        return None

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def create_neutron_sync_record(self, sync_start_time, expected_completion_time, hostname='', session=None):
        """create neutron sync record"""
        if not session:
            session = self.get_session('write')

        if not sync_start_time:
            LOG.error(_LE("The sync start time is not specified."))
            raise AssertionError("The sync start time is not specified.")
        if not expected_completion_time:
            LOG.error(_LE("The completion time is not specified."))
            raise AssertionError("The completion time is not specified.")

        record = dbschema.ACNeutronSyncSchema(
            sync_start_time=sync_start_time, expected_completion_time=expected_completion_time, hostname=hostname)
        session.add(record)
        session.flush()
        LOG.debug("Added neutron sync record for start time: %s, completion time: %s", sync_start_time,
                  expected_completion_time)
        return record

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def update_neutron_sync_record(self, expected_completion_time=None, sync_end_time=None, sync_id=None, session=None):
        """update neutron sync record"""
        if not session:
            session = self.get_session('write')

        if not expected_completion_time and not sync_end_time:
            LOG.error(_LE("The completion time is not specified."))
            raise AssertionError("The completion time is not specified.")

        if sync_id:
            record = session.query(dbschema.ACNeutronSyncSchema).filter_by(sync_id=sync_id).first()
        else:
            records = session.query(dbschema.ACNeutronSyncSchema).all()
            if records:
                record = records[-1]
            else:
                return None

        if expected_completion_time:
            LOG.debug("Updating neutron sync %s completion time from %s to %s", record.sync_id,
                      record.expected_completion_time, expected_completion_time)
            record.expected_completion_time = expected_completion_time
        if sync_end_time:
            record.sync_end_time = sync_end_time
            LOG.debug("Updating neutron sync %s end time to %s", record.sync_id, sync_end_time)
        session.merge(record)
        session.flush()

        return record

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def create_config_record(self, key, value, comment=None, session=None):
        """Insert the config record into db.

        :param key:         Key of a record in the config table to be created.
        :param value:       Value of a record in the config table to be created.
        :param comment:     Comment of a record in the config table to be created.
        :param session:     Current session info where the operation is performed. Used to access db.
        :return: None
        """

        if not session:
            session = self.get_session('write')

        if not key:
            LOG.error(_LE("The record in config table should have a key."))
            raise AssertionError("Key is not specified in config record")

        rec = dbschema.ACConfigSchema(key=key, value=value, comment=comment)
        session.add(rec)
        session.flush()
        LOG.debug("Added config record for key:%s,value:%d,comment:%s", key, value, comment)
        return rec

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def update_config_record(self, key, value, comment=None, session=None):
        """Update the config record into db.

        :param key:         Key of a record in the config table to be updated.
        :param value:       Value of a record in the config table to be updated.
        :param comment:     Comment of a record in the config table to be updated.
        :param session:     Current session info where the operation is performed. Used to access db.
        :return: None
        """

        if not session:
            session = self.get_session('write')

        if not key:
            LOG.error(_LE("The record in config table should have a key."))
            raise AssertionError("Key is not specified in config record")

        rec = session.query(dbschema.ACConfigSchema).filter_by(key=key).first()

        if rec:
            if value is not None:
                LOG.debug("Updating config record %s from %d to %d", rec.key, rec.value, value)
                rec.value = value

            if comment is not None:
                LOG.debug("Updating config record %s from %s to %s", rec.key, rec.comment, comment)
                rec.comment = comment

            session.merge(rec)
            session.flush()
        else:
            rec = self.create_config_record(key, value, comment, session)

        return rec

    def get_config_record(self, key, session=None):
        """Query a single config record from db.

        :param key:         Key of a record in the config table to query the record.
        :param session:     Current session info where the operation is performed. Used to access db.
        :return: Single config record
        """

        if not session:
            session = self.get_session()
        if key:
            return session.query(dbschema.ACConfigSchema).filter_by(key=key).first()
        return None

    # Delete the records from the config table.
    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=1, retry_on_deadlock=True)
    def delete_config_record(self, key, session=None):
        """Delete config records from db based on the conditions.

        :param session:   Current session info where the operation is performed. Used to access db.
        :param seq_num:   Sequence number of a record in the config table as a condition to delete the records.
        :param rec:       Instead of sequence number user can provide the complete record also as a condition to delete.
        :return: None
        """
        if not session or ncu.get_ops_version() in [ac_const.FSP_6_5]:
            db_session = self.get_session('write')
        else:
            db_session = session

        rec_in_db = db_session.query(dbschema.ACConfigSchema).filter_by(key=key).first()

        if rec_in_db:
            LOG.debug("Deleting the config record for key:%s.", rec_in_db.key)
            db_session.delete(rec_in_db)
            db_session.flush()

    def get_duplicate_plugin_record(self, session, res_id, operation, res_type):
        """Query a single security group or security group rule create plugin record from db.

        :param session:    Current session info where the operation is performed. Used to access db.
        :param res_id:     Resource uuid in which the user performed the operation.
        :param operation:  The operation which user has performed, such as create, update or delete.
        :param res_type:   The type of the resource on which the user operation is performed.
        :return: Single plugin record
        """
        if not session:
            session = self.get_session()

        securitygroups = [ac_const.NW_HW_SEC_GRP, ac_const.NW_HW_SEC_GRP_RULE]
        if res_type not in securitygroups or operation != ac_const.OPER_CREATE:
            return None

        if res_id:
            return session.query(dbschema.ACPluginSchema).filter_by(
                res_uuid=res_id, res_type=res_type, user_oper=operation).first()
        return None

    # Get the server record, to check the own server state for neutron-sync
    def get_server_record(self, session=None, neutron_id=None, neutron_name=None):
        """Get the neutron server state record.

        :param session: Current session info where the operation is performed.Used to access db.
        :param neutron_id: Neutron id to identify the record.
        :param neutron_name: Neutron name to identify the record.
        :return: Single server record
        """
        if not session:
            session = self.get_session()

        if neutron_id:
            return session.query(dbschema.ACNeutronStateSchema).filter_by(neutron_id=neutron_id).first()

        if neutron_name:
            return session.query(dbschema.ACNeutronStateSchema).filter_by(neutron_name=neutron_name).first()
        return None

    # Delete the records from the plugin table.
    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=1, retry_on_deadlock=True)
    def delete_plugin_record(self, session, seq_num=None, rec=None, new_session=False):
        """Delete plugin records from db based on the conditions.

        :param session:  Current session info where the operation is performed. Used to access db.
        :param seq_num:  Sequence number of a record in the plugin table as a condition to delete the records.
        :param rec:      Instead of sequence number user can provide the complete record also as a condition to delete.
        :return: None
        """
        if new_session or not session or ncu.get_ops_version() in [ac_const.FSP_6_5]:
            db_session = self.get_session('write')
        else:
            db_session = session

        if seq_num:
            db_seq_num = seq_num
        if rec:
            db_seq_num = rec.seq_num

        rec_in_db = db_session.query(dbschema.ACPluginSchema).filter_by(seq_num=db_seq_num).first()

        if rec_in_db:
            LOG.debug("Deleting the plugin record for seq num:%d,resource id:%s,operation:%s,state:%s,"
                      "on neutron server %d",
                      rec_in_db.seq_num, rec_in_db.res_uuid, rec_in_db.user_oper, rec_in_db.state, rec_in_db.neutron_id)
            db_session.delete(rec_in_db)
            db_session.flush()
            return True
        return False

    # Get all neutron server list to check the alive status
    def get_server_record_list(self, session, state=None):
        """Get the neutron server state record.

        :param session: Current session info where the operation is performed.Used to access db.
        :param state: State of the record to be queried with
        :return: Server record list
        """

        if not session:
            session = self.get_session()

        query_condition = {}
        if state:
            query_condition['state'] = state

        return session.query(dbschema.ACNeutronStateSchema).filter_by(**query_condition).all()

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def create_dependency(self, session, res_seq_num, dep_seq_num):
        """Insert the dependency record into db.

        :param session: Current session info where the operation is performed.Used to access db.
        :param res_seq_num: Current resource seq_number, will get from the plugin table.
        :param dep_seq_num: Sequence number of the plugin record which it is dependent on.
        :return: None
        """

        session = self.get_session('write')
        LOG.debug("Creating dependency for resource:%s with resource:%s", res_seq_num, dep_seq_num)
        rec = dbschema.ACValidationSchema(res_seq_num=res_seq_num, dep_seq_num=dep_seq_num)
        session.add(rec)
        session.flush()

    def read_dependency(self, session, res_seq_num=None, dep_seq_num=None):
        """Read the dependency from the db.

        :param session: Current session info where the operation is performed.Used to access db.
        :param res_seq_num: Current resource seq_number, will get from the plugin table.
        :param dep_seq_num: Dependent resource seq_number, will get from the plugin table.
        :return: Dependency record list
        """
        session = self.get_session()
        if res_seq_num:
            return session.query(dbschema.ACValidationSchema).filter_by(res_seq_num=res_seq_num).all()

        if dep_seq_num:
            return session.query(dbschema.ACValidationSchema).filter_by(dep_seq_num=dep_seq_num).all()

        return session.query(dbschema.ACValidationSchema).all()

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=1, retry_on_deadlock=True)
    def delete_dependency(self, dep_seq_num=None, rec=None):
        """Delete the dependency from the db using the dependent sequence number.

        :param session: Current session info where the operation is performed.Used to access db.
        :param dep_seq_num: Dependent resource seq_number to delete the dependency. all the record with this
                            dependency will be deleted.
        :return: None
        """

        db_session = self.get_session('write')

        if dep_seq_num:
            num = dep_seq_num
        if rec:
            num = rec.dep_seq_num

        query_condition = {'dep_seq_num': num}
        if rec and rec.res_seq_num:
            query_condition['res_seq_num'] = rec.res_seq_num

        db_session.query(dbschema.ACValidationSchema).filter_by(**query_condition).delete()
        db_session.flush()

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def optimise_dependent_record(self):
        """optimise dependent record"""
        LOG.debug("[AC] Begin to optimise dependent record.")
        sql_del = "delete from huawei_ac_validation where dep_seq_num not in (select seq_num from huawei_ac_plugin);"
        session = self.get_session('write')
        with session.begin(subtransactions=True):
            LOG.info(_LI("[AC] Begin to delete validation records"))
            session.execute(sql_del)

    def check_is_neutron_sync_in_progress(self, session):
        """check is neutron sync in progress"""
        if not session:
            session = self.get_session()
        query_cond = (dbschema.ACNeutronStateSchema.state.in_([ac_const.NEUTRON_SYNC, ac_const.CONSISTENCY_CHECK]),)
        server = session.query(dbschema.ACNeutronStateSchema).filter(*query_cond).first()
        if server:
            return True, server
        return False, None


class ACdbInterfaceUtil(ACdbInterfaceCommon):
    """AC db Interface util"""

    def __init__(self):
        self.neutron_name = ACCommonUtil.get_neutron_server_name()
        self.vpc_conn_db = VpcConnectionDbMixin()
        self.l2br_db = L2brDbMixin()

    def get_alive_server_record_list(self, session):
        """Get all alive neutron server list except self
        :param session: Current session info where the operation is performed.Used to access db.
        :param is_leader: to query all leader records
        :return: Server record list
        """

        if not session:
            session = self.get_session()

        query_condition = (dbschema.ACNeutronStateSchema.state != ac_const.ERROR,)

        rec_list = session.query(dbschema.ACNeutronStateSchema).filter(*query_condition).order_by(
            asc(dbschema.ACNeutronStateSchema.create_time)).all()
        return rec_list

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def create_server_record(self, session=None):
        """Create the neutron server state record which is used for neutron
            status updation. neutron_id will be allocated automatically.

        :param session: Current session info where the operation is performed.Used to access db.
        :return: None
        """
        LOG.debug("Creating neutron server state record for neutron : %s", self.neutron_name)
        if not session:
            session = self.get_session('write')

        create_time = session.execute(utcnow()).scalar()
        rec = dbschema.ACNeutronStateSchema(neutron_name=self.neutron_name, create_time=create_time,
                                            sync_type=cfg.CONF.huawei_ac_config.neutron_sync_type)
        session.add(rec)
        session.flush()

    def get_dead_neutron_server_list(self, session):
        """ Find and update dead servers. And return need_neutron_sync
        if any dead server in NEUTRON_SYNC identified as ERROR
        :param session:
        :return: need_neutron_sync
        """
        query_condition = ()
        if not session:
            session = self.get_session()
        now = session.execute(utcnow()).scalar()
        timedelta_s = datetime.timedelta(
            minutes=ac_const.SERVER_ALIVE_CHECK_INTERVAL)

        query_condition += (dbschema.ACNeutronStateSchema.neutron_name != self.neutron_name,
                            dbschema.ACNeutronStateSchema.state != ac_const.ERROR,
                            dbschema.ACNeutronStateSchema.access_time < (now - timedelta_s),)

        server_list = session.query(dbschema.ACNeutronStateSchema).filter(*query_condition).all()
        return server_list

    def check_is_leader(self, session):
        """check is leader"""
        if not session:
            session = self.get_session()
        query_condition = (dbschema.ACNeutronStateSchema.neutron_name == self.neutron_name,
                           dbschema.ACNeutronStateSchema.is_leader == sa.true(),)
        leader = session.query(dbschema.ACNeutronStateSchema).filter(*query_condition).first()
        return bool(leader)

    def get_local_neutron_id(self, session=None):
        """Get the local neutron server id.

        :param session: Current session info where the operation is performed.Used to access db.
        :return: Local neutron server id
        """
        if not session:
            session = self.get_session()

        neutron_serv = self.get_server_record(session, neutron_name=self.neutron_name)
        if neutron_serv:
            return neutron_serv.neutron_id
        return -1

    # If user changed the neutron name and restarted the neutron server, then a new record will be added, but the old
    # one will still remain in the db for some time, and after a period of time, this has to be deleted as it is not
    # useful. So we use this API to delete the neutron server state record in this case.
    @classmethod
    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def delete_server_record(cls, session, neutron_id=None, rec=None):
        """Delete the neutron server state record.

        :param session: Current session info where the operation is performed.Used to access db.
        :param neutron_id: Neutron id to identify the record.
        :param rec: Instead of neutron_id user can provide the complete record also as a condition to delete.
        :return: None
        """
        if neutron_id:
            rec = session.query(dbschema.ACNeutronStateSchema).filter_by(neutron_id=neutron_id).first()

        if rec:
            LOG.debug("Deleting server state record for neutron id:%d and neutron name:%s", rec.neutron_id,
                      rec.neutron_name)
            session.delete(rec)
            session.flush()

    @classmethod
    def get_neutron_read_dict(cls):
        """get neutron read dict.

        :return: None
        """
        if hasattr(securitygroups_db, 'SecurityGroup'):
            ncu.NEUTRON_READ_DICT.update({ac_const.NW_HW_SEC_GRP: securitygroups_db.SecurityGroup})
        else:
            try:
                from neutron.db.models.securitygroup import SecurityGroup
                ncu.NEUTRON_READ_DICT.update({ac_const.NW_HW_SEC_GRP: SecurityGroup})
            except Exception as ex:
                LOG.error(_LE('[AC] SecurityGroup import failed: %s'), ex)

        if hasattr(securitygroups_db, 'SecurityGroupRule'):
            ncu.NEUTRON_READ_DICT.update({ac_const.NW_HW_SEC_GRP_RULE: securitygroups_db.SecurityGroupRule})
        else:
            try:
                from neutron.db.models.securitygroup import SecurityGroupRule
                ncu.NEUTRON_READ_DICT.update({ac_const.NW_HW_SEC_GRP_RULE: SecurityGroupRule})
            except Exception as ex:
                LOG.error(_LE('[AC] SecurityGroupRule import failed: %s'), ex)
        cls._update_neutron_dict_with_vpn_fiwrewall()
        try:
            from neutron.db.qos import models as qos_db_model
            ncu.NEUTRON_READ_DICT[ac_const.NW_HW_QOS_POLICY] = qos_db_model.QosPolicy
        except ImportError:
            LOG.debug('Unable to read QOS policy db.')

    @classmethod
    def _update_neutron_dict_with_vpn_fiwrewall(cls):
        try:
            # ignore VPN db import error
            ncu.NEUTRON_READ_DICT.update({
                ac_const.NW_HW_IKE_POLICY: vpn_db.IKEPolicy,
                ac_const.NW_HW_IPSEC_POLICY: vpn_db.IPsecPolicy,
                ac_const.NW_HW_VPN_SERVICE: vpn_db.VPNService,
                ac_const.NW_HW_IPSEC_SITE_CONNECTION: vpn_db.IPsecSiteConnection})
        except Exception as ex:
            LOG.info(_LI('[AC] VPN plugin not configured: %s'), ex)
        try:
            # ignore endpoint group import error
            ncu.NEUTRON_READ_DICT.update({ac_const.NW_HW_EP_GROUPS: vpn_db.VPNEndpointGroup})
        except Exception as ex:
            LOG.info(_LI('[AC] Endpoint group not configured: %s'), ex)
        try:
            # ignore firewall db import error
            ncu.NEUTRON_READ_DICT.update({
                ac_const.NW_HW_FIREWALL: fw_db.Firewall,
                ac_const.NW_HW_FIREWALL_RULE: fw_db.FirewallRule,
                ac_const.NW_HW_FIREWALL_POLICY: fw_db.FirewallPolicy})
        except Exception as ex:
            LOG.info(_LI('[AC] Firewall not configured: %s'), ex)

    def get_neutron_record(self, session, res_id, res_type, status=None):
        """Get the local neutron server id.

        :param session: Current session info where the operation is performed.Used to access db.
        :param res_id:  Resource id of the data to identify.
        :param res_type: Type of the resource, such as router, network, etc.
        :param status: Status of the record.
        :return: Neutron db record
        """
        if not res_id and not res_type:
            LOG.debug("Update neutron API has been called with invalid res_id, res_type inputs")
            return []

        if not session:
            session = self.get_session()
        if res_type not in ncu.NEUTRON_READ_DICT:
            return []
        query_condition = {}
        res_model = ncu.NEUTRON_READ_DICT.get(res_type)
        if res_id:
            if res_type == ac_const.NW_HW_ROUTER_EXTRA_ATTRIBUTES:
                query_condition['router_id'] = res_id
            else:
                query_condition['id'] = res_id
            rec = session.query(res_model).filter_by(**query_condition).first()
        else:
            if status:
                query_condition['status'] = status
            rec = session.query(res_model).filter_by(**query_condition).all()
        return rec

    @classmethod
    def get_fip_neutron_record(cls, session, res_id=None):
        """Get neutron db records."""
        if res_id:
            query_condition = {'id': res_id}
            rec = session.query(FloatingIP).filter_by(**query_condition).first()
        else:
            query_cond = (FloatingIP.fixed_port_id != '',)
            rec = session.query(FloatingIP).filter(*query_cond).all()
        return rec

    @classmethod
    def get_router_neutron_record_by_gw_port(cls, session, gw_port):
        """get router neutron record by gw port"""
        query_condition = {'gw_port_id': gw_port}
        return session.query(Router).filter_by(**query_condition).all()

    def _get_db_state_flag(self, res_id, res_type, state):
        """get db state flag"""
        if not res_id or not state or not res_type:
            LOG.debug("Update neutron API has been called with invalid res_id, res_type or state inputs")
            return True

        ctx = context.get_admin_context()
        if res_type == ac_const.NW_HW_VPC_CONNECTION:
            self.vpc_conn_db.update_db_vpc_connection_status(ctx, res_id, state)
            return True
        elif res_type == ac_const.NW_HW_L2BR:
            self.l2br_db.update_db_l2br_status(ctx, res_id, state)
            return True
        return False

    @staticmethod
    def _get_neutron_update_dict():
        """get neutron update dict"""

        neutron_update_dict = {ac_const.NW_HW_NETWORKS: models_v2.Network,
                               ac_const.NW_HW_PORTS: models_v2.Port,
                               ac_const.NW_HW_ROUTERS: Router,
                               ac_const.NW_HW_FIP: FloatingIP,
                               ac_const.NW_HW_DNAT: DNAT, }

        try:
            # ignore firewall db import error
            neutron_update_dict.update({ac_const.NW_HW_FIREWALL: fw_db.Firewall})
        except Exception as ex:
            LOG.info(_LE('[AC] Failed to add firewall to neutron update dict: %s'), ex)

        try:
            # ignore VPN db import error
            neutron_update_dict.update({ac_const.NW_HW_VPN_SERVICE: vpn_db.VPNService,
                                        ac_const.NW_HW_IPSEC_SITE_CONNECTION: vpn_db.IPsecSiteConnection, })
        except Exception as ex:
            LOG.info(_LE('[AC] VPN not configured: %s'), ex)
        return neutron_update_dict

    @classmethod
    def get_fw_rule_record_by_policy(cls, session, fw_policy_id):
        """get fw rule record by policy"""
        query_condition = {'firewall_policy_id': fw_policy_id}
        return session.query(fw_db.FirewallRule).filter_by(**query_condition).order_by(asc(fw_db.FirewallRule.position))

    def update_neutron_db_update_time(self, session, res_id, update_at, res_type):
        """update neutron db update time"""
        if not session:
            session = self.get_session('write')
        resource = self.get_neutron_record(session, res_id, res_type)
        if resource:
            resource.standard_attr.updated_at = update_at
            session.merge(resource)
            session.flush()
            LOG.info(_LI("Updating %s update_at to %s for id : %s"), res_type, update_at, res_id)

    @classmethod
    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def update_active_ac_record(cls, session, ip_addr):
        """update active ac record"""
        rec = session.query(dbschema.ACClusterSchema).filter_by(id=ac_const.DEFAULT_AC_ID).first()
        if rec:
            rec.active_ip = ip_addr
            session.merge(rec)
            session.flush()
            LOG.debug("Updated active AC record for IP : %s .", ip_addr)
        else:
            LOG.debug("Active AC record is not exist in huawei_ac_cluster.Create record for new active IP:%s.", ip_addr)
            rec_new = dbschema.ACClusterSchema(id=ac_const.DEFAULT_AC_ID, active_ip=ip_addr)
            session.add(rec_new)
            session.flush()

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def create_active_ac_record(self, session):
        """create active ac record"""
        ac_rec = session.query(dbschema.ACClusterSchema).filter_by(id=ac_const.DEFAULT_AC_ID).first()
        if ac_rec:
            self.update_active_ac_record(session, ac_const.DEFAULT_AC_IP)
        else:
            rec = dbschema.ACClusterSchema(id=ac_const.DEFAULT_AC_ID, active_ip=ac_const.DEFAULT_AC_IP)
            session.add(rec)
            session.flush()
            LOG.debug("Added active AC record for default IP:%s,", ac_const.DEFAULT_AC_IP)

    @classmethod
    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def delete_active_ac_record(cls, session, host):
        """delete active ac record"""
        rec = session.query(dbschema.ACClusterSchema).filter_by(id=ac_const.DEFAULT_AC_ID, active_ip=host).first()
        if rec:
            session.delete(rec)
            session.flush()
            LOG.debug("Deleted active AC record.")

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def delete_failed_resource(self, res_id, session=None):
        """Delete failed resource.

        :param res_id: Id of failed resource
        :param session: Current session info where the operation is performed. Used to access db.
        :return: None
        """

        if not session:
            session = self.get_session('write')
        with session.begin(subtransactions=True):
            failed_resource = session.query(dbschema.ACFailedResources).filter_by(id=res_id).first()
            if failed_resource:
                session.delete(failed_resource)
                LOG.debug("[AC] Deleted failed resource for res_id:%s.", res_id)

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def create_cloud_alarm(self, alarm_data, session=None):
        """Create cloud alarm record.

        :param alarm_data: data of the alarm
        :param session: Current session info where the operation is performed. Used to access db.
        :return: None
        """

        cloud_alarm = dbschema.ACAlarms(
            inner_id=alarm_data['inner_id'],
            type=alarm_data['type'],
            level=alarm_data['level'],
            host_id=alarm_data['host_id'],
            occur_time=alarm_data['occur_time'],
            description=alarm_data['description'],
            retry_count=0)
        if not session:
            session = self.get_session('write')
        with session.begin(subtransactions=True):
            session.add(cloud_alarm)
            LOG.debug("Created cloud alarm record: {inner_id=%s, type=%s,level=%s, host_id=%s, description=%s}.",
                      alarm_data['inner_id'], alarm_data['type'], alarm_data['level'], alarm_data['host_id'],
                      alarm_data['description'])

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def save_websocket_msg(self, websocket_msg, session=None):
        """Create websocket message record.

        :param websocket_msg: websocket message
        :param session: Current session info where the operation is performed. Used to access db.
        :return: None
        """
        if not session:
            session = self.get_session('write')

        report_time = ACCommonUtil.get_standard_current_time()
        with session.begin(subtransactions=True):
            websocket_msg_db = dbschema.ACWebsocketMsg(
                id=websocket_msg['res_id'],
                res_type=websocket_msg['res_type'],
                operation=websocket_msg['operation'],
                status=websocket_msg['status'],
                msg_time=report_time)
            session.add(websocket_msg_db)
            LOG.info("Create websocket message record:{res_id=%s,res_type=%s,operation=%s,status=%s,report_time=%s}.",
                     websocket_msg['res_id'], websocket_msg['res_type'], websocket_msg['operation'], websocket_msg,
                     report_time)


class ACdbInterface(ACdbInterfaceUtil):
    """AC db Interface"""

    def __init__(self):
        super(ACdbInterface, self).__init__()
        self.ops_version = ncu.get_ops_version()
        self._notifer = n_rpc.get_notifier('network')

    def _update_plugin_record_time(self, session, entry_info, res_param):
        """update plugin record time"""
        res_id, operation, res_type = res_param
        network_id = update_time = None
        if res_type == ac_const.NW_HW_PORTS and 'network-id' in entry_info:
            network_id = entry_info['network-id']
        if 'updated-at' in entry_info:
            if isinstance(entry_info['updated-at'], datetime.datetime):
                update_time = entry_info['updated-at'].strftime(ac_const.ISO8601_TIME_FORMAT)
            else:
                update_time = str(entry_info['updated-at'])
            update_time = datetime.datetime.strptime(
                update_time, ac_const.ISO8601_TIME_FORMAT)

        # not change timestamp when security_group sync
        if operation == ac_const.OPER_CREATE and res_type == ac_const.NW_HW_SEC_GRP:
            if self.get_res_attribute(session, res_id, res_type):
                return

        if res_type != ac_const.NW_HW_EXROUTE and operation != ac_const.OPER_DELETE:
            self.update_attribute_record(session, res_id, res_type, updated_at=update_time, network_id=network_id)

    # When plugin provide a new record to the MRD this API will allow to add the new record.
    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def create_plugin_record(self, session, entry_info, res_param, state):
        """Insert the plugin record into db.

        :param session:    Current session info where the operation is performed. Used to access db.
        :param entry_info: The user data which need to kept in the db for AC sync.
        :param res_param:    res_id, operation, res_type
        :param state:      Current state of the resource, whether in sync or wait etc.
        :return: None
        """
        res_id, operation, res_type = res_param

        if not session or self.ops_version in [ac_const.FSP_6_5]:
            session = self.get_session('write')

        if state != ac_const.NEUTRON_SYNC:
            # Update the attribute table. state check above is added to identify plugin records which are created for
            # the state-full resources's plugin entry creation while neutron-sync.
            self._update_plugin_record_time(session, entry_info, (res_id, operation, res_type))

        neutron_rec = self.get_server_record(neutron_name=self.neutron_name)
        if neutron_rec is None:
            LOG.error(_LE("Neutron server is not initialized.Please initialize it first."))
            raise AssertionError("Neutron server not initialized in plugin")

        rec = dbschema.ACPluginSchema(neutron_id=neutron_rec.neutron_id, res_uuid=res_id, res_type=res_type,
                                      user_oper=operation, data=entry_info, retry_count=0, state=state)
        session.add(rec)
        session.flush()
        LOG.debug("Added plugin record for resource type:%s,resource id:%s,operation:%s,state:%s,retry count:%s,"
                  "on neutron server %d", res_type, res_id, operation, state, 0, neutron_rec.neutron_id)
        return rec

    # If need update the access time or retry_count we can use this API to update the record. It internally will
    # update the same
    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def update_plugin_record(self, session, new_value=dbschema.ACPluginSchema(), old_value=None):
        """Update the plugin record into db.

        :param session: Current session info where the operation is performed.Used to access db.
        :param new_value: ACPluginSchema,keep new value,may update those filed:
            seq_num:     Sequence number of a record in the plugin table to be updated.
            rec:         Instead of sequence number user can provide the complete record also to update.
            state:       New state of the resource, whether in sync or wait, error-retry, complete, suspend etc.
            retry_count: New retry count which need to be updated in the plugin table.
            neutron_id:  New neutron_id of the record in case if one server is dead.
        :param old_value: ACPluginSchema,old value,should update.
        :return: None
        """
        if not session:
            session = self.get_session('write')

        if new_value.seq_num:
            rec = session.query(dbschema.ACPluginSchema).filter_by(seq_num=new_value.seq_num).first()
        else:
            rec = old_value

        if not rec:
            return rec

        if new_value.state:
            rec.state = new_value.state
        if new_value.retry_count is not None:
            rec.retry_count = new_value.retry_count
        if new_value.neutron_id is not None:
            rec.neutron_id = new_value.neutron_id

        session.merge(rec)
        session.flush()

        return rec

    # Get the next plugin record from the db based on the seq number
    def get_next_plugin_record(self, session, db_filter=dbschema.ACPluginSchema(), time_delta=None):
        """Query next single plugin record from db.

        :param session:     Current session info where the operation is performed. Used to access db.
        :param db_filter:      ACPluginSchema,May contain those parameter:
            neutron_id:  Neutron id of a record in the plugin table to
                              query the records.
                              -1    - means all record
                              None  - means get the local neutron-id
                              other - will return all record of that neutron server id
           state:       State of the record to query with
        :param time_delta:  Update time delta to query with
        :return: Single plugin record
        """

        if not session:
            session = self.get_session('write')
        if not db_filter.seq_num:
            return None

        query_condition = []
        if not db_filter.neutron_id:
            neutron_server = self.get_server_record(session, neutron_name=self.neutron_name)
            if neutron_server:
                query_condition.append(dbschema.ACPluginSchema.neutron_id == neutron_server.neutron_id)
        elif db_filter.neutron_id != -1:
            query_condition.append(dbschema.ACPluginSchema.neutron_id == db_filter.neutron_id)

        if db_filter.state:
            query_condition.append(dbschema.ACPluginSchema.state == db_filter.state)

        if time_delta:
            now = session.execute(utcnow()).scalar()
            timedelta_s = datetime.timedelta(seconds=time_delta)
            query_condition.append(dbschema.ACPluginSchema.update_time < (now - timedelta_s))

        if db_filter.seq_num == ac_const.PLUGIN_INVALID_SEQ_NUM:
            return session.query(dbschema.ACPluginSchema).filter(*query_condition).order_by(
                asc(dbschema.ACPluginSchema.seq_num)).first()

        query_condition.append(dbschema.ACPluginSchema.seq_num > db_filter.seq_num)
        return session.query(dbschema.ACPluginSchema).filter(*query_condition).order_by(
            asc(dbschema.ACPluginSchema.seq_num)).first()

    # Get a list of plugin records, can be based on neutron id or other conditions.
    def get_plugin_record_list(self, session, db_filter=dbschema.ACPluginSchema()):
        """Query plugin records from db based on the conditions.

        :param session:     Current session info where the operation is performed. Used to access db.
        :param db_filter:      ACPluginSchema,Query condition,May contain those:
              neutron_id:  Neutron id of a record in the plugin table to query the records.
                              -1    - means all record
                              None  - means get the local neutron-id
                              other - will return all record of that neutron server id
              res_id:      Instead of sequence number user can provide the res_uuid to query the data.
              user_oper:   Instead of sequence number user can provide the operation also to query the data.
              res_type:    Instead of sequence number user can provide the res_type also to query the data.
              state:       Current state of the records.
        :return: List of plugin records or None
        """
        query_condition = {}
        if not session:
            session = self.get_session()
        if not db_filter.neutron_id:
            neutron_serv = self.get_server_record(session, neutron_name=self.neutron_name)
            if neutron_serv:
                query_condition['neutron_id'] = neutron_serv.neutron_id
        elif db_filter.neutron_id != -1:
            query_condition['neutron_id'] = db_filter.neutron_id

        if db_filter.res_uuid or db_filter.res_uuid == '':
            query_condition['res_uuid'] = db_filter.res_uuid

        if db_filter.user_oper:
            query_condition['user_oper'] = db_filter.user_oper
        if db_filter.res_type:
            query_condition['res_type'] = db_filter.res_type
        if db_filter.state:
            query_condition['state'] = db_filter.state

        return session.query(dbschema.ACPluginSchema).filter_by(**query_condition).all()

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES,
                               retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def delete_plugin_with_res_id(self, session, res_id):
        """delete plugin records from db based on the conditions.

        :param session:     Current session info where the operation is performed. Used to access db.
        :param db_filter:      res_id:      Instead of sequence number user can provide the res_uuid to query the data.
        """
        if not session:
            session = self.get_session()
        query_condition = {"res_uuid": res_id}
        with session.begin(subtransactions=True):
            session.query(dbschema.ACPluginSchema).\
                filter_by(**query_condition).delete()
            session.flush()

    # Update the server state or new leader in case if one is dead.
    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_UPDATE_SERVER_MAX_RETRIES,
                               retry_interval=ac_const.DB_UPDATE_SERVER_RETRY_INTERVAL,
                               max_retry_interval=ac_const.DB_UPDATE_SERVER_MAX_RETRY_INTERVAL,
                               inc_retry_interval=True, retry_on_disconnect=True, retry_on_deadlock=True)
    def update_server_record(self, new_value, session=None):
        """Update the neutron server state record.

        :param session: Current session info where the operation is performed.Used to access db.
        :param neutron_id: Neutron id of the record to identify the record.
        :param new_value: ACNeutronStateSchema,new value which should update.
            rec: Instead of neutron_id user can provide the complete record also as a condition to delete.
            state: New state of the neutron server which need to be updated.
            is_leader: New leader status for the record
            sync_res: What resource is been transferred now as part of neutron sync. If port is been transferred,
                         the network id will also be added to the 'port_' string to identify which network the
                         operation is being performed.
            create_time: happen restart,update the create_time
            sync_type: To update the neutron_sync_type of the neutron server.
        :return: None
        """
        if not session:
            session = self.get_session('write')

        # If no neutron-id and rec, then get the neutron id from config env
        if not new_value.neutron_id:
            server_rec = self.get_server_record(neutron_name=self.neutron_name)
            if server_rec:
                new_value.neutron_id = server_rec.neutron_id

        if new_value.neutron_id:
            rec = session.query(dbschema.ACNeutronStateSchema).filter_by(
                neutron_id=new_value.neutron_id).first()
        else:
            return
        LOG.debug("Updating server for neutron(id=%s,name=%s):%s-->%s", rec.neutron_id, rec.neutron_name, rec,
                  new_value)
        if new_value.sync_res or new_value.sync_res == '':
            rec.sync_res = new_value.sync_res
        if new_value.state:
            rec.state = new_value.state
        if new_value.is_leader is not None:
            rec.is_leader = sa.true() if new_value.is_leader else sa.false()
        if new_value.create_time is not None:
            rec.create_time = session.execute(new_value.create_time).scalar()
        if new_value.sync_type:
            rec.sync_type = new_value.sync_type
        session.merge(rec)
        session.flush()

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def update_access_time_server_record(self, session):
        """Update the neutron server access_time.
        :param session: Current session info where the operation is performed.Used to access db.
        :param neutron_id: Neutron id of the record to identify the record.
        :return: None
        """
        if not session:
            session = self.get_session('write')
        rec = session.query(dbschema.ACNeutronStateSchema).filter_by(
            neutron_name=self.neutron_name).first()
        if rec:
            if rec.state == ac_const.ERROR:
                LOG.info('Change neutron server state from ERROR to NORMAL.')
                rec.state = ac_const.NORMAL
            LOG.debug('Updating access time for neutron server: %s, id: %d,state: %s.', rec.neutron_name,
                      rec.neutron_id, rec.state)
            rec.access_time = session.execute(utcnow()).scalar()
            session.merge(rec)
            session.flush()
        else:
            LOG.error(_LE('Neutron server record with neutron_name %s not found while updating access time.'),
                      self.neutron_name)
            try:
                self.create_server_record()
            except Exception as ex:
                LOG.error(_LE('[AC] Create neutron server state record failed: %s'), ex)
                raise ex

    # Update neutron db records.
    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def update_neutron_db_state(self, session, res_id, res_type, state):
        """Get the local neutron server id.

        :param session: Current session info where the operation is performed.Used to access db.
        :param res_id:  Resource id of the data to identify.
        :param res_type: Type of the resource, such as router, network, etc.
        :param state:   New state of the system.
        currently processing. Used for extra validation.
        :return: None
        """
        db_state_flag = self._get_db_state_flag(res_id, res_type, state)
        if db_state_flag:
            return None

        if not session:
            session = self.get_session('write')
        neutron_update_dict = ACdbInterface._get_neutron_update_dict()
        if res_type not in neutron_update_dict:
            return None
        res_model = neutron_update_dict.get(res_type)
        rec = session.query(res_model).filter_by(id=res_id).first()
        if not rec:
            return None

        if res_type == ac_const.NW_HW_PORTS:
            rec_list = self.get_plugin_record_list(
                session, dbschema.ACPluginSchema(res_uuid=res_id, neutron_id=-1, user_oper=ac_const.OPER_DELETE))
            if rec_list:
                return None
            admin_ctx = context.get_admin_context()
            need_send_message, port_info = self._send_rpc_port_update_start(admin_ctx, rec, state, res_id)

        # By the condition update ensuring the record not changed.
        with session.begin(subtransactions=True):
            query_res = session.query(res_model).filter_by(id=res_id, status=rec.status)
            if query_res:
                query_res.update({'status': state})
        LOG.info(_LI("Updating neutron db status to %s for res type %s and id : %s"), state, res_type, rec.id)
        session.flush()

        if res_type == ac_const.NW_HW_PORTS and need_send_message:
            self._notifer.info(admin_ctx, 'port.update.end', {"port": port_info})
            LOG.info("[AC] Send port.update.end MQ message: %s", {"port": port_info})
        return None

    def _send_rpc_port_update_start(self, admin_ctx, rec, state, res_id):
        if ncu.IS_FSP and state != rec.status and state in \
                [ac_const.NEUTRON_STATUS_DOWN, ac_const.NEUTRON_STATUS_ACTIVE, ac_const.NEUTRON_STATUS_ERROR]:
            try:
                port_info = ncu.get_core_plugin().get_port(
                    admin_ctx, res_id)
                port_info.update({"status": state})
                LOG.info("[AC] Send port.update.start MQ message: %s", {"port": port_info})
                self._notifer.info(admin_ctx, 'port.update.start', {"port": port_info})
                return True, port_info
            except Exception as ex:
                LOG.error("[AC] Send port.update.start MQ port_id:%s message failed: %s", res_id, ex)
        return False, None

    @staticmethod
    def get_active_ac_ip(session):
        """get active ac ip"""
        host_list = cfg.CONF.huawei_ac_agent_config.rpc_server_ip.replace(' ', '').split(',')
        if len(host_list) == 1:
            return host_list[0]

        try:
            rec = session.query(dbschema.ACClusterSchema).filter_by(id=ac_const.DEFAULT_AC_ID).first()
            if rec:
                return rec.active_ip
        except Exception:
            LOG.error(_LE("Can not query active AC record."))
        return None

    @classmethod
    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def create_bm_bond_mode_record(cls, session, port_id, bond_mode):
        """create BM bond mode record"""
        bond_mode_rec = session.query(dbschema.ACBMbondmodeSchema).filter_by(port_id=port_id).first()
        if not bond_mode_rec:
            new_rec = dbschema.ACBMbondmodeSchema(port_id=port_id, bond_mode=bond_mode)
            session.add(new_rec)
            session.flush()
            LOG.debug("Created bare metal bond mode record for port : %s, ", port_id)

    @classmethod
    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def delete_bm_bond_mode_record(cls, session, port_id):
        """delete BM bond mode record"""
        bond_mode_rec = session.query(dbschema.ACBMbondmodeSchema).filter_by(port_id=port_id).first()
        if bond_mode_rec:
            session.delete(bond_mode_rec)
            session.flush()
            LOG.debug("Deleted bare metal bond mode record for port:%s, ", port_id)

    def get_bm_bond_mode(self, session, port_id):
        """get BM bond mode"""
        if not session:
            session = self.get_session()

        bond_mode_rec = session.query(dbschema.ACBMbondmodeSchema).filter_by(port_id=port_id).first()
        if bond_mode_rec:
            return bond_mode_rec.bond_mode
        return None

    def confirm_standard_res_attributes(self, session, resource, res_param):
        """Function to update the attribute table data in the resource

        :param session: Current session info where the operation is performed. Used to access db.
        :param res_param: Resource type, res_id, operation
        :param resource: resource data which needs to be updated.
        :return:
        """
        res_type, res_id, operation = res_param
        if not session:
            session = self.get_session()
        resource_log = copy.deepcopy(resource)
        resource_log = validate.validate_log_record(resource_log, res_type)
        LOG.debug('Update the attribute data in the resource: %s', resource_log)
        updated_at = ACCommonUtil.get_standard_current_time().strftime(ac_const.ISO8601_TIME_FORMAT)
        if res_type == ac_const.NW_HW_PORTS:
            rec = self.get_port_attribute(session, res_id)
        else:
            rec = self.get_res_attribute(session, res_id, res_type)
        if rec:
            # not change timestamp when security_group sync
            if operation == ac_const.OPER_CREATE and rec.update_time and res_type == ac_const.NW_HW_SEC_GRP:
                updated_at = rec.update_time.strftime(ac_const.ISO8601_TIME_FORMAT)
            resource['created-at'] = rec.create_time.strftime(ac_const.ISO8601_TIME_FORMAT)
        else:
            resource['created-at'] = updated_at
        resource['updated-at'] = updated_at
        LOG.debug('Updated request info fields created-at:%s,updated-at:%s', resource['created-at'],
                  resource['updated-at'])
        return resource

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def update_attribute_record(self, session, res_id, res_type, **kwargs):

        """Function to update or insert the attribute-tables

        :param session: Current session info where the operation is performed. Used to access db.
        :param res_id: Unique identifier of the resource
        :param res_type: Resource-type
        :param kwargs: updated_at/network_id
        :return:
        """
        if not session:
            session = self.get_session('write')
        with session.begin(subtransactions=True):
            updated_at = kwargs.get("updated_at")
            if updated_at:
                update_time = updated_at
            else:
                update_time = ACCommonUtil.get_standard_current_time()

            if isinstance(update_time, datetime.datetime):
                update_time = datetime.datetime.strptime(
                    update_time.strftime(ac_const.ISO8601_TIME_FORMAT), ac_const.ISO8601_TIME_FORMAT)

            if res_type == ac_const.NW_HW_PORTS:
                port_rec = self.get_port_attribute(session, res_id)
                if port_rec:
                    port_rec.update_time = update_time
                else:
                    port_rec = dbschema.ACPortAttributes(uuid=res_id, network_id=kwargs.get("network_id"),
                                                         create_time=update_time, update_time=update_time)
                session.merge(port_rec)
                session.flush()
            else:
                attr_rec = self.get_res_attribute(session, res_id, res_type)
                if attr_rec:
                    attr_rec.update_time = update_time
                else:
                    attr_rec = dbschema.ACResourceAttributes(
                        uuid=res_id, res_type=res_type, create_time=update_time, update_time=update_time)
                session.merge(attr_rec)
                session.flush()
        LOG.debug("Updated time for id:%s, res-type: %s, update-time:%s", res_id, res_type, update_time)
        return

    def get_res_attribute(self, session=None, res_id=None, res_type=None):
        """ Function to get the resource attribute from
        huawei_ac_res_attributes

        :param session:Current session info where the operation is performed. Used to access db.
        :param res_id: Unique identifier of the resource
        :param res_type: Resource-type
        :return: record from huawei_ac_res_attributes
        """
        query_condition = {}
        if not session:
            session = self.get_session()
        if res_type:
            query_condition['res_type'] = res_type
        if res_id:
            query_condition['uuid'] = res_id
            return session.query(dbschema.ACResourceAttributes).filter_by(**query_condition).first()
        return session.query(dbschema.ACResourceAttributes).filter_by(**query_condition).all()

    def get_port_attribute(self, session=None, res_id=None, network_id=None):
        """Function to get the resource attribute from huawei_ac_port_attributes

        :param session: Current session info where the operation is performed. Used to access db.
        :param res_id: Unique identifier of the resource
        :param network_id: network-id to which the port belongs to
        :return: record from huawei_ac_port_attributes
        """
        query_condition = {}
        if not session:
            session = self.get_session()
        if network_id:
            query_condition['network_id'] = network_id
        if res_id:
            query_condition['uuid'] = res_id
            return session.query(dbschema.ACPortAttributes).filter_by(**query_condition).first()
        return session.query(dbschema.ACPortAttributes).filter_by(**query_condition).all()

    def delete_standard_res_attributes(self, res_type, res_id, session=None):
        """Function to delete the resource attribute from attribute-tables

        :param res_type: Resource-type
        :param res_id: Unique identifier of the resource
        :param session: Current session info where the operation is performed. Used to access db.
        """
        LOG.debug("Deleting the attribute start")
        if not session or self.ops_version in [ac_const.FSP_6_5]:
            session = self.get_session('write')
        with session.begin(subtransactions=True):
            if res_type == ac_const.NW_HW_PORTS:
                rec = self.get_port_attribute(session, res_id)
            else:
                rec = self.get_res_attribute(session, res_id, res_type)
            if rec:
                session.delete(rec)
                LOG.debug("Deleting the attribute in DB")
        LOG.debug("Deleting the attribute record for res-id:%s,res-type:%s", res_id, res_type)

    def get_ac_process_req(self):
        """Function to get req seq and update time.
        """
        try:
            session = self.get_session()
            seqs = session.query(dbschema.ACPluginSchema.seq_num, dbschema.ACPluginSchema.update_time).filter(
                dbschema.ACPluginSchema.state.in_(
                    [ac_const.COMPLETE, ac_const.SUSPEND, ac_const.NEUTRON_SYNC, ac_const.IN_PROCESS])).all()
            len_of_seqs = len(seqs)
            LOG.debug("[AC]There are %s requests that AC is processing", len_of_seqs)
            seq_index = {}
            for seq in seqs:
                seq_index[seq[0]] = {'timeout_increase': len_of_seqs * ac_const.TIME_NEED_PER_SEQ,
                                     'update_time': seq.update_time}
            return seq_index
        except Exception as ex:
            LOG.error("[AC]when get plugin requests,catch:%s", ex)
            return {}

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def update_ac_connection_state_record(self, neutron_name, ac_connection_state):
        """Update the neutron server access_time.
        :param neutron_name: Neutron id of the record to identify the record.Used to access db.
        :param ac_connection_state: connection ac state can be 0 and 1.
        :return: None
        """

        session = self.get_session('write')
        rec = session.query(dbschema.ACNeutronStateSchema).filter_by(neutron_name=neutron_name).first()
        if rec:
            rec.ac_connection_state = ac_connection_state
            session.merge(rec)
            session.flush()
        else:
            LOG.error(_LE('Neutron server record with neutron_name %s not found while updating access time.'),
                      neutron_name)

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=False,
                               max_retry_interval=0)
    def create_or_update_failed_resource(self, resource_data, force_recovery=True, session=None):
        """Create of update failed resource record.

        :param resource_data: ACFailedResources,data of failed resource
        :param force_recovery:  whether recovery data exist or not
        :param session: Current session info where the operation is performed. Used to access db.
        :return: None
        """
        LOG.debug("[AC]start deal failed resource record:%s.", resource_data)
        # 如果资源未指定了重试次数则读取默认配置文件中配置
        error_retry_count = resource_data.retry_count if resource_data.retry_count else int(
            cfg.CONF.huawei_ac_config.error_retry_count)
        if error_retry_count == 0:
            LOG.debug("[AC]error_retry_count=0,no need to create failed resource record: %s.", resource_data.id)
            return
        if not session:
            session = self.get_session('write')

        (status, server) = self.check_is_neutron_sync_in_progress(session)
        if status and server:
            LOG.info("[AC] Neutron sync is running, Not create failed resource record.")
            return

        failed_resource = session.query(dbschema.ACFailedResources).filter_by(id=resource_data.id).first()
        failed_time = ACCommonUtil.get_standard_current_time()
        with session.begin(subtransactions=True):
            if failed_resource:
                # 状态上报报错，不更新重试次数
                # 接口报错，刷新重试次数
                if not force_recovery:
                    return
                failed_resource.operation = resource_data.operation
                failed_resource.retry_count = error_retry_count
                failed_resource.failed_time = failed_time
                LOG.info("Updated failed resource record:%s", failed_resource)
            else:
                failed_resource = dbschema.ACFailedResources(
                    id=resource_data.id, res_type=resource_data.res_type, operation=resource_data.operation,
                    failed_time=failed_time, retry_count=error_retry_count)
                session.add(failed_resource)
                LOG.info("Created failed resource record:%s", failed_resource)

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES, retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def update_subport_device_info(self, subport_id, subport_device_id, subport_device_owner, session=None):
        """update_subport_device_info.

        :param subport_id: subport id
        :param subport_device_id: subport device_id
        :param device_owner: subport device_owner
        :param session: Current session info where the operation is performed. Used to access db.
        :return: None
        """
        if not session:
            session = self.get_session('write')
        with session.begin(subtransactions=True):
            LOG.info(_LI("update subport in db start : %s."), subport_id)
            port = session.query(models_v2.Port).filter_by(id=subport_id).first()
            if port:
                port.device_id = subport_device_id
                port.device_owner = subport_device_owner
                session.merge(port)
                session.flush()
            LOG.info(_LI("update subport in db end : %s."), subport_id)

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES,
                               retry_interval=0, inc_retry_interval=0,
                               max_retry_interval=0)
    def handle_segments_records(self, session=None):
        """clean invalid segments.
        """
        LOG.info('Begin to clean invalid segments')
        if not session:
            session = self.get_session('write')
        with session.begin(subtransactions=True):
            bind_levels = session.query(models.PortBindingLevel).all()
            segments_used = [bind_level.segment_id for bind_level in bind_levels]
            segments_invalid = session.query(ncu.NetworkSegment).filter(
                ncu.NetworkSegment.is_dynamic == 1,
                ncu.NetworkSegment.id.notin_(segments_used))
            LOG.info('Delete invalid segments: %s', segments_invalid.all())
            for segment_invalid in segments_invalid:
                session.delete(segment_invalid)
        LOG.info('Clean invalid segments done.')
