#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""bgp router"""

import abc

import six
from neutron.api import extensions
from neutron.api.v2 import base

from networking_huawei.drivers.ac.extensions.sync_result.sync_result import \
    Sync_result

try:
    from neutron_lib.api import validators
except ImportError:
    validators = None
try:
    from neutron.api.v2 import attributes
except ImportError:
    attributes = None

try:
    from neutron.api.extensions import ExtensionDescriptor
except ImportError:
    from neutron_lib.api.extensions import ExtensionDescriptor

try:
    from neutron.common.exceptions import NotFound
except ImportError:
    from neutron_lib.exceptions import NotFound

try:
    from neutron.common.exceptions import NeutronException
except ImportError:
    from neutron_lib.exceptions import NeutronException

try:
    from neutron.services.service_base import ServicePluginBase
except ImportError:
    from neutron_lib.services.base import ServicePluginBase

from networking_huawei.drivers.ac.common import validate as validator
from networking_huawei.drivers.ac.extensions import bgp_route \
    as bgp_route_extension
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei._i18n import _

LOG = logging.getLogger(__name__)


class BgpRouteNotFound(NotFound):
    """Bgp Route not found class"""
    message = _("Bgp route %(bgp_route_id)s could not be found.")


extensions.append_api_extensions_path(bgp_route_extension.__path__)

BGP_ROUTE = 'bgp-route'

OPS_VERSION = ncu.get_ops_version()


def _validate_range(data, valid_values=None):
    msg = None
    data = str(data)
    if data.isdigit():
        data = int(data)
        min_value = valid_values[0]
        max_value = valid_values[1]
        if min_value is not None and data < min_value:
            msg = _("'%(data)s' is too small - must be at least "
                    "'%(limit)d'") % {'data': data, 'limit': min_value}
        if max_value is not None and data > max_value:
            msg = _("'%(data)s' is too large - must be no larger than "
                    "'%(limit)d'") % {'data': data, 'limit': max_value}
    else:
        msg = _("'%s' is not an integer") % data
    if msg:
        LOG.debug(msg)
    return msg


def _validate_hold_time(data, valid_values=None):
    LOG.debug("validate hold time :%s", data)
    if data is 0:
        return ''
    msg = _validate_range(data, valid_values)
    return msg


if validators and hasattr(validators, 'add_validator') and \
        'type:hold_time' not in validators.validators:
    validators.add_validator('hold_time', _validate_hold_time)
if validators and hasattr(validators, 'add_validator') and \
        'type:ip_address_list_or_nodata' not in validators.validators:
    validators.add_validator('ip_address_list_or_nodata',
                             validator.validate_source_ips)

if attributes and hasattr(attributes, 'validators') and \
        'type:hold_time' not in attributes.validators:
    attributes.validators['type:hold_time'] = _validate_hold_time
if attributes and hasattr(attributes, 'validators') and \
        'type:ip_address_list_or_nodata' not in attributes.validators:
    attributes.validators['type:ip_address_list_or_nodata'] = \
        validator.validate_source_ips


class InvalidIntegerInput(NeutronException):
    """Invalid input for operation"""
    message = _("Invalid input for operation: %(error_message)s.")


def convert_to_int(data):
    """convert to int"""
    try:
        return int(data)
    except (ValueError, TypeError):
        msg = _("'%s' is not an integer") % data
        raise InvalidIntegerInput(error_message=msg)


# Attribute Map
RESOURCE_ATTRIBUTE_MAP = {
    "bgp_routes": {
        'id': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:uuid_or_none': None},
            'is_visible': True, 'primary_key': True,
            'default': None},
        'name': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:string_or_none': 255},
            'default': '',
            'is_visible': True},
        'description': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:string_or_none': 255},
            'is_visible': True,
            'default': ''},
        'tenant_id': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:string': 255},
            'is_visible': True},
        'router_id': {
            'allow_post': True, 'allow_put': False,
            'is_visible': True,
            'validate': {'type:uuid': None}},
        'router_interface_id': {
            'allow_post': True, 'allow_put': False,
            'is_visible': True, 'default': None,
            'validate': {'type:uuid_or_none': None}},
        'ip_version': {
            'allow_post': True, 'allow_put': False,
            'is_visible': True, 'default': -1,
            'convert_to': convert_to_int,
            'validate': {'type:values': [-1, 4, 6]}},
        'peer_ip_address': {
            'allow_post': True, 'allow_put': False,
            'is_visible': True,
            'validate': {'type:ip_address': None}},
        'peer_as_number': {
            'allow_post': True, 'allow_put': False,
            'is_visible': True,
            'validate': {'type:range': [1, 4294967295]}},
        'bfd': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': False,
            'validate': {'type:boolean': None}},
        'peer_type': {
            'allow_post': True, 'allow_put': False,
            'is_visible': True, 'default': 0,
            'validate': {'type:values': [0, 1]}},
        'session_attribute': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
            'validate': {'type:dict_or_nodata': {
                'keepalive_time': {
                    'type:range': [0, 21845],
                    'required': False,
                    'default': 60},
                'hold_time': {
                    'type:hold_time': [3, 65535],
                    'required': False,
                    'default': 180},
                'auth': {'type:dict_or_nodata': {
                    'auth_type': {
                        'type:values': ['password', "none"],
                        'required': False,
                        'allow_put': False,
                        'default': None},
                    'password_type': {
                        'type:values': ['cipher'],
                        'required': False,
                        'allow_put': False,
                        'default': 'cipher'},
                    'password_text': {
                        'type:string': 255,
                        'allow_put': False,
                        'required': False}, }, },
                'suppress': {
                    'type:boolean': None,
                    'required': False},
            }}},
        'source_ips': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:ip_address_list_or_nodata': None},
            'is_visible': True, 'default': None},
    }
}


class Bgp_route(ExtensionDescriptor):
    """Consistency report generation"""

    @classmethod
    def get_name(cls):
        """Name for this extension"""
        return "Bgp route"

    @classmethod
    def get_alias(cls):
        """core_plugin class to load the extension"""
        return 'bgp-route'

    @classmethod
    def get_description(cls):
        """A small description about this extension"""
        return "Support connecting site with datacenter by bgp"

    @classmethod
    def get_namespace(cls):
        """The XML namespace for this extension"""
        return ""

    @classmethod
    def get_updated(cls):
        """Specify when was this extension last updated"""
        return "2017-04-18T21:10:50-00:00"

    @classmethod
    def get_resources(cls):
        """Returns Ext Resources."""
        try:
            from neutron.common import resource_norecord as norec
            dict_norec = {}
            dict_norec['bgp_route'] = \
                [['bgp_route', 'session_attribute', 'auth', 'password_text']]
            norec.Resource_plain_password_norecord(). \
                set_res_norecord(**dict_norec)
        except Exception as ex:
            LOG.debug("Recource can not be unrecorded %s", ex)

        allow_pagination, allow_sorting, plugin = Sync_result.get_resource_before(
            BGP_ROUTE)

        resources = []
        for resource_name in ["bgp_route"]:
            collection_name = BGP_ROUTE + "s"
            params = RESOURCE_ATTRIBUTE_MAP.get("bgp_routes", dict())
            member_actions = {}
            controller = base.create_resource(
                collection_name, resource_name, plugin, params, allow_bulk=True,
                member_actions=member_actions, allow_sorting=allow_sorting,
                allow_pagination=allow_pagination)
            resources.append(extensions.ResourceExtension(
                collection_name, controller, member_actions=member_actions,
                attr_map=params))

        return resources

    @classmethod
    def get_plugin_interface(cls):
        """get plugin interface"""
        return BgpRouteBase

    def get_extended_resources(self, version):
        """get extended resources"""
        if version != "2.0":
            return {}
        return RESOURCE_ATTRIBUTE_MAP

    def update_attributes_map(self, attributes, extension_attrs_map=None):
        """update attributes map"""
        super(Bgp_route, self).update_attributes_map(
            attributes, extension_attrs_map=RESOURCE_ATTRIBUTE_MAP)


@six.add_metaclass(abc.ABCMeta)
class BgpRouteBase(ServicePluginBase):
    """Bgp route class"""

    def get_plugin_type(self):
        """get plugin type"""
        pass

    def get_plugin_description(self):
        """get plugin description"""
        pass

    def get_plugin_name(self):
        """get plugin name"""
        pass

    @abc.abstractmethod
    def create_bgp_route(self, context, bgp_route):
        """the abstractmethod to create bgp route"""
        pass

    @abc.abstractmethod
    def update_bgp_route(self, context, id, bgp_route):
        """the abstractmethod to update bgp route"""
        pass

    @abc.abstractmethod
    def delete_bgp_route(self, context, id):
        """the abstractmethod to delete bgp route"""
        pass

    @abc.abstractmethod
    def get_bgp_route(self, context, id, fields=None):
        """the abstractmethod to get bgp route"""
        pass

    @abc.abstractmethod
    def get_bgp_routes(self, context, filters=None, fields=None):
        """the abstractmethod to get bgp routes"""
        pass
