#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""L2brBase class"""

import abc
import six
from neutron import manager
try:
    from neutron.common.exceptions import NotFound, InvalidInput
except ImportError:
    from neutron_lib.exceptions import NotFound, InvalidInput
from neutron.api.v2 import base
try:
    from neutron_lib.plugins import directory
except Exception:
    directory = None
try:
    from neutron.api.extensions import ExtensionDescriptor
except ImportError:
    from neutron_lib.api.extensions import ExtensionDescriptor
try:
    from neutron.api import extensions as extensions_neutron_api
except ImportError:
    extensions_neutron_api = None
try:
    from neutron.services.service_base import ServicePluginBase
except ImportError:
    from neutron_lib.services.base import ServicePluginBase

from oslo_config import cfg

import networking_huawei.drivers.ac.common.constants as ac_constants
from networking_huawei.drivers.ac.extensions import l2br as l2br_extension
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu


class L2brNotFound(NotFound):
    """L2brNotFound class"""
    message = "L2br %(l2br_id)s could not be found."


class L2brDeviceInfoParamInvalid(InvalidInput):
    """L2brDeviceInfoParamInvalid class"""
    message = "Invalid value for l2br: you need input only one parameter " \
              "(device_mac,device_ip or device_id)."


class L2brServiceInfoParamInvalid(InvalidInput):
    """L2brDeviceInfoParamInvalid class"""
    message = "Invalid value for l2br: you need input only one parameter " \
              "(network_id or vni)."


NAME_MAX_LEN = 64
DEVICE_ID_MAX_LEN = 255
DEVICE_MAC_MAX_LEN = 64
DEVICE_IP_MAX_LEN = 64
INTERFACE_ID_MAX_LEN = 255

extensions_neutron_api.append_api_extensions_path(l2br_extension.__path__)

RESOURCE_ATTRIBUTE_MAP = {
    'l2brs': {
        'id': {'allow_post': False, 'allow_put': False,
               'validate': {'type:uuid': None},
               'is_visible': True,
               'primary_key': True},
        'name': {'allow_post': True, 'allow_put': True,
                 'validate': {'type:string': NAME_MAX_LEN},
                 'is_visible': True, 'default': ''},
        'tenant_id': {'allow_post': True, 'allow_put': False,
                      'validate': {'type:string': None},
                      'required_by_policy': True,
                      'is_visible': True, 'default': ''},
        'device_info': {'allow_post': True, 'allow_put': True,
                        'is_visible': True, 'default': None,
                        'validate': {'type:dict_or_nodata': {
                            'device_id': {
                                'type:string': DEVICE_ID_MAX_LEN,
                                'required': False},
                            'device_mac': {
                                'type:string': DEVICE_MAC_MAX_LEN,
                                'required': False},
                            'device_ip': {
                                'type:string': DEVICE_IP_MAX_LEN,
                                'required': False},
                            'interface_id': {
                                'type:string': INTERFACE_ID_MAX_LEN,
                                'required': True},
                        }}},
        'service_info':
        {'allow_post': True, 'allow_put': True,
         'is_visible': True, 'default': None,
         'validate': {
             'type:dict_or_none': {
                 'vlan_id': {'type:range': [0, 4094], 'required': True},
                 'vni': {'type:range': [1, 16777215], 'required': False},
                 'network_id': {'type:uuid': None, 'required': False},
             }
         }},
        'status': {'allow_post': False, 'allow_put': False,
                   'validate': {'type:string': None},
                   'is_visible': True, 'default': ''},
    }
}


class L2br(ExtensionDescriptor):
    """L2br class"""

    @classmethod
    def get_name(cls):
        """get name"""
        return "Neutron L2br"

    @classmethod
    def get_alias(cls):
        """get alias"""
        return "l2br"

    @classmethod
    def get_description(cls):
        """get description"""
        return "Huawei Agile Controller L2br"

    @classmethod
    def get_namespace(cls):
        """the function to get namespace"""
        return ""

    @classmethod
    def get_updated(cls):
        """the function to get updated"""
        return "2016-05-30T10:00:00-00:00"

    @classmethod
    def get_resources(cls):
        """Returns Ext Resources."""
        ops_version = ncu.get_ops_version()
        if ops_version in ac_constants.OPS_VERSION_O_PQRTW_6_21:
            allow_pagination = True
            allow_sorting = True
            plugin = directory.get_plugin(ac_constants.NW_HW_L2BR)
        else:
            plugin = \
                manager.NeutronManager.get_service_plugins()[
                    ac_constants.NW_HW_L2BR]
            allow_pagination = cfg.CONF.allow_pagination
            allow_sorting = cfg.CONF.allow_sorting

        resources = []

        for resource_name in ['l2br']:
            collection_name = resource_name + 's'
            params = RESOURCE_ATTRIBUTE_MAP.get(collection_name, dict())
            member_actions = {}
            controller = base.create_resource(
                collection_name, resource_name, plugin, params,
                allow_bulk=True,
                member_actions=member_actions,
                allow_pagination=allow_pagination,
                allow_sorting=allow_sorting)
            res = extensions_neutron_api.ResourceExtension(
                collection_name,
                controller,
                member_actions=member_actions,
                attr_map=params)
            resources.append(res)

        return resources

    @classmethod
    def get_plugin_interface(cls):
        """ get plugin interface """
        return L2brBase

    def update_attributes_map(self, attributes, extension_attrs_map=None):
        """ update attributes map """
        super(L2br, self).update_attributes_map(
            attributes, extension_attrs_map=RESOURCE_ATTRIBUTE_MAP)

    def get_extended_resources(self, version):
        """ get extended resources """
        if version != '2.0':
            return {}
        return RESOURCE_ATTRIBUTE_MAP


@six.add_metaclass(abc.ABCMeta)
class L2brBase(ServicePluginBase):
    """L2brBase class"""

    def get_plugin_type(self):
        """ get plugin type """
        return ac_constants.NW_HW_L2BR

    def get_plugin_name(self):
        """ get plugin name """
        pass

    def get_plugin_description(self):
        """ get plugin description """
        return 'L2BR plugin'

    @abc.abstractmethod
    def create_l2br(self, context, l2br):
        """the abstractmethod to create l2br"""
        pass

    @abc.abstractmethod
    def update_l2br(self, context, l2br_id, l2br):
        """the abstractmethod to update l2br"""
        pass

    @abc.abstractmethod
    def delete_l2br(self, context, l2br_id):
        """the abstractmethod to delete l2br"""
        pass

    @abc.abstractmethod
    def get_l2br(self, context, l2br_id, fields=None):
        """the abstractmethod to get l2br"""
        pass

    @abc.abstractmethod
    def get_l2brs(self, context, filters=None, fields=None):
        """the abstractmethod to get l2brs"""
        pass
