# !/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""
| 功能：
| 版本：2022-06-20 09:48 创建
"""
import re
import traceback

import netaddr
from neutron.db.models_v2 import IPAllocation

from networking_huawei.drivers.ac.common import constants as ac_cnst
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler
from networking_huawei.drivers.ac.common.ac_config_utils import ConfigParas
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei.drivers.ac.db.exroute.exroute import RouterExRoute, PATTERN_BFD_NAME
from networking_huawei.drivers.ac.db.exroute.exroute_mec_db import ExroutesDbMixinMec
from networking_huawei.drivers.ac.extensions.bfd_route.ac_bfd_route_ext import Ac_bfd_route_ext
from networking_huawei.drivers.ac.extensions.exroute.l3_exroutes import ParamNotSpecified, ParamInvalid, \
    InvalidParam, MissParam, ExRoutePortNotFound
from networking_huawei.drivers.ac.model.exroute_model import ACExRouteModel
from networking_huawei.drivers.ac.plugins.abstract_neutron_service_plugin import AbstractServicePlugin
from networking_huawei.drivers.ac.plugins.bfd_route import ac_bfd_validator
from networking_huawei.drivers.ac.plugins.l3.l3_router_util import ExRoutesException
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI

try:
    # try to import neutron callbacks
    from neutron.callbacks import registry
    from neutron.callbacks.resources import ROUTER, ROUTER_INTERFACE
except ImportError:
    # there is no neutron callbacks in fusionsphere
    from neutron_lib.callbacks import registry
    from neutron_lib.callbacks.resources import ROUTER, ROUTER_INTERFACE

try:
    from neutron.common.exceptions import BadRequest
except ImportError:
    from neutron_lib.exceptions import BadRequest

LOG = ncu.ac_log.getLogger(__name__)


class AcBfdRoutePlugin(AbstractServicePlugin, ExroutesDbMixinMec):
    """AC 配置插件类"""

    def __init__(self):
        super(AcBfdRoutePlugin, self).__init__(Ac_bfd_route_ext())
        config_parser = ConfigParas(ac_cnst.HUAWEI_CONFIG_FILE)
        # bfd前缀
        self._pre_bfd_name = config_parser.get_config_detail('huawei_ac_config', 'bfd_name', default=None)
        if self._pre_bfd_name:
            self._pre_bfd_name = self._pre_bfd_name.strip()
        self.ac_reliability = ACReliabilityAPI(ac_cnst.NW_HW_EXROUTE)
        self.db_if = ACdbInterface()

    def _generate_bfd_name(self, context, bfd_route):
        ip_allocations = context.session.query(IPAllocation).filter(IPAllocation.ip_address == bfd_route.nexthop)
        LOG.debug('[AC] Generate bfd name, route: %s, ip_allocations.count: %s', bfd_route, ip_allocations.count())
        if ip_allocations.count() != 1:  # 多个Port的fix_ips.ip_address==bfd_route.nexthop
            raise ExRoutePortNotFound(ex_route=bfd_route)
        port_id = ip_allocations.first().port.id
        LOG.info('[AC] Generate bfd name, port_id: %s', port_id)
        result = '%s-v%s-%s' % (self._pre_bfd_name, bfd_route.ip_version, port_id)
        if not re.match(PATTERN_BFD_NAME, result):
            raise InvalidParam(name='bfd_name', value=result)
        LOG.debug('[AC] Generate bfd name, result:%s', result)
        return result

    def _create_before(self, context, router_id, bfd_routes):
        result = []
        for elem in bfd_routes:
            add_elem = RouterExRoute.from_dict(router_id, elem)
            if add_elem.type == 401:
                add_elem.ports = self._validate_ports_custom(context, elem, router_id)
            elif add_elem.type == 402:
                if 'bfd' in elem:
                    raise InvalidParam(name='bfd', value=elem)
                if add_elem.source_ips:
                    raise InvalidParam(name='source_ips', value=elem)
            extra_opt = {'health_detect': add_elem.bfd}
            if add_elem.source_ips:
                extra_opt['source_ips'] = add_elem.source_ips
            if add_elem.bfd and not add_elem.bfd_name:  # 需要指定bfd名称，并标记为首port
                bfd_name = self._generate_bfd_name(context, add_elem)
                add_elem.bfd_name = bfd_name
                # 标记首port
                extra_opt['bfd_leader'] = True
            extra_opt['bfd_name'] = add_elem.bfd_name
            add_elem.extra_opt = extra_opt
            result.append(add_elem)
        return result

    def create_core(self, request, body=None, **kwargs):
        """创建资源

        :param request: neutron.api.v2.resource.Request,请求
        :param body: dict,请求体。
        :param kwargs: dict,其它参数。neutron.api.v2.resource.resource()以键值对参数传递。
        :return: dict
        """
        ac_bfd_validator.validate_pre_bfd_name(self._pre_bfd_name)
        LOG.debug('[AC]create bfd router start:%s,body=%s,kwargs=%s', repr(request), body, kwargs)
        router_id = kwargs.get('id')
        context = request.context
        req_bfd_routes = body.get('exroutes', [])
        try:
            bfd_routes = self._create_before(context, router_id, req_bfd_routes)
        except (ParamNotSpecified, ParamInvalid, InvalidParam, MissParam) as e:
            LOG.error('[AC]create before step failed:%s', traceback.format_exc())
            raise BadRequest(resource='exroutes', msg=str(e))

        with context.session.begin(subtransactions=True):
            for elem in bfd_routes:
                context.session.add(elem)

        result = []
        ac_bfd_routes = []
        for elem in bfd_routes:
            result.append(elem.show())
            ac_bfd_routes.append(elem.to_dict())
        self._create_exroutes_after(context, router_id, ac_bfd_routes)
        LOG.debug('[AC]create bfd router result:%s', result)
        return {'exroutes': result}

    def _create_exroutes_after(self, context, router_id, ex_routes):
        exroutes_failed = []
        if hasattr(context, 'session'):
            session = context.session
        else:
            session = self.db_if.get_session('write')
        rec_plugin_list, exroute_info_list, exroute_list = self._generate_exroute_list(
            context, session, ex_routes, router_id)
        for flag_number, _ in enumerate(exroute_info_list):
            exroute = exroute_list[flag_number]
            try:
                self.ac_reliability.update_plugin_record(
                    context, router_id, exroute_info_list[flag_number],
                    ac_cnst.NW_HW_CREATE_EXROUTE, rec_plugin=rec_plugin_list[flag_number])
            except Exception as ex:
                LOG.error('[AC] Huawei AC create exroute %(route)s '
                          'failed: %(ex)s', {'route': exroute, 'ex': ex})
                self._optisize_exroute_info(exroute)
                exroutes_failed.append(self._get_exroute_format_custom(exroute))
        self._delete_exroute_plugin_record(session, rec_plugin_list)
        self._handle_create_exroutes_failure(context, router_id, exroutes_failed)

    @classmethod
    def _optisize_exroute_info(cls, exroute):
        key_blacklist = ['segment_type', 'segment_id', 'priority', 'extra_opt']
        for key in key_blacklist:
            exroute.pop(key, None)

    def _delete_exroute_plugin_record(self, session, rec_plugin_list):
        for plugin in rec_plugin_list:
            try:
                self.db_if.delete_plugin_record(session, plugin.seq_num)
            except Exception as ex:
                LOG.error('Huawei AC delete exroute plugin_record: %s failed: %s',
                          plugin.seq_num, str(ex))

    def _handle_create_exroutes_failure(self, context, router_id, exroutes_failed):
        if exroutes_failed:
            LOG.error('[AC] Huawei AC create exroutes %s failed.',
                      exroutes_failed)
            self.delete_mec_exroutes(
                context, router_id, {'exroutes': exroutes_failed})
            ac_osprofiler.record_chain_exception_end("create exroutes fail")
            raise ExRoutesException(opr='create', ex=exroutes_failed)

    def _generate_exroute_list(self, context, session, exroutes_data, router_id):
        rec_plugin_list = []
        exroute_info_list = []
        exroute_list = []
        for exroute in exroutes_data:
            if ncu.get_ops_version() in [ac_cnst.FSP_6_1, ac_cnst.FSP_6_3_0]:
                self._update_exroute_ip_version(exroute)
            interface_id = ACCommonUtil.get_exroute_interface_id(
                context, exroute.get('nexthop'), exroute.get('destination'))
            exroute_info = ACExRouteModel.ac_model_format(exroute, router_id,
                                                          interface_id)
            rec_plugin_tmp = self.db_if.create_plugin_record(
                session, exroute_info,
                (router_id, ac_cnst.OPER_CREATE, ac_cnst.NW_HW_EXROUTE),
                ac_cnst.IN_PROCESS)
            rec_plugin_list.append(rec_plugin_tmp)
            exroute_info_list.append(exroute_info)
            exroute_list.append(exroute)
        return rec_plugin_list, exroute_info_list, exroute_list

    @classmethod
    def _update_exroute_ip_version(cls, exroute):
        if not exroute.get('ip_version') and exroute.get('destination'):
            exroute['ip_version'] = netaddr.IPNetwork(
                exroute['destination']).version

    def list_core(self, request, **kwargs):
        """查询列表入口

        :param request: neutron.api.v2.resource.Request,请求
        :param kwargs: dict,其它参数。neutron.api.v2.resource.resource()以键值对参数传递。
        :return: list
        """
        ac_bfd_validator.validate_pre_bfd_name(self._pre_bfd_name)
        LOG.debug('[AC]list bfd router start:%s,kwargs=%s', repr(request), kwargs)
        result = self.list_mec_exroutes(request.context, kwargs.get('id'), type_value=kwargs.get('type_value'),
                                        nexthop_value=kwargs.get('nexthop_value'))
        result.pop('exroutes_list', None)
        result.pop('id', None)
        LOG.debug('[AC]list bfd router result:%s', result)
        return result

    def delete_core(self, request, router_id, **kwargs):
        """查询列表入口

        :param request: neutron.api.v2.resource.Request,请求
        :param router_id: str,route id
        :param kwargs: dict,其它参数。neutron.api.v2.resource.resource()以键值对参数传递。
        :return: list
        """
        context = request.context
        exroutes_info = kwargs.get("body")
        ac_bfd_validator.validate_pre_bfd_name(self._pre_bfd_name)
        LOG.info('[AC] Begin to delete exroutes %(ex)s for router %(id)s',
                 {'ex': exroutes_info, 'id': router_id})
        ac_osprofiler.record_chain_start("delete exroutes start,router id:" +
                                         router_id + str(exroutes_info))
        exroutes_deleted = []
        exroutes_failed = []
        exroutes_dict_del = self.pre_delete_exroutes(context, router_id, exroutes_info)

        for exroute in exroutes_dict_del.get('exroutes_del'):
            exroute_to_format = self._get_exroute_format_custom(exroute)
            try:
                exroute_info = ACExRouteModel.ac_model_format(
                    exroute_to_format, router_id)
                exroute_info.pop('ecmp-enable', None)
                self.ac_reliability.update_plugin_record(
                    context, router_id, exroute_info,
                    ac_cnst.NW_HW_DELETE_EXROUTE)
                exroutes_deleted.append(exroute_to_format)
            except Exception as ex:
                LOG.error('[AC] Huawei AC delete exroute %(route)s '
                          'failed: %(ex)s', {'route': exroute, 'ex': ex})
                exroutes_failed.append(exroute)

                LOG.error('[AC] Huawei AC delete exroutes %s failed.',
                          exroutes_failed)
                ac_osprofiler.record_chain_exception_end("delete exroutes fail")

        exroutes_dict = self.delete_mec_exroutes(
            context, router_id, {'exroutes': exroutes_deleted})
        if exroutes_failed:
            LOG.error('[AC] Huawei AC remove exroutes %s failed.',
                      exroutes_failed)
            ac_osprofiler.record_chain_exception_end("remove exroutes fail")
            raise BadRequest(resource='exroutes', msg=exroutes_failed)

        exroutes_dict.pop('exroutes_list', None)
        exroutes_dict.pop('id', None)

        LOG.info('[AC] Huawei AC delete exroutes successfully.')
        ac_osprofiler.record_chain_end_with_reason("delete exroute success")
        return exroutes_dict

    def _get_exroute_format_custom(self, exroute):
        exroute_format = {'type': exroute.get('type')}
        if 'nexthop' in exroute:
            exroute_format['nexthop'] = \
                str(netaddr.IPAddress(exroute['nexthop']))
        if 'destination' in exroute:
            exroute_format['destination'] = self._get_destination(exroute)
        if 'ports' in exroute:
            exroute_format['ports'] = exroute['ports']
        return exroute_format

    @classmethod
    def _get_destination(cls, exroute):
        return str(netaddr.IPNetwork(exroute['destination']).cidr)
