# !/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
"""
查询、创建、删除 dryRun 任务
2022-06-02 16:24 创建
"""

import requests
from oslo_config import cfg
from oslo_serialization import jsonutils

from networking_huawei.common.exceptions import ACHttpException

try:
    from neutron.common.exceptions import NotFound
except ImportError:
    from neutron_lib.exceptions import NotFound

from networking_huawei.common.constants import REST_GET, REST_POST, REST_DELETE
from networking_huawei.drivers.ac.client.restclient import ACReSTClient
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.common.neutron_compatible_util import ac_log
from networking_huawei.drivers.ac.extensions.dry_run.dry_run_task import Dry_run_task
from networking_huawei.drivers.ac.plugins.abstract_neutron_service_plugin import AbstractServicePlugin

LOG = ac_log.getLogger(__name__)


class DryRunTasksNotFound(NotFound):
    """DryRunTasks not found"""
    message = "DryRunTasks %(dry_run_task_id)s could not be found."


class DryRunTaskModel:

    def __init__(self, **kwargs):
        self.id = kwargs.get('id')
        self.name = kwargs.get('name')
        self.status = kwargs.get('status', 'ERROR').upper()
        if self.status not in {'ACTIVE', 'BUILDING', 'ERROR', 'DELETING'}:
            raise ValueError('The value of status is illegal:%s' % kwargs)
        self.reason = kwargs.get('reason')
        self.created_at = kwargs.get('created_at')

    def to_dict(self):
        return {
            "dry_run_task": {
                'id': self.id,
                'name': self.name,
                'status': self.status,
                'reason': self.reason,
                'created_at': self.created_at
            }
        }


class DryRunTaskPlugin(AbstractServicePlugin):
    """dry run 任务插件"""

    def __init__(self):
        super(DryRunTaskPlugin, self).__init__(Dry_run_task())
        # 客户端
        validate.validate_rpc_server_ip()
        self.host_list = cfg.CONF.huawei_ac_agent_config.rpc_server_ip.replace(' ', '').lower().split(',')
        self.client = ACReSTClient()
        ac_ip = self.host_list[0] if len(self.host_list) == 1 else ac_const.DEFAULT_AC_IP
        self.urls = {
            'GET': 'https://%s:%s/controller/dc/v2/neutronapi/dry-run-tasks' % (ac_ip, ac_const.rest_server_port),
            'GET_ALL': 'https://%s:%s/controller/dc/v2/neutronapi/dry-run-task' % (ac_ip, ac_const.rest_server_port),
            'POST': 'https://%s:%s/controller/dc/v2/neutronapi/dry-run-task' % (ac_ip, ac_const.rest_server_port),
            'DELETE': 'https://%s:%s/controller/dc/v2/neutronapi/dry-run-tasks' % (ac_ip, ac_const.rest_server_port)
        }

    def show_core(self, request, pk_id, **kwargs):
        LOG.info("dryRun#request=%s,pk_id=%s,kwargs=%s", repr(request), pk_id, kwargs)
        try:
            # request: neutron.api.v2.resource.Request,response: requests.Response
            response = self.client.send(REST_GET, '%s/%s' % (self.urls.get('GET'), pk_id), pk_id, "")
            LOG.info('dryRun#ac response:%s %s', response, response.content)
            if response.status_code == requests.codes.not_found:
                raise DryRunTasksNotFound(dry_run_task_id=pk_id)
            if requests.codes.ok <= response.status_code < requests.codes.multiple_choices:
                result = DryRunTaskModel(**jsonutils.loads(response.content) if response.content else None)
                LOG.info('dryRun#show dry run task of %s success,request:%s,response:%s,result:%s', pk_id,
                         repr(request), response, result)
                return result.to_dict()
            else:
                LOG.info('dryRun#show dry run task of %s failed,request:%s,ac response:%s %s', pk_id, repr(request),
                         response, response.content)
                raise ACHttpException(response)
        except Exception as ex:
            LOG.error('query neutron status of dry run failed:%s', ex)
            raise

    def list_core(self, request, **kwargs):
        LOG.info("dryRun#request=%s,kwargs=%s", repr(request), kwargs)
        try:
            response = self.client.send(REST_GET, self.urls.get('GET_ALL'), "", "")
            LOG.info('dryRun#ac response:%s %s', response, response.content)
            if requests.codes.ok <= response.status_code < requests.codes.multiple_choices:
                ac_result = jsonutils.loads(response.content) if response.content else None
                LOG.info('dryRun#list dry run tasks success,request:%s,response:%s,result:%s',
                         repr(request), response, ac_result)
                result = dict()
                result['total_num'] = ac_result.get('totalNum')
                result['dryrun_tasks'] = ac_result.get('tasks')
                return result
            else:
                LOG.info('dryRun#list dry run tasks failed,request:%s,ac response:%s %s', repr(request),
                         response, response.content)
                raise ACHttpException(response)
        except Exception as ex:
            LOG.error('list dry run tasks failed:%s', ex)
            raise

    def create_core(self, request, body=None, **kwargs):
        # request: neutron.api.v2.resource.Request,response: requests.Response
        LOG.info("dryRun#request=%s,body=%s,kwargs=%s", repr(request), body, kwargs)
        dry_run_task = body.get('dry_run_task', {})
        ac_body = {'id': dry_run_task.get('id'), 'name': dry_run_task.get('name')}
        LOG.debug("dryRun#%s %s,body=%s", REST_POST, self.urls.get('POST'), ac_body)
        headers = {
            "Content-type": "application/json",
            "Accept": "application/json",
            "Accept-Language": "en-US",
            "xx-local-forward": "dcnservice"
        }
        response = self.client.send(REST_POST, self.urls.get('POST'), '', jsonutils.dumps(ac_body), headers=headers)
        LOG.info('dryRun#ac response:%s %s', response, response.content)
        if requests.codes.ok <= response.status_code < requests.codes.multiple_choices and \
                response.status_code != requests.codes.already_reported:
            result = DryRunTaskModel(**jsonutils.loads(response.content) if response.content else None).to_dict()
        else:
            LOG.info('dryRun#create dry run task of failed,request:%s,ac response:%s %s', repr(request), response,
                     response.content)
            raise ACHttpException(response)
        LOG.info('dryRun#create dry run task,request:%s,response:%s,%s', repr(request), response, result)
        return result

    def delete_core(self, request, pk_id, **kwargs):
        LOG.info("dryRun#request=%s,pk_id=%s,kwargs=%s", repr(request), pk_id, kwargs)
        response = self.client.send(REST_DELETE, '%s/%s' % (self.urls.get('DELETE'), pk_id), pk_id, "")
        if response.status_code < requests.codes.ok or response.status_code > requests.codes.multiple_choices:
            LOG.info('dryRun#delete dry run task failed,request:%s,ac response:%s %s', repr(request), response,
                     response.content)
            raise ACHttpException(response)
        LOG.info('dryRun#delete dry run task success,request:%s', repr(request))
        return None
