#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2020 Huawei Technologies Co. Ltd. All rights reserved.
"""
    huawei flow mirror plugin
"""

import copy

from oslo_log import log as logging

from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.client.service import DryRunStateEnum
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler
from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.db.flow_mirror import flow_mirror_db as \
    tass_db
from networking_huawei.drivers.ac.extensions.flowmirror import \
    flow_mirror as taas
from networking_huawei.drivers.ac.model.flow_mirror_model import \
    ACFlowMirrorModel
from networking_huawei.drivers.ac.plugins.dry_run import dry_run_util
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI

LOG = logging.getLogger(__name__)


def get_dry_run_state(tap_service):
    if tap_service is None:
        dry_run_state = DryRunStateEnum.NO_EXIST
    else:
        if dry_run_util.is_dry_run_data(tap_service, ac_constants.NW_HW_TAP_SERVICE):
            dry_run_state = DryRunStateEnum.DRY_RUN_DATA
        else:
            dry_run_state = DryRunStateEnum.REAL_DATA
    return dry_run_state


class HuaweiFlowMirrorPlugin(tass_db.FlowMirrorDbMixin, taas.FlowMirrorBase):
    """Implementation of the Huawei AC Flow Mirror Service Plugin."""
    supported_extension_aliases = ['huawei-flow-mirror']

    def __init__(self):
        LOG.info(_LI("[AC] Init huawei flow_mirror plugin."))
        super(HuaweiFlowMirrorPlugin, self).__init__()
        self.ac_reliability = ACReliabilityAPI(ac_constants.NW_HW_FLOW_MIRROR)
        LOG.info(_LI("[AC] Initialization finished successfully "
                     "for huawei flow mirror plugin."))

    def get_plugin_type(self):
        """get plugin type"""
        return ac_constants.NW_HW_FLOW_MIRROR

    def get_plugin_description(self):
        """ get plugin description """
        return 'Huawei flow mirror service plugin'

    def create_tap_service(self, context, tap_service):
        """ create tap service """
        LOG.info(_LI("[AC]Begin to create TAP SERVICE:%s,context=%s"), tap_service, context)
        ac_osprofiler.record_chain_start("create tap service start")

        tap_service_dict = super(HuaweiFlowMirrorPlugin, self).create_db_tap_service(context, tap_service)
        LOG.info(_LI("[AC] Create tap service record in Neutron DB successful: %s "), tap_service)
        ac_osprofiler.record_chain_start("tap service id:" + tap_service_dict['id'])
        tap_service_info = ACFlowMirrorModel.ac_model_format_service(tap_service_dict)
        try:
            LOG.debug("Start to send the request of create tap service")
            self.ac_reliability.update_plugin_record(context, tap_service_info.get('uuid'), tap_service_info,
                                                     ac_constants.NW_HW_CREATE_TAP_SERVICE)
        except Exception as ex:
            LOG.error(_LE("[AC]Huawei AC create tap service failed.Roll back:delete tap service in Neutron DB."
                          "catch:%s"), ex)
            ac_osprofiler.record_chain_exception_end("create tap service fail")
            super(HuaweiFlowMirrorPlugin, self).delete_db_tap_service(context, tap_service_info.get('uuid'))
            raise ex

        LOG.info(_LI("[AC] Huawei AC create tap service successful."))
        ac_osprofiler.record_chain_end_with_reason("create tap service success")
        return tap_service_dict

    def create_tap_flow(self, context, tap_flow):
        """ create tap flow """
        tap_flow_log = copy.deepcopy(tap_flow)['tap_flow']
        tap_flow_log = validate.validate_log_record(tap_flow_log, ac_constants.NW_HW_TAP_FLOW)
        LOG.info(_LI("[AC] Begin to create TAP FLOW: %s "), tap_flow_log)
        ac_osprofiler.record_chain_start("create tap flow start")

        tap_flow_dict = super(HuaweiFlowMirrorPlugin, self).create_db_tap_flow(context, tap_flow)
        tap_flow_log = copy.deepcopy(tap_flow_dict)
        tap_flow_log = validate.validate_log_record(tap_flow_log, ac_constants.NW_HW_TAP_FLOW)
        LOG.info(_LI("[AC] Create tap flow record in Neutron DB successful: %s "), tap_flow_log)
        ac_osprofiler.record_chain_start("tap flow id:" + tap_flow_dict['id'])
        LOG.info("tap_flow_dict:%s", tap_flow_dict)
        tap_flow_info = ACFlowMirrorModel.ac_model_format_flow(tap_flow_dict)
        try:
            LOG.debug("Start to send the request of create tap flow")
            self.ac_reliability.update_plugin_record(context, tap_flow_info.get('uuid'), tap_flow_info,
                                                     ac_constants.NW_HW_CREATE_TAP_FLOW)
        except Exception as ex:
            LOG.error(_LE("[AC]Huawei AC create tap flow failed.Roll back: delete tap flow in Neutron DB.catch: %s"),
                      ex)
            ac_osprofiler.record_chain_exception_end("create tap flow fail")
            super(HuaweiFlowMirrorPlugin, self).delete_db_tap_flow(context, tap_flow_info.get('uuid'))
            raise ex

        LOG.info(_LI("[AC] Huawei AC create tap flow successful."))
        ac_osprofiler.record_chain_end_with_reason("create tap flow success")
        return tap_flow_dict

    def delete_tap_service(self, context, tap_service_id):
        """ delete tap service """
        LOG.info(_LI("[AC] Begin to delete tap_service: %s"), tap_service_id)
        ac_osprofiler.record_chain_start("delete tap service start,ID:" + tap_service_id)
        tap_service = self.get_tap_service_by_id(context, tap_service_id)
        dry_run_state = get_dry_run_state(tap_service)
        try:
            LOG.debug("Start to send the request of delete tap_service")
            self.ac_reliability.update_plugin_record(context, tap_service_id, {}, ac_constants.NW_HW_DELETE_TAP_SERVICE,
                                                     dry_run_state=dry_run_state)
        except Exception as ex:
            LOG.error(_LE("[AC] AC delete TAP SERVICE failed for %s."), ex)
            ac_osprofiler.record_chain_exception_end("delete tap service fail")
            raise ex

        LOG.info(_LI("[AC] AC delete tap service successful."))

        try:
            # 多查询了一次，后面再优化
            super(HuaweiFlowMirrorPlugin, self).delete_db_tap_service(context, tap_service_id)
        except Exception as ex:
            LOG.error(_LE("[AC] Failed to delete tap service in Neutron DB for %s."), ex)
            raise ex
        ac_osprofiler.record_chain_end_with_reason("delete tap service success")
        LOG.info(_LI("[AC] Delete tap service record in Neutron DB successful."))

    def delete_tap_flow(self, context, tap_flow_id):
        """ delete tap flow """
        LOG.info(_LI("[AC] Begin to delete tap_flow: %s"), tap_flow_id)
        ac_osprofiler.record_chain_start("delete tap flow start,ID:" + tap_flow_id)
        tap_flow = self.get_tap_flow_by_id(context, tap_flow_id)
        if tap_flow is None:
            dry_run_state = DryRunStateEnum.NO_EXIST
        else:
            if dry_run_util.is_dry_run_data(tap_flow, ac_constants.NW_HW_TAP_FLOW):
                dry_run_state = DryRunStateEnum.DRY_RUN_DATA
            else:
                dry_run_state = DryRunStateEnum.REAL_DATA
        try:
            LOG.debug("Start to send the request of delete tap_flow")
            self.ac_reliability.update_plugin_record(context, tap_flow_id, {},
                                                     ac_constants.NW_HW_DELETE_TAP_FLOW, dry_run_state=dry_run_state)
        except Exception as ex:
            LOG.error(_LE("[AC] AC delete TAP FLOW failed for %s."), ex)
            ac_osprofiler.record_chain_exception_end("delete tap flow fail")
            raise ex

        LOG.info(_LI("[AC] AC delete tap flow successful."))

        try:
            super(HuaweiFlowMirrorPlugin, self).delete_db_tap_flow(context, tap_flow_id)
        except Exception as ex:
            LOG.error(_LE("[AC] Failed to delete tap flow in Neutron DB for %s."), ex)
            raise ex
        ac_osprofiler.record_chain_end_with_reason("delete tap flow success")
        LOG.info(_LI("[AC] Delete tap flow record in Neutron DB successful."))

    def update_tap_service(self, context, tap_service_id, tap_service):
        dry_run_util.check_whether_dryrun_to_product(
            ac_constants.NW_HW_TAP_SERVICE, ac_constants.OPER_UPDATE,
            self.get_tap_service_by_id(ncu.neutron_context.get_admin_context(), tap_service_id),
            tap_service.get('tap_service'))
        LOG.info(_LI("[dryRun]Begin to update tap service:id=%s,body=%s"), tap_service_id, tap_service)
        old_tap_service = self.get_tap_service_by_id(context, tap_service_id)
        dry_run_state = get_dry_run_state(old_tap_service)
        old_tap_service['description'] = None
        LOG.info(_LI("[dryRun]update tap service:%s,state=%s"), old_tap_service, dry_run_state)
        result = self._make_tap_service_dict(old_tap_service)
        try:
            if dry_run_state == DryRunStateEnum.DRY_RUN_DATA:
                ac_body = ACFlowMirrorModel.ac_model_format_service(result)
                LOG.info(_LI("[dryRun]update tap service to ac:%s"), ac_body)
                self.ac_reliability.update_plugin_record(
                    context, tap_service_id, ac_body, ac_constants.NW_HW_UPDATE_TAP_SERVICE,
                    dry_run_state=dry_run_state)
                # 只更新description
                super(HuaweiFlowMirrorPlugin, self).update_tap_service(context, tap_service_id, result)
        except Exception as ex:
            import traceback
            LOG.error(_LE("[dryRun]Failed to update tap service in Neutron DB for %s."), traceback.format_exc())
            raise ex
        LOG.info(_LI("[dryRun]update tap service end:%s"), result)
        return result

    def update_tap_flow(self, context, tap_flow_id, tap_flow):
        dry_run_util.check_whether_dryrun_to_product(
            ac_constants.NW_HW_TAP_FLOW, ac_constants.OPER_UPDATE,
            self.get_tap_flow_by_id(ncu.neutron_context.get_admin_context(), tap_flow_id), tap_flow.get('tap_flow'))
        LOG.info(_LI("[dryRun]Begin to update tap flow:id=%s,body=%s"), tap_flow_id, tap_flow)
        old_tap_flow = self.get_tap_flow_by_id(context, tap_flow_id)
        dry_run_state = get_dry_run_state(old_tap_flow)
        old_tap_flow['description'] = None
        LOG.info(_LI("[dryRun]update tap flow:%s,state=%s"), old_tap_flow, dry_run_state)
        result = self._make_tap_flow_dict(old_tap_flow)
        try:
            if dry_run_state == DryRunStateEnum.DRY_RUN_DATA:
                ac_body = ACFlowMirrorModel.ac_model_format_flow(result)
                LOG.info(_LI("[dryRun]update tap flow to ac:%s"), ac_body)
                self.ac_reliability.update_plugin_record(context, tap_flow_id, ac_body,
                                                         ac_constants.NW_HW_UPDATE_TAP_FLOW,
                                                         dry_run_state=dry_run_state)
                # 只更新description
                super(HuaweiFlowMirrorPlugin, self).update_tap_flow(context, tap_flow_id, result)
        except Exception as ex:
            import traceback
            LOG.error(_LE("[dryRun]Failed to update tap flow in Neutron DB for %s."), traceback.format_exc())
            raise ex
        LOG.info(_LI("[dryRun]update tap flow end:%s"), result)
        return result
