#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""ml2 driver"""

import re
import copy
from six.moves import range
from neutron.db.securitygroups_db import SecurityGroupDbMixin
try:
    from neutron.db.common_db_mixin import CommonDbMixin
except ImportError:
    from networking_huawei.drivers.ac.common.common_db_mixin import CommonDbMixin
try:
    from neutron.services.timestamp.timestamp_db import TimeStamp_db_mixin
except ImportError:
    TimeStamp_db_mixin = None
try:
    from neutron.common.exceptions import BadRequest
except ImportError:
    from neutron_lib.exceptions import BadRequest

from networking_huawei._i18n import _LE
from networking_huawei._i18n import _LI
from networking_huawei.drivers.ac.common import constants as ac_cnst
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI
from networking_huawei.drivers.ac.external.ext_if import ACKeyStoneIf
from networking_huawei.drivers.ac.model.security_group_model \
    import ACSecurityGroupModel, ACSecurityGroupRuleModel
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu

LOG = ncu.ac_log.getLogger(__name__)
TIME_FORMAT_WHOLE_SECONDS = '%Y-%m-%dT%H:%M:%S'


def _extend_resource_dict_timestamp(resource_res, resource_db):
    if (resource_db and resource_db.created_at and
            resource_db.updated_at):
        resource_res['created_at'] = \
            (resource_db.created_at.strftime(TIME_FORMAT_WHOLE_SECONDS)) + 'Z'
        resource_res['updated_at'] = \
            (resource_db.updated_at.strftime(TIME_FORMAT_WHOLE_SECONDS)) + 'Z'


class SecurityGroupDbManager(SecurityGroupDbMixin, CommonDbMixin):
    """security group Db Manager"""
    pass


class MechanismDriverSecurityGroupUtil(object):
    """Security group util"""
    security_group_db = SecurityGroupDbManager()
    ac_reliability_api = ACReliabilityAPI()

    def __init__(self):
        pass

    @classmethod
    def make_security_group_dict(cls, security_group_dict, context):
        """make security group dict"""
        try:
            if TimeStamp_db_mixin:
                security_group = cls.security_group_db.\
                    _get_security_group(context, security_group_dict['id'])
                if ncu.get_ops_version() in [ac_cnst.OPS_P, ac_cnst.OPS_Q,
                                             ac_cnst.OPS_R, ac_cnst.OPS_T, ac_cnst.OPS_W,
                                             ac_cnst.FSP_6_5, ac_cnst.FSP_21_0]:
                    _extend_resource_dict_timestamp(
                        security_group_dict, security_group)
                else:
                    TimeStamp_db_mixin().extend_resource_dict_timestamp(
                        None, security_group_dict, security_group)
        except Exception as ex:
            LOG.error(_LE('[AC] Failed to make security group dict: %s'), ex)

    @classmethod
    def create_security_group(cls, resource, event, trigger, **kwargs):
        """create security group"""
        LOG.info(_LI('[AC] Begin to create security group: %s'), kwargs.get('security_group'))
        security_group_dict = copy.deepcopy(kwargs.get('security_group'))
        ac_osprofiler.record_chain_start("create security group start,ID:"
                                         + security_group_dict['id'])
        context = kwargs.get('context')
        security_group_rule_list = \
            security_group_dict.pop('security_group_rules', [])
        cls.make_security_group_dict(security_group_dict, context)
        if not security_group_dict['tenant_id']:
            return
        security_group_info = \
            ACSecurityGroupModel.ac_model_format(security_group_dict)
        try:
            cls.ac_reliability_api. \
                update_plugin_record(context,
                                     security_group_dict['id'],
                                     security_group_info,
                                     ac_cnst.NW_HW_CREATE_SEC_GRP)
        except Exception as ex:
            LOG.error(_LE('[AC] Failed to create security group '
                          'in huawei driver: %s'), ex)
            ac_osprofiler.record_chain_exception_end(str(ex))
            raise ex
        LOG.info(_LI('[AC] Huawei AC create security group successfully.'))
        for security_group_rule in security_group_rule_list:
            rule_kwargs = {
                'context': context,
                'security_group_rule': security_group_rule
            }
            try:
                cls.create_security_group_rule(resource, event, trigger,
                                               **rule_kwargs)
            except Exception as ex:
                LOG.error(_LE('[AC] Failed to create security group rule '
                              'in security group creating: %s'), ex)

    @classmethod
    def update_security_group(cls, resource, event, trigger, **kwargs):
        """update security group"""
        LOG.info(_LI('[AC] Begin to update security group: %s, resource: %s, '
                     'event: %s, trigger: %s'), kwargs.get('security_group'),
                 resource, event, trigger)
        security_group_dict = kwargs.get('security_group')
        context = kwargs.get('context')
        cls.make_security_group_dict(security_group_dict, context)
        if not security_group_dict['tenant_id']:
            return
        security_group_info = \
            ACSecurityGroupModel.ac_model_format(security_group_dict)
        security_group_info.pop('security-rules', None)
        try:
            ac_osprofiler.record_chain_start("update security group start,ID:"
                                             + security_group_dict['id'])
            cls.ac_reliability_api. \
                update_plugin_record(context,
                                     security_group_dict['id'],
                                     security_group_info,
                                     ac_cnst.NW_HW_UPDATE_SEC_GRP)
        except Exception as ex:
            LOG.error(_LE('[AC] Failed to update security group '
                          'in huawei driver: %s'), ex)
            ac_osprofiler.record_chain_exception_end("update security group "
                                                     "fail:" + str(ex))
            raise ex
        LOG.info(_LI('[AC] Huawei AC update security group successfully.'))

    @classmethod
    def delete_security_group(cls, resource, event, trigger, **kwargs):
        """delete security group"""
        LOG.info(_LI('[AC] Begin to delete security group: %s, resource: %s, '
                     'event: %s, trigger: %s'), kwargs.get('security_group_id'),
                 resource, event, trigger)
        security_group_id = kwargs.get('security_group_id')
        context = kwargs.get('context')
        try:
            ac_osprofiler.record_chain_start("delete security group start,ID:"
                                             + security_group_id)
            cls.ac_reliability_api. \
                update_plugin_record(context, security_group_id, {},
                                     ac_cnst.NW_HW_DELETE_SEC_GRP)
        except Exception as ex:
            LOG.error(_LE('[AC] Failed to delete security group '
                          'in huawei driver: %s'), ex)
            ac_osprofiler.record_chain_exception_end("delete security group "
                                                     "fail:" + str(ex))
            raise ex
        LOG.info(_LI('[AC] Huawei AC delete security group successfully.'))

    @classmethod
    def validate_security_group_rule(cls, resource, event, trigger, **kwargs):
        """validate security group rule"""
        LOG.info(_LI('[AC] Begin to validate security group rule: %s, resource: '
                     '%s, event: %s, trigger: %s'), kwargs.get('security_group_rule'),
                 resource, event, trigger)
        security_group_rule_dict = kwargs.get('security_group_rule')
        if security_group_rule_dict.get('ethertype') == 'IPv6':
            return
        if security_group_rule_dict.get('remote_ip_prefix'):
            remote_ip, netmask = \
                str(security_group_rule_dict['remote_ip_prefix']).split('/', 1)
            if remote_ip in ac_cnst.REJECT_REMOTE_IP_PREFIXES:
                msg = 'loopback or broadcast remote ip prefix is not allowed'
                LOG.error(_LE('[AC] Failed to validate security group rule: '
                              '%s.'), msg)
                raise BadRequest(resource='security group rule', msg=msg)
            remote_ip_list = [int(ip) for ip in remote_ip.split('.')]
            LOG.info(_LI('[AC] Remote ip list: %s'), remote_ip_list)
            binary_netmask_list = \
                re.findall(r'.{8}', '1' * int(netmask) + '0' * (32 - int(netmask)))
            netmask_list = [int(mask, 2) for mask in binary_netmask_list]
            LOG.info(_LI('[AC] Net mask list: %s'), netmask_list)
            for i in range(ac_cnst.REMOTE_IP_PREFIX_LEN):
                if remote_ip_list[i] != (netmask_list[i] & remote_ip_list[i]):
                    msg = 'remote ip prefix format is wrong'
                    LOG.error(_LE('[AC] Failed to validate security group rule: '
                                  '%s.'), msg)
                    raise BadRequest(resource='security group rule', msg=msg)
        LOG.info(_LI('[AC] Huawei AC validate security group rule successfully.'))

    @classmethod
    def create_security_group_rule(cls, resource, event, trigger, **kwargs):
        """create security group rule"""
        LOG.info(_LI('[AC] Begin to create security group rule: %s, resource: %s, '
                     'event: %s, trigger: %s'), kwargs.get('security_group_rule'),
                 resource, event, trigger)
        security_group_rule_dict = kwargs.get('security_group_rule')
        context = kwargs.get('context')
        if not security_group_rule_dict['tenant_id']:
            return
        security_group_rule_info = \
            ACSecurityGroupRuleModel.ac_model_format(security_group_rule_dict)
        try:
            ac_osprofiler.record_chain_start("crate security group rule,ID:"
                                             + security_group_rule_dict['id'])
            cls.ac_reliability_api. \
                update_plugin_record(context,
                                     security_group_rule_dict['id'],
                                     security_group_rule_info,
                                     ac_cnst.NW_HW_CREATE_SEC_GRP_RULE)
        except Exception as ex:
            LOG.error(_LE('[AC] Failed to create security group rule '
                          'in huawei driver: %s'), ex)
            ac_osprofiler.record_chain_exception("create security group rule "
                                                 "fail:" + str(ex))
            raise ex
        LOG.info(_LI('[AC] Huawei AC create security group rule '
                     'successfully.'))

    @classmethod
    def delete_security_group_rule(cls, resource, event, trigger, **kwargs):
        """delete security group rule"""
        LOG.info(_LI('[AC] Begin to delete security group rule: %s, resource: %s, '
                     'event: %s, trigger: %s'),
                 kwargs.get('security_group_rule_id'), resource, event, trigger)
        security_group_rule_id = kwargs.get('security_group_rule_id')
        context = kwargs.get('context')
        try:
            ac_osprofiler.record_chain_start("delete security group rule,ID:"
                                             + security_group_rule_id)
            cls.ac_reliability_api. \
                update_plugin_record(context, security_group_rule_id, {},
                                     ac_cnst.NW_HW_DELETE_SEC_GRP_RULE)
        except Exception as ex:
            LOG.error(_LE('[AC] Failed to delete security group rule '
                          'in huawei driver: %s'), ex)
            ac_osprofiler.record_chain_exception("delete security group rule "
                                                 "fail:" + str(ex))
            raise ex
        LOG.info(_LI('[AC] Huawei AC delete security group rule '
                     'successfully.'))

    @classmethod
    def _make_security_group_rule_dict(cls, security_group_rule, fields=None):
        """make security group rule dict """
        from neutron.db import _utils as db_utils
        res = {'tenant_id': security_group_rule['tenant_id'],
               'id': security_group_rule['id'],
               'ethertype': security_group_rule['ethertype'],
               'security_group_id': security_group_rule['security_group_id'],
               'direction': security_group_rule['direction'],
               'port_range_min': security_group_rule['port_range_min'],
               'protocol': security_group_rule['protocol'],
               'remote_ip_prefix': security_group_rule['remote_ip_prefix'],
               'remote_group_id': security_group_rule['remote_group_id'],
               'port_range_max': security_group_rule['port_range_max'], }
        return db_utils.resource_fields(res, fields)

    @classmethod
    def _make_security_group_dict(cls, security_group, fields=None):
        """make security group dict """
        from neutron.db import _utils as db_utils
        res = {'id': security_group['id'],
               'name': security_group['name'],
               'tenant_id': security_group['tenant_id'],
               'description': security_group['description']}
        res['security_group_rules'] = [
            cls._make_security_group_rule_dict(r.db_obj)
            for r in security_group.rules]
        return db_utils.resource_fields(res, fields)

    @classmethod
    def _get_security_groups(cls, context, filters=None, fields=None,
                             sorts=None, limit=None,
                             marker=None, page_reverse=False):
        from neutron.objects import base as base_obj
        from neutron.objects import securitygroup as sg_obj
        filters = filters or {}
        pager = base_obj.Pager(
            sorts=sorts, limit=limit, marker=marker, page_reverse=page_reverse)
        sg_objs = sg_obj.SecurityGroup.get_objects(
            context, _pager=pager, validate_filters=False, **filters)
        return [cls._make_security_group_dict(obj, fields)
                for obj in sg_objs]

    @classmethod
    def sync_default_security_groups(cls):
        """synchronise default security groups"""
        LOG.info(_LI('[AC] Begin to synchronise default security groups.'))
        tenants = ACKeyStoneIf.get_tenant_list_from_keystone()
        tenants_dict = dict(zip([tenant.id for tenant in tenants],
                                [tenant.name for tenant in tenants]))
        context = ncu.neutron_context.get_admin_context()
        if ncu.get_ops_version() in [ac_cnst.FSP_6_5, ac_cnst.FSP_6_5_NFVI]:
            security_groups = cls._get_security_groups(context)
        else:
            security_groups = cls.security_group_db.get_security_groups(context)
        for security_group in security_groups:
            try:
                context.tenant_name = \
                    tenants_dict.get(security_group['tenant_id'])
            except Exception:
                context.tenant_name = security_group['tenant_id']
            if security_group['name'] == 'default':
                kwargs = {
                    'context': context,
                    'security_group': security_group,
                    'is_default': True
                }
                try:
                    cls.create_security_group(None, None, None, **kwargs)
                except Exception as ex:
                    LOG.error(_LE('[AC] Failed to create security group %s'
                                  'in default security groups synchronising: '
                                  '%s' % (security_group, ex)))

        LOG.info(_LI('[AC] Huawei AC default security groups synchronisation'
                     'is finished.'))

    @classmethod
    def security_group_no_change(cls, original, current):
        """check security group updated or no change"""
        for key in set(original.keys()):
            if key in ['revision_number', 'created_at', 'updated_at',
                       'security_groups', 'status', 'dns_assignment']:
                continue
            if original[key] != current[key]:
                return False
        return True
