#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
""" AC Reliability API """

import copy
import time

from neutron.plugins.ml2 import models
from oslo_config import cfg

from networking_huawei._i18n import _LI, _LE
from networking_huawei.common.exceptions import BlockedByNeutronSyncException
from networking_huawei.drivers.ac.client.restclient import ACReSTClient
from networking_huawei.drivers.ac.client.service import ACReSTService, DryRunStateEnum
from networking_huawei.drivers.ac.client.service import ACRestUtils
from networking_huawei.drivers.ac.client.service import RequestServiceParas
from networking_huawei.drivers.ac.common import constants as ac_constants, neutron_version_util
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import osprofiler_warp as ac_osprofiler
from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei.drivers.ac.db.lock import lockedobjects_db as lock_db
from networking_huawei.drivers.ac.db.schema import ACPluginSchema
from networking_huawei.drivers.ac.sync import http_heart_beat
from networking_huawei.drivers.ac.sync.util import ACUtil
from networking_huawei.drivers.ac.sync.validation import ACValidation

try:
    from neutron import context
except ImportError:
    from neutron_lib import context

LOG = ncu.ac_log.getLogger(__name__)

VIF_MIGRATED = 'network-vif-migrated'
WEBSOCKET_MAX_RETRY_TIMES = 5


class ACReliabilityAPI(object):
    """ AC Reliability API """

    def __init__(self, res_plugin=None):
        if res_plugin in ac_constants.NW_HW_AC_PLUGIN_RES_LIST:
            ac_constants.NW_HW_NEUTRON_SYNC_SUPPORT_RES.extend(
                ac_constants.NW_HW_AC_PLUGIN_RES_LIST.get(res_plugin))
            if res_plugin == ac_constants.NW_HW_L3 and ncu.IS_FSP:
                ac_constants.NW_HW_NEUTRON_SYNC_SUPPORT_RES.extend(
                    ac_constants.NW_HW_AC_PLUGIN_RES_LIST.get(ac_constants.NW_HW_EXROUTE))
                ac_constants.NW_HW_NEUTRON_SYNC_SUPPORT_RES.extend(
                    ac_constants.NW_HW_AC_PLUGIN_RES_LIST.get(ac_constants.NW_HW_SNATS))

        LOG.debug('Neutron sync support resource registered: %s',
                  ac_constants.NW_HW_NEUTRON_SYNC_SUPPORT_RES)

        self.db_if = ACdbInterface()
        self.validation = ACValidation()
        self.ac_util = ACUtil()
        self.rest_service = ACReSTService()
        self.client = ACReSTClient()
        self.ops_version = ncu.get_ops_version()
        self.http_heart_beat = http_heart_beat.HttpHeatBeat2AC()

        # Update the hostname
        ACCommonUtil.get_neutron_server_name()

    def update_plugin_record(self, context_info, res_id, entry_info, operation, rec_plugin=None,
                             dry_run_state=DryRunStateEnum.DEFAULT):
        """Plugin thread call this api to send entry_info to AC.

        :param context_info: context info, we can get session info from context info
        :param res_id:     Resource uuid in which the user performed the operation.
        :param entry_info: The user data which need to kept in the db for AC sync.
        :param operation:  The operation which user has performed,such as create_network, update_subnet or delete_port.
        :param rec_plugin:  The user data has been kept in the db before(huawei_ac_plugin), used for exroute now.
        :param dry_run_state: DryRunStateEnum,state use in dry run for description the state of data in database.
        :return: None
        """
        try:
            self.http_heart_beat.block_send_request_to_ac()
            self.process_plugin_record(context_info, entry_info, (res_id, operation), rec_plugin=rec_plugin,
                                       dry_run_state=dry_run_state)
        except Exception:
            LOG.exception(_LE('Exception in message reliability driver'))
            raise

    def process_plugin_record(self, context_info, entry_info, res_param, rec_plugin=None,
                              dry_run_state=DryRunStateEnum.DEFAULT):
        """Plugin thread call this api to send entry_info to AC.

        :param context_info: context info, we can get session info from context info
        :param res_param:   res_id, operation
        :param entry_info: The user data which need to kept in the db for AC sync.
        :param rec_plugin:  The user data has been kept in the db before(huawei_ac_plugin), used for exroute now.
        :param dry_run_state: DryRunStateEnum,state use in dry run for description the state of data in database.
        :return: None
        """

        res_id, operation = res_param
        (opr, res_type) = ACUtil.split_operation(operation)
        entry_log = copy.deepcopy(entry_info)
        entry_log = validate.validate_log_record(entry_log, res_type)
        LOG.info(_LI('AC request to update plugin record, '
                     'operation: %s, rest_info: %s'), operation, entry_log)
        if hasattr(context_info, 'session'):
            session = context_info.session
        else:
            session = self.db_if.get_session('write')

        if res_type != ac_constants.NW_HW_EXROUTE:
            self.db_if.confirm_standard_res_attributes(
                session, entry_info, (res_type, res_id, opr))
            # Optimise plugin record before insert to plugin db if need
            self._optimise_when_add_new_entry(session, res_id, operation)

        # Check if new record is dependent
        dep_seq_num_list = self.validation.validate_new_data(
            session, res_id, entry_info, operation)

        self.check_neutron_sync(context_info, opr, res_type, res_id)
        # Dependent resource
        # change dryrun data to real data, not check dependency
        if dep_seq_num_list and not (dry_run_state == DryRunStateEnum.DRY_RUN_DATA and opr == "update"):
            # ignore duplicate security group rule create request
            if self.db_if.get_duplicate_plugin_record(session, res_id, opr,
                                                      res_type):
                LOG.debug("[AC] ignore duplicate security group rule create "
                          "request")
                ac_osprofiler.record_call_chain("ignore duplicate security "
                                                "group create")
                return

            self._process_dependent_record(
                session, entry_info, dep_seq_num_list, (res_id, opr, res_type))
            ac_osprofiler.record_call_chain(
                res_type + " is dependent on other resource,will using "
                           "scheduler to send AC, return success first")
        # independent resource
        else:
            # ignore duplicate security group create request
            sec_grp = self.db_if.get_duplicate_plugin_record(
                session, res_id, opr, res_type)

            if sec_grp:
                self.db_if.update_plugin_record(
                    session, ACPluginSchema(state=ac_constants.WAIT),
                    old_value=sec_grp)
                LOG.debug("[AC] ignore duplicate security group create request")
                ac_osprofiler.record_call_chain("ignore duplicate security "
                                                "group create")
                return

            self._process_independent_record(session, entry_info, (res_id, opr, res_type), rec_plugin,
                                             dry_run_state=dry_run_state)

    def _is_blocked_by_neutron_sync(self, ctx):
        """ whether neutron sync is running """
        if hasattr(ctx, 'session'):
            session = ctx.session
        else:
            session = self.db_if.get_session('write')
        sync_res = self.ac_util.get_current_neutron_sync_res(
            session)
        if sync_res:
            return True
        return False

    def check_neutron_sync(self, ctx, operation='', res_type='', res_id=''):
        """ check neutron sync """
        if self._is_blocked_by_neutron_sync(ctx):
            LOG.error("[AC] Neutron sync is running, operation is blocked")
            alarm_description = {
                'operation': operation,
                'res_type': res_type,
                'res_id': res_id,
                'reason': "Neutron sync is running, operation is blocked"
            }
            self.client.send_cloud_alarm(alarm_description)
            raise BlockedByNeutronSyncException

    def _process_dependent_record(self, session, entry_info, dep_seq_num_list,
                                  res_param):
        res_id, opr, res_type = res_param
        LOG.debug("Add the record in plugin db, id:%s, "
                  "opr:%s, res_type:%s", res_id, opr, res_type)
        rec = self.db_if.create_plugin_record(
            session, entry_info, (res_id, opr, res_type), ac_constants.WAIT)
        seq_num = rec.seq_num
        for dep_seq_num in dep_seq_num_list:
            LOG.debug("Add the dependency in validation db, "
                      "seq_num:%s, dep_seq_num:%s",
                      seq_num, dep_seq_num)
            self.validation.add_validations(session,
                                            seq_num,
                                            dep_seq_num)
            ac_osprofiler.record_call_chain(res_id + " dependent on " +
                                            str(dep_seq_num))

        if ACUtil.check_resource_have_state(res_type):
            if res_type == ac_constants.NW_HW_PORTS:
                port = self.db_if.get_neutron_record(session, res_id, res_type)
                if ACReliabilityAPI.no_need_set_pending_port(port, opr):
                    return
            self.set_pending_status(session, seq_num,
                                    (opr, res_type, res_id))

    @staticmethod
    def no_need_set_pending_port(port, opr):
        """ whether port need set pending status """
        if ACUtil.is_compute_port(port):
            LOG.debug("[AC] VM port need not set pending status.")
            return True
        if port and not port.get('device_id') and \
                not port.get('device_owner') and opr == 'update':
            LOG.debug(
                "[AC] Need not set pending status when update port"
                " without device_owner/device_id.")
            return True
        return False

    def set_pending_status(self, session, seq_num, res_param):
        """ set pending status """
        opr, res_type, res_id = res_param
        (method, _) = ACUtil.get_method_and_resource_url(
            '%s_%s' % (opr, res_type))
        pending_status = ACRestUtils.get_operation_pending_status(
            method)
        if self.ops_version in [ac_constants.OPS_EZ_M] and res_type in \
                [ac_constants.NW_HW_NETWORKS, ac_constants.NW_HW_ROUTERS]:
            pending_status = ac_constants.NEUTRON_STATUS_BUILD
        if self.ops_version not in [ac_constants.OPS_EZ_M] or \
                (self.ops_version in ac_constants.OPS_EZ_M and
                 res_type != ac_constants.NW_HW_PORTS):
            if not seq_num:
                return
            if not self.db_if.get_plugin_record(session, seq_num):
                return
            self.db_if.update_neutron_db_state(
                session, res_id, res_type, pending_status)

    def _process_independent_record(self, session, entry_info, res_param, rec_plugin_input=None,
                                    dry_run_state=DryRunStateEnum.DEFAULT):
        """ process independent record """
        res_id, opr, res_type = res_param
        LOG.debug("Add the record in plugin db, "
                  "id:%s, opr:%s, res_type:%s",
                  res_id, opr, res_type)
        if rec_plugin_input:
            rec_plugin = rec_plugin_input
        else:
            if opr == ac_constants.OPER_DELETE and \
                    res_type == ac_constants.NW_HW_PORTS:
                session = self.db_if.get_session('write')
            # make sure plugin data is deleted in case normal_sync processes the plugin data
            if opr == ac_constants.OPER_DELETE and \
                    res_type not in [ac_constants.NW_HW_EXROUTE, ac_constants.NW_HW_ROUTERS]:
                self.db_if.delete_plugin_with_res_id(session, res_id)
                self.db_if.delete_failed_resource(res_id)
            rec_plugin = self.db_if.create_plugin_record(
                session, entry_info, (res_id, opr, res_type),
                ac_constants.IN_PROCESS)

        if opr == ac_constants.OPER_DELETE:
            self.handle_delete(res_type, res_id)

        (method, resource_url) = ACUtil.get_method_and_resource_url(
            "%s_%s" % (opr, res_type))
        # send independent request to AC
        pool_id = None
        if res_type == ac_constants.NW_HW_MEMBER:
            pool_id = entry_info.pop(ac_constants.URL_POOL_ID, None)
        if res_type in [ac_constants.NW_HW_INSERT_RULE,
                        ac_constants.NW_HW_REMOVE_RULE]:
            url, entry_info = ACReliabilityAPI._get_url_and_entry_info(
                ac_constants.NW_HW_RESTFUL_V3, method, resource_url,
                res_id, entry_info)
        elif res_type in [ac_constants.NW_HW_TAP_SERVICE,
                          ac_constants.NW_HW_TAP_FLOW]:
            url, entry_info = \
                ACReliabilityAPI._get_url_and_entry_info(
                    ac_constants.NW_HW_RESTFUL_MIRROR, method,
                    resource_url, res_id, entry_info)
        elif res_type == ac_constants.NW_HW_EXROUTE:
            url = ac_constants.ADD_EXROUTES_URL if opr == 'create' \
                else ac_constants.REMOVE_EXROUTES_URL
        elif res_type == ac_constants.NW_HW_EXTERNAL_WHITELIST:
            url = ACReliabilityAPI.get_external_white_list_url(opr, res_id)
        else:
            device_owner = entry_info.get('device-owner') \
                if res_type == ac_constants.NW_HW_PORTS else None
            url = ACUtil.form_resource_url(
                ac_constants.NW_HW_URL, method, resource_url, res_id,
                pool_id=pool_id, device_owner=device_owner)
        rest_paras = RequestServiceParas(session, rec_plugin.seq_num, res_type, method, url, res_id, entry_info,
                                         dry_run_state=dry_run_state)
        self.rest_service.request_send_service(rest_paras)

    def handle_delete(self, res_type, res_id):
        """ handle delete """

        def _empty_or_only_deleting(rec_list):
            return not rec_list or (len(rec_list) == 1 and rec_list[0].user_oper == ac_constants.OPER_DELETE)

        retry_count = 0
        if res_type == ac_constants.NW_HW_EXROUTE:
            while True:
                rec_list = self.db_if.get_plugin_record_list(
                    self.db_if.get_session('write'),
                    ACPluginSchema(res_type=ac_constants.NW_HW_EXROUTE,
                                   user_oper=ac_constants.OPER_CREATE,
                                   state=ac_constants.IN_PROCESS))
                retry_count = retry_count + 1
                if retry_count >= ac_constants.QUERY_RETRY_COUNT or not rec_list:
                    break
                time.sleep(ac_constants.QUERY_SLEEP_TIMES)
            if retry_count == ac_constants.QUERY_RETRY_COUNT:
                LOG.error("[AC] query status in_process exroute timeout")
        else:
            while True:
                rec_list = self.db_if.get_plugin_record_list(
                    self.db_if.get_session('write'),
                    ACPluginSchema(res_uuid=res_id, neutron_id=-1,
                                   state=ac_constants.IN_PROCESS))
                retry_count = retry_count + 1
                if retry_count >= ac_constants.QUERY_RETRY_COUNT or _empty_or_only_deleting(
                        rec_list):
                    break
                time.sleep(ac_constants.QUERY_SLEEP_TIMES)
            if retry_count == ac_constants.QUERY_RETRY_COUNT:
                LOG.error("[AC] query status in_process port timeout,res_id:%s", res_id)

    @staticmethod
    def get_external_white_list_url(opr, res_id):
        """ get external white list url """
        url = ""
        if opr == 'create':
            url = ac_constants.EXTERNAL_WHITELIST_URL
        elif opr in ['delete', 'update']:
            url = ac_constants.EXTERNAL_WHITELIST_URL + "/" + res_id
        elif opr in ac_constants.EXTERNAL_WHITELIST_URL_SUFFIX:
            url = '%s/%s/%s' % (ac_constants.EXTERNAL_WHITELIST_URL,
                                res_id, ac_constants.EXTERNAL_WHITELIST_URL_SUFFIX.get(opr))
        return url

    @staticmethod
    def _get_url_and_entry_info(base_url, method, resource_url, res_id,
                                entry_info):
        """ get url and entry info """
        url = ACUtil.form_restful_url(
            base_url, method, resource_url, res_id)
        if base_url == ac_constants.NW_HW_RESTFUL_MIRROR:
            entry_info['created_at'] = entry_info['created-at']
            entry_info['updated_at'] = entry_info['updated-at']
        entry_info.pop('created-at')
        entry_info.pop('updated-at')
        return url, entry_info

    def _optimise_when_add_new_entry(self, session, res_id, operation):
        """ optimise when add new entry """
        (opr, res_type) = ACUtil.split_operation(operation)
        if opr == ac_constants.OPER_CREATE:
            return

        # When PUT, delete any old PUT data of the same uuid, when its status
        # is not in-process
        elif opr == ac_constants.OPER_UPDATE:
            rec_list = self.db_if.get_plugin_record_list(
                session,
                ACPluginSchema(res_uuid=res_id, res_type=res_type,
                               user_oper=ac_constants.OPER_UPDATE))
            for rec in rec_list:
                if rec.state == ac_constants.WAIT or \
                        rec.state == ac_constants.ERROR_RETRY:
                    dependency = self.db_if.read_dependency(
                        session, dep_seq_num=rec.seq_num)
                    if not dependency:
                        self.ac_util.delete_one_record_in_plugin_db(
                            session, rec.seq_num)

        # when DELETE, delete all record of the same uuid
        # if it is not in-process
        # if the same uuid record is being created or updated in-process
        # then throw exception
        elif opr == ac_constants.OPER_DELETE:
            rec_list = self.db_if.get_plugin_record_list(
                session, ACPluginSchema(res_uuid=res_id))
            for rec in rec_list:
                if rec.state != ac_constants.IN_PROCESS:
                    self.ac_util.delete_one_record_in_plugin_db(
                        session, rec.seq_num, new_session=True)
        return

    @staticmethod
    def _notify_nova_vif_plugged_failed(port, current_status):
        """ notify nova vif plugged failed """
        from neutron.notifiers import nova
        LOG.info(_LI('[AC] Begin to notify nova vif plugged, port %(port)s,'
                     'current status %(status)s'),
                 {'port': port, 'status': current_status})
        nova_notifier = nova.Notifier()
        _notify_event = [
            {'server_uuid': port.device_id,
             'name': nova.VIF_PLUGGED,
             'status': 'failed',
             'tag': port.id}]
        nova_notifier.send_events(_notify_event)
        return

    def notify_nova_vif_plugged_and_update_status(self, res_id, current_status,
                                                  migrated=False):
        """ notify nova vif plugged and update status """
        session = self.db_if.get_session('write')
        port = self.db_if.get_neutron_record(
            session, res_id, ac_constants.NW_HW_PORTS)
        if not port:
            return
        if port.status == ac_constants.NEUTRON_STATUS_ERROR:
            return
        if port.status in [ac_constants.NEUTRON_STATUS_DOWN, ac_constants.NEUTRON_STATUS_ACTIVE]:
            from neutron.notifiers import nova
            nova_notifier = nova.Notifier()
            _notify_event = [
                {'server_uuid': port.device_id,
                 'name': VIF_MIGRATED if migrated else nova.VIF_PLUGGED,
                 'status': nova.NEUTRON_NOVA_EVENT_STATUS_MAP.get(
                     current_status),
                 'tag': port.id}]
            nova_notifier.send_events(_notify_event)
            self.db_if.update_neutron_db_state(
                session, res_id, ac_constants.NW_HW_PORTS, current_status)

    def get_port_binding_info(self, port, session, status, res_id):
        """ get port binding info """
        if port['status'] == ac_constants.NEUTRON_STATUS_UNBOUND:
            ACReliabilityAPI._notify_nova_vif_plugged_failed(port, status)
        from neutron.plugins.ml2 import db
        if self.ops_version in [ac_constants.OPS_O, ac_constants.OPS_P]:
            ctx = context.get_admin_context()
            port_db, binding = db.get_locked_port_and_binding(ctx, res_id)
        elif self.ops_version in [ac_constants.OPS_Q, ac_constants.OPS_R, ac_constants.OPS_W,
                                  ac_constants.OPS_T, ac_constants.FSP_6_5,
                                  ac_constants.FSP_21_0]:
            ctx = context.get_admin_context()
            port_db = db.get_port(ctx, res_id)
            binding = (ctx.session.query(models.PortBinding).
                       enable_eagerloads(False).
                       filter_by(port_id=res_id).
                       with_lockmode('update').
                       first())
        else:
            port_db, binding = db.get_locked_port_and_binding(session, res_id)
        return port_db, binding

    def _need_return(self, session, res_id, res_type, status):
        """status of sriov and ironic port with error status, sriov port with
        admin state down, dhcp port with unbound status, will not be updated
        """
        port = self.db_if.get_neutron_record(session, res_id, res_type)
        if ACUtil.is_compute_port(port):
            port_db, binding = self.get_port_binding_info(port, session, status, res_id)
            if port_db is None or binding is None:
                return True
            if binding.vnic_type not in ["direct", "baremetal"] \
                    and port['status'] != ac_constants.NEUTRON_STATUS_ERROR \
                    and not cfg.CONF.huawei_ac_config.vhost_user:
                return True
            if binding.vnic_type in ['direct'] and not port['admin_state_up']:
                return True
        if self._check_need_return(session, port, res_id, res_type, status):
            return True
        if port['device_owner'] == 'network:dhcp' and port['status'] == 'UNBOUND':
            return True
        return False

    @neutron_version_util.restrict_version(ac_constants.FSP_21_0)
    def _check_need_return(self, *args):
        port = args[1]
        try:
            from neutron.objects.trunk import models as neutron_models
        except ImportError:
            return False
        if ACUtil.is_subport(port):
            return self._check_need_return_subport(neutron_models, *args)
        return self._check_need_return_compute_port(neutron_models, *args)

    def _check_need_return_subport(self, neutron_models, *args):
        session, port, res_id, res_type, status = args
        trunk_info = session.query(neutron_models.Trunk).filter_by(id=port['device_id']).first()
        parent_port_id = trunk_info.get('port_id', None)
        if not parent_port_id:
            return False
        parent_port_info = ncu.get_port_binding(session, parent_port_id)
        # subport状态上报时，如果对应的parent port的host为空，但是返回的status为ACTIVE，则将子port状态更新为DOWN
        if parent_port_info and not parent_port_info.host and status == ac_constants.NEUTRON_STATUS_ACTIVE:
            LOG.info('[AC] Ignore updating subport status from AC, port_id: %s, rcv status: %s', res_id,
                     status)
            self.db_if.update_neutron_db_state(session, res_id, res_type, ac_constants.NEUTRON_STATUS_DOWN)
            return True
        # subport状态上报时，如果对应的parent port的host不为空，但是返回的status为DOWN，则不修改该port的状态
        if parent_port_info and parent_port_info.host and status == ac_constants.NEUTRON_STATUS_DOWN:
            subport_info = ncu.get_port_binding(session, res_id)
            LOG.info('[AC] Ignore updating subport status from AC, port_id: %s, rcv status: %s, cur status: %s', res_id,
                     status, subport_info.status)
            self.db_if.update_neutron_db_state(session, res_id, res_type, subport_info.status)
            return True
        return False

    def _check_need_return_compute_port(self, neutron_models, *args):
        session, port, res_id, res_type, status = args
        trunk_info = session.query(neutron_models.Trunk).filter_by(port_id=res_id).first()
        if not trunk_info:
            return False
        port_binding = ncu.get_port_binding(session, res_id)
        # 带trunk信息的port，如果port的host为空，但是返回的status为ACTIVE，则将该port状态更新为DOWN
        if port_binding and not port_binding.host and status == ac_constants.NEUTRON_STATUS_ACTIVE:
            LOG.info('[AC] Ignore updating parent port status from AC, port_id: %s, rcv status: %s', res_id,
                     status)
            self.db_if.update_neutron_db_state(session, res_id, res_type, ac_constants.NEUTRON_STATUS_DOWN)
            return True
        # 带trunk信息的port，如果port的host不为空，但是返回的status为DOWN，则不修改该port的状态
        if port_binding and port_binding.host and status == ac_constants.NEUTRON_STATUS_DOWN:
            LOG.info('[AC] Ignore updating parent port status from AC, port_id: %s, rcv status: %s, cur status: %s',
                     res_id, status, port_binding.status)
            self.db_if.update_neutron_db_state(session, res_id, res_type, port_binding.status)
            return True
        return False

    @lock_db.wrap_db_lock(lock_db.RESOURCE_STATUS_REPORT)
    def update_resource_status(self, res_id, opr, res_type, status):
        """AC Agent call this api to operate plugin db &.validation db
        & neutron db

        :param res_id:         Resource uuid in which the user
        performed the operation.

        :param opr:        The operation which user has performed,
        such as create, update.

        :param res_type    resource type ,such as port

        :param status      record 's neutron db status, such as ACTIVE, DOWN

        :return:           None
        """
        LOG.info(_LI("[AC]Get the record from plugin db, id:%s, opr:%s, "
                     "res_type:%s, status:%s"), res_id, opr, res_type, status)
        retry_time = 0
        session = self.db_if.get_session('write')
        while retry_time < WEBSOCKET_MAX_RETRY_TIMES - 1:
            rec_list = self.db_if.get_plugin_record_list(
                session,
                ACPluginSchema(neutron_id=-1, res_uuid=res_id, user_oper=opr,
                               res_type=res_type))

            if not rec_list:
                LOG.debug("Resource is empty, wait for request reponse")
                retry_time += 1
                time.sleep(ac_constants.REPORT_STATUS_INTERVAL_TIME)
                continue
            retry_time_add, rtn_flg = self._update_resource_status_plugin_rec(
                rec_list, (session, res_id, res_type, status))
            if rtn_flg:
                return
            retry_time += retry_time_add
            if retry_time < WEBSOCKET_MAX_RETRY_TIMES - 1:
                session = self.db_if.get_session('write')

        websocket_msg = {"res_id": res_id,
                         "operation": opr,
                         "res_type": res_type,
                         "status": status}
        self.db_if.save_websocket_msg(websocket_msg, session)

    def _update_resource_status_plugin_rec(self, rec_list, prams):
        """update_resource_status call"""
        session, res_id, res_type, status = prams
        result = 0
        for plugin_rec in rec_list:
            if (plugin_rec.state == ac_constants.COMPLETE) or \
                    (plugin_rec.state == ac_constants.SUSPEND) or \
                    (plugin_rec.state == ac_constants.NEUTRON_SYNC):
                seq_num = plugin_rec.seq_num
                if status == ac_constants.NEUTRON_STATUS_ERROR:
                    self.ac_util.delete_one_record_in_plugin_db(
                        session, seq_num, on_error=True, new_session=True)
                    LOG.debug("Resource error, delete all resources "
                              "that depend on this")
                else:
                    deleted_flag = self.db_if.delete_plugin_record(
                        session, rec=plugin_rec, new_session=True)
                    if not deleted_flag:
                        continue
                    LOG.debug("Delete the dependency from validation "
                              "db, seq_num:%s", seq_num)
                    self.db_if.delete_dependency(seq_num)
                LOG.debug("Update neutron db state res_type:%s, "
                          "status:%s", res_type, status)
                # For VM port, port status is managed by OVS, not AC-AGENT
                if res_type == ac_constants.NW_HW_PORTS \
                        and (status != ac_constants.NEUTRON_STATUS_ERROR) \
                        and self._need_return(session, res_id, res_type,
                                              status):
                    return result, True

                self.db_if.update_neutron_db_state(session, res_id,
                                                   res_type, status)
                return result, True
            else:
                continue

        result += 1
        time.sleep(ac_constants.REPORT_STATUS_INTERVAL_TIME)
        return result, False
