#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""AC util"""

import datetime

from oslo_config import cfg
from oslo_utils import uuidutils
from six.moves import urllib

from networking_huawei._i18n import _LI, _LE
from networking_huawei.common import constants
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.db.ac_proc_status.ac_proc_status_db import ACProStatusDbMixin
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei.drivers.ac.db.schema import ACPluginSchema

LOG = ncu.ac_log.getLogger(__name__)

NEUTRON_SYNC_RESULT_DICT = {
    ac_constants.NW_HW_SUCCESS: 'success',
    ac_constants.NW_HW_ERROR: 'failure',
    ac_constants.NW_HW_TIMEOUT: 'time-out',
    ac_constants.NW_HW_SKIP_SYNC: 'skip-sync'
}
NEUTRON_SYNC_METHOD_DICT = {
    ac_constants.OPER_CREATE: constants.REST_POST,
    ac_constants.OPER_UPDATE: constants.REST_UPDATE,
    ac_constants.OPER_DELETE: constants.REST_DELETE
}


class ACUtil(object):
    """ ac util """

    def __init__(self):
        self._dbInf = ACdbInterface()

    # split operation to two parameters, method and resource url
    @staticmethod
    def get_method_and_resource_url(operation):
        """ get method and resource url """
        if operation not in ac_constants.NW_HW_NEUTRON_RESOURCES.keys():
            raise AssertionError("Operation invalid!")
        opr = ac_constants.NW_HW_NEUTRON_RESOURCES.get(operation)['method']
        res_type = ac_constants.NW_HW_NEUTRON_RESOURCES.get(operation)['rsrc']
        return opr, res_type

    @staticmethod
    def split_operation(operation):
        """ split operation to two parameters, opr&resource_type """
        if operation not in ac_constants.NW_HW_NEUTRON_RESOURCES.keys():
            raise AssertionError("Operation invalid!")
        result_list = operation.split('_', 1)
        return result_list[0], result_list[1]

    def delete_one_record_in_plugin_db(
            self, session, seq_num, on_error=None, new_session=False):
        """
        Find all dependency and clear all dependency and dependent records.
        on_error=True will help to set the neutron-db as ERROR
        """
        seq_num_list = [seq_num]

        for seq_value in seq_num_list:
            try:
                # Set the neutron-db as error for failure scenarios
                self._on_error_process(session, seq_value, on_error)
            finally:
                # delete this rec in plugin db
                LOG.debug("Delete plugin record, seq_num:%s", seq_value)
                current = self._dbInf.get_plugin_record(session, seq_value)
                self._dbInf.delete_plugin_record(session, seq_value,
                                                 new_session=new_session)

                # Store the other records, which is depend on this for deletion
                dependencies = self._dbInf.read_dependency(
                    session, dep_seq_num=seq_value)
                for dependency in dependencies:
                    self._deal_dependency_record(session, seq_num_list,
                                                 dependency, current)

                # process dependencies that this rec depends on
                dependencies = self._dbInf.read_dependency(
                    session, res_seq_num=seq_value)
                for dependency in dependencies:
                    LOG.debug("Delete dependency seq_num:%s, dep_num:%s",
                              dependency.res_seq_num, dependency.dep_seq_num)
                    self._dbInf.delete_dependency(rec=dependency)

    def _deal_dependency_record(self, session, seq_num_list, dependency, current):
        depend = self._dbInf.get_plugin_record(
            session, dependency.res_seq_num)
        no_need_append = self._if_need_append(current, depend)
        if dependency.res_seq_num in seq_num_list:
            return
        if not no_need_append:
            seq_num_list.append(dependency.res_seq_num)
        else:
            LOG.debug("Delete dependency seq_num:%s, "
                      "dep_num:%s", dependency.res_seq_num,
                      dependency.dep_seq_num)
            self._dbInf.delete_dependency(rec=dependency)

    def _on_error_process(self, session, seq_value, on_error):
        """ on error process """
        if not on_error:
            return
        plugin_rec = self._dbInf.get_plugin_record(
            session, seq_num=seq_value)
        if plugin_rec:
            if plugin_rec.res_type in \
                    ac_constants.RESOURCE_HAVE_STATE:
                self._dbInf.update_neutron_db_state(
                    session,
                    plugin_rec.res_uuid,
                    plugin_rec.res_type,
                    ac_constants.NEUTRON_STATUS_ERROR)
            else:
                LOG.info(_LI("The resource %s don't need to "
                             "update status"), plugin_rec.res_type)
            # Check if need to update ac process status
            self._update_ac_process_state(plugin_rec.res_type,
                                          plugin_rec.res_uuid)

    def _if_need_append(self, current, depend):
        """ if need append """
        no_need_append = self._if_append_dhcp(current, depend) or \
                         self._if_append_network(current, depend)
        return no_need_append

    @staticmethod
    def _if_append_dhcp(current, depend):
        """ if append dhcp """
        no_need_append = (
                current is not None and depend is not None and
                current.res_type == ac_constants.NW_HW_PORTS and
                current.user_oper == ac_constants.OPER_UPDATE and
                depend.res_type == ac_constants.NW_HW_SUBNETS and
                depend.user_oper == ac_constants.OPER_DELETE and
                current.data.get('device-owner') == 'network:dhcp' and
                current.data.get('network-id') ==
                depend.data.get('network-id')
        )
        return no_need_append

    @staticmethod
    def _if_append_network(current, depend):
        """ if append network """
        no_need_append = (
                current is not None and depend is not None and
                current.res_type == ac_constants.NW_HW_SUBNETS and
                current.user_oper == ac_constants.OPER_DELETE and
                depend.res_type == ac_constants.NW_HW_NETWORKS and
                depend.user_oper == ac_constants.OPER_DELETE and
                current.data.get('network-id') ==
                depend.data.get('uuid')
        )
        return no_need_append

    @staticmethod
    def _update_ac_process_state(res_type, res_id):
        """support port resource."""
        if res_type not in [ac_constants.NW_HW_PORTS]:
            return
        # check if has configed extension plugin for ac process status
        if (ncu.IS_FSP and ac_constants.NW_HW_AC_STATUS_PORT in
                cfg.CONF.ml2.extension_drivers):
            device_dict = {
                'id': res_id,
                'type': res_type,
                'ac_proc_status': 'ERROR'
            }
            ac_proc_db_func = ACProStatusDbMixin()
            ac_status_dict = ac_proc_db_func. \
                update_db_ac_proc_status(device_dict)
            LOG.info(_LI("[AC] get sync msg timeout or error, update ac "
                         "process status to ERROR, %s."), ac_status_dict)

    @staticmethod
    def form_neutron_sync_url(base_url, request_type):
        """ form neutron sync url """
        return '%s/huawei-ac-netnsync:%s' % (base_url, request_type)

    @staticmethod
    def form_restful_url(base_url, method, res_type, res_id=None):
        """ form restful url """
        LOG.info(_LI('[AC] Begin to form RESTFUL URL for method: %s, res_type:'
                     ' %s'), method, res_type)
        if res_type in [ac_constants.NW_HW_INSERT_RULE,
                        ac_constants.NW_HW_REMOVE_RULE]:
            container = ac_constants.NW_HW_RESTFUL_FIREWALL_POLICY
            return "%s%s/%s/%s" % (base_url, container, res_id, res_type)
        elif res_type in [ac_constants.NW_HW_TAP_SERVICE,
                          ac_constants.NW_HW_TAP_FLOW]:
            if res_id and method == constants.REST_DELETE:
                return ac_constants.NW_HW_FLOW_MIRROR_URL_PREFIX \
                    + res_type + 's/' + res_id
            return ac_constants.NW_HW_FLOW_MIRROR_URL_PREFIX + res_type + 's'
        else:
            LOG.error(_LE('[AC] This res_type: %s does not support restful '
                          'url.'))
            return None

    @staticmethod
    def form_resource_url(base_url, method, res_type, res_id=None, **kwargs):
        """ form resource url """
        conditions = {'limit': kwargs.get('limit', 1),
                      'offset': kwargs.get('offset', 1)}
        if kwargs.get('cond'):
            conditions.update(kwargs.get('cond'))
        cond_url = urllib.parse.urlencode(conditions)
        huawei_cfg = "huawei-ac-neutron"
        res_containers = "neutron-cfg"
        if kwargs.get('pool_id'):
            res_containers += "/pools/pool/%s" % str(kwargs.get('pool_id'))
        container = ACUtil.get_container(res_type)
        if method == constants.REST_POST:
            return "%s/%s:%s/%s" % (
                base_url, huawei_cfg, res_containers, container)
        elif method == constants.REST_GET:
            if res_id:
                return "%s/%s:%s/%s/%s/%s?%s" % (
                    base_url, huawei_cfg, res_containers, container, res_type,
                    res_id, cond_url)
            return "%s/%s:%s/%ss?%s" % (
                base_url, huawei_cfg, res_containers, res_type, cond_url)
        else:
            url = "%s/%s:%s/%s/%s/%s" % (base_url, huawei_cfg, res_containers,
                                         container, res_type, res_id)
            if kwargs.get('device_owner') and method == constants.REST_DELETE and \
                    res_type == ac_constants.NW_HW_PORTS:
                url += "?device-owner=%s" % kwargs.get('device_owner')
            return url

    @staticmethod
    def get_container(res_type):
        """ get_container """
        if res_type == ac_constants.NW_HW_IKE_POLICY:
            container = ac_constants.NW_HW_IKE_POLICIES
        elif res_type == ac_constants.NW_HW_IPSEC_POLICY:
            container = ac_constants.NW_HW_IPSEC_POLICIES
        elif res_type == ac_constants.NW_HW_VPN_SERVICE:
            container = ac_constants.NW_HW_VPN_SERVICES
        elif res_type == ac_constants.NW_HW_IPSEC_SITE_CONNECTION:
            container = ac_constants.NW_HW_IPSEC_SITE_CONNECTIONS
        elif res_type == ac_constants.NW_HW_L2BRS:
            container = ac_constants.NW_HW_L2BRS
        elif res_type[-1] == 'y':
            container = res_type[:-1] + 'ies'
        else:
            container = "%ss" % res_type
        return container

    @staticmethod
    def calculate_incremental_timeout(timeout):
        """ ncrease timeout time """
        # 2017.8.24: 1) remove the incremental time
        return timeout

    @staticmethod
    def find_neutron_sync_interval():
        """ find neutron sync interval """
        neutron_sync_time = cfg.CONF.huawei_ac_config.neutron_sync_time
        interval = ac_constants.NEUTRON_SYNC_DEFAULT_TIMEOUT_DAILY
        try:
            datetime.datetime.strptime(neutron_sync_time, '%H:%M:%S')
        except ValueError:
            try:
                datetime.datetime.strptime(neutron_sync_time, '%a %H:%M:%S')
                interval = ac_constants.NEUTRON_SYNC_DEFAULT_TIMEOUT_WEEKLY
            except ValueError:
                datetime.datetime.strptime(neutron_sync_time, '%A %H:%M:%S')
                interval = ac_constants.NEUTRON_SYNC_DEFAULT_TIMEOUT_WEEKLY

        return interval

    @staticmethod
    def calculate_neutron_sync_interval():
        """
        Calculate the time interval for the neutron server based on the time
        configured in the configuration parameter.
        """
        neutron_sync_time = cfg.CONF.huawei_ac_config.neutron_sync_time
        weekday = weekday_num = None
        interval = ac_constants.NEUTRON_SYNC_DEFAULT_TIMEOUT_DAILY
        try:
            config_time = datetime.datetime.strptime(
                neutron_sync_time, '%H:%M:%S')
        except ValueError:
            try:
                config_time = datetime.datetime.strptime(
                    neutron_sync_time, '%a %H:%M:%S')
                weekday = neutron_sync_time.split(' ', 1)[0]
            except ValueError:
                config_time = datetime.datetime.strptime(
                    neutron_sync_time, '%A %H:%M:%S')
                weekday = neutron_sync_time.split(' ', 1)[0]
        if weekday:
            dow = ['mon', 'monday', 'tue', 'tuesday', 'wed', 'wednesday',
                   'thu', 'thursday', 'fri', 'friday', 'sat', 'saturday',
                   'sun', 'sunday']
            if weekday.lower() in dow:
                weekday_num = dow.index(weekday.lower()) // 2
            interval = ac_constants.NEUTRON_SYNC_DEFAULT_TIMEOUT_WEEKLY
        current_time = datetime.datetime.utcnow()
        month = current_time.month
        year = current_time.year
        day_offset = current_time.day

        diff_time = datetime.datetime(year=year,
                                      month=month,
                                      day=day_offset,
                                      hour=config_time.hour,
                                      minute=config_time.minute,
                                      second=config_time.second)

        if weekday_num is not None:
            if weekday_num < current_time.weekday() or \
                    (weekday_num == current_time.weekday() and
                     current_time.time() >= diff_time.time()):
                day_offset = (7 - current_time.weekday()) + weekday_num
            else:
                day_offset = weekday_num - current_time.weekday()
            diff_time += datetime.timedelta(days=day_offset)
        if weekday_num is None:
            if (current_time.time()) >= config_time.time():
                diff_time += datetime.timedelta(days=1)

        # Finding the timeout interval to start neutron sync in seconds
        timeout = (diff_time - current_time).total_seconds()
        # Ensuring a minimum of 5 seconds to start the neutron-sync
        timeout = timeout if timeout > ac_constants.NEUTRON_SYNC_INITIAL_INTERVAL else \
            ac_constants.NEUTRON_SYNC_INITIAL_INTERVAL
        return timeout, interval

    @staticmethod
    def check_resource_have_state(res_type):
        """check resource have state"""
        return res_type in ac_constants.RESOURCE_HAVE_STATE

    @staticmethod
    def get_status_by_type(res_neutron, res_type):
        """ get status by type """
        if res_type == ac_constants.NW_HW_ROUTER_IF:
            return res_neutron.port.status
        return res_neutron.status

    # Compare two dict
    @staticmethod
    def compare_dict(dict1, dict2):
        """ compare dict """
        dict1_keys = set(dict1.keys())
        dict2_keys = set(dict2.keys())
        intersect_keys = dict1_keys.intersection(dict2_keys)
        added_elements = dict1_keys - dict2_keys
        removed_elements = dict2_keys - dict1_keys
        modified_elements = [res_type for res_type in intersect_keys if dict1[res_type] != dict2[res_type]]
        same_elements = [res_type for res_type in intersect_keys if dict1[res_type] == dict2[res_type]]
        LOG.debug('added_elements: %s', added_elements)
        LOG.debug('removed_elements: %s', removed_elements)
        LOG.debug('modified_elements: %s', modified_elements)
        LOG.debug('same_elements: %s', same_elements)
        compare_dict_result = list(added_elements), list(removed_elements), modified_elements, same_elements
        return compare_dict_result

    @staticmethod
    def get_leader_in_server_list(server_list):
        """ get leader in server list """
        for server in server_list:
            if server.is_leader:
                return server
        return None

    def get_current_neutron_sync_res(self, session):
        """ get current neutron sync res"""
        (status, server) = self._dbInf.check_is_neutron_sync_in_progress(
            session)
        if status and server:
            LOG.debug('check_is_neutron_sync_in_progress returns: status:%s, '
                      'server:%s', status, server)
        if not status:
            return None
        if server.state in {ac_constants.NEUTRON_SYNC}:
            return server.sync_res
        return None

    def clear_dependent_suspended_records(self, session, res_type):
        """ clear dependent suspended records """
        if res_type and res_type is ac_constants.NW_HW_PORTS:
            return

        rec_list = self._dbInf.get_plugin_record_list(
            session, ACPluginSchema(neutron_id=-1, res_type=res_type,
                                    state=ac_constants.SUSPEND))
        for record in rec_list:
            self._dbInf.delete_plugin_record(session, seq_num=record.seq_num)
            self._dbInf.delete_dependency(dep_seq_num=record.seq_num)

    def clear_neutron_sync_records(self, session, neutron_id):
        """ clear neutron sync records """
        rec_list = self._dbInf.get_plugin_record_list(
            session, ACPluginSchema(neutron_id=neutron_id,
                                    state=ac_constants.NEUTRON_SYNC))
        for record in rec_list:
            LOG.error(_LE('AC failed to update the status within configured '
                          'response time. Deleting the plugin record and '
                          'updating the neutron db status error for uuid:%s, '
                          'res_type: %s'), record.res_uuid, record.res_type)
            self._dbInf.update_neutron_db_state(
                session, record.res_uuid, record.res_type,
                ac_constants.NEUTRON_STATUS_ERROR)
            self._dbInf.delete_plugin_record(session, rec=record)

    @staticmethod
    def is_compute_port(port, is_model=False):
        """ is compute port """
        try:
            device_id = port.get('device_id') if not is_model \
                else port.get('device-id')
            device_owner = port.get('device_owner') if not is_model \
                else port.get('device-owner')
            if not device_id or not device_owner:
                return False
            if uuidutils.is_uuid_like(device_id) and device_owner.startswith('compute:'):
                return True
        except Exception:
            LOG.debug('Invalid port data.')
        return False

    @staticmethod
    def is_subport(port):
        """is subport"""
        try:
            device_id = port.get('device_id')
            device_owner = port.get('device_owner')
            if not device_id or not device_owner:
                return False
            if uuidutils.is_uuid_like(device_id) and device_owner == 'trunk:subport':
                return True
        except Exception:
            LOG.info('[AC] Invalid subport data.')
        return False

    @staticmethod
    def is_dhcp_port(port):
        """ is dhcp port """
        try:
            if port and (port.get('device_owner') == 'network:dhcp'):
                return True
        except Exception:
            LOG.debug('Invalid port data.')
        return False

    @staticmethod
    def is_baremetal_port(port, is_model=False):
        """check is baremetal port"""
        try:
            device_owner = port.get('device_owner') if not is_model else port.get('device-owner')
            vnic_type = port.get('binding:vnic_type') if not is_model else port.get('vnic-type')
            return device_owner.startswith("baremetal:") or \
                (vnic_type == "baremetal" and not device_owner.startswith('network:f5'))
        except Exception as e:
            LOG.info('[AC] Invalid port data, error: %s', str(e))
        return False

    @classmethod
    def _get_time_for_sync_record(cls, res):
        """ get time for sync record """
        if res['created_at'] and str(res['created_at'])[-1] == 'Z':
            created_time = str(res['created_at'])[:-1]
        else:
            created_time = str(res['created_at'])
        if res['updated_at'] and str(res['updated_at'])[-1] == 'Z':
            updated_time = str(res['updated_at'])[:-1]
        else:
            updated_time = str(res['created_at'])
        return created_time.split('.')[0], updated_time.split('.')[0]

    @classmethod
    def _handle_sync_record_list(cls, record_list, sync_record_list, res,
                                 res_sync_info):
        res_type, method, result, sync_type = res_sync_info
        created_time, updated_time = cls._get_time_for_sync_record(res)
        data_info = \
            "Type: %s, Operation: %s, Id: %s, Name: %s, " \
            "TenantName: %s, CreateTime: %s, UpdateTime: %s, " \
            "Result: %s\n" % \
            (res_type, NEUTRON_SYNC_METHOD_DICT.get(method), res['id'], res['name'],
             res['tenant_name'], created_time,
             updated_time, NEUTRON_SYNC_RESULT_DICT.get(result))
        record_list.append(data_info)
        data_info_dict = {'Type': res_type,
                          'Operation': NEUTRON_SYNC_METHOD_DICT.get(method),
                          'Id': res['id'],
                          'Name': res['name'],
                          'TenantName': res['tenant_name'],
                          'CreateTime': created_time,
                          'UpdateTime': updated_time,
                          'Result': NEUTRON_SYNC_RESULT_DICT.get(result),
                          'sync_type': sync_type,
                          'exec_result': 'success',
                          'error_message': ''}
        sync_record_list.append(data_info_dict)

    @classmethod
    def _get_id_msg_and_name(cls, res):
        """ get_id_msg_and_name """
        if res['res'] == ac_constants.NW_HW_EXROUTE:
            id_msg = "{'router_id':'%s', 'nexthop':'%s', " \
                     "'destination':'%s','type':'%s'}" % \
                     (res['uuid'], res['nexthop'],
                      res['destination'], res.get('type'))
        elif res['res'] == ac_constants.NW_HW_BINDPORTS:
            id_msg = 'port-id:%s,network-id:%s,host:%s,' \
                     'ops-vlan:%s,ac-vlan:%s' % \
                     (res['port_id'], res['network_id'],
                      res['host_id'], res['segmentation_id'],
                      res['ac_segment'])
        else:
            id_msg = res['uuid']
        name = res.get('name', ac_constants.EMPTY_STR)
        return id_msg, name

    @classmethod
    def get_inconsistent_rec_info(cls, res):
        """ get inconsistent resource info """
        id_msg, name = cls._get_id_msg_and_name(res)
        tenant_name = res.get('tenant-name', ac_constants.EMPTY_STR)
        created_at, updated_at = cls._get_time_for_inconsistent_rec_info(res)
        rec_info = id_msg, name, tenant_name, created_at, updated_at
        return rec_info

    @classmethod
    def _get_time_for_inconsistent_rec_info(cls, res):
        """ get created and updated time """
        from networking_huawei.drivers.ac.common import constants as ac_const
        if res.get('created_at'):
            if isinstance(res['created_at'], datetime.datetime):
                created_at = res['created_at'].strftime(ac_const.ISO8601_TIME_FORMAT)
            elif str(res['created_at'])[-1] == 'Z':
                created_at = str(res['created_at'])[:-1]
            else:
                created_at = str(res['created_at'])
        else:
            created_at = ac_constants.EMPTY_STR
        if res.get('updated_at'):
            if isinstance(res['updated_at'], datetime.datetime):
                updated_at = res['updated_at'].strftime(ac_const.ISO8601_TIME_FORMAT)
            elif str(res['updated_at'])[-1] == 'Z':
                updated_at = str(res['updated_at'])[:-1]
            else:
                updated_at = str(res['updated_at'])
        else:
            updated_at = ac_constants.EMPTY_STR
        return created_at.split('.')[0], updated_at.split('.')[0]
