#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""validation"""

from neutron.db import models_v2
from oslo_config import cfg
from oslo_serialization import jsonutils

from networking_huawei._i18n import _LI, _LE
from networking_huawei.common import exceptions as ml2_exc
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei.drivers.ac.db.schema import ACPluginSchema
from networking_huawei.drivers.ac.sync.util import ACUtil

try:
    from neutron.db.l3_db import FloatingIP
except ImportError:
    from neutron.db.models.l3 import FloatingIP

LOG = logging.getLogger(__name__)

upper_level_dict = {
    ac_constants.NW_HW_NETWORKS:              '_get_upper_level_network',
    ac_constants.NW_HW_SUBNETS:               '_get_upper_level_subnet',
    ac_constants.NW_HW_PORTS:                 '_get_upper_level_port',
    ac_constants.NW_HW_SEC_GRP_RULE:          '_get_upper_level_sec_grp_rule',
    ac_constants.NW_HW_ROUTERS:               '_get_upper_level_router',
    ac_constants.NW_HW_FIP:                   '_get_upper_level_fip',
    ac_constants.NW_HW_SNATS:                 '_get_upper_level_snat',
    ac_constants.NW_HW_VPC_CONNECTION:      '_get_upper_level_vpc_connection',
    ac_constants.NW_HW_IPSEC_SITE_CONNECTION: '_get_upper_level_ipsec',
    ac_constants.NW_HW_VPN_SERVICE:           '_get_upper_level_vpn',
    ac_constants.NW_HW_EP_GROUPS:             '_get_upper_level_ep_group',
    ac_constants.NW_HW_FIREWALL:              '_get_upper_level_firewall',
    ac_constants.NW_HW_FIREWALL_POLICY: '_get_upper_level_firewall_policy',
    ac_constants.NW_HW_LOADBALANCER:          '_get_upper_level_loadbalancer',
    ac_constants.NW_HW_LISTENER:              '_get_upper_level_listener',
    ac_constants.NW_HW_POOL:                  '_get_upper_level_pool',
    ac_constants.NW_HW_MEMBER:                '_get_upper_level_member',
    ac_constants.NW_HW_HEALTH_MONITOR:   '_get_upper_level_health_monitor',
    ac_constants.NW_HW_DNAT:                  '_get_upper_level_fip',
    ac_constants.NW_HW_EXTERNAL_SEGMENT:      '_get_upper_level_gbp_es',
    ac_constants.NW_HW_EXTERNAL_POLICY:       '_get_upper_level_gbp_ep',
    ac_constants.NW_HW_IPV6_NS_QOS_POLICY: '_get_upper_level_ipv6_ns_qos_policy',
}


class ACValidation(object):
    """AC validator"""

    def __init__(self):
        self.db_if = ACdbInterface()

    def validate_new_data(self, session, res_id, entry_info, operation):
        """validate new data"""
        LOG.info(_LI('[AC] Check the dependency of the request data '))
        (opr, res_type) = ACUtil.split_operation(operation)
        dep_upper_level = []
        dep_on_same_rec = []
        if opr == ac_constants.OPER_CREATE:
            dep_upper_level.extend(self._create_validate(session, entry_info,
                                                         (res_id, operation, res_type)))
        elif opr == ac_constants.OPER_UPDATE:
            dep_on_same_rec_result, dep_upper_level_result = self._update_valid(
                session, entry_info, (res_id, operation, res_type))
            dep_on_same_rec.extend(dep_on_same_rec_result)
            dep_upper_level.extend(dep_upper_level_result)
        dep_upper_level.extend(dep_on_same_rec)
        return dep_upper_level

    def _create_validate(self, session, entry_info, res_param):
        res_id, operation, res_type = res_param
        dep_upper_level = []
        upper_level_id_list = \
            self._get_upper_level_list(session, entry_info, operation)
        # when upper level resource is deleted in plugindb,
        # then throw exception to user
        # when upper level resource is created in plugindb,
        #  or upper level resource is update in plugindb
        # then this resource should wait
        for upper_level_id in upper_level_id_list:
            rec_list = self.db_if.get_plugin_record_list(
                session, ACPluginSchema(res_uuid=upper_level_id, neutron_id=-1))
            LOG.info(_LI("[AC] Get dependency for %s, type: %s"), res_id, res_type)
            for rec in rec_list:
                # No need to make dependency with neutron-sync records.
                if rec.state == ac_constants.NEUTRON_SYNC:
                    continue
                if res_type == ac_constants.NW_HW_VPC_CONNECTION and \
                        rec.res_type == ac_constants.NW_HW_EXROUTE:
                    continue
                if rec.user_oper == ac_constants.OPER_DELETE:
                    LOG.error(_LE('[AC] Upper level resource is being '
                                  'deleted, this resource can not be '
                                  'create now.'))
                    self._throw_exception(
                        ac_constants.OPER_CREATE,
                        res_type,
                        upper_level_id,
                        res_id)
                if rec.user_oper in [ac_constants.OPER_CREATE,
                                     ac_constants.OPER_UPDATE]:
                    dep_upper_level.append(rec.seq_num)
        return dep_upper_level

    def _update_valid(self, session, entry_info, res_param):
        res_id, operation, res_type = res_param
        dep_on_same_rec = []
        dep_upper_level = []
        # if new request is 'UPDATE', check if this uuid 'POST' or 'UPDATE'
        # is in plugin db
        rec_list = self.db_if.get_plugin_record_list(
            session, ACPluginSchema(res_uuid=res_id, neutron_id=-1))
        for rec in rec_list:
            # No need to make dependency with neutron-sync records.
            if rec.state == ac_constants.NEUTRON_SYNC:
                continue
            if rec.user_oper in [ac_constants.OPER_CREATE,
                                 ac_constants.OPER_UPDATE]:
                if self._need_processing_dependency(res_type, entry_info):
                    LOG.info(_LI("[AC] No need to process dependency of itself"))
                    continue
                else:
                    dep_on_same_rec.append(rec.seq_num)
        # when upper level resource is created in plugindb, add dependency
        # when upper level resource is deleted in plugindb, thow exception
        upper_level_id_list = \
            self._get_upper_level_list(session, entry_info, operation)
        for upper_level_id in upper_level_id_list:
            rec_list = self.db_if.get_plugin_record_list(
                session, ACPluginSchema(res_uuid=upper_level_id, neutron_id=-1))
            self._update_other_valid(rec_list, res_param, entry_info,
                                     dep_upper_level, upper_level_id)
        return dep_on_same_rec, dep_upper_level

    def _update_other_valid(self, rec_list, res_param, entry_info, dep_upper_level, upper_level_id):
        res_id, _, res_type = res_param
        for rec in rec_list:
            # No need to make dependency with neutron-sync records.
            if rec.state == ac_constants.NEUTRON_SYNC:
                continue
            if res_type == ac_constants.NW_HW_PORTS and \
                    'port-parent-id' in entry_info.get('profile'):
                profile = jsonutils.loads(entry_info.get('profile'))
                parent_port_id = profile.get('port-parent-id')
                if parent_port_id == rec.res_uuid:
                    dep_upper_level.append(rec.seq_num)
                continue
            if rec.user_oper == ac_constants.OPER_CREATE:
                dep_upper_level.append(rec.seq_num)
            elif rec.user_oper == ac_constants.OPER_DELETE:
                LOG.error(_LE('[AC] Upper level resource is being '
                              'deleted this resource can not be '
                              'updated now.'))
                self._throw_exception(
                    ac_constants.OPER_UPDATE,
                    res_type,
                    upper_level_id,
                    res_id)

    def add_validations(self, session, res_seq_num, dep_seq_num):
        """add check, when add dependency record, check the dependency record
        exist in plugin db or not
        """
        dep_res = self.db_if.get_plugin_record(
            session, seq_num=dep_seq_num)
        if dep_res:
            LOG.debug("[AC] Add dependency in validation db, "
                      "seq_num = %d, dep_seq_num = %d",
                      res_seq_num, dep_seq_num)
            self.db_if.create_dependency(
                session, res_seq_num, dep_seq_num)
        else:
            LOG.debug("Depency request is not exsit: %s",
                      dep_seq_num)

    @classmethod
    def _throw_exception(cls, opr, res_type, upper_level_id, res_id):
        (exception_method, resource_url) = \
            ACUtil.get_method_and_resource_url(
                "%s_%s" % (opr, res_type))
        exception_url = ACUtil.form_resource_url(
            ac_constants.NW_HW_URL,
            exception_method,
            resource_url,
            res_id)
        message = '%s %s error because %s is being processed. ' \
                  % (opr, res_id, upper_level_id)
        raise ml2_exc.MechanismDriverError(
            method=exception_method,
            url=exception_url,
            error=message)

    @classmethod
    def _need_processing_dependency(cls, res_type, entry_info):
        """ whether to process dependency of itself """
        if res_type == ac_constants.NW_HW_PORTS:
            if ACUtil.is_baremetal_port(entry_info, is_model=True):
                return True
        if cfg.CONF.huawei_ac_config.error_retry_count != 0:
            return False
        if res_type == ac_constants.NW_HW_PORTS:
            if ACUtil.is_compute_port(entry_info, is_model=True):
                return True
        return False

    def _get_upper_level_list(self, session, entry_info, operation):
        id_list = self._get_upper_level(
            session,
            entry_info,
            operation)
        upper_level_id_list = []
        for elem in id_list:
            upper_level_id_list.append(elem)
        upper_level_id_list = list(set(upper_level_id_list))
        return upper_level_id_list

    def _get_upper_level(self, session, entry_info, operation):
        (_, res_type) = ACUtil.split_operation(operation)
        if res_type not in upper_level_dict.keys():
            return []
        upper_levels = []
        func = getattr(self, upper_level_dict.get(res_type))
        func(session, entry_info, upper_levels)
        return upper_levels

    def _get_upper_level_network(self, _, entry_info, upper_levels):
        if 'qos-policy-id' in entry_info:
            qos_policy_id = entry_info['qos-policy-id']
            self._add_item(qos_policy_id, upper_levels)

    def _get_upper_level_subnet(self, _, entry_info, upper_levels):
        if 'network-id' in entry_info:
            network_id = entry_info['network-id']
            self._add_item(network_id, upper_levels)

    def _get_upper_level_port(self, _, entry_info, upper_levels):
        ctx = ncu.neutron_context.get_admin_context()
        if 'network-id' in entry_info:
            network_id = entry_info["network-id"]
            self._add_item(network_id, upper_levels)
        for fixed_ip in entry_info.get('fixed-ips', []):
            if 'subnet-id' in fixed_ip:
                self._add_item(fixed_ip['subnet-id'], upper_levels)
        if 'qos-policy-id' in entry_info:
            qos_policy_id = entry_info['qos-policy-id']
            self._add_item(qos_policy_id, upper_levels)
        for security_group_id in entry_info.get('security-groups', []):
            self._add_item(security_group_id, upper_levels)
        if entry_info.get('device-owner') == 'network:router_interface' and \
                entry_info.get('device-id'):
            self._add_item(entry_info['device-id'], upper_levels)
        if entry_info.get('device-owner') == 'trunk:subport' and\
                entry_info.get('device-id') and 'port-parent-id' in entry_info.get('profile'):
            try:
                trunk_info = ncu.get_service_plugin()['trunk'].\
                    get_trunk(ctx, entry_info.get('device-id'))
                self._add_item(trunk_info.get('port_id'), upper_levels)
            except Exception as ex:
                LOG.error(_LE("[AC] Can not find the trunk info: %s."), ex)

    def _get_upper_level_sec_grp_rule(self, _, entry_info, upper_levels):
        if 'security-group-id' in entry_info:
            security_group_id = entry_info['security-group-id']
            self._add_item(security_group_id, upper_levels)
        if 'remote-group-id' in entry_info:
            security_group_id = entry_info['remote-group-id']
            self._add_item(security_group_id, upper_levels)

    def _get_upper_level_router(self, _, entry_info, upper_levels):
        if 'gateway-port-id' in entry_info:
            ex_port_id = entry_info['gateway-port-id']
            self._add_item(ex_port_id, upper_levels)
        if 'interfaces' in entry_info:
            interfaces = entry_info['interfaces']
            for interface in interfaces:
                if 'port-id' in interface:
                    inner_port_id = interface['port-id']
                    self._add_item(inner_port_id, upper_levels)

    def _get_upper_level_fip(self, session, entry_info, upper_levels):
        if 'port-id' in entry_info:
            vm_port_id = entry_info['port-id']
            self._add_item(vm_port_id, upper_levels)
        if 'router-id' in entry_info:
            router_id = entry_info['router-id']
            if router_id:
                self._add_item(router_id, upper_levels)
        with session.begin(subtransactions=True):
            fip_qry = session.query(FloatingIP)
        floating_ips = fip_qry.filter_by(id=entry_info['uuid']).all()
        if floating_ips:
            floating_ip = floating_ips[0]
            self_port_id = floating_ip['floating_port_id']
            self._add_item(self_port_id, upper_levels)

    def _get_upper_level_snat(self, _, entry_info, upper_levels):
        ctx = ncu.neutron_context.get_admin_context()
        if entry_info.get('router-id'):
            self._add_item(entry_info['router-id'], upper_levels)
        if 'orignal-cidrs' in entry_info:
            l3_plugin = ncu.get_service_plugin()['L3_ROUTER_NAT']
            router = l3_plugin._get_router(ctx, entry_info['router-id'])
            if ncu.get_ops_version() in [ac_constants.OPS_Q,
                                         ac_constants.OPS_R,
                                         ac_constants.OPS_T,
                                         ac_constants.OPS_W,
                                         ac_constants.FSP_6_5,
                                         ac_constants.FSP_21_0]:
                router_attached_ports = router.attached_ports
            else:
                router_attached_ports = router.attached_ports.all()
            for rp in router_attached_ports:
                port_info = l3_plugin._core_plugin.get_port(ctx, rp.port_id)
                fixed_ips = port_info.get('fixed_ips', [{}])
                if not fixed_ips:
                    return
                subnet_id = fixed_ips[0].get('subnet_id')
                subnet = l3_plugin._core_plugin.get_subnet(ctx, subnet_id)
                subnet_cidr = subnet.get('cidr')
                self._deal_cidr_for_snat(entry_info, subnet_cidr, port_info,
                                         upper_levels)

    def _deal_cidr_for_snat(self, entry_info, subnet_cidr, port_info,
                            upper_levels):
        for orignal_cidr in entry_info.get('orignal-cidrs', {}):
            if subnet_cidr == orignal_cidr:
                self._add_item(port_info.get('id'), upper_levels)

    def _get_upper_level_vpc_connection(self, _, entry_info, upper_levels):
        if 'local-router-id' in entry_info:
            self._add_item(entry_info['local-router-id'], upper_levels)
        if 'peer-router-id' in entry_info:
            self._add_item(entry_info['peer-router-id'], upper_levels)
        if 'local-subnets' in entry_info:
            for lc_subnet in entry_info['local-subnets']:
                self._add_item(lc_subnet, upper_levels)
        if 'peer-subnets' in entry_info:
            for peer_subnet in entry_info['local-subnets']:
                self._add_item(peer_subnet, upper_levels)

    def _get_upper_level_ipsec(self, _, entry_info, upper_levels):
        if 'vpnservice-id' in entry_info:
            vpnservice_id = entry_info['vpnservice-id']
            self._add_item(vpnservice_id, upper_levels)
        if 'ikepolicy-id' in entry_info:
            ikepolicy_id = entry_info['ikepolicy-id']
            self._add_item(ikepolicy_id, upper_levels)
        if 'ipsecpolicy-id' in entry_info:
            ipsecpolicy_id = entry_info['ipsecpolicy-id']
            self._add_item(ipsecpolicy_id, upper_levels)
        if 'local-ep-group-id' in entry_info:
            ep_group_id = entry_info['local-ep-group-id']
            self._add_item(ep_group_id, upper_levels)
        if 'peer-ep-group-id' in entry_info:
            ep_group_id = entry_info['peer-ep-group-id']
            self._add_item(ep_group_id, upper_levels)

    def _get_upper_level_vpn(self, session, entry_info, upper_levels):
        if 'router-id' in entry_info:
            router_id = entry_info['router-id']
            router = self.db_if.get_neutron_record(
                session,
                router_id,
                ac_constants.NW_HW_ROUTERS)
            if router:
                self._add_item(router_id, upper_levels)
                if router.get('gw_port_id'):
                    gateway_port_id = router['gw_port_id']
                    self._add_item(gateway_port_id, upper_levels)
                port_list = session.query(models_v2.Port). \
                    filter_by(device_id=router_id).all()
                port_id_list = []
                for port in port_list:
                    port_id_list.append(port.id)
                subnetid = entry_info.get("subnet-id")
                port_list2 = session.query(models_v2.IPAllocation). \
                    filter_by(subnet_id=subnetid).all()
                port_id_list2 = []
                for port in port_list2:
                    port_id_list2.append(port.port_id)
                ports = set(port_id_list) & set(port_id_list2)
                for port_id in ports:
                    self._add_item(port_id, upper_levels)

    def _get_upper_level_ep_group(self, _, entry_info, upper_levels):
        if entry_info.get('type') == 'subnet':
            for subnet_id in entry_info.get('endpoints'):
                self._add_item(subnet_id, upper_levels)

    def _get_upper_level_firewall(self, _, entry_info, upper_levels):
        if "firewall-policy-id" in entry_info:
            firewall_policy_id = entry_info["firewall-policy-id"]
            self._add_item(firewall_policy_id, upper_levels)
        if "router-ids" in entry_info:
            router_ids = entry_info["router-ids"]
            for router_id in router_ids:
                self._add_item(router_id, upper_levels)

    def _get_upper_level_firewall_policy(self, _, entry_info, upper_levels):
        if "firewall-rules" in entry_info:
            firewall_rules = entry_info["firewall-rules"]
            for firewall_rule_id in firewall_rules:
                self._add_item(firewall_rule_id, upper_levels)

    def _get_upper_level_loadbalancer(self, _, entry_info, upper_levels):
        if "vip-subnet-id" in entry_info:
            subnet_id = entry_info["vip-subnet-id"]
            self._add_item(subnet_id, upper_levels)
        if "vip-port-id" in entry_info:
            vip_port_id = entry_info["vip-port-id"]
            self._add_item(vip_port_id, upper_levels)

    def _get_upper_level_listener(self, _, entry_info, upper_levels):
        if 'loadbalancers' in entry_info:
            loadbalancers = entry_info['loadbalancers']
            for loadbalancer in loadbalancers:
                self._add_item(loadbalancer, upper_levels)

    def _get_upper_level_pool(self, _, entry_info, upper_levels):
        if 'listeners' in entry_info:
            listeners = entry_info['listeners']
            for listener in listeners:
                self._add_item(listener, upper_levels)
        if 'loadbalancers' in entry_info:
            loadbalancers = entry_info['loadbalancers']
            for loadbalancer in loadbalancers:
                self._add_item(loadbalancer, upper_levels)

    def _get_upper_level_member(self, _, entry_info, upper_levels):
        if 'subnet-id' in entry_info:
            subnet_id = entry_info['subnet-id']
            self._add_item(subnet_id, upper_levels)
        if ac_constants.URL_POOL_ID in entry_info:
            pool_id = entry_info[ac_constants.URL_POOL_ID]
            self._add_item(pool_id, upper_levels)

    def _get_upper_level_health_monitor(self, _, entry_info, upper_levels):
        if 'pools' in entry_info:
            pools = entry_info['pools']
            for pool in pools:
                self._add_item(pool, upper_levels)

    def _get_upper_level_gbp_es(self, _, entry_info, upper_levels):
        if 'subnet-id' in entry_info:
            subnet_id = entry_info['subnet-id']
            self._add_item(subnet_id, upper_levels)

    def _get_upper_level_gbp_ep(self, _, entry_info, upper_levels):
        for external_segment_id in entry_info.get('external-segments', []):
            self._add_item(external_segment_id, upper_levels)

    def _get_upper_level_ipv6_ns_qos_policy(self, _, entry_info,
                                            upper_list):
        if 'router-id' in entry_info:
            self._add_item(entry_info['router-id'], upper_list)
        if 'port-id' in entry_info:
            self._add_item(entry_info['port-id'], upper_list)

    @classmethod
    def _add_item(cls, res_id, upper_levels):
        upper_levels.append(res_id)
