#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Websocket messages sync"""

import threading
import time
import eventlet

from networking_huawei.drivers.ac.db.schema import ACPluginSchema
from oslo_db import api as oslo_db_api

from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.db import dbif
from networking_huawei.drivers.ac.db import schema as dbschema
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI
from networking_huawei.drivers.ac.db.lock import lockedobjects_db as lock_db

LOG = ncu.ac_log.getLogger(__name__)
WEBSOCKET_MSG_SYNC_THREAD_WAIT = 15
HANDLE_WEBSOCKET_MSG_TIMEOUT = 150


class WebsocketMsgSync(object):
    """Websocket message Sync"""

    def __init__(self):
        self._db_if = dbif.ACdbInterface()
        self.reliability_api = ACReliabilityAPI(ac_const.NW_HW_ML2)
        timer_websocket_msg_sync = threading.Timer(
            0, self.websocket_msg_sync_process)
        timer_websocket_msg_sync.start()
        LOG.info("Websocket message sync thread initialization done.")

    def websocket_msg_sync_process(self):
        """Basic processing of the websocket_msg_sync_process thread.

        :return: None
        """
        try:
            with eventlet.Timeout(2 * HANDLE_WEBSOCKET_MSG_TIMEOUT, False):
                self._deal_websocket_msgs()
        except Exception as ex:
            LOG.error("[AC] Handle websocket messages with exception: %s", ex)
        finally:
            timer = threading.Timer(WEBSOCKET_MSG_SYNC_THREAD_WAIT,
                                    self.websocket_msg_sync_process)
            timer.start()

    @staticmethod
    def _is_invalid_message(websocket_message):
        msg_time = websocket_message.get("msg_time")
        msg_time_sec = int(time.mktime(msg_time.timetuple()))
        time_diff = int(time.time()) - msg_time_sec
        return time_diff > HANDLE_WEBSOCKET_MSG_TIMEOUT * 2

    def _deal_websocket_msgs(self):
        admin_ctx = ncu.neutron_context.get_admin_context()
        messaages = self._get_websocket_msgs(admin_ctx.session)
        for websocket_message in messaages:
            LOG.info("[AC] Begin to handle websocket messages: %s",
                     websocket_message)
            if self._is_invalid_message(websocket_message):
                LOG.error("[AC] Handle websocket message: %s timeout.",
                          websocket_message)
                self.delete_websocket_msg(websocket_message, admin_ctx.session)
                continue
            if self._handle_websocket_msg(websocket_message["id"],
                                          websocket_message, admin_ctx.session):
                LOG.info("[AC] Handle websocket message: %s success.",
                         websocket_message)
                self.delete_websocket_msg(websocket_message, admin_ctx.session)
                continue

    def _get_websocket_msgs(self, session=None):
        if not session:
            session = self._db_if.get_session('read')
        return session.query(dbschema.ACWebsocketMsg).all()

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES,
                               retry_interval=0,
                               inc_retry_interval=0,
                               max_retry_interval=0)
    def delete_websocket_msg(self, query_condition, session=None):
        """Create websocket message record.

        :param websocket_msg: websocket message
        :param session: Current session info where the operation is
                        performed. Used to access db.
        :return: None
        """
        if not session:
            session = self._db_if.get_session('write')

        with session.begin(subtransactions=True):
            websocket_msg = session.query(dbschema.ACWebsocketMsg). \
                filter_by(**query_condition).first()
            if websocket_msg:
                session.delete(websocket_msg)
                LOG.info("[AC] Deleted websocket message: %s", websocket_msg)

    @lock_db.wrap_db_lock(lock_db.RESOURCE_STATUS_REPORT)
    def _handle_websocket_msg(self, res_id, websocket_message, session=None):
        if not session:
            session = self._db_if.get_session('write')
        rec_list = self._db_if.get_plugin_record_list(
            session, ACPluginSchema(neutron_id=-1, res_uuid=res_id,
                                    user_oper=websocket_message["operation"],
                                    res_type=websocket_message["res_type"]))

        if not rec_list:
            LOG.info("Websocket sync msg: %s, Resource is empty.", websocket_message)
            return False

        _, rtn_flg = self.reliability_api._update_resource_status_plugin_rec(
            rec_list, (session, res_id, websocket_message["res_type"],
                       websocket_message["status"]))
        return rtn_flg
