#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""upgrade neutron db"""


from __future__ import print_function
import site
import os.path
from os import listdir
import shutil

NETWORKING_HUAWEI_MIGRATE_PATH = 'networking_huawei/drivers/ac/db/migration/' \
    'alembic_migrations/versions'
NEUTRON_MIGRATE_PATH = 'neutron/db/migration/alembic_migrations/versions'


def get_dependency_path(package_path):
    """get dependency path"""
    for site_path in site.getsitepackages():
        absolute_path = os.path.join(site_path, package_path)
        if os.path.exists(absolute_path):
            return absolute_path
    print('ERROR: Path not exists:', NEUTRON_HEAD)
    raise IOError


NEUTRON_PATH = get_dependency_path(NEUTRON_MIGRATE_PATH)
HUAWEI_PLUGIN_PATH = get_dependency_path(NETWORKING_HUAWEI_MIGRATE_PATH)
NEUTRON_HEAD = os.path.join(NEUTRON_PATH, 'HEAD')


def run():
    """run"""
    # Read the neutron head file
    if not os.path.isfile(NEUTRON_HEAD):
        print('ERROR: File not exists:', NEUTRON_HEAD)
        raise IOError
    with open(NEUTRON_HEAD) as fp_neutron_head:
        version = fp_neutron_head.read()
        if not version:
            print('ERROR: No version information in ', NEUTRON_HEAD)
            raise IOError
        neutron_version = version.rstrip('\n')
    # Read the huawei head file
    huawei_head = os.path.join(HUAWEI_PLUGIN_PATH, 'HEAD')
    if not os.path.isfile(huawei_head):
        print('ERROR: File not exists:', huawei_head)
        raise IOError
    with open(huawei_head) as fp_huawei_head:
        version = fp_huawei_head.read()
        if not version:
            print('ERROR: No version information in', huawei_head)
            raise IOError
        huawei_version = version.rstrip('\n')
    # Reading the list of huawei upgrade files
    directory_list = listdir(HUAWEI_PLUGIN_PATH)
    upgrade_files = []
    initial_file = ''
    for up_file in directory_list:
        if up_file.endswith('.py') and not up_file.startswith('__') \
                and os.path.isfile(os.path.join(HUAWEI_PLUGIN_PATH, up_file)):
            upgrade_files.append(up_file)
            if up_file.endswith('initial_branch.py'):
                initial_file = os.path.join(NEUTRON_PATH, up_file)
    if not initial_file or not upgrade_files:
        print('ERROR: Essential upgrade files not present in',
              HUAWEI_PLUGIN_PATH)
        raise IOError
    # Copying the upgrade files to the neutron directory
    for up_file in upgrade_files:
        src_path = os.path.join(HUAWEI_PLUGIN_PATH, up_file)
        dst_path = os.path.join(NEUTRON_PATH, up_file)
        shutil.copy2(src_path, dst_path)
    # Updating the previous version of plugin init file as neutron version
    with open(initial_file, 'r') as init_file:
        file_data = init_file.read()
    file_data = file_data.replace(
        'Revises: huawei', "Revises: '%s'" % neutron_version)
    file_data = file_data.replace(
        'down_revision = None', "down_revision = '%s'" % neutron_version)
    flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
    with os.fdopen(os.open(initial_file, flags, 0o400), 'w') as init_file:
        init_file.write(file_data)
    # Writing huawei plugin head as neutron head now
    with os.fdopen(os.open(NEUTRON_HEAD, flags, 0o400), 'w') as init_file:
        init_file.write(huawei_version)
    print('INFO: Copy upgrade files to neutron migration success!')


if __name__ == '__main__':
    run()
