#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2018 Huawei Technologies Co. Ltd. All rights reserved.
"""update tool"""

from __future__ import print_function

import logging
import os
import re
import shutil
import stat
import subprocess
import sys
import time
from distutils.sysconfig import get_python_lib, get_python_version

import six.moves

from networking_huawei.drivers.ac.common.file_utils import FileMeta

try:
    import commands
except ImportError:
    import subprocess as commands
try:
    import selinux
except ImportError:
    selinux = None

LOG_FILE_OLD = os.path.realpath(r'/var/log/patch/update.log')
LOG_FILE_NAME = r'update.log'
ROOT_PATH = r'/'
BACKUP_CONF_LIST = ['/etc/neutron/neutron.conf',
                    '/etc/neutron/huawei_driver_config.ini']
TIMESTAMP = time.strftime("%Y-%m-%d_%H-%M-%S", time.localtime())
CODE_MAIN_NAME = 'networking_huawei'
VERSION_FILE_NAME = 'version.info'

NEW_PARAM_RULE = {
    "encrypted_keys": [
        "ac_auth_password",
        "keystone_passwd",
        "websocket_key_password",
        "websocket_private_key"
    ]
}

OLD_PARAM_RULE = {
    "encrypted_keys": [
        "ac_auth_password",
        "keystone_passwd"
    ]
}

NORMAL_NEUTRON_STOP_TIMEOUT = 10
NEUTRON_CHECK_TIMES = 30
CHECK_INTERVAL_TIME = 2
FSP_ENCRYPT_WEBSOCKET_KEY_PWD = 'fsp_encrypt_websocket_key_pwd'
AC_ENCRYPT_WEBSOCKET_KEY_PWD = 'ac_encrypt_websocket_key_pwd'

CPS_EXT_PARAM_ADD = ["cps", "template-ext-params-add", "--service", "neutron",
                     "neutron-server", "--parameter"]

CPS_T = ['cps', '', '--service', 'neutron', 'neutron-server', '--parameter', '']

TEMPLATE_EXT_PARAMS_ADD = 'template-ext-params-add'

CPS_EXT_PARAM_DEL = ["cps", "template-ext-params-del", "--service",
                     "neutron", "neutron-server", "--parameter", "", ""]

CPS_EXT_PARAM_UPDATE = ["cps", "template-ext-params-update", "--service",
                        "neutron", "neutron-server", "--parameter",
                        "huawei_driver_config.__param_rule='", "", "'"]

CPS_COMMIT = 'cps commit'

DRIVER_CFG_PATH = os.path.realpath('/etc/neutron/huawei_driver_config.ini')

DRIVER_PREFIX = "huawei_driver_config.huawei_ac_config."

CODE_PATH = 'networking_huawei/drivers/ac/'

BEFORE_FSP_6_3_1 = ['FusionSphere6.1', 'FusionSphere6.3.0']

AC_ML2_DRIVER_PATH = 'networking_huawei/drivers/ac/plugins/ml2/driver.py'

PYTHON_VERSION = get_python_version()

LOCAL_DIST_PACKAGES_PATH = os.path.realpath(
    '/usr/local/lib/python' + PYTHON_VERSION + '/dist-packages')

SITE_PACKAGES_PATH = os.path.realpath(
    '/usr/local/lib/python' + PYTHON_VERSION + '/site-packages')

DIST_PACKAGES_PATH = os.path.realpath(
    '/usr/lib/python' + PYTHON_VERSION + '/dist-packages')

CERT_FILE = ['client.cer', 'client_key.pem', 'ssl_cacert.pem', 'trust.cer']

LOGGER = logging.getLogger(__name__)
FORMATTER = logging.Formatter('%(asctime)s - %(levelname)s - %(message)s')
# console log
CONSOLE_HANDLER = logging.StreamHandler(sys.stdout)
CONSOLE_HANDLER.setFormatter(FORMATTER)
LOGGER.addHandler(CONSOLE_HANDLER)
LOGGER.setLevel(logging.INFO)


class UpdateFunBase(object):
    """update function base"""

    def __init__(self):
        self.neutron_server_cmd_flag = False

    def get_history_path(self):
        """get history path"""
        history_path = os.path.join(self.get_package_home_path(), "huawei_plugin_update_history")
        if not os.path.exists(history_path):
            os.makedirs(history_path)
        elif os.path.isfile(history_path):
            os.remove(history_path)
            os.makedirs(history_path)
        return history_path

    def get_version_file(self):
        """get version file"""
        return os.path.join(self.get_code_main_path(), VERSION_FILE_NAME)

    def get_code_main_path(self):
        """get code _main path"""
        return os.path.join(self.get_package_home_path(), CODE_MAIN_NAME)

    # prepare
    def get_current_version(self):
        """get current version"""
        update_log = os.path.join(self.get_history_path(), LOG_FILE_NAME)
        version_info_now = 'default'
        if not os.path.exists(update_log):
            # create the new directory and file.
            status, output = commands.getstatusoutput('touch %s' % update_log)
            if status:
                LOGGER.error(output)
                exit(0)
        else:
            version_file = self.get_version_file()
            if not os.path.isfile(version_file):
                return version_info_now
            status, output = commands.getstatusoutput("awk '{print;exit}' %s" % version_file)
            if status == 0 and output:
                version_info_now = output
        return version_info_now

    def get_package_home_path(self):
        """get package home path"""
        if os.path.isdir(os.path.join(LOCAL_DIST_PACKAGES_PATH, CODE_MAIN_NAME)):
            return LOCAL_DIST_PACKAGES_PATH
        if os.path.isdir(os.path.join(SITE_PACKAGES_PATH, CODE_MAIN_NAME)):
            return SITE_PACKAGES_PATH
        if os.path.isdir(os.path.join(DIST_PACKAGES_PATH, CODE_MAIN_NAME)):
            return DIST_PACKAGES_PATH
        self.neutron_server_cmd_flag = True
        return get_python_lib()

    @classmethod
    def get_update_patch_path_version(cls):
        """get patch file path and version_new info.

        :return: .../site-packages-XXX  version
        """
        patch_main_path = patch_update_version = ''
        update_tool_path = commands.getoutput('pwd')
        patch_next_folder = os.listdir(update_tool_path)
        for next_name in patch_next_folder:
            # site-packages-XXX
            patch_main_path = os.path.join(update_tool_path, next_name)
            if os.path.isdir(patch_main_path):
                if '-' not in next_name:
                    continue
                patch_update_version = next_name.strip().split('-')[2]
                break
        return patch_main_path, patch_update_version

    @classmethod
    def search_file(cls, filepath):
        """search files in update packages"""
        result = []
        for dir_path, _, filenames in os.walk(filepath):
            for filename in filenames:
                result.append(os.path.join(dir_path, filename))
        if not result:
            LOGGER.info('there is no new file in update-package.')
            sys.exit(0)
        return result

    @classmethod
    def check_package_file_list(cls):
        """check files of the update_tool package before update"""
        LOGGER.info('Begin to check file hash of update-package.')
        status, result = commands.getstatusoutput("bash %s/verify_file_list.sh" % sys.path[0])
        if status != 0:
            LOGGER.error('The update-package is not absolutely correct,abort to update: %s', result)
            sys.exit(1)
        LOGGER.info('Complete to check file hash of update-package.')

    @classmethod
    def config_cmcc_pike(cls):
        with os.fdopen(os.open(DRIVER_CFG_PATH, os.O_RDONLY, stat.S_IRUSR), 'r') as file:
            content_list = file.readlines()
            original_config_content = ''.join(content_list)
            updated_config_content = []
            if 'cmcc_env' not in original_config_content:
                for line in content_list:
                    updated_config_content.append(line)
                    updated_config_content.append('\ncmcc_env = true\n') if 'OPS_version' in line else None
            else:
                for line in content_list:
                    updated_config_content.append('cmcc_env = true\n') if 'cmcc_env' in line \
                        else updated_config_content.append(line)
        with os.fdopen(os.open(DRIVER_CFG_PATH, os.O_WRONLY, stat.S_IWUSR), 'w') as file:
            file.write(''.join(updated_config_content))

    def check_if_cmcc_pike(self, ops_version):
        if 'Pike' != ops_version:
            return False
        file_meta = FileMeta(os.path.join(self.get_package_home_path(), AC_ML2_DRIVER_PATH))
        if 'openstack' == file_meta.user and 'openstack' == file_meta.user_group:
            return True
        return False

    def restart_check(self, restart=True):
        """restart check"""
        if not restart:
            return
        config = six.moves.configparser.ConfigParser()
        config.read(DRIVER_CFG_PATH)
        try:
            ops_version = config.get('huawei_ac_config', 'OPS_version').strip()
        except Exception:
            ops_version = 'Liberty'
        LOGGER.info('ops_version:%s', ops_version)
        LOGGER.info('restarting...')

        if self.check_if_cmcc_pike(ops_version):
            self.config_cmcc_pike()

        if re.match('^FusionSphere.+', ops_version):
            start_cmd = ['cps', 'host-template-instance-operate', '--action',
                         'start', '--service', 'neutron', 'neutron-server']
            stop_cmd = ['cps', 'host-template-instance-operate', '--action',
                        'stop', '--service', 'neutron', 'neutron-server']
            self.start_or_stop(stop_cmd, start_cmd)

        else:
            stop_cmd = ['systemctl', 'stop', 'neutron-server']
            start_cmd = ['systemctl', 'start', 'neutron-server']
            if not self.neutron_server_cmd_flag:
                stop_cmd = ['service', 'neutron-server', 'stop']
                start_cmd = ['service', 'neutron-server', 'start']
            self.start_or_stop(stop_cmd, start_cmd)

    def start_or_stop(self, stop_cmd, start_cmd):
        """ start or stop and check"""
        if self.check_process() == 0:
            UpdateFunBase.stop_and_check(stop_cmd)
            if self.check_process() == 0:
                LOGGER.error('stop neutron-server failed!please check first')
                # neutron-server进程还在的情况，等待5s后，启动neutron-server
                time.sleep(5)
        self.start_and_check(start_cmd)

    @staticmethod
    def stop_and_check(stop_cmd):
        """stop and check"""
        LOGGER.info('stop neutron server,this may cost in 30 seconds..')
        subprocess.call(stop_cmd)
        time.sleep(NORMAL_NEUTRON_STOP_TIMEOUT)
        status, output = commands.getstatusoutput("ps -ef | grep -v grep |grep 'neutron-server'")
        LOGGER.info("stop-check status: %s,output: %s", status, output)
        if status == 0 and output.strip() != '':
            LOGGER.info("start to kill neutron-server process...")
            cnt = 0
            while cnt < NEUTRON_CHECK_TIMES:
                cmd_ps = [os.path.realpath('/bin/pgrep'), 'neutron-server']
                for line in subprocess.Popen(args=cmd_ps, stdout=subprocess.PIPE).stdout.readlines():
                    subprocess.call([os.path.realpath('/bin/kill'), '-9', line.decode().rstrip()])
                time.sleep(CHECK_INTERVAL_TIME)
                kill_status, kill_output = commands.getstatusoutput("ps -ef | grep -v grep |grep 'neutron-server'")
                LOGGER.info("kill-check status: %s,output: %s", kill_status, kill_output)
                if kill_status != 0 and kill_output.strip() == '':
                    break
                cnt += 1

    def start_and_check(self, start_cmd):
        """start and check neutron server"""
        LOGGER.info('started neutron server...')
        subprocess.call(start_cmd)
        time.sleep(10)
        for _ in six.moves.range(0, 5):
            if self.check_process() == 0:
                LOGGER.info("restart neutron-server success")
                LOGGER.info('-' * 50)
                return
            LOGGER.info("wait for 10 seconds and check")
            time.sleep(10)

        LOGGER.info("restart neutron-server failed, please check service manually")
        LOGGER.info('-' * 50)
        exit(1)

    @classmethod
    def check_process(cls):
        """check neutron process"""
        status, output = commands.getstatusoutput("ps -ef | grep -v grep "
                                                  "|grep 'neutron-server'")
        LOGGER.info("status : %s,output:%s", status, output)
        if status != 0:
            if output.strip() == '':
                return 1
            LOGGER.error('check neutron process catch an exception:%s', output)
            exit(-1)
        return 0

    def write_log(self, log_record, log_file):
        """write log"""
        flags = os.O_WRONLY | os.O_CREAT | os.O_APPEND
        with os.fdopen(os.open(log_file, flags, 0o400), 'a') as file_log:
            file_log.write(log_record)
        user_info = os.stat(os.path.join(self.get_package_home_path(), AC_ML2_DRIVER_PATH))
        os.chown(log_file, user_info.st_uid, user_info.st_gid)


class BackupFiles(UpdateFunBase):
    """back up files"""

    def __init__(self, package_home_path, current_version, update_patch_version):
        self.current_version = current_version
        self.update_patch_version = update_patch_version
        self.package_home_path = package_home_path
        super(BackupFiles, self).__init__()

    def get_code_backup_path(self):
        """get code backup path"""
        return os.path.join(self.get_history_path(),
                            "%s_to_%s_%s" % (self.current_version, self.update_patch_version, TIMESTAMP))

    @staticmethod
    def get_child_path(input_path, child_name):
        """get child path"""
        for dir_path, _, _ in os.walk(input_path):
            if dir_path.endswith(child_name):
                return dir_path
        LOGGER.info('dirname \'%s\' does not exist', input_path)
        return None

    def copy_backup_files(self):
        """
        copy files include:(1) networking_huawei code
                           (2) conf file like:neutron conf etc.
        :return:
        """
        LOGGER.info('package home path:%s', self.package_home_path)

        # make sure code path
        code_path = os.path.join(self.package_home_path, CODE_MAIN_NAME)
        if not os.path.isdir(code_path):
            LOGGER.error("code path: %s not exist", code_path)
            exit(0)

        # check backup path
        backup_path = self.get_code_backup_path()
        if not os.path.isdir(backup_path):
            status = commands.getstatusoutput("mkdir -p %s" % backup_path)[0]
            if status != 0:
                LOGGER.error('mkdir backup path : %s failed!', backup_path)
                exit(0)

        # copy code
        copy_version_cmd = r'\cp -axfp %s %s' % (code_path, backup_path)
        copy_status, output = commands.getstatusoutput(copy_version_cmd)
        if copy_status != 0:
            LOGGER.error('copy codes path files: %s to %s failed:%s', code_path, backup_path, output)
            self.roll_back_update_operate(self.current_version)
            exit(0)

        backup_log_file = os.path.join(backup_path, "backup.log")
        if os.path.exists(backup_log_file):
            backup_log_file_status = commands.getstatusoutput('touch %s' % backup_log_file)[0]
            if backup_log_file_status != 0:
                LOGGER.error('create file %s failed:', backup_log_file)
                self.roll_back_update_operate(self.current_version)
                exit(0)
        LOGGER.info('back up version:%s,path:%s' % (self.current_version, backup_path))

        # copy conf files
        for conf_file in BACKUP_CONF_LIST:
            cp_cmd = r'\cp -axfp %s %s' % (conf_file, backup_path)
            status, output = commands.getstatusoutput(cp_cmd)
            if status != 0:
                LOGGER.error("copy conf file error:%s", output)
                self.roll_back_update_operate(self.current_version)
                exit(-1)
            LOGGER.info('backup version:%s,conf path:%s' % (self.current_version, conf_file))

    def add_version(self):
        """add_version"""
        code_version_file = os.path.join(self.package_home_path, CODE_MAIN_NAME, VERSION_FILE_NAME)
        if os.path.exists(code_version_file):
            sed_status, sed_output = commands.getstatusoutput('sed -i \'1d\' %s' % code_version_file)
            if sed_status != 0:
                LOGGER.error(sed_output)
                self.roll_back_update_operate(self.current_version)
                exit(-1)
        try:
            self.write_log(self.update_patch_version, code_version_file)
        except Exception:
            self.roll_back_update_operate(self.current_version)

    def backup(self):
        """back up"""
        LOGGER.info('version now: %s ', self.current_version)

        LOGGER.info('version updating: %s', self.update_patch_version)
        if self.current_version == self.update_patch_version:
            LOGGER.info('version now is %s already.', self.update_patch_version)

        # # find all new files need in update-packages
        LOGGER.info('-' * 50)
        LOGGER.info('start to backup files ')
        self.copy_backup_files()
        self.add_version()

    def roll_back_update_operate(self, version):
        """roll back update operate"""
        version_path = os.path.join(self.get_history_path(), version)
        if not os.path.isdir(version_path):
            LOGGER.error("unknown version : %s", version)
            exit(1)
        LOGGER.info("start to remove backup code :%s", version_path)
        shutil.rmtree(version_path)


def change_pwd_grp_mode(path, mode, uid, gid, ctx):
    """
    traverse all dirs and change mode all dirs/files
    :param ctx:
    :param gid:
    :param uid:
    :param path:
    :param mode:

    :return:
    """
    for root, dirs, files in os.walk(path):
        for file_name in files:
            if file_name[0] == ".":
                continue
            file_path = os.path.join(root, file_name)
            size = int(round(os.path.getsize(file_path)))
            if file_name in CERT_FILE:
                # 证书文件600
                LOGGER.info("update file : '%s ',size : %s,mode : %s", file_path, size, stat.S_IRUSR | stat.S_IWUSR)
                os.chmod(file_path, stat.S_IRUSR | stat.S_IWUSR)
            else:
                LOGGER.info("update file : '%s ',size : %s,mode : %s", file_path, size, mode)
                os.chmod(file_path, mode)
            os.chown(file_path, uid, gid)
            if selinux and os.path.exists(file_path) and ctx:
                selinux.chcon(file_path, ctx)
        for dir_name in dirs:
            if dir_name[0] == ".":
                continue
            dir_path = os.path.join(root, dir_name)
            # dir mode 500
            LOGGER.info("update dir_name : '%s '", dir_path)
            os.chmod(dir_path, stat.S_IRUSR | stat.S_IXUSR)
            os.chown(dir_path, uid, gid)


def copytree(src, dst, rm_dir=True):
    """复制目录到另一个目录

    :param src: str,目录路径
    :param dst: str,目录
    :param rm_dir: bool,是否要清空目录
    """
    if rm_dir and os.path.exists(dst):
        LOGGER.info("remove:%s" % dst)
        shutil.rmtree(dst)
    LOGGER.info("copy:%s --> %s" % (src, dst))
    shutil.copytree(src, dst)

    # 修改文件所有者和分组
    for root_dir, _, files in os.walk(dst):
        src_dir = root_dir.replace(dst, src, 1)
        file_meta = os.stat(src_dir)
        LOGGER.info("chown:%s to %s(%s)" % (root_dir, file_meta, src_dir))
        os.chown(root_dir, file_meta.st_uid, file_meta.st_gid)
        for file in files:
            file_meta = os.stat(os.path.join(src_dir, file))
            LOGGER.info("chown:%s to %s(%s)" % (os.path.join(root_dir, file), file_meta, os.path.join(src_dir, file)))
            os.chown(os.path.join(root_dir, file), file_meta.st_uid, file_meta.st_gid)


def copyfile(src, dst):
    """复制文件到另一个路径

    :param src: str,文件路径
    :param dst: str,文件或目录
    """
    LOGGER.info("copy:%s --> %s" % (src, dst))
    shutil.copy2(src, dst)

    # 修改文件所有者和分组
    filepath = dst
    if not os.path.isfile(filepath):  # dst是一个目录
        filepath = os.path.join(dst, os.path.basename(src))
    file_meta = os.stat(src)
    LOGGER.info("chown:%s to %s(%s)" % (filepath, file_meta, src))
    os.chown(filepath, file_meta.st_uid, file_meta.st_gid)


class UpdateFunction(UpdateFunBase):
    """update function"""

    def __init__(self, package_home_path, current_version, update_patch_version, backup_path, patch_main_path):
        self.package_home_path = package_home_path
        self.example_file_path = os.path.join(self.package_home_path, AC_ML2_DRIVER_PATH)
        self.example_mode = self.get_path_mode(self.example_file_path)
        self.backup_path = backup_path
        self.current_version = current_version
        self.update_patch_version = update_patch_version
        self.patch_main_path = patch_main_path

        config = six.moves.configparser.ConfigParser()
        config.read(DRIVER_CFG_PATH)
        super(UpdateFunction, self).__init__()

    @staticmethod
    def get_file_rwx_mode(st_mode):
        if st_mode == 0:
            LOGGER.error('cal example file mode error')
            exit(0)
        config_file = os.path.realpath("/etc/neutron/huawei_driver_config.ini")
        config = six.moves.configparser.ConfigParser()
        config.read(config_file)
        ops = None
        if config.has_option('huawei_ac_config', 'OPS_version'):
            ops = config.get('huawei_ac_config', 'OPS_version').strip()
        if ops is None:
            return st_mode
        if ops == "EZ_Mitaka":
            # 444
            return stat.S_IRUSR | stat.S_IRGRP | stat.S_IROTH
        else:
            # 400
            return stat.S_IRUSR
        return st_mode

    @staticmethod
    def get_result_mode(st_mode):
        """get result mode"""
        result_mode = 0
        if st_mode:
            for elem in [stat.S_IRUSR, stat.S_IWUSR, stat.S_IXUSR, stat.S_IRGRP, stat.S_IWGRP, stat.S_IXGRP,
                         stat.S_IROTH, stat.S_IWOTH, stat.S_IXOTH]:
                if elem != 0:
                    result_mode = result_mode | elem
        return result_mode

    @staticmethod
    def get_path_mode(path):
        """
        # write original mode
        :param path:
        :return:
        """
        return os.stat(path).st_mode

    @staticmethod
    def get_file_uid(path):
        """
        # write original mode
        :param path:
        :return:
        """
        return os.stat(path).st_uid

    @staticmethod
    def get_file_gid(path):
        """
        # write original mode
        :param path:
        :return:
        """
        return os.stat(path).st_gid

    @staticmethod
    def get_file_context(path):
        """get file context"""
        if selinux and os.path.exists(path):
            try:
                context = selinux.getfilecon(path)[-1]
            except Exception:
                context = None
            return context
        return 'unconfined_u:object_r:lib_t:s0'

    def update_patch(self):
        """update patch"""
        code_main_path = os.path.join(self.patch_main_path, CODE_MAIN_NAME)
        LOGGER.info('example file mode is : %s', oct(self.example_mode))
        model_rxw_mode = self.get_file_rwx_mode(self.example_mode)
        model_context = self.get_file_context(self.example_file_path)
        LOGGER.info('example file context is : %s', model_context)
        uid = self.get_file_uid(self.example_file_path)
        gid = self.get_file_gid(self.example_file_path)
        LOGGER.info('example file uid,gid is : %s,%s', uid, gid)
        change_pwd_grp_mode(code_main_path, model_rxw_mode, uid, gid, model_context)

        # copy all codes
        dst_main_path = os.path.join(self.package_home_path, CODE_MAIN_NAME)
        LOGGER.info('copy dir: %s to dir: %s', code_main_path, dst_main_path)
        cp_cmd = r'\cp -axfp %s/* %s' % (code_main_path, dst_main_path)
        copy_update_status, output = commands.getstatusoutput(cp_cmd)

        if copy_update_status != 0:
            LOGGER.error(output)
            self.roll_back()
            exit(-1)

        # write update log
        update_log_info = '%s,%s\n' % (self.update_patch_version, self.backup_path)
        update_log = os.path.join(self.get_history_path(), LOG_FILE_NAME)
        LOGGER.info("write update log to file :%s ", update_log)
        self.write_log(update_log_info, update_log)

    def roll_back(self):
        """roll back"""
        # get system site-packages path
        LOGGER.info("update failed,start to rollback env")
        dst_main_path = os.path.join(self.package_home_path, CODE_MAIN_NAME)
        back_name = os.path.join(self.backup_path, CODE_MAIN_NAME)
        LOGGER.info("start to recover code env to version :%s", self.current_version)
        LOGGER.info("%s --> %s", back_name, dst_main_path)
        copytree(back_name, dst_main_path)
        # conf file list callback
        LOGGER.info("start to rollback conf file")
        for conf_file in BACKUP_CONF_LIST:
            src_conf_file = os.path.join(self.backup_path, os.path.basename(conf_file))
            LOGGER.info("copy conf file '%s' to '%s'", src_conf_file, conf_file)
            copyfile(src_conf_file, conf_file)
        LOGGER.info("start to remove backup dir")
        shutil.rmtree(self.backup_path)
        LOGGER.info("start to restart neutron")
        self.restart_check()

    @staticmethod
    def get_huawei_tools_paths():
        """
        :return: list of huawei tools paths in current system
        """
        paths = []
        for dir_path, dir_names, _ in os.walk(ROOT_PATH):
            if not dir_path.endswith('networking-huawei'):
                continue
            for dirname in dir_names:
                if dirname == 'tools':
                    paths.append(os.path.join(dir_path, dirname))
        return paths

    @staticmethod
    def get_huawei_tools_files(tools_paths, patch_file, patch_path):
        """
        :param tools_paths: list of huawei tools paths
        :param patch_file: full path of patch file
        :param patch_path: full path of patch directory
        :return: list of huawei tools files in current system
        """
        patch_tools_path = os.path.join(patch_path, 'tools/')
        patch_file_name = patch_file.replace(patch_tools_path, '')
        tools_files = []
        for tools_path in tools_paths:
            if patch_file_name in os.listdir(tools_path):
                tools_files.append(os.path.join(tools_path, patch_file_name))
        return tools_files

    def backup_huawei_tools(self, org_files):
        """
        :param org_files: list of original huawei tools files
        :return: list of backup files
        """
        backup_files = []
        version = '.' + self.current_version + '-' + self.update_patch_version
        for _file in org_files:
            if not os.path.exists(_file):
                LOGGER.info('file does not exist: %s', _file)
                continue
            backup_file = _file + version
            backup_files.append(backup_file)
            shutil.copy(_file, backup_file)
        return backup_files

    @classmethod
    def update_huawei_tools(cls, org_files, patch_file):
        """
        :param org_files: list of original huawei tools files
        :param patch_file: full path of huawei tools patch
        :return: None
        """
        if not os.path.exists(patch_file):
            LOGGER.info('file does not exist: %s', patch_file)
            return
        for _file in org_files:
            if not os.path.exists(_file):
                LOGGER.info('file does not exist: %s', _file)
                return
            shutil.copy(patch_file, _file)

    def add_huawei_tools(self, tools_paths, patch_file):
        """
        :param tools_paths: list of huawei tools paths
        :param patch_file: full path of huawei tools patch
        :return: None
        """
        if not os.path.exists(patch_file):
            LOGGER.error('file does not exist: %s', patch_file)
            return
        new_files = []
        backup_files = []
        version = '.' + self.current_version + '-' + self.update_patch_version
        for _path in tools_paths:
            new_file = os.path.join(_path, patch_file.split('tools/')[-1])
            if not os.path.exists(new_file):
                new_files.append(new_file)
                shutil.copy(patch_file, _path)
            else:
                shutil.copy(patch_file, _path)
                backup_file = new_file + version
                backup_files.append(backup_file)

        LOGGER.info('add files: %s', new_files)
        LOGGER.info('backup files: %s', backup_files)

    def patch_tools(self):
        """
        :return: None
        """
        all_patch_files = self.search_file(self.patch_main_path)
        LOGGER.info('patching huawei tools start...')
        tools_paths = self.get_huawei_tools_paths()
        for patch_file in all_patch_files:
            if patch_file.replace(self.patch_main_path, '').startswith('/tools'):
                tools_files = self.get_huawei_tools_files(tools_paths, patch_file, self.patch_main_path)
                LOGGER.info('files in sys: %s', tools_files)

                if len(tools_files) != len(tools_paths):
                    self.add_huawei_tools(tools_paths, patch_file)
                    tools_files = self.get_huawei_tools_files(tools_paths, patch_file, self.patch_main_path)
                else:
                    # backup tool files
                    backup_files = self.backup_huawei_tools(tools_files)
                    LOGGER.info('backup files: %s', backup_files)
                    self.update_huawei_tools(tools_files, patch_file)

                # change file mode
                self.change_tool_mode(tools_files)
                if tools_files:
                    _, output = commands.getstatusoutput('stat %s' % tools_files[0])
                    LOGGER.info(output.split('\n')[3])
        LOGGER.info('patching huawei tools end...')

    @classmethod
    def change_tool_mode(cls, tool_files):
        """change tool mode"""
        for tool_file in tool_files:
            # key 600
            if tool_file.endswith('.pem'):
                os.chmod(tool_file, stat.S_IRUSR | stat.S_IWUSR)
            # conf 640
            elif tool_file.endswith('.txt') or str(tool_file).endswith('.ini'):
                os.chmod(tool_file, stat.S_IRUSR | stat.S_IWUSR | stat.S_IRGRP)
            # scripts/code 550
            else:
                os.chmod(tool_file, stat.S_IRUSR | stat.S_IXUSR | stat.S_IRGRP | stat.S_IXGRP)

    @classmethod
    def flush_db_sql(cls, upgrade=True):
        """flush db sql"""
        if not upgrade:
            return
        cmd = 'neutron-db-manage --subproject networking-huawei upgrade head'
        status, output = commands.getstatusoutput(cmd)
        if status != 0:
            if 'No such revision or branch' in output:
                print('warning: the database already be upgraded, no need to upgrade again')
            else:
                LOGGER.error(output)
                exit(1)


class UpdateProcess(object):
    """update process"""

    def __init__(self):
        self.fun_base = UpdateFunBase()
        self.fun_base.check_package_file_list()
        self.package_home_path = self.fun_base.get_package_home_path()
        self.current_version = self.fun_base.get_current_version()

        patch_main_path, update_patch_version = self.fun_base.get_update_patch_path_version()

        self.backup = BackupFiles(self.package_home_path, self.current_version, update_patch_version)
        backup_path = self.backup.get_code_backup_path()
        self.func = UpdateFunction(self.package_home_path, self.current_version, update_patch_version, backup_path,
                                   patch_main_path)

        self.backup_log_file = os.path.join(backup_path, "backup.log")
        self.create_log_file(backup_path, self.backup_log_file)

        file_handler = logging.FileHandler(self.backup_log_file)
        file_handler.setFormatter(FORMATTER)
        LOGGER.addHandler(file_handler)
        self.init_update_log()

    @classmethod
    def create_log_file(cls, backup_path, backup_log_file):
        """create log file"""
        if not os.path.isdir(backup_path):
            status = commands.getstatusoutput("mkdir -p %s" % backup_path)[0]
            if status != 0:
                LOGGER.error('mkdir backup path : %s failed!', backup_path)
                exit(0)
        if not os.path.exists(backup_log_file):
            backup_log_file_status = commands.getstatusoutput('touch %s' % backup_log_file)[0]
            if backup_log_file_status != 0:
                LOGGER.error('create file %s failed:', backup_log_file)
                exit(0)

    def init_update_log(self):
        """init update log"""
        history_path = self.backup.get_history_path()
        next_path = os.listdir(history_path)
        for path in next_path:
            if path == 'update.log':
                return
        shutil.move(LOG_FILE_OLD, history_path)

    def process(self, flush_db_flg=True, restart_check_flg=True):
        """start process"""
        # backup files
        LOGGER.info('-' * 50)
        try:
            self.backup.backup()
        except Exception as error:
            LOGGER.error("back up operate catch an exception:%s", error)
            self.backup.roll_back_update_operate(self.current_version)
        # patch files
        LOGGER.info('-' * 50)
        try:
            self.func.update_patch()
            # patch tools
            self.func.patch_tools()
            # flush db
            self.func.flush_db_sql(flush_db_flg)
        except Exception as error:
            LOGGER.error("update patch catch an exception:%s", error)
            self.func.roll_back()
        # restart and check
        LOGGER.info('-' * 50)
        self.func.restart_check(restart_check_flg)


def main():
    """main function"""
    update = UpdateProcess()
    if len(sys.argv) == 1:
        update.process()
    elif len(sys.argv) == 3:
        flush_db_flg = sys.argv[1].strip().lower()
        restart_check_flg = sys.argv[2].strip().lower()
        if flush_db_flg in {'true', 'false'} and restart_check_flg in {'true', 'false'}:
            update.process(flush_db_flg == 'true', restart_check_flg == 'true')
        else:
            LOGGER.error("Input parameter error")
    else:
        LOGGER.error("Wrong number of input parameters.")


if __name__ == '__main__':
    main()
