#!/usr/bin/env bash

IGNORE_FILES=(
"tools/networking_huawei_config.txt"
)

CURRENT_PATH=$(cd $(dirname $0); pwd)
cd $CURRENT_PATH

if [[ -f "$CURRENT_PATH/update_tool.py" ]]; then
    CHECK_BASE_PATH=$CURRENT_PATH
else
    CHECK_BASE_PATH=$(cd ..; pwd)
fi
cd $CHECK_BASE_PATH

LIST_FILE_NAME="./verify/filelist.txt"
ROOT_CA_FILE="./verify/HuaweiRootCAsipCRLs.crl"
HW_CMS_TOOL="./verify/hwcmstool"

LIST_FILE_CMS="${LIST_FILE_NAME}.cms"
LIST_FILE_CRL="${LIST_FILE_NAME}.crl"

VERSION=""
CREATOR=""
SUPPORT_CREATOR="Huawei Technology Inc"

#*************************************************************#
# Name:         verify_cms_file                               #
# Description:  verify cms function                           #
#*************************************************************#
function verify_cms_file()
{
    echo "Start verify list file."
    #Get package path and it could not be null

    CHECK_FILES=(
        "${LIST_FILE_NAME}"
        "${LIST_FILE_CMS}"
        "${LIST_FILE_CRL}"
        "${ROOT_CA_FILE}"
        "${HW_CMS_TOOL}"
    )

    for check_file_name in ${CHECK_FILES[@]}
    do
        if [ ! -f "${check_file_name}" ]
        then
            echo "Error: ${check_file_name} does not exist."
            return 1
        fi
    done

    chmod +x ${HW_CMS_TOOL}
    local ret_content=""
    ${HW_CMS_TOOL} --verify --source "${LIST_FILE_NAME}" --sign "${LIST_FILE_CMS}" --crl "${LIST_FILE_CRL}" --cacrl "${ROOT_CA_FILE}"
    if [ $? -ne 0 ]
    then
        echo "Error: Verify file ${LIST_FILE_NAME} failed with CMS"
        return 1
    fi
    echo "Successfully verified list file ${LIST_FILE_NAME} with CMS"
    return 0
}

function fn_compare_hash()
{
    local file_name=$1
    echo ${IGNORE_FILES[@]}|grep -w ${file_name} > /dev/null
    if [ $? -eq 0 ]; then
        echo "Need not to check ${file_name}."
        return 0
    fi
    local hash_cmd=$2
    local hash_code=$3
    local file_path=${CHECK_BASE_PATH}/${file_name}
    gen_hash=`${hash_cmd} ${file_path}`
    new_hash_code=`echo ${gen_hash} | awk '{print $1}'`
    if [ ${new_hash_code} = ${hash_code} ]; then
        echo "Verify ${hash_cmd} ${file_name} is OK."
        return 0
    else
        echo "Verify ${hash_cmd} ${file_name} is wrong."
        return 1
    fi
}

function check_file_v1_0_0()
{
    # 读取文件清单内容
    local file_name=""
    local hash_type=""
    local hash_code=""
    local hash_cmd=""
    while read line;
    do
        key=`echo ${line} | awk -F ":" '{print $1}'|awk '$1=$1'`
        value=`echo ${line} | awk -F ":" '{print $2}'|awk '$1=$1'`
        if [ `echo ${line} | awk -F ":" '{print NF}'` != "2" ]; then
            # 可变文件的场景，没有hash值，解析的结果只有1列，这种情况不校验hash，继续处理后续文件
            # 考虑到已读取的值可能干扰后续的hash比较，所以把下列变量初始化
            file_name=""
            hash_type=""
            hash_code=""
            hash_cmd=""
            continue
        fi

        if [ "${key}" = "Manifest Version" ] || [ "$key" = "Create By" ] ; then
            continue
        elif [ "${key}" = "Name" ] ; then
            file_name=${value}
            continue
        else
            hash_type=${key}
            hash_code=${value}
        fi

        # 检查hash算法是否在可选范围内(可选范围:SHA256、SHA384, SHA512)
        case "${hash_type}" in
        "SHA-256-Digest")
            hash_cmd="sha256sum";;
        "SHA-384-Digest")
            hash_cmd="sha384sum";;
        "SHA-512-Digest")
            hash_cmd="sha512sum";;
        *) echo "[ERROR]:algorithm: ${hash_type} is not support."; return 1
        esac

        if [ "${file_name}" != "" ] && [ "${hash_code}" != "" ] && [ "${hash_cmd}" != "" ] ; then
            fn_compare_hash ${file_name} ${hash_cmd} ${hash_code}
            if [ $? != 0 ];then
                return 1
            fi
            file_name=""
            hash_type=""
            hash_code=""
            hash_cmd=""
        fi
    done < ${LIST_FILE_NAME}
}


#*************************************************************#
# Name:         check_file_list                               #
# Description:  check if files in list are modified           #
#*************************************************************#
function check_file_list()
{
    #清单文件前两行必须包含Manifest Version, Create By
    # 读取文件清单内容
    VERSION=`head -n 2 ${LIST_FILE_NAME} | grep "Manifest Version" | awk -F ":" '{print $2}'|awk '$1=$1' `
    CREATOR=`head -n 2 ${LIST_FILE_NAME} | grep "Create By" | awk -F ":" '{print $2}'|awk '$1=$1' `
    if [ "${CREATOR}_" != "${SUPPORT_CREATOR}_" ]; then
        echo "[ERROR]: Creator name(${CREATOR}) error"
        echo "${SUPPORT_CREATOR}\."
        return 1
    fi

    case "${VERSION}" in
    "1.0")
        check_file_v1_0_0
        return $?;;
    *) echo "[ERROR]: list file version: ${VERSION} is not support."
    esac
    return 1
}

#==============================================================
# main entrance
# 校验文件列表签名
verify_cms_file || exit 1
check_file_list || exit 1

exit 0