#!/usr/bin/python
# -*- coding: utf-8 -*-
import time
import traceback
import json
import uuid

from getDBConnection import get_zenith_session
from common_tasks.base_task import BaseTask

ONE_PACKAGE = 500


class UpgradeNetconfPara(BaseTask):
    def __init__(self, product_name="NCE"):
        super(UpgradeNetconfPara, self).__init__()
        self.set_product_name(product_name)
        self.info("UpgradeNetconfPara init product_name is %s" % product_name)
        self.src_db_session = get_zenith_session('neresmgmtdb', 'neresmgmtdb', product_name)
        if self.src_db_session is None:
            self.error("neresmgmtdb is None")
            return
        self.src_db_session.autocommit(True)
        self.src_db_cursor = self.src_db_session.cursor()

        self.dst_db_session = get_zenith_session('neresdb', 'neresdb', product_name)
        if self.dst_db_session is None:
            self.error("neresdb_sess is None")
            return
        self.dst_db_session.autocommit(True)
        self.dst_db_cursor = self.dst_db_session.cursor()

        self.src_table = "TBL_NECOMMUPARAM"
        self.src_table_cols = ("neResId", "neNativeId", "neIP", "commuParamType", "commuParamRemark",
                               "commuParamContent")
        self.src_table_cols_index = {y:x for x, y in enumerate(self.src_table_cols)}
        self.dst_table = "NetConfPara"
        self.dst_table_cols = ("modelId", "fModelId", "tenantId", "neResId", "neId", "purpose",
                               "connectStatus", "userName", "password", "port", "userPrivateKey",
                               "userPrivateKeyPwd", "authMode", "loginTimeout", "responseTimeout",
                               "createAt", "updateAt")

    def convert_data(self, paras):
        # neresmgmtdb表中存在多种协议类型的参数，升级过程中应该只处理NETCONF协议的参数
        commu_param_type = paras[self.src_table_cols_index.get("commuParamType")]
        if "NETCONF" != commu_param_type:
            self.warning("the commuParamType is: %s, not NETCONF" % commu_param_type)
            return []

        modelId = str(uuid.uuid1())
        fModelId = None
        tenantId = "default-organization-id"
        neResId = paras[self.src_table_cols_index.get("neResId")]
        neId = paras[self.src_table_cols_index.get("neNativeId")].lstrip("NE=")
        if neId and not str(neId).isdigit():
            self.debug("neId is not number, data is ignored: %s" % neId)
            return []
        purpose = "default"
        connectStatus = "-1"

        # commuParamContent字段的值可能是None或者非法的json字符串，需要进行异常处理
        commuParamContent_text = paras[self.src_table_cols_index.get("commuParamContent")]
        try:
            commuParamContent = json.loads(commuParamContent_text)
            userName = commuParamContent.get("userName")
            password = commuParamContent.get("pwd")
            port = commuParamContent.get("port")
            userPrivateKey = commuParamContent.get("userPrivateKey")
            userPrivateKeyPwd = commuParamContent.get("userPrivateKeyPwd")
            authMode = commuParamContent.get("authMode")
            loginTimeout = commuParamContent.get("loginTimeout")
            responseTimeout = commuParamContent.get("responseTimeout")
        except TypeError as te:
            self.warning(te)
            return []
        except json.decoder.JSONDecodeError as jde:
            self.warning(jde)
            return []
        except AttributeError as ae:
            self.warning(ae)
            return []

        createAt = str(int(time.time() * 1000))
        updateAt = str(int(time.time() * 1000))

        return tuple(None if x is None else str(x) for x in (modelId, fModelId, tenantId, neResId, neId, purpose,
                               connectStatus, userName, password, port, userPrivateKey,
                               userPrivateKeyPwd, authMode, loginTimeout, responseTimeout,
                               createAt, updateAt))

    def to_UpgradePara(self, datas):
        col_names = "`" + ("`, `".join(self.dst_table_cols)) + "`"
        val_ids = ":" + (",:".join((str(x+1) for x in range(len(self.dst_table_cols)))))
        insert_stmt = "INSERT INTO `%s` (%s) VALUES(%s)" % (self.dst_table, col_names, val_ids)
        self.debug("insert sql stmt: %s" % insert_stmt)
        list_datas = []
        for data in datas:
            data = self.convert_data(data)
            if len(data) == len(self.dst_table_cols):
                tuple_data = tuple(data)
                list_datas.append(tuple_data)
            if len(list_datas) == ONE_PACKAGE:
                self.exec_sql(insert_stmt, list_datas)
                list_datas = []

        if len(list_datas) != 0:
            self.exec_sql(insert_stmt, list_datas)

    def close_session(self):
        self.dst_db_cursor.close()
        self.dst_db_session.close()
        self.src_db_cursor.close()
        self.src_db_session.close()

    def do(self):
        try:
            self.info('UpgradeNetconfPara::do start!')
            select_stmt = "select %s from %s" % (",".join(self.src_table_cols), self.src_table)
            self.debug("execute sql: %s" % select_stmt)
            try:
                self.src_db_cursor.execute(select_stmt)
                datas = self.src_db_cursor.fetchall()
            except BaseException as be:
                self.warning("execute sql failed, err is: %s" % str(be))
                datas = []

            final_data = []
            if datas:
                final_data = list(datas)
            self.info('get para data success count: %d' % len(final_data))

            self.to_UpgradePara(final_data)
            self.info('to_UpgradePara done')
            self.close_session()
        except Exception as e:
            self.close_session()
            self.error('UpgradeNetconfPara got exception')
            self.error("$s" % traceback.format_exc())
            return -1
        self.info('UpgradeNetconfPara::do done')
        return 0

if __name__ == '__main__':

    tool = UpgradeNetconfPara()
    print('[INFO] UpgradeNetconfPara start>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>')
    tool.do()
    print('[INFO] UpgradeNetconfPara finished<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<')
