#!/usr/bin/python
# -*- coding: utf-8 -*-
import sys
import time
import uuid
import traceback

from getDBConnection import get_zenith_session
from common_tasks.base_task import BaseTask

ONE_PACKAGE = 500


class UpgradeStelnetTempalte(BaseTask):
    def __init__(self, product_name="NCE"):
        super(UpgradeStelnetTempalte, self).__init__()
        self.set_product_name(product_name)
        self.info("UpgradeStelnetTempalte init product_name is %s" % product_name)
        # R21C00以前版本：升级前所属数据库实例：nmsdbsvr，所属数据库：MCDB
        # R21C00及以后版本：升级前所属数据库实例：nmsdbsvr，所属数据库：UMDB
        self.src_db_session = get_zenith_session('UMDB', 'UMDB', product_name)
        if self.src_db_session is None:
            self.error("UMDB is None")
            return

        self.src_db_session.autocommit(True)
        self.src_db_cursor = self.src_db_session.cursor()

        self.dst_db_session = get_zenith_session('neresdb', 'neresdb', product_name)
        if self.dst_db_session is None:
            self.error("neresdb_sess is None")
            return
        self.dst_db_session.autocommit(True)
        self.dst_db_cursor = self.dst_db_session.cursor()

        self.src_table = "tbl_TelnetTemplate"
        self.src_table_cols = (
            "TemplateID", "TemplateName", "AuthMode", "LoginUser", "LoginPwd", "ProtocolVersion","IsEnable",
            "PrivilegeLevel", "PrivilegePwd", "LoginTimeout","ResponseTimeout", "Port", "ProtocolType",
            "UserPrivateKey", "UserPrivateKeyPwd", "TemplateType")
        self.src_table_cols_index = {y:x for x, y in enumerate(self.src_table_cols)}

        self.dst_table = "StelnetTemplate"
        self.dst_table_cols = (
            "modelId", "fModelId", "tenantId", "templateName", "templateDesc", "accessMode", "authMode",
            "userName", "password", "privilegeState", "privilegeLevel", "privilegePwd", "loginTimeout",
            "responseTimeout", "port", "protocolType", "userPrivateKey", "userPrivateKeyPwd", "createAt",
            "updateAt")

    def convert_data(self, paras):
        modelId = str(uuid.uuid1())
        fmodelId = None
        tenantId = "default-organization-id"
        templateName = paras[self.src_table_cols_index.get("TemplateName")]
        templateDesc = None
        accessMode = paras[self.src_table_cols_index.get("TemplateType")]
        authModeMap = {
            "1": "NOAuth",
            "2": "Password",
            "3": "User",
            "4": "PrivateKey",
            "5": "PrivateKeyPassword",
        }
        authMode = authModeMap.get(str(paras[self.src_table_cols_index.get("AuthMode")]))
        userName = paras[self.src_table_cols_index.get("LoginUser")]
        password = paras[self.src_table_cols_index.get("LoginPwd")]
        privilegeStateMap = {
            "0": False,
            "1": True
        }
        privilegeState = privilegeStateMap.get(str(paras[self.src_table_cols_index.get("IsEnable")]))
        privilegeLevel = paras[self.src_table_cols_index.get("PrivilegeLevel")]
        privilegePwd = paras[self.src_table_cols_index.get("PrivilegePwd")]
        # 特权使能开关关闭的时候且特权密码全是空格，则置空
        if not privilegeState and privilegePwd and privilegePwd.isspace():
            privilegePwd = None
        loginTimeout = paras[self.src_table_cols_index.get("LoginTimeout")]
        responseTimeout = paras[self.src_table_cols_index.get("ResponseTimeout")]
        port = paras[self.src_table_cols_index.get("Port")]
        protocolTypeMap = {
            "0": "Telnet",
            "1": "STelnet",
        }
        protocolType = protocolTypeMap.get(str(paras[self.src_table_cols_index.get("ProtocolType")]))
        userPrivateKey = paras[self.src_table_cols_index.get("UserPrivateKey")]
        if userPrivateKey and isinstance(userPrivateKey, bytes):
            self.info("[%s] userPrivateKey is bytes, trans to str" % str(templateName))
            userPrivateKey = str(userPrivateKey.decode('utf-8'))
        userPrivateKeyPwd = paras[self.src_table_cols_index.get("UserPrivateKeyPwd")]
        createAt = int(time.time() * 1000)
        updateAt = int(time.time() * 1000)
        return tuple(None if x is None else str(x) for x in (
            modelId, fmodelId, tenantId, templateName, templateDesc, accessMode, authMode, userName, password,
            privilegeState, privilegeLevel, privilegePwd, loginTimeout, responseTimeout, port,protocolType,
            userPrivateKey, userPrivateKeyPwd, createAt, updateAt))

    def to_UpgradeStelnetTempalte(self, stelnet_templates):
        col_names = "`" + ("`, `".join(self.dst_table_cols)) + "`"
        val_ids = ":" + (",:".join((str(x+1) for x in range(len(self.dst_table_cols)))))
        insert_stmt = "INSERT INTO `%s` (%s) VALUES(%s)" % (self.dst_table, col_names, val_ids)
        self.debug("%s" % insert_stmt)
        list_datas = []
        dst_cols_len = len(self.dst_table_cols)
        for tempalte in stelnet_templates:
            data = self.convert_data(tempalte)
            if len(data) == dst_cols_len:
                tuple_data = tuple(data)
                list_datas.append(tuple_data)
            else:
                self.warning("coverted data length not equals dst table cols, to be ignored. ")
            if len(list_datas) == ONE_PACKAGE:
                self.exec_sql(insert_stmt, list_datas)
                list_datas = []

        if len(list_datas) != 0:
            self.exec_sql(insert_stmt, list_datas)

    def close_session(self):
        self.dst_db_cursor.close()
        self.dst_db_session.close()
        self.src_db_cursor.close()
        self.src_db_session.close()

    def do(self):
        try:
            self.info('UpgradeStelnetTempalte::do start!')
            select_stmt = 'select %s from %s' % (",".join(self.src_table_cols), self.src_table)
            templates = self.exec_query_sql(self.src_db_cursor, select_stmt)
            self.info('get stelnet templates data success count:%d'%len(templates))

            self.to_UpgradeStelnetTempalte(templates)
            self.info('to_UpgradeStelnetTempalte done')
            self.close_session()
        except Exception as e:
            self.close_session()
            self.error('UpgradeStelnetTempalte got exception')
            self.error(traceback.format_exc())
            return -1
        self.info('UpgradeStelnetTempalte::do done')
        return 0


if __name__ == '__main__':
    tool = UpgradeStelnetTempalte()
    print('[INFO] UpgradeStelnetTempalte start>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>')
    tool.do()
    print('[INFO] UpgradeStelnetTempalte finished<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<')
