# -*- coding:utf-8 -*-
"""检查os补丁包一致性"""
import re
import logging
from collections import Counter

from utils.log_util import EasysuiteLogger
from func.upgrade.upload_pkg_mgr.dao.upload_package_dao import UploadPackageDao

local_logger = logging.getLogger(__name__)


def execute_task_check(kvs):
    """
    功能描述: 执行检查任务
    :return: 执行结果
    """

    # 执行任务
    logger = EasysuiteLogger.get_logger(kvs, 'task_check_os_package_upgrade')
    logger.easysuite_init('Start to initialize the os patch upgrade check task.')
    logger.easysuite_start("Start to check the os patch upgrade.")
    ret = check_os_upgrade_is_allow(kvs, logger)
    if ret is False:
        logger.easysuite_error("Failed to check the os upgrade patch package.")
        return False
    else:
        logger.easysuite_finish("Check the os upgrade patch package successfully.")
    return True


def check_os_upgrade_is_allow(kvs, logger):
    """
    功能描述: 检查os补丁包一致性
    :return: 执行结果
    """

    # 从数据库读取软件包列表信息
    logger.info('Start to read os software package version.')
    task_id = kvs["easysuite.task_id"]
    all_packages = UploadPackageDao.query_upload_all_package(task_id, 'UPLOAD')
    os_type = kvs.get('os_type')

    patch_version = []
    patchtool_version = []

    for package in all_packages:
        if os_type == "Euler":
            rest_patch = re.match('(iMasterNCE.*)(_EulerOS[0-9.]+Patch)(-[a-zA-Z0-9-]+)(.*\..*)', package.real_pkg_name)
            rest_patchtool = re.match('(iMasterNCE.*)(_OSPatchtool_EulerOS[0-9.]+)(-[a-zA-Z0-9-]+)(.*\..*)', package.real_pkg_name)
        else:
            rest_patch = re.match('(iMasterNCE.*)(_suse[0-9.]+Patch)(-[a-zA-Z0-9-]+)(.*\..*)', package.real_pkg_name)
            rest_patchtool = re.match('(iMasterNCE.*)(_OSPatchtool_suse[0-9.]+)(-[a-zA-Z0-9-]+)(.*\..*)', package.real_pkg_name)
        if rest_patch:
            patch_version.append(f'{rest_patch.group(1)}{rest_patch.group(3)}')
        if rest_patchtool:
            patchtool_version.append(f'{rest_patchtool.group(1)}{rest_patchtool.group(3)}')

    logger.info(f'OSPatch version: {patch_version}')
    logger.info(f'OSPatchtool version: {patchtool_version}')

    if not patch_version:
        logger.easysuite_error('The OSPatch package version information was not found.')
        return False

    # 适配只有补丁包的企业场景
    if not patchtool_version:
        logger.info('No OSPatchtool package of the matching version was found. Skip this check.')
        return True

    if len(patch_version) != len(patchtool_version):
        logger.easysuite_error('The number of OSPatch packages and OSPatchtool packages is inconsistent.')
        return False

    # 判断补丁包版本一致性
    if Counter(patch_version) != Counter(patchtool_version):
        logger.easysuite_error('The OSPatch version does not match the OSPatchtool version. '
                               'The upgrade is not allowed.')
        return False

    first_version = patch_version[0].strip().split('-', 1)[0]
    for version in patch_version + patchtool_version:
        curr_version = version.strip().split('-', 1)[0]
        if curr_version != first_version:
            logger.easysuite_error(f'The patch package has an inconsistent version: {curr_version} {first_version}. '
                                   'The upgrade is not allowed.')
            return False
    return True


def check(kvs):
    """
    功能描述：判断此任务是否需要执行
    :param kvs: 全局字典
    :return: 检查结果
    """
    os_type = kvs.get('os_type')
    upgrade_os = 'id_upgrade_os'
    upgrade_content = kvs.get('upgrade_item', None)
    return upgrade_os in upgrade_content and os_type in ("Euler", "Suse")


def exe(kvs):
    ret = list()
    result = execute_task_check(kvs)
    if result:
        ret.append([('', '', True)])
    else:
        ret.append([('', '', False)])
    return ret
