#!/bin/bash
#########################################################
# Copyright © Huawei Technologies Co., Ltd. 1998-2020. All rights reserved. 
# File name     : migrate_databse.sh
# Description   : 数据库数据迁移框架
###########################################################

# 初始化脚本工作路径和脚本名
SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)

# 当前任务时间戳
TAIL=$(date +%Y%m%d%H%M%S)
    
# 初始化日志路径和日志文件名
LOG_PATH="/opt/oss/log/manager/easysuite_upgrade/scriptlog"
LOG_FILE="${LOG_PATH}/${SELF_FILE//.sh/}_${TAIL}.log"
CONFIG_DIR="/opt/oss/log/manager/easysuite_upgrade_config"
[ ! -e "${CONFIG_DIR}" ] && mkdir -p "${CONFIG_DIR}"
DBM_CMD="dbsvc""_""adm"
OSS_USER=$(id -nu 3001)

##################################################
# 校验执行用户
# 脚本要求使用ossadm用户执行	
##################################################
function check_user()
{
    local user=$(whoami)
    oss_user=$(id -nu 3001)
    if [ "${user}" != "${oss_user}" ]
    then
        echo "[$(date +'%Y-%m-%d %H:%M:%S')]| User have no permission to run this script" | tee -a "${LOG_FILE}"
        return 1
    fi
}

##################################################
# 记录Easysuite前台展示日志
# @param
#   $1:日志级别 ERROR/WARN/INFO
#   $2:日志内容
##################################################
function TASK_LOG()
{
    BASH_PID=$$
    if [ "$1" = "ERROR" -o "$1" = "error" ]
    then
        Level="ERROR"
    elif [ "$1" = "WARN" -o "$1" = "warn" ]
    then
        Level="WARN"
    else
        Level="INFO"
    fi
    echo "[$(date +%Y)-$(date +%m)-$(date +%d) $(date +%H):$(date +%M):$(date +%S)] [${BASH_PID}] | ${Level} $2" | tee -a "${TASK_LOG_FILE}"
}

##################################################
# 记录Easysuite前台展示日志及后台日志
# @param
#   $1:日志级别 ERROR/WARN/INFO
#   $2:日志内容
##################################################
function DOUBLE_LOG()
{
    TASK_LOG "$1" "$2"
    LOG "$1" "$2"
}

##################################################
# 定义相关全局变量
##################################################
function init_variable()
{
    # 平台ssh命令工具
    SSHTOOL="${OSS_ROOT}/tools/common/sshmgr/sshmgr.sh"
    
    # 迁移工作目录
    MIGRATEPATH="/opt/upgrade/easysuite_upgrade/databasemigrate"
    
    # 驱动目录
    MIGRATETOOL="${REPO_ROOT}/Services/Common/DataMigrateTool"
    GAUSSDRIVER="${REPO_ROOT}/Custom/DatabaseDriver_gauss"
    ZENITHDRIVER="${REPO_ROOT}/Custom/DatabaseDriver_zenith"

    # 修改containlist接口文件
    PYTHONC_TOOL="${SCRIPT_PATH}/pyscripts/modify_containlist.pyc"
    PYTHON_TOOL="${SCRIPT_PATH}/pyscripts/modify_containlist.py"

    # Easysuite前台进度值
    PROGRESS=0

    # Easysuite任务相关日志/进度/状态文件
    TASK_ID_PATH="/opt/upgrade/easysuite_upgrade/taskmgr/${TASKID}"
    TASK_LOG_FILE="${TASK_ID_PATH}/task.log"
    TASK_PROGRESS_FILE="${TASK_ID_PATH}/task.progress"
    TASK_STATUS_FILE="${TASK_ID_PATH}/task.status"

    # 是否是新任务标志位
    NEW_TASK="TRUE"
    [ -d "${TASK_ID_PATH}" ] && NEW_TASK="FALSE"
}

##################################################
# 入参格式化，将相关参数赋值给对应变量
##################################################
function init_params()
{
    local num=$#
    if [ $((${num}%2)) -eq 1 ]
    then
        num=$((${num}/2+1))
    else
        num=$((${num}/2))
    fi
    local count=1
    for((i=1;i<=${num};i++))
    do
        [[ ${1#-} == "config" ]] && { MIGRATE_CONFIG=$2;shift 2; continue; }
        [[ ${1#-} == "taskid" ]] && { TASKID=$2;shift 2; continue; }
        [[ ${1#-} == "start_gauss" ]] && { STARTGAUSS=$2;shift 2; continue; }
    done
}

##################################################
# 校验入参
# @param
#   config     --product1*src_version1*des_version1*migrate_config1@product2*src_version2*des_version2*migrate_config2
##################################################
function check_input()
{
    init_params "$@"
    if [ $? -ne 0 ]
    then
        echo "Example: bash ${SELF_FILE} -config NCE-IP*R19C00CP6025*R20C10SPC300*NCE-Analysis:SPC602@NCE-Super*R19C00CP6025*R20C10SPC300*NCE-aPaaS:CP6025 -taskid c6bfc6d0a769c2b319f9"
        DOUBLE_LOG "ERROR" "The input param is invalid."
        return 1
    fi
}

##################################################
# 刷新Easysuite任务标志文件
# @param
#   $1:进度 0-100
#   $2:状态 fail/running/finish
##################################################
function fresh_result()
{
    echo "Progress=$1" >"${TASK_PROGRESS_FILE}" || DOUBLE_LOG "ERROR" "Failed to echo 'Progress=$1' to ${TASK_PROGRESS_FILE}"
    echo "Status=$2" >"${TASK_STATUS_FILE}" || DOUBLE_LOG "ERROR" "Failed to echo 'Status=$2' to ${TASK_STATUS_FILE}"
}

##################################################
# 刷新Easysuite任务标志文件为“失败”状态
##################################################
function fresh_fail()
{
    fresh_result "100" "fail"
    DOUBLE_LOG "ERROR" "Failed to excute task:${TASKID}."
}

##################################################
# 刷新Easysuite任务标志文件为“完成”状态
##################################################
function fresh_finish()
{
    fresh_result "100" "finish"
    DOUBLE_LOG "INFO" "Finished to excute task:${TASKID}."
}

##################################################
# 刷新Easysuite任务进度文件
# @param
#   $1:进度增加值，达到100后不再增加
##################################################
function fresh_progress()
{
    local addnum=$1
    local base=$(expr 100 - ${addnum})
    if [ ${PROGRESS} -le ${base} ]
    then
        PROGRESS=$(expr ${PROGRESS} + ${addnum})
        echo "Progress=${PROGRESS}">"${TASK_PROGRESS_FILE}"
    fi
}

##################################################
# 检查迁移工具是否存在
##################################################
function pre_check()
{
    DOUBLE_LOG "INFO" "Start to check necessary files."
    if [ ! -d "${MIGRATETOOL}" ]
    then
        DOUBLE_LOG "ERROR" "The MigrateTool does not exist."
        return 1
    fi
    if [ ! -d "${GAUSSDRIVER}" ]
    then
        DOUBLE_LOG "WARN" "The GaussDB driver does not exist."
    fi
    if [ ! -d "${ZENITHDRIVER}" ]
    then
        DOUBLE_LOG "WARN" "The ZenithDB driver does not exist."
    fi
    DOUBLE_LOG "INFO" "Finished to check necessary files."
}

##################################################
# 查询产品信息
##################################################
function query_product_info()
{
    local product="$1"
    DOUBLE_LOG "INFO" "Start to query ${product} infomation."
    TMP_INFO="${CONFIG_DIR}/${product}_infomation_${TAIL}"
    [ -d "${TMP_INFO}" ] && rm -rf "${TMP_INFO}"
    mkdir -p "${TMP_INFO}"
    bash "${OSS_ROOT}"/tools/resmgr/queryproduct.sh -pn "${product}" -output "${TMP_INFO}" >/dev/null 2>&1
    if [ $? -eq 0 ]
    then
        DOUBLE_LOG "INFO" "Finished to query ${product} infomation."
        fresh_progress 1
    else
        DOUBLE_LOG "ERROR" "Failed to query ${product} infomation."
        return 1
    fi
}

##################################################
# 检查上次任务是否完成，完成直接退出
##################################################
function check_finish()
{
    if [ "${NEW_TASK}" = "FALSE" ]
    then
        grep -q "Finished to excute task:${TASKID}." "${TASK_LOG_FILE}"
        if [ $? -eq 0 ]
        then
            fresh_finish
            return 0
        fi
    fi
    return 1
}

##################################################
# 查询产品高斯进程是否启动
# 未启动需要启动进程
##################################################
function start_product_gauss()
{
    local product="$1"
    DOUBLE_LOG "INFO" "Start to check GaussDB status(${product})."
    local gauss_dbips=($(${OSS_ROOT}/apps/DBAgent/bin/dbsvc_adm -cmd query-db-instance -tenant ${product} | grep gauss | awk '{print $6}' | sort -u))
    if [ -n "${gauss_dbips}" ]
    then
        local query_info="${CONFIG_DIR}/query_gaussinfo_${TAIL}.info"
        "${OSS_ROOT}"/apps/DBAgent/bin/${DBM_CMD} -cmd query-db-instance -tenant "${product}" | grep gauss | awk '{print $3 "\t" $6}' > ${query_info}
        while read line
        do
            LOG "INFO" "Variable(line):${line}"
            local instname=$(echo ${line} | awk '{print $1}')
            local instip=$(echo ${line} | awk '{print $2}')
            bash "${SSHTOOL}" -exectype cmd -ip "${instip}" -cmd "ps -ef | grep /opt/gauss | grep -v grep | grep -w ${instname}" -u ${OSS_USER} -timeout 3600 -encrypt N >>"${LOG_FILE}" 2>&1
            if [ $? -ne 0 ]
            then
                bash "${SSHTOOL}" -exectype cmd -ip "${instip}" -cmd "sudo -u dbuser bash -c \"cd ~; source appgsdb.bashrc&&/opt/gauss/app/bin/gs_ctl start -D /opt/gauss/data/${instname}\"" -u ${OSS_USER} -timeout 3600 -encrypt N >>"${LOG_FILE}" 2>&1
                if [ $? -ne 0 ]
                then
                    rm -f ${query_info}
                    DOUBLE_LOG "ERROR" "Failed to start ${instname} on ${instip}."
                    return 1
                fi
            fi
        done <${query_info}
        rm -f ${query_info}
        fresh_progress 5
        DOUBLE_LOG "INFO" "Finished to check GaussDB status."
    else
        DOUBLE_LOG "WARN" "Product ${product} does not use the GaussDB."
    fi
}

##################################################
# 根据containlist倒换zenith实例
##################################################
function switch_zenith()
{
    ${OSS_ROOT}/apps/DBAgent/bin/${DBM_CMD} -cmd query-db-instance | grep zenith | grep -qi 'Slave'
    if [ $? -ne 0 ]
    then
        DOUBLE_LOG "INFO" "No switchover is required."
        return 0
    fi

    DOUBLE_LOG "INFO" "Start to switch zenith DB."
    info_file="${CONFIG_DIR}/zenith_master_${TAIL}.info"

    python >${info_file} << PEND
import json
file='/opt/upgrade/backup/migrate/new/containerlist.json'
with open(file, 'r') as f:
    contain=json.load(f)
for item in contain.keys():
    if contain[item]['containerType'] == 'zenith':
        if contain[item].get('dbList'):
            str='{} {}'.format(contain[item].get('dbInstanceID'),item)
            print(str)
PEND
    echo "Master info:" >> ${LOG_FILE}
    cat ${info_file} >> ${LOG_FILE}

    local query_info="${CONFIG_DIR}/query_dbinfo_${TAIL}.info"
    ${OSS_ROOT}/apps/DBAgent/bin/${DBM_CMD} -cmd query-db-instance | grep zenith | grep -w Master | grep -v managedbsvr | awk '{print $1 "\t" $3}' > ${query_info}
    while read line
    do
        LOG "INFO" "Variable(line):${line}"
        local db_instance=$(echo ${line} | awk '{print $1}')
        local master=$(echo ${line} | awk '{print $2}')
        # 根据containlist倒换
        grep ${db_instance} ${info_file} | grep -qw ${master}
        if [ $? -ne 0 ]
        then
            local real_master=$(grep ${db_instance} ${info_file} | awk '{print $2}')
            [ -z "${real_master}" ] && { DOUBLE_LOG "WARN" "real master is empty.Skip switch ${db_instance}"; continue ; }
            DOUBLE_LOG "INFO" "Switch ${db_instance} master to ${real_master}."
            ${OSS_ROOT}/apps/CloudbMgmtService/bin/switchtool.sh -cmd switchover -instid ${db_instance} -newmaster ${real_master} &>>${LOG_FILE}
            if [ $? -ne 0 ]
            then
                rm -f ${query_info}
                DOUBLE_LOG "ERROR" "Failed to switch zenith DB."
                return 1
            fi
        fi
    done <${query_info}
    DOUBLE_LOG "INFO" "Finished to switch zenith DB."
    
    DOUBLE_LOG "INFO" "Start to check Zenith DB."
    # 检查主备状态，300s超时
    local count=0
    while true
    do
        local flag=0
        ${OSS_ROOT}/apps/DBAgent/bin/${DBM_CMD} -cmd query-db-instance | grep zenith | grep -v managedbsvr | awk '{print $3 "\t" $11}' > ${query_info}
        while read line
        do
            local db=$(echo ${line} | awk '{print $1}')
            local status=$(echo ${line} | awk '{print $2}')
            grep -qi ${db} ${info_file}
            local res=$?
            if [ ${res} -eq 0 ]
            then
                if [ "${status}" == "Master" ]
                then
                    continue
                else
                    flag=1
                    break
                fi
            fi
        done <${query_info}
        if [ ${flag} -eq 0 ]
        then
            rm -f ${info_file} ${query_info}
            DOUBLE_LOG "INFO" "Finished to check Zenith DB."
            return 0
        fi
        let count++
        if [ ${count} -eq 30 ]
        then
            rm -f ${info_file} ${query_info}
            DOUBLE_LOG "ERROR" "Failed to switch zenith db."
            return 1
        fi
        sleep 10
    done
}

##################################################
# 根据产品导出环境数据库配置连接文件
##################################################
function export_new_containerlist()
{
    local product="$1"
    DOUBLE_LOG "INFO" "Start to export ${product} DB config."

    local backup_path="/opt/upgrade/backup/migrate/new"
    mkdir -p "${backup_path}" &>>"${LOG_FILE}"
    if [ $? -ne 0 ]
    then
        DOUBLE_LOG "ERROR" "Failed to mkdir ${backup_path}."
        return 1
    fi

    python >>"${LOG_FILE}" 2>&1 << PEND
import easyhttputil
exportParam = {}
exportParam['tenant'] = '${product}'
backup_file='/opt/upgrade/backup/migrate/new/containerlist.json'
http_func = easyhttputil.httppostWithRetry if hasattr(easyhttputil, 'httppostWithRetry') else easyhttputil.http_post_with_retry
response = http_func('/rest/plat/deploy-proxy/v1/containerlist/action?action-id=export-containerlist',exportParam, retry = 3, sleep = 5)
f = open(backup_file,'wb')
f.write(response)
f.close()
PEND
    if [ $? -eq 0 ]
    then
        DOUBLE_LOG "INFO" "Finished to export ${product} DB config."
        fresh_progress 1
    else
        DOUBLE_LOG "ERROR" "Failed to export ${product} DB config."
        return 1
    fi
    chmod 600 /opt/upgrade/backup/migrate/new/containerlist.json
    rm -f /opt/upgrade/backup/migrate/new/containerlist_new.json
}

##################################################
# 导出全量环境数据库配置连接文件
##################################################
function export_all_containerlist()
{
    local product="$1"
    DOUBLE_LOG "INFO" "Start to export ${product} DB config."

    local backup_path="/opt/upgrade/backup/migrate/new"
    mkdir -p "${backup_path}" &>>"${LOG_FILE}"
    if [ $? -ne 0 ]
    then
        DOUBLE_LOG "ERROR" "Failed to mkdir ${backup_path}."
        return 1
    fi

    python >>"${LOG_FILE}" 2>&1 << PEND
import easyhttputil
exportParam = {}
backup_file='/opt/upgrade/backup/migrate/new/containerlist.json'
http_func = easyhttputil.httppostWithRetry if hasattr(easyhttputil, 'httppostWithRetry') else easyhttputil.http_post_with_retry
response = http_func('/rest/plat/deploy-proxy/v1/containerlist/action?action-id=export-containerlist',exportParam, retry = 3, sleep = 5)
f = open(backup_file,'wb')
f.write(response)
f.close()
PEND
    if [ $? -eq 0 ]
    then
        DOUBLE_LOG "INFO" "Finished to export ${product} DB config."
        fresh_progress 1
    else
        DOUBLE_LOG "ERROR" "Failed to export ${product} DB config."
        return 1
    fi
}

##################################################
# 停止对应产品服务
##################################################
function stop_app()
{
    local product="$1"
    DOUBLE_LOG "INFO" "Start to stop ${product} application."
    local nodeip=$(cat "${TMP_INFO}"/nodes_${product}.json 2>/dev/null | python -c "import json; import sys; obj=json.load(sys.stdin); print(obj['hostlist'][0]['nodemgrip'])" 2>/dev/null)
    rm -rf "${TMP_INFO}"
    LOG "INFO" "Variable(nodeip):${nodeip}"
    if [ -z "${nodeip}" ]
    then
        DOUBLE_LOG "ERROR" "Failed to query product IP."
        return 1
    fi
    local start_time=$(date "+%s")
    local time_out=$(expr ${start_time} + 1800)
    local cmd_result="true"
    for i in {1..2}
    do
        local current_time=$(date "+%s")
        if [ ${current_time} -ge ${time_out} ]
        then
            # 如果超过30分钟,不再重试
            break
        fi
        STOP_APP_CMD="ssh -o ConnectTimeout=3600 -o stricthostkeychecking=no -o ConnectionAttempts=3 -o ServerAliveInterval=10 ${nodeip} bash ${OSS_ROOT}/agent/bin/ipmc_adm -cmd stopnodes -tenant ${product} -type app"
        LOG "INFO" "Start to send cmd:${STOP_APP_CMD}"
        ${STOP_APP_CMD}>>"${LOG_FILE}" 2>&1
        result="$?"
        LOG "INFO" "CMD:${STOP_APP_CMD},Result:${result}"
        if [ ${result} -eq 0 ]
        then
            cmd_result="true"
            break
        else
            cmd_result="false"
        fi
    done
    if [ ${cmd_result} == "true" ]
    then
        DOUBLE_LOG "INFO" "Finished to stop ${product} application."
        fresh_progress 10
    else
        DOUBLE_LOG "ERROR" "Failed to stop ${product} application."
        return 1
    fi
    return 0
}

##################################################
# 为DataMigrateTool准备高斯证书文件
##################################################
function gauss_trust()
{
    local product=$1
    local targetPath=$2
    local gauss_dbips=($(${OSS_ROOT}/apps/DBAgent/bin/dbsvc_adm -cmd query-db-instance -tenant ${product} | grep gauss | awk '{print $6}' | sort -u))
    if [ -n "${gauss_dbips}" ]
    then
        local dbip="${gauss_dbips[0]}"
        bash "${SSHTOOL}" -exectype cmd -ip "${dbip}" -cmd "sudo -u dbuser cat /opt/gauss/ssl/trust.cer 2>/dev/null" -u ${OSS_USER} -timeout 3600 -encrypt N > "${targetPath}"/trust.cer 2>/dev/null
        if [ $? -ne 0 ]
        then
            DOUBLE_LOG "ERROR" "Failed to obtain the GaussDB Certificate from ${dbip}."
            return 1
        fi
    fi
}

##################################################
# 初始化迁移工具所需相关文件
# 每个迁移任务工作区独立
##################################################
function init_migrate_file()
{
    local domain=$1
    local targetPath=$2
    local product=$3
    DOUBLE_LOG "INFO" "Start to init ${domain} migrate files."
    tool=$(ls ${MIGRATETOOL}/*.zip 2>/dev/null | sort -V | sed -n '$p')
    gauss=$(ls ${GAUSSDRIVER}/*.zip 2>/dev/null | sort -V | sed -n '$p')
    zenith=$(ls ${ZENITHDRIVER}/*.zip 2>/dev/null | sort -V | sed -n '$p')

    # 准备MigrateTool工具
    if [ -f "${tool}" ]
    then
        unzip -o "${tool}" -d "${targetPath}" >/dev/null 2>&1
        if [ $? -ne 0 ]
        then
            DOUBLE_LOG "ERROR" "Failed to unzip MigrateTool."
            return 1
        fi
        fresh_progress 1
    else
        DOUBLE_LOG "ERROR" "The MigrateTool does not exist."
        return 1
    fi
    
    mkdir -p "${targetPath}"/lib &>>"${LOG_FILE}"
    if [ $? -ne 0 ]
    then
        DOUBLE_LOG "ERROR" "Failed to mkdir ${targetPath}."
        return 1
    fi
    fresh_progress 1

    # 源环境非R19升级到R21C10，数据库数据迁移前不需要准备高斯证书
    if [ "${STARTGAUSS}" == "true" ]
    then
        DOUBLE_LOG "INFO" "The source environment is R19."
        # 准备高斯证书
        gauss_trust "${product}" "${targetPath}"
        [ $? -ne 0 ] && return 1
        fresh_progress 1

        # 准备gauss驱动
        if [ -f "${gauss}" ]
        then
            tmp_gauss="${CONFIG_DIR}/easysuite_gaussdriver_${TAIL}"
            [ -d "${tmp_gauss}" ] && rm -rf "${tmp_gauss}"
            unzip -o "${gauss}" -d "${tmp_gauss}" >/dev/null 2>&1
            if [ $? -ne 0 ]
            then
                DOUBLE_LOG "ERROR" "Failed to unzip GaussDriver."
                return 1
            fi
            cp -f "${tmp_gauss}"/lib/* "${targetPath}"/lib >/dev/null 2>&1
            if [ $? -ne 0 ]
            then
                DOUBLE_LOG "ERROR" "Failed to copy GaussDriver to ${targetPath}."
                return 1
            fi
            [ -d "${tmp_gauss}" ] && rm -rf "${tmp_gauss}"
            fresh_progress 1
        fi
    else
        DOUBLE_LOG "INFO" "The source environment is not R19. No need to prepare Gauss Certificate."
    fi
    
    # 准备zenith驱动
    if [ -f "${zenith}" ]
    then
        tmp_zenith="${CONFIG_DIR}/easysuite_zenithdriver_${TAIL}"
        [ -d "${tmp_zenith}" ] && rm -rf "${tmp_zenith}"
        unzip -o "${zenith}" -d "${tmp_zenith}" >/dev/null 2>&1
        if [ $? -ne 0 ]
        then
            DOUBLE_LOG "ERROR" "Failed to unzip ZenithDriver."
            return 1
        fi
        cp -f "${tmp_zenith}"/lib/* "${targetPath}"/lib >/dev/null 2>&1
        if [ $? -ne 0 ]
        then
            DOUBLE_LOG "ERROR" "Failed to copy ZenithDriver to ${targetPath}."
            return 1
        fi
        [ -d "${tmp_zenith}" ] && rm -rf "${tmp_zenith}"
        fresh_progress 1
    fi

    # 准备数据库连接修改接口文件
    if [ -f "${PYTHONC_TOOL}" ]
    then
        cp -f "${PYTHONC_TOOL}" "${targetPath}" &>>"${LOG_FILE}"
        if [ $? -ne 0 ]
        then
            DOUBLE_LOG "ERROR" "Failed to copy PYTHONC_TOOL to ${targetPath}."
            return 1
        fi
    elif [ -f "${PYTHON_TOOL}" ]
    then
        cp -f "${PYTHON_TOOL}" "${targetPath}" &>>"${LOG_FILE}"
        if [ $? -ne 0 ]
        then
            DOUBLE_LOG "ERROR" "Failed to copy PYTHON_TOOL to ${targetPath}."
            return 1
        fi
    else
        DOUBLE_LOG "ERROR" "The ModifyTool does not exist."
        return 1
    fi
    fresh_progress 1

    # 刷脚本格式为unix
    if [ -d "${targetPath}" ]
    then
        find "${targetPath}" -name "*.sh" -exec dos2unix {} \; &>>"${LOG_FILE}"
        if [ $? -ne 0 ]
        then
            DOUBLE_LOG "ERROR" "Failed to dos2unix ${targetPath} files"
            return 1
        fi
    fi
    fresh_progress 1

    DOUBLE_LOG "INFO" "Finished to init ${domain} migrate files."
}

##################################################
# 调用子域脚本，记录其标准输出及错误输出
# @param
#   $1:脚本路径
#   $2:脚本名称
##################################################
function exc_one_shell()
{
    local exc_file_path="${1}"
    local exc_file_name="${2}"
    local domain="${3}"
    local work_path="${4}"
    LOG "INFO" "Variable(exc_file_path):${exc_file_path};(exc_file_name):${exc_file_name};(work_path):${work_path}"
    if [ -n "${exc_file_name}" -a -f "${exc_file_path}"/"${exc_file_name}" ]
    then
        DOUBLE_LOG "INFO" "Start to excute ${exc_file_name}[solution due to ${domain}]."
        bash "${exc_file_path}"/"${exc_file_name}" -input "${work_path}" >> "${LOG_PATH}/${exc_file_name//.sh/}_${TAIL}.log" 2>&1
        result=$?
        if [ ${result} -eq 0 ]
        then
            DOUBLE_LOG "INFO" "Finished to excute ${exc_file_name}[solution due to ${domain}]."
            fresh_progress 3
        else
            DOUBLE_LOG "ERROR" "Failed to excute ${exc_file_name}[solution due to ${domain}]. Return Code:${result}"
            return 1
        fi
    fi
}

##################################################
# 调用某个目录下子域数据迁移脚本
# @param
#   $1:目录路径
##################################################
function exc_domain_shell()
{
    local exc_dir=$1
    local domain=$2
    local work_path=$3
    cd "${exc_dir}" &>>"${LOG_FILE}"
    if [ $? -ne 0 ]
    then
        DOUBLE_LOG "ERROR" "Failed to cd ${exc_dir}."
        return 1
    fi
    local prefile=$(ls *.sh 2>/dev/null | grep "^pre_migrate" | sed -n 1p 2>/dev/null)
    local excfile=$(ls *.sh 2>/dev/null | grep "^exc_migrate" | sed -n 1p 2>/dev/null)
    local postfile=$(ls *.sh 2>/dev/null | grep "^post_migrate" | sed -n 1p 2>/dev/null)
    
    # 执行前置脚本
    exc_one_shell "${exc_dir}" "${prefile}" "${domain}" "${work_path}" || return 1
    
    # 执行迁移脚本
    exc_one_shell "${exc_dir}" "${excfile}" "${domain}" "${work_path}" || return 1
    
    # 执行后置脚本
    exc_one_shell "${exc_dir}" "${postfile}" "${domain}" "${work_path}" || return 1
}

##################################################
# 迁移完成的情况下，停止gauss进程
# 停止失败不报错，仅记录日志
##################################################
function stop_product_gauss()
{
    local product="$1"
    local gauss_dbips=($(${OSS_ROOT}/apps/DBAgent/bin/${DBM_CMD} -cmd query-db-instance -tenant ${product} | grep gauss | awk '{print $6}' | sort -u))
    if [ -n "${gauss_dbips}" ]
    then
        DOUBLE_LOG "INFO" "Start trying to stop GaussDB."
        "${OSS_ROOT}"/apps/DBAgent/bin/${DBM_CMD} -cmd query-db-instance -tenant "${product}" | grep gauss | awk '{print $3 "\t" $6}' | while read line
        do
            LOG "INFO" "Variable(line):${line}"
            local instname=$(echo ${line} | awk '{print $1}')
            local instip=$(echo ${line} | awk '{print $2}')
            bash "${SSHTOOL}" -exectype cmd -ip "${instip}" -cmd "ps -ef | grep /opt/gauss | grep -v grep | grep -w ${instname}" -u ${OSS_USER} -timeout 3600 -encrypt N >>"${LOG_FILE}" 2>&1
            if [ $? -eq 0 ]
            then
                bash "${SSHTOOL}" -exectype cmd -ip "${instip}" -cmd "sudo -u dbuser bash -c \"cd ~; source appgsdb.bashrc&&/opt/gauss/app/bin/gs_ctl stop -D /opt/gauss/data/${instname}\"" -u ${OSS_USER} -timeout 3600 -encrypt N >>"${LOG_FILE}" 2>&1
                if [ $? -ne 0 ]
                then
                    DOUBLE_LOG "WARN" "Failed to stop ${instname} on ${instip}.Please manually stop the ${instname}."
                fi
            fi
        done
        DOUBLE_LOG "INFO" "Finish trying to stop GaussDB."
    fi
}

##################################################
# 正常退出前备份子域日志
##################################################
function backup_log()
{
    local zip_name="migrate_log_${TAIL}.zip"
    local zip_path="${LOG_PATH}/${zip_name}"
    local tmp_zip_path="${CONFIG_DIR}/migrate_log_${TAIL}"
    mkdir "${tmp_zip_path}" 2>/dev/null
    find "${MIGRATEPATH}" -type f -name '*.log' | xargs cp --parents -t "${tmp_zip_path}"
    cd "${tmp_zip_path}"/"${MIGRATEPATH}"
    zip -r -q "${zip_path}" ./
    chmod 440 "${zip_path}"
    rm -rf "${tmp_zip_path}"
}

##################################################
# 复制旧containerlist文件
##################################################
function copy_old_containerlist() {
    local product="$1"
    old_migrate_path="/opt/upgrade/backup/migrate/old"
    for _ in $(seq 3)
    do
      cp -fp ${old_migrate_path}/containerlist_${product}.json ${old_migrate_path}/containerlist.json &>>${LOG_FILE}
      res=$?
      [ ${res} -ne 0 ] && { sleep 1; continue ; }
      break
    done
    if [ ${res} -ne 0 ]
    then
        DOUBLE_LOG "ERROR" "Failed to copy containerlist_${product}.json to ${old_migrate_path}."
        return 1
    fi
}

##################################################
# 数据迁移前的必要准备
##################################################
function pre_migrate_one_product()
{
    local product="$1"
    local work_path="$2"

    # 查询产品信息
    query_product_info "${product}" || return 1

    # 查询产品高斯进程是否启动,未启动需要启动进程(默认启动, 支持定制)
    if [ "${STARTGAUSS}" == "true" ]
    then
        start_product_gauss "${product}" || return 1
        if [ $? -ne 0 ]
        then
            fresh_fail
            return 1
        fi
    fi

	  # 导出全部数据库连接文件
    export_all_containerlist

    # 数据库倒换
    switch_zenith || return 1

	  # 生成单产品的数据库迁移配置文件
    export_new_containerlist "${product}" || return 1

    # 将旧连接配置文件复制到/opt/upgrade/backup/migrate/old
    copy_old_containerlist "${product}" || return 1

    # 非R19C00版本不停止产品服务
    echo "${SRC_VERSION}" | grep -q "^V100R019C00" || return 0

    # 停止产品服务
    stop_app
    if [ $? -ne 0 ]
    then
        fresh_fail
        return 1
    fi
}

##################################################
# 根据Easysuite解析的迁移参数，准备迁移工具
# 然后调用子域迁移脚本
##################################################
function exec_migrate_one_product()
{
    local product="$1"
    local work_path="$2"
    local config="$3"
    LOG "INFO" "Variable(config):${config}"
    
    # 启动迁移任务
    for paths in ${config//,/ }
    do
        local domain=$(echo "${paths}" | awk -F: '{print $1}')
        local exc_config=$(echo "${paths}" | awk -F: '{print $2}')
        local exc_dir="${MIGRATEPATH}/${domain}/${exc_config}"
        LOG "INFO" "Variable(domain):${domain};(exc_config):${exc_config};(exc_dir):${exc_dir}"
        
        # 初始化迁移工具所需相关文件
        init_migrate_file "${domain}" "${exc_dir}" "${product}"
        if [ $? -eq 0 ]
        then
            # 调用目录下子域数据迁移脚本
            exc_domain_shell "${exc_dir}" "${domain}" "${work_path}" || return 1
        else
            DOUBLE_LOG "ERROR" "Failed to init ${domain} migrate files."
            return 1
        fi
    done

}

##################################################
# 迁移后置处理
##################################################
function post_migrate_one_product()
{
    local product="$1"
    # 停止当前产品产品高斯数据库
    stop_product_gauss "${product}"
}

##################################################
# 执行单个产品迁移任务
##################################################
function migrate_one_product()
{
    local product="$1"
    local src="$2"
    local des="$3"
    local configs="$4"
    # 拼接入参目录
    local work_path="/opt/upgrade/easysuite_upgrade/workpath/${src}-${des}/workpath-${product}"
    DOUBLE_LOG "INFO" "Start the data migration of the ${product} product."

    # 数据迁移前置准备
    pre_migrate_one_product "${product}" "${work_path}" || return 1

    # 进行数据迁移
    exec_migrate_one_product "${product}" "${work_path}" "${configs}" || return 1

    # 迁移成功后置操作
    post_migrate_one_product "${product}" || return 1

    # 记录成功记录
    DOUBLE_LOG "INFO" "${product} migration succeeded."
}

##################################################
# 检查单个产品迁移执行情况，成功跳过
##################################################
function check_one_product_result()
{
    local product="$1"
    if [ "${NEW_TASK}" = "FALSE" ]
    then
        grep -q "${product} migration succeeded." "${TASK_LOG_FILE}" && return 0
    fi
    return 1
}

##################################################
# 执行迁移任务
##################################################
function migrate_database()
{
    for one_product in ${MIGRATE_CONFIG//@/ }
    do
        local product=$(echo ${one_product} | awk -F '*' '{print $1}')
        check_one_product_result "${product}" && continue
        local src=$(echo ${one_product} | awk -F '*' '{print $2}')
        local des=$(echo ${one_product} | awk -F '*' '{print $3}')
        local configs=$(echo ${one_product} | awk -F '*' '{print $4}')
        migrate_one_product "${product}" "${src}" "${des}" "${configs}"
        if [ $? -ne 0 ]
        then
            fresh_fail
            return 1
        fi
    done

    # 导出全部数据库连接文件
    export_all_containerlist

    # 备份迁移工具日志文件
    backup_log

    # 任务完成退出，刷新标志文件
    fresh_finish
}

##################################################
# 初始化相关变量，并进行前置检查
##################################################
function init()
{
    # 校验调用用户
    check_user || return 1

    # 加载公共方法
    . "${SCRIPT_PATH}"/common.sh
    LOG "INFO" "Start to init ${LOG_FILE}."
    
    # 校验入参，初始化参数
    check_input "$@" || return 1

    # 导入平台环境变量
    . /opt/oss/manager/bin/engr_profile.sh
    
    # 定义任务相关全局变量
    init_variable

    echo "${MIGRATE_CONFIG}; ${TASKID}; ${STARTGAUSS}" >>${TASK_LOG_FILE}
    
    # 初始化任务文件夹
    init_taskmgr "${TASKID}"

    # 检查数据库迁移工具是否存在
    pre_check
    if [ $? -ne 0 ]
    then
        fresh_fail
        return 1
    fi
}

##################################################
# 入口函数
##################################################
function main()
{
    # 初始化及前置检查
    init "$@" || return 1

    # 重入结果检查
    check_finish && return 0
    
    # 启动数据库数据迁移
    migrate_database
    return $?
}

main "$@"
exit $?