# encoding=utf-8
"""
功 能：记录日志
版权信息：华为技术有限公司，版本所有(C) 2019-2029
修改记录：2019-12-11 12:00 创建
"""
import os
import sys
import json
import time
import threading

from commonlog import Logger
from uniep_taskmgr import Unieptaskmgr
from taskmgr_util import Taskmgrutil

RECORD_LOGGER = Logger().getinstance(sys.argv[0])


class OperateApp:
    """
    启停服务
    """
    def __init__(self, e_task_path):
        self.task_path = e_task_path
        self.start_time = int(time.time())
        self.time_out = 3600
        self.func_result = False

    def set_task(self, status='', progress='', msg=''):
        """
        功能说明：设置任务状态和信息
        :param status:
        :param progress:
        :param msg:
        :return:
        """
        if status:
            Taskmgrutil.set_e_taskstatus(self.task_path, status)
        if progress:
            Taskmgrutil.set_e_taskprogress(self.task_path, progress)
        if msg:
            msg = f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] [{os.getpid()}] {msg}"
            Taskmgrutil.set_msg_append(self.task_path, msg)

    @staticmethod
    def get_action_id(action):
        """
        功能说明：获取action-id
        :param action:
        :return:
        """
        action_id = "STOP"
        if action == 'start':
            action_id = 'START'
        return action_id

    @staticmethod
    def get_expect_status(action):
        """
        功能说明：获取预期状态
        :param action:
        :return:
        """
        # start -> RUNNING, stop -> STOPPED
        return 'RUNNING' if action == 'start' else 'STOPPED'

    @staticmethod
    def get_response(response):
        """
        功能说明：请求响应
        :return:
        """
        if not response:
            return False
        try:
            json.loads(response)
        except Exception:
            RECORD_LOGGER.error(f"Failed to load json from \"{response}\"")
            self.set_task(msg=f'The task response is \"{response}\"')
            return False
        return True

    def record_exit_response(self, status, response):
        """
        功能说明:记录最后一次退出的响应详情打印到终端
        :param status:
        :param response:
        :return:
        """
        if not (status and response and self.get_response(response) and json.loads(response)):
            return
        # 查询请求成功并且响应包含具体不符合预期服务信息，退出时打印一次
        self.set_task(msg=f'The query task details:')
        for one_node in json.loads(response):
            node_ip = one_node.get('manageIp')
            service_name = one_node.get('svrName', '')
            process_name = one_node.get('instanceName', '')
            node_service_status = one_node.get('status')
            self.set_task(msg=f'The node({node_ip}) service({service_name},'
                              f'process({process_name})) '
                              f'status is {node_service_status}')
        return

    def create_task(self, product_name, action, time_out=1200, retry_count=2):
        """
        功能说明:创建任务
        :param product_name:
        :param action:
        :param time_out:
        :param retry_count:
        :return:
        """
        self.set_task(msg=f'Start to create {action} service task')
        action_id = OperateApp.get_action_id(action)
        create_task_url = f"/rest/plat/omp/v1/main/monitor/products/{product_name}" \
                          f"/action?action-id={action_id}"
        max_time = self.start_time + time_out
        result = False
        response = '{}'
        while True:
            # 超时打印超时报错退出
            if int(time.time()) >= max_time:
                RECORD_LOGGER.info("Time_out")
                self.set_task(msg=f'Failed to create the task for {action}ing the product service.'
                                  f'The task execution timeout exceeds {time_out} seconds')
                break

            # 等待一秒
            time.sleep(1)

            # 失败已有重试
            status, response = Unieptaskmgr().send_post_request(create_task_url, {'type': 'app'})
            RECORD_LOGGER.info(f"create_task:status {status} response \"{response}\"")

            # 非200 -> 207,并且返回无数据打印报错退出
            if not status or not response:
                self.set_task(msg=f'Failed to create the task for {action}ing the product service.'
                                  f'Task execution response is \"{response}\"')
                RECORD_LOGGER.info(f"create_task: send request failed")
                break

            if not self.get_response(response):
                # json,解析失败重试
                continue

            # 过程信息打印详情
            print_data = json.loads(response)
            print_data.pop('uri')
            self.set_task(msg=f'The task response is {print_data}')

            # 界面打印信息不打印uri请求地址,后续代码会使用到uri,response重新实例化
            res_data = json.loads(response)
            progress = res_data.get('progress')
            ret_code = res_data.get('retCode')
            # progress == 100 -> retCode == 0
            if progress == 100 and ret_code == "0":
                result = True
                RECORD_LOGGER.info(f"create_task: send request success")
                self.set_task(msg=f'Finished to create the task for {action}ing the product '
                                  f'service')
                break

            # progress == 100 -> retCode == 0
            if progress == 100 and ret_code != "0":
                RECORD_LOGGER.info(f"create_task: request return failed")
                self.set_task(msg=f'Failed to create the task for {action}ing the product '
                                  f'service.Task execution response is \"{json.loads(response)}\"')
                break

        if result is False and retry_count > 0:
            RECORD_LOGGER.info(f"Retry to create_task.{retry_count}")
            self.set_task(msg=f"Retry to create_task")
            # 等待3分钟,给系统缓冲,避免cpu高问题
            time.sleep(180)
            retry_count = retry_count - 1
            # 已发布版本重试保证稳定性,历史版本各种服务启动依赖,重试的任务创建超时时间只有5分钟
            return self.create_task(product_name, action, time_out + 480, retry_count=retry_count)

        return result, response

    @staticmethod
    def check_is_new_version(product_name, action):
        """
        功能说明：查询任务状态
        :param product_name:
        :param action:
        :return:
        """
        expect_status = OperateApp.get_expect_status(action)
        check_url = f'/rest/appcontroller/v1/status/unexpected/app?projectName={product_name}' \
                    f'&expectStatus={expect_status}'
        status, response = Unieptaskmgr().send_get_request(check_url)
        RECORD_LOGGER.info(f"check_is_new_version: status {status} response \"{response}\"")
        if not status:
            RECORD_LOGGER.info(f"check_is_new_version: False")
            # 接口调用不通,为旧版本
            return False
        RECORD_LOGGER.info(f"check_is_new_version: True")
        # 接口调用通，为新版本
        return True

    def query_by_old_url(self, response, action):
        """
        功能说明：使用旧接口查询服务状态
        :param response:
        :param action:
        :return:
        """
        self.set_task(msg='Start to query the service status')
        expect_status = OperateApp.get_expect_status(action)
        check_url = json.loads(response).get('uri')
        result = False
        max_time = self.start_time + self.time_out
        while True:
            # 超时打印超时报错退出
            if int(time.time()) >= max_time:
                RECORD_LOGGER.info("Time_out")
                self.set_task(msg=f'Failed to query the service status.'
                                  f'The task execution timeout exceeds {self.time_out} seconds')
                break

            # 查询频次放慢点五秒
            time.sleep(5)

            status, response = Unieptaskmgr().send_get_request(check_url)
            RECORD_LOGGER.info(f"query_by_old_url:status {status} response \"{response}\"")

            # 请求失败
            if not status or not response:
                self.set_task(msg=f'The query task response is \"{response}\"')
                continue

            if not self.get_response(response):
                # json,解析失败重试
                continue

            if status and response and json.loads(response).get('progress') == 100:
                # 请求成功,并且有响应信息
                print_data = json.loads(response)
                # 老接口的progress无实际含义非100为缓存数据
                print_data.pop('progress')
                self.set_task(msg=f'The query task response is \"{print_data}\"')

            # 查询成功,预期满足要求退出
            if status and response and json.loads(response).get('progress') == 100 \
                    and json.loads(response).get('status') == expect_status:
                result = True
                RECORD_LOGGER.info(f"query_by_old_url: the service status meets the expectation.")
                self.set_task(msg=f'Finished to query the service status')
                break
        return result

    def query_by_new_url(self, product_name, action):
        """
        功能说明：使用新接口查询服务状态
        :param product_name:
        :param action:
        :param time_out:
        :return:
        """
        self.set_task(msg='Start to query the service status')
        expect_status = OperateApp.get_expect_status(action)
        check_url = f'/rest/appcontroller/v1/status/unexpected/app?projectName={product_name}' \
                    f'&expectStatus={expect_status}'
        result = False
        status = False
        response = '{}'
        max_time = self.start_time + self.time_out
        while True:
            # 超时打印超时报错退出
            if int(time.time()) >= max_time:
                RECORD_LOGGER.info("Time_out")
                self.set_task(msg=f'Failed to query the service status.'
                                  f'The task execution timeout exceeds {self.time_out} seconds')
                break

            # 查询频次放慢点五秒
            time.sleep(5)

            status, response = Unieptaskmgr().send_get_request(check_url)
            RECORD_LOGGER.info(f"query_by_new_url:status {status} response \"{response}\"")

            # 请求失败
            if not status or not response:
                self.set_task(msg=f'The query task response is \"{response}\"')
                continue

            if not self.get_response(response):
                # json,解析失败重试
                continue

            # 查询成功,无异常服务退出
            if not json.loads(response):
                result = True
                RECORD_LOGGER.info(f"query_by_new_url: the service status meets the expectation.")
                self.set_task(msg=f'Finished to query the service status')
                break

            # 过程记录节点状态即可
            self.set_task(msg=f'The query task details:')
            nodes = {one_node.get('manageIp'): one_node.get('status')
                     for one_node in json.loads(response)}
            for node_ip, node_status in nodes.items():
                self.set_task(msg=f'The node({node_ip}) service status is {node_status}')

        # 记录最后一次退出微服务详情
        self.record_exit_response(status, response)
        return result

    def operate_app(self, product_name, action):
        """
        功能描述：启停服务
        :param product_name
        :param action
        返回： response
        修改记录：新增方法
        """
        try:
            self.set_task(msg=f"[ES_START] Start to {action} service")
            status, response = self.create_task(product_name, action)
            # 创建任务失败,报错退出
            if not status:
                RECORD_LOGGER.info(f'{response}')
                self.set_task(msg=f"[ES_FINISH] Failed to {action} service")
                return

            # 检查是否有新版本查询接口
            if not self.check_is_new_version(product_name, action):
                self.func_result = self.query_by_old_url(response, action)
            else:
                self.func_result = self.query_by_new_url(product_name, action)

            # 查询任务执行结果失败,退出
            if not self.func_result:
                self.set_task(msg=f"[ES_FINISH] Failed to {action} service")
                return

            # 查询任务执行结果成功
            self.set_task(msg=f"[ES_FINISH] Finished to {action} service")
            return
        except Exception as e_msg:
            self.set_task(msg=f"[ES_ERROR] Failed to {action} the product service due to "
                              f"exception({e_msg}).")
            return

    def operate_app_task(self, product_name, action):
        """
        功能说明:创建任务并挂起刷新进度
        :param product_name:
        :param action:
        :return:
        """
        # 并发下任务
        child_task = threading.Thread(target=self.operate_app,
                                      args=(product_name, action,))
        child_task.start()
        # 15分钟,常规耗时
        default_cost_time = 900
        # 进程活跃,一直刷新进度
        while child_task.is_alive():
            current_cost_time = time.time() - int(self.start_time)
            # 0 -> 100
            self.set_task(progress=str(min(99, int(current_cost_time / default_cost_time * 100))))
            # 进度刷新,5秒刷新一次
            time.sleep(5)
        return self.func_result


def main(argv):
    """
    main函数
    """
    product_name = argv[1]
    action = argv[2]
    e_task_id = argv[3]
    e_taskmgr_path = "/opt/upgrade/easysuite_upgrade/taskmgr"
    e_task_path = os.path.join(e_taskmgr_path, e_task_id)

    # 初始化任务
    Taskmgrutil().init_e_taskmgr(e_task_path)
    Taskmgrutil.set_e_taskstatus(e_task_path, "RUNNING")
    Taskmgrutil.set_e_taskprogress(e_task_path, "0")

    if action not in ["start", 'stop']:
        OperateApp(e_task_path).set_task(status='error', progress='100',
                                         msg=f"The input parameter(action:{action}) is incorrect.")
        return

    if not OperateApp(e_task_path).operate_app_task(product_name, action):
        OperateApp(e_task_path).set_task(status='error', progress='100')
    else:
        OperateApp(e_task_path).set_task(status='success', progress='100')
    return


if __name__ == '__main__':
    main(sys.argv)

